/*
Copyright (C) 2016, 2017 Siep Kroonenberg

This file is part of TLaunch.

TLaunch is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

TLaunch is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with TLaunch.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "tla_utils.h"
#include "tlaunch.h"

////////////////////////////////////////////////
// parsing ini file and related utilities

/*
While parsing, we temporarily store some variables in the
environment. After parsing, the savedenv structs let us go back to
the original environment if we want to.
*/
typedef struct {
  wchar_t * name;
  wchar_t * old;
  wchar_t * new;
} SavedEnv;
static SavedEnv * savedenv[MAX_ENV];
static short nenvs;

static wchar_t * trim_spaces(wchar_t * s) {
  wchar_t * p, * q;
  p = s;
  while (*p==L' ') p++;
  // now *p is either 0 or a non-space character
  if (*p==L'\0') return NULL;
  q = p;
  while (*(++q)!=L'\0'); // find terminating 0
  while (*(--q)==L' ') *q = L'\0';
  return p;
} // trim_spaces

static void add_temp_env( wchar_t * e, wchar_t * v ) {
  // Note. If the value v is NULL, the environment variable is deleted;
  // In Windows, empty means undefined.
  SavedEnv * senv;
  BOOL did_it;
  if ( nenvs >= MAX_ENV ) tldie( NULL, L"Too many strings" );
  senv = calloc_fatal( 1, sizeof( SavedEnv ));
  savedenv[ nenvs++ ] = senv;
  senv->name = tl_wcscpy( e );
  senv->new = ( v ? tl_wcscpy( v ) : NULL );
  senv->old = get_env( e );
  did_it = SetEnvironmentVariable(e, senv->new);
  if (senv->new && !did_it)
    tldie( NULL, L"Failed to set environment variable %ls", e);
}

static BOOL uses_tlp( wchar_t * command ) {
  wchar_t * prog;
  BOOL uses_t = FALSE;
  prog = prog_from_cmd( command );
  if( same_file( prog, tlperl )) uses_t = TRUE;
  if( same_file( prog, tlwperl )) uses_t = TRUE;
  free( prog );
  return uses_t;
}

void get_tl_release() {
  // read release from <tlroot>/tlpkg/texlive.tlpdb
  wchar_t * dbfilename;
  char * cptr;
  int i, j, l, l2, ok;
  FILE * dbfile;
  char dbline[200];

  // full path of texlive.tlpdb
  dbfilename = tl_wcsnconcat( 2, tlroot, L"/tlpkg/texlive.tlpdb" );
  // wchar_subst( dbfilename, L'/', L'\\' );

  // read start of release file; assume ansi/ascii
  dbfile = _wfopen( dbfilename, L"r" );
  FREE0( dbfilename );
  if ( !dbfile ) return; // version remains NULL

  for ( i=0; i<200; i++ ) {
    memset( dbline, 0, sizeof( dbline ));
    cptr = fgets( dbline, sizeof( dbline ), dbfile );
    if( !cptr ) {
      // i.e. EOF or error
      fclose( dbfile );
      // version remains NULL
      return;
    }
    l = strlen( "depend release/" );
    if( !strncmp( "depend release/", dbline, l)) {
      // remove CR|LF
      cptr = strrchr( dbline, '\n' );
      if( cptr ) *cptr = 0;
      cptr = strrchr( dbline, '\r' );
      if( cptr ) *cptr = 0;
      l2 = strlen( dbline ) - l;
      version = calloc_fatal( l2 + 1, sizeof( wchar_t ));
      // convert characters one by one to wchar_t
      for( j=0; j<l2; j++ ) {
        ok = mbtowc( version + j, &( dbline[l+j] ), 1 );
        if( ok == -1 ) { // give up
          FREE0( version );
          break;
        }
      }
      fclose( dbfile );
      return;
    }
  }
  // if we got here, version should still be NULL
  fclose( dbfile );
  return;
} // get_tl_release

// initializations which depend on the 'tlroot' variable
static void tlroot_inits() {
  wchar_t * pspath, * rt;
  add_temp_env( L"tlroot", tlroot );
  if( !tlbin ) tlbin = tl_wcsnconcat( 2, tlroot, L"\\bin\\win32" );
  tlperl = tl_wcsnconcat( 2, tlroot, L"\\tlpkg\\tlperl\\bin\\perl.exe" );
  tlwperl = tl_wcsnconcat( 2, tlroot, L"\\tlpkg\\tlperl\\bin\\wperl.exe" );
  add_temp_env( L"tlperl", tlperl );
  add_temp_env( L"tlwperl", tlwperl );
  // adjust process search path (not using add_temp_env)
  pspath = get_env( L"path" );
  if ( !pspath ) tldie( NULL, L"Failed to adjust searchpath" );
  rt = get_env( L"tlroot" );
  if ( !rt ) tldie( NULL, L"Failed to adjust searchpath" );
  // new_searchpath global
  tl_wasprintf( &new_searchpath, L"%ls\\bin\\win32;%ls", rt, pspath );
  if( !new_searchpath ) tldie( NULL, L"Failed to create new searchpath" );
  if( !SetEnvironmentVariable( L"PATH", new_searchpath ))
      tldie( NULL, L"Failed to set new searchpath, error %u", GetLastError() );
  // tlp_searchpath also global
  tl_wasprintf( &tlp_searchpath, L"%ls\\tlpkg\\tlperl\\bin;%ls",
      rt, new_searchpath );
  if( !tlp_searchpath ) tldie( NULL,
      L"Failed to create extended new searchpath" );
  // set version, a global variable, from <tlroot>/tlpkg/texlive.tlpdb
  get_tl_release( );
  if( !version || !version[0] )
    tldie( NULL, L"Cannot figure out TeX Live version" );
  add_temp_env( L"version", version );
  free( pspath );
  free( rt );
} // tlroot_inits

// calculate basename from command
// whether or not command refers to a filename
wchar_t * get_basename( wchar_t * command ) {
   wchar_t * prog = NULL, * basename = NULL;
   prog = prog_from_cmd( command ); // only looks at syntax!
   if( prog && ( wcslen( prog ) == wcslen( command ) ||
       ( wcslen( prog ) == ( wcslen( command ) - 2 ) &&
       command[0] == L'"' && command[ wcslen( command ) - 1 ] == L'"' ))) {
   // command is a possibly quoted filename; safe to register as app
   wchar_t * bsl = wcsrchr( prog, L'\\' );
   if( !bsl ) bsl = wcsrchr( prog, L'/' );
   if( bsl ) basename = tl_wcscpy( bsl + 1 );
   // else basename remains NULL
  }
  FREE0 ( prog );
  return basename;
} // get_basename

// We 'slurp' the configuration file into one temporary string.
// During parsing, we chop it up in-place into pieces which will be copied to
// a more permanent location during the parsing process.

void parse_ini ( wchar_t * config ) {
  wchar_t *p_cur, *p_next, *ptr, *section, *section0; // pointers into config
  int j, nlines = 0, maxlines = 10000; // guard against infinite loop
  SubMenu * cur_menu = NULL;

  // If the exe is the launcher itself, tlroot is already known.
  // If forgetter, tlroot must be read from the ini file.
  if( tlroot ) tlroot_inits();

  // allow %java% if we can find it
  { wchar_t * prog = NULL, * cmd = NULL;
    cmd = get_assoc_cmd_ext( L".jar" );
    if( cmd ) {
      prog = prog_from_cmd( cmd );
      FREE0( cmd );
    }
    if( !prog ) prog = find_registered_app( L"javaw.exe" );
    if( !prog ) prog = find_registered_app( L"java.exe" );
    if ( prog ) {
      add_temp_env( L"java", prog );
      free( prog );
    } else {
      add_temp_env( L"java", NULL );
    }
  }
  /*
  Tokenization:
  Mingw wcstok is non-standard and has no 3rd parameter;
  use wcspbrk instead to find the next delimiter

  We create pointers into config
  and replace delimiters with null characters.
  The resulting pointers/substrings are copied
  to various freshly allocated structure members.
  */
  p_next = config;
  section = NULL;
  if( !tlroot ) section = L"Root";
  if (p_next && *p_next) do {
    if (++nlines >= maxlines) tldie( NULL, L"Ini file too long");
    p_cur = p_next; // *p_cur guaranteed non-zero
    p_next = wcspbrk(p_cur, L"\n\r"); // next line delimiter char
    if (p_next) { // line delimiter found
      // zero this line delimiter
      *p_next = (wchar_t)0;
      // skip at least the present character and
      // stop skipping at non-match or at terminating zero
      j=0;
      do {
        p_next++; j++;
        if (j>=10000) tldie( NULL, L"Cannot find next line");
      } while (*p_next && wcschr(L"\n\r", *p_next));

    }
    // now the unchanged p_cur is/points to the current line as a string,
    // and p_next either points to the next line or to the end or is null
    p_cur = trim_spaces(p_cur); // trim_spaces returns NULL if no non-spaces
    if (!p_cur) continue;
    if (*p_cur==L';') continue; // comment line
    if (*p_cur==L'[') { // [section]
      section0 = ++p_cur;
      ptr = wcschr(p_cur, L']');
      if (!ptr) tldie( NULL, L"Section line lacks terminating ]: %ls", section);
      *ptr = (wchar_t)0;
      section0 = trim_spaces(section0);
      if (!section0) tldie( NULL, L"Empty section name" );
      if( section ) FREE0( section );
      section = expand_env_vars( section0 );
      // Check for section name. We do not test for duplicate names.
      if (!wcsncmp(section, L"MN:", 3)) { //menu
        if( !ed_initdone ) initialize_editors();
        if (nmenus >= MAX_MENUS) tldie( NULL, L"Too many menus: %ls", section);
        menus[nmenus] = calloc_fatal( 1, sizeof( SubMenu ));
        cur_menu = menus[nmenus++];
        cur_menu->name = tl_wcscpy( section + 3 ); // skip prefix
        cur_menu->nitems = 0;
        cur_menu->first_item = ncontrols;
      } else if (!_wcsicmp(section, L"Buttons" )) {
        if( !ed_initdone ) initialize_editors();
        nbuttons = 0;
        first_button = ncontrols;
      } else if (!wcsncmp(section, L"FT:", 3)) { // filetype / progid
        Pgid * pg;
        if (nprogs >= MAX_PGIDS)
            tldie( NULL, L"Too many filetypes: %ls", section);
        pg = calloc_fatal( 1, sizeof( Pgid ));
        pgids[nprogs++] = pg;
        pg->progid = tl_wcscpy( section + 3 );
        pg->primary = TRUE;
        pg->path_prefix = FALSE;
        /*
        We do not test for duplicates.
        At worst, we might get multiple Pgid structs for one filetype,
        overwriting each other's information in the registry.
        */
      } else if (!wcsncmp(section, L"SC:", 3)) { // utility script
        Script * sc;
        if (nsc >= MAX_SC) tldie( NULL, L"Too many utility scripts: %ls", section);
        sc = calloc_fatal( 1, sizeof( Script ));
        scripts[nsc++] = sc;
        sc->name = tl_wcscpy( section + 3 );
        sc->splashtext = L"Working..."; // might get overwritten
      } else if( _wcsicmp( section, L"strings" ) &&
           _wcsicmp( section, L"general" ) &&
           _wcsicmp( section, L"root" )) {
          tldie( NULL, L"Illegal section: %ls", section);
      }
      // free( section );
    } else { // should be line key=value
      if (!section) tldie( NULL, L"%ls not in any section", p_cur);
      ptr = wcschr(p_cur, L'=');
      if (ptr) {
        wchar_t * k0, * k = NULL, * v0, * v = NULL;
        *ptr = L'\0';
        k0 = trim_spaces( p_cur );
        v0 = trim_spaces( ptr + 1 );
        // a lone '=' means separator; only allowed for menus and buttons
        // these cases will be handled further down.
        if (( !k0 || !v0 ) && wcsncmp(section, L"MN:", 3) &&
            _wcsicmp(section, L"Buttons") )
          tldie( NULL, L"Key or value missing in '%ls'", p_cur );
          // if one is NULL, both should be
        if(( k0 && !v0 ) || ( !k0 && v0 ))
          tldie( NULL, L"Key or value missing in '%ls'", p_cur );
        if ( k0 && v0 ) {
          // Normal cases: non-empty k[ey] and v[alue], handled per section.
          // Copy and expand k0 to k and v0 to v, both with allocation.
          // Where appropriate, test whether a command is available.
          // If not, null the entry's command member but do not bail out.
          v = expand_env_vars( v0 );
          k = expand_env_vars( k0 );
          if( !k || !v ) tldie( NULL, L"Expanded key or value empty in '%ls'",
              p_cur );
        }
        if ( !_wcsicmp(section, L"Root" )) {
          if( !_wcsicmp( k, L"tlroot" )) {
            tlroot = tl_wcscpy( v );
            tlbin = tl_wcsnconcat( 2, tlroot, L"\\bin\\win32" );
          } else if( !_wcsicmp( k, L"texmfvar" )) {
            texmfvar = tl_wcscpy( v );
          } else if( !_wcsicmp( k, L"texmfconfig" )) {
            texmfconfig = tl_wcscpy( v );
          } else if( !_wcsicmp( k, L"version" )) {
            version = tl_wcscpy( v );
          }
          add_temp_env( k, v );
          FREE0( k );
          FREE0( v );
        } else if (!_wcsicmp(section, L"Strings")) {
          add_temp_env( k, v );
          free( k );
          free( v );
        } else if (!_wcsicmp(section, L"General")) {
          if (!_wcsicmp(k, L"FILETYPES")) {
            if (!_wcsicmp(v, L"none"))
              assoc_mode = 0;
            else if (!_wcsicmp(v, L"new"))
              assoc_mode = 1;
            else if (!_wcsicmp(v, L"overwrite"))
              assoc_mode = 2;
            else
              tldie( NULL, L"Illegal value FILETYPES %ls\nFILETYPES"
                 L" should have value none, new or overwrite", v );
          } else if( !_wcsicmp(k, L"SEARCHPATH" )) {
            if( !wcscmp( v, L"1" ))
                do_searchpath = 1;
            else if( !wcscmp( v, L"0" ))
                do_searchpath = 0;
            else
                tldie( NULL, L"Parameter for SEARCHPATH should be 0 or 1" );
          } else if( !_wcsicmp(k, L"REMEMBER" )) {
            if( !wcscmp( v, L"1" ))
                create_rememberer = 1;
            else if( !wcscmp( v, L"0" ))
                create_rememberer = 0;
            else
                tldie( NULL, L"Parameter for SEARCHPATH should be 0 or 1" );
          } else if( !_wcsicmp(k, L"KEEPTEMPS" )) {
            if( !wcscmp( v, L"1" ))
                keep_capts = 1;
            else if( !wcscmp( v, L"0" ))
                keep_capts = 0;
            else
                tldie( NULL, L"Parameter for KEEPTEMPS should be 0 or 1" );
          } else {
            tldie( NULL, L"Illegal parameter for general section", k );
          }
          free( v );
          free( k );
        } else if (!wcsncmp(section, L"FT:", 3)) { // file type
          Pgid * pg = pgids[ nprogs - 1 ];
          if( !_wcsicmp(k, L"command")) {
            // calculate basename now, even if the program
            // turns out not to exist. Needed for forgetting.
            pg->basename = get_basename( v );
            // normalize_command includes existence check
            pg->command = normalize_command( v );
            free( v );
            // create shell command if not yet defined
            if( pg->command && ! pg->shell_cmd )
              pg->shell_cmd = tl_wcsnconcat( 2, pg->command, L" \"%1\"" );
            // define iconspec if not yet defined
            if( pg->command && ! pg->iconspec ) {
              wchar_t * prog;
              prog = prog_from_cmd( pg->command );
              pg->iconspec = tl_wcsnconcat( 2, prog, L",0" );
              FREE0( prog );
            }
          } else if( !_wcsicmp( k, L"shell_cmd" )) {
            // first remove shell command if it exists
            if( pg->shell_cmd ) free( pg->shell_cmd );
            // normalize_command implies existence check
            pg->shell_cmd = normalize_command( v );
            free( v );
          } else if( !_wcsicmp( k, L"name" )) {
            // first remove name if it exists
            if( pg->name ) free( pg->name );
            pg->name = v;
          } else if( !_wcsicmp( k, L"icon" )) {
            pg->iconspec = v;
          } else if( !_wcsicmp( k, L"extensions" )) {
            pg->extensions = v;
          } else if( !_wcsicmp( k, L"primary" )) {
            if( !wcscmp( v, L"1" ))
                pg->primary = TRUE;
            else if( !wcscmp( v, L"0" ))
                pg->primary = FALSE;
            else
                tldie( NULL, L"Parameter for PRIMARY should be 0 or 1" );
            FREE0( v );
          } else if( !_wcsicmp( k, L"path_prefix" )) {
            if( !wcscmp( v, L"1" ))
                pg->path_prefix = TRUE;
            else if( !wcscmp( v, L"0" ))
                pg->path_prefix = FALSE;
            else
                tldie( NULL, L"Parameter for PATH_PREFIX should be 0 or 1" );
            FREE0( v );
          } else {
            tldie( NULL, L"Illegal entry %ls for %ls", k, section);
          }
          free( k );
        } else if (!wcsncmp(section, L"SC:", 3)) { // utility script
          Script * sc = scripts[ nsc - 1 ];
          if (!_wcsicmp(k, L"splash")) {
            sc->splashtext = v;
          } else if (!_wcsicmp( k, L"command" )) {
            // normalize_command includes existence check
            sc->command = normalize_command( v );
            free( v );
          } else if (!_wcsicmp(k, L"postmessage" )) {
            free( v ); // parameter accepted but ignnored
          } else {
            tldie( NULL, L"Illegal entry %ls for %ls", k, section);
          }
          free( k );
        } else if (!wcsncmp(section, L"MN:", 3) ||
              !wcsicmp( section, L"Buttons")) { // controls
          Control * ctrl_item;
          int is_button;
          is_button = _wcsicmp( section, L"Buttons") ? 0 : 1;
          // '=' in buttons section originally intended for grouping
          // but for now ignored:
          if( !k0 && !v0 && is_button ) continue;
          if ( ncontrols >= MAX_CONTROLS )
              tldie( NULL, L"Too many controls in menu: %ls", section + 3);
          ctrl_item = calloc_fatal( 1, sizeof( Control ));
          controls[ ncontrols++] = ctrl_item;
          if( is_button ) nbuttons += 1;
          else cur_menu->nitems += 1;
          if( !k0 && !v0 ){
            ctrl_item->type = L'N'; // N => None
            continue; // separator
          } else if (!k0 ) {
            tldie( NULL, L"Illegal entry =%ls", v0 );
          } else if (!v0 ) {
            tldie( NULL, L"Illegal entry %ls=", k0 );
          }
          ctrl_item->display = k;
          ctrl_item->enabled = TRUE;
          ctrl_item->action = NULL; // command or 'shell object'
          ctrl_item->pgid = NULL;
          ctrl_item->hw = NULL;
          // ctrl_item->siid = -1; // stock icon index
          ctrl_item->script = NULL;
          ctrl_item->fun = NULL;
          if( !wcsncmp( v, L"FT:", 3 )) {
            int i;
            ctrl_item->type = L'P';
            for( i=0; i< nprogs; i++ ) {
              if( !_wcsicmp( v + 3, pgids[i]->progid )) {
                ctrl_item->pgid = pgids[i];
                if( !( pgids[i]->command )) {
                  ctrl_item->enabled = FALSE;
                } else {
                  ctrl_item->uses_tlperl = uses_tlp( pgids[i]->command );
                }
                free( v );
                break;
              }
            }
            if( !ctrl_item->pgid ) tldie( NULL,
                L"Menu entry %ls refers to undefined filetype %ls", k, v+3 );
          } else if( !wcsncmp( v, L"SC:", 3 )) {
            int i;
            ctrl_item->type = L'S';
            for( i=0; i< nsc; i++ ) {
              if( !_wcsicmp( v + 3, scripts[i]->name )) {
                ctrl_item->script = scripts[i];
                if( !( scripts[i]->command ))
                  ctrl_item->enabled = FALSE;
                else
                  ctrl_item->uses_tlperl = uses_tlp( scripts[i]->command );
                break;
              }
            }
            if( !ctrl_item->script ) tldie( NULL,
                L"Menu entry %ls refers to undefined script %ls", k, v );
            free( v );
          } else if( !wcsncmp( v, L"SO:", 3 )) {
            // note. `shell objects' are not predefined in the ini file
            ctrl_item->type = L'D';
            if( is_url( v + 3 )) {
              ctrl_item->type = L'D';
              ctrl_item->action = tl_wcscpy( v + 3 );
            } else if( PathIsDirectory( v + 3 ) == FILE_ATTRIBUTE_DIRECTORY ) {
              ctrl_item->type = L'C';
              ctrl_item->action = tl_wcsnconcat(
                3, L"Explorer.exe \"", v + 3, L"\"" );
            } else if( FILE_EXISTS( v + 3 )) {
              ctrl_item->type = L'D';
              ctrl_item->action = tl_wcscpy( v + 3 );
            } else {
              ctrl_item->enabled = FALSE;
            }
            free( v );
          } else if( !wcsncmp( v, L"FU:", 3 )) {
            // Functions: quit_tl, init_tl, clear_tl, ed_sel_tl,
            // uninst_tl, about_tl.
            // There is no function editor_tl.
            // The corresponding control invokes the then-default editor.
            ctrl_item->type = L'F';
            if( !_wcsicmp( v+3, L"quit" )) {
              ctrl_item->fun = quit_tl;
            } else if( !_wcsicmp( v+3, L"initialize" )) {
              ctrl_item->fun = init_tl;
            } else if( !_wcsicmp( v+3, L"clear" )) {
              ctrl_item->fun = clear_tl;
            } else if( !_wcsicmp( v+3, L"editor_select" )) {
              ctrl_item->fun = ed_sel_tl;
            } else if( !_wcsicmp( v+3, L"default_editor" )) {
              ctrl_item->type = L'E';
              ctrl_item->pgid = ed_pgid;
              if( !ctrl_item->pgid->command ) ctrl_item->enabled = FALSE;
              else ctrl_item->uses_tlperl =
                  uses_tlp( ctrl_item->pgid->command );
            } else if( !_wcsicmp( v+3, L"about" )) {
              ctrl_item->fun = about_tl;
            } else if( !_wcsicmp( v+3, L"uninst_all" )) {
              ctrl_item->fun = uninst_all_tl;
            } else if( !_wcsicmp( v+3, L"uninst" )) {
              ctrl_item->fun = uninst_tl;
            } else {
              add_to_mess( &errmess, L"Undefined function %ls", v+3 );
            }
            free( v );
          } else {
            // normalize_command includes existence check
            ctrl_item->action = normalize_command( v );
            free( v );
            if( !( ctrl_item->action ))
              ctrl_item->enabled = FALSE;
            else
              ctrl_item->uses_tlperl = uses_tlp( ctrl_item->action );
            ctrl_item->type = L'C';
          }
        } // MN:- or Buttons section
      } // if ptr
    } // key=value
  } while ( p_next && *p_next);
  if( section ) FREE0( section );
  if( ncontrols < 1 ) tldie( NULL, L"No controls" );

  // copy some strings to globals
  tl_name = get_env( L"TLNAME" );
  if( !tl_name ) tl_name = tl_wcsnconcat( 2, L"TeX Live ", version );
  configdir = get_env( L"TLCONFIG" );
  if( !configdir )
    tldie( NULL, L"TLCONFIG not defined in ini file" );
  wchar_subst( configdir, L'/', L'\\' );
  // if( !configdir ) {
  //   wchar_t * appd;
  //   appd = get_shell_folder( FOLDERID_RoamingAppData );
  //   if( !appd ) tldie( NULL, L"Cannot get or set config directory" );
  //   configdir = tl_wcsnconcat( 3, appd, L"\\", tl_name );
  //   FREE0( appd );
  // }

  // pre|post [un]install
  pre_config = get_env( L"PRE_CONFIG" );
  post_config = get_env( L"POST_CONFIG" );
  pre_forget = get_env( L"PRE_FORGET" );

  // announcement string
  tl_announce = get_env( L"ANNOUNCE" );

  { // restore environment variables from savedenv; ignore errors
    int i;
    SavedEnv * senv;

    for ( i=0; i<nenvs; i++ ) {
      senv = savedenv[i];
      if( senv ) {
        if( senv->name ) {
          SetEnvironmentVariable( senv->name, senv->old );
          free( senv->name );
          senv->name = NULL;
        }
        if( senv->old ) { free( senv->old ); senv->old = NULL; }
        if( senv->new ) { free( senv->new ); senv->new = NULL; }
        free( senv );
        savedenv[i] = NULL;
      }
    }
  }
  // save any perl-related environment variables in perl_envs
  { int i;
    for( i=0; i<nperl_envs; i++ )
        perl_envs[i].val = get_env( perl_envs[i].name );
  }
} // parse_ini
