/*
Copyright (C) 2016, 2017 Siep Kroonenberg

This file is part of TLaunch.

TLaunch is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

TLaunch is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with TLaunch.  If not, see <http://www.gnu.org/licenses/>.
*/

#undef _WIN32_WINNT
#define _WIN32_WINNT _WIN32_WINNT_WIN7
#undef UNICODE
#define UNICODE
#undef _UNICODE
#define _UNICODE

//////////////////////////////////////////////

// capturing output:
// size of read buffer in bytes and chunk size for reading
/*
// small values for testing overflow
#define READSIZE 512
#define RDBUF 256
*/
// larger values for production
#define READSIZE 32760
#define RDBUF 4096

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
#include <wchar.h>
#include <wctype.h>
#include <stdlib.h>
#include <stdarg.h> // for functions with optional arguments
#include <process.h> // for threads; not needed for CreateProcess
#include <windows.h>
#include <shlwapi.h>
#include <commctrl.h>
#include <knownfolders.h>

// do not include here header files with inline functions.
// If an inline directive is ignored, there would be multiple copies
// of the function concerned, which causes a link error

#include "resource.h"

////////////////////////////////////////////////////

/*
extended length paths: 32767 + null char. This includes a prefix
\\?\ or \\?\UNC\ as in \\?\UNC\server\share.  Such paths will not
be parsed: use no '/', no '.' or '..'.  Search MSDN for MAX_PATH

Notable quotes from MSDN:

I. There is no need to perform any Unicode normalization on path and
file name strings for use by the Windows file I/O API functions
because the file system treats path and file names as an opaque
sequence of WCHARs.

II. The shell and the file system have different requirements. It is
possible to create a path with the Windows API that the shell user
interface is not able to interpret properly.
*/

////////////////////////////
// GENERALITIES

// extern storage class specifier usually omitted i.e. implied

#define TL_MAX_STR 32768 // including terminating null char

#define MAX_TMP_PATH (MAX_PATH - 13)
// including slot for terminating null byte

// calloc wrapper which terminates upon failure
void * calloc_fatal( size_t nitems, size_t it_size );

// free and set to NULL
#define FREE0(p) { if( p ) { free( p ); p = NULL; }}

// maximum tries, with 100ms intervals
#define PATIENCE 50

// global variable for error message; initialize in main program.
wchar_t * errmess;
// errkeep set by caller, cleared by set_err,
// signals to set_err to append to existing value
int errkeep;

// logfile
wchar_t * logpath;
FILE * logstream;
void wr_log( wchar_t * fmt, ... );

////////////////////////////////////////////////////
// From tla_utils_[w|c].c (windows and command-line version resp.);
// command-line versions for the sake of testing

void tlinfo( HWND hwnd, wchar_t * fmt, ... );
void tlwarn( HWND hwnd, wchar_t * fmt, ... );
void tldie( HWND hwnd, wchar_t * fmt, ... );

// GUI: messagebox (if short) or custom dialog with edit control (if long)
// (tla_utils_w.c)
// CMD: just fwprintf (tla_utils_c.c)
void display_string( HWND parent, wchar_t * str, wchar_t * capt );

////////////////////////////////////////
// The module tla_COM requires COM and is written in C++,
// although a rewrite in C is possible.

BOOL init_com( void );
void uninit_com( void );
int make_shortcut( wchar_t * target, wchar_t * parms, wchar_t * link_loc );

// The remaining functions defined in tla_utils.c

////////////////////////////////////////////////////
// string functions with allocation. caller frees memory

// copy n wchars to new string and add null character, with allocation.
wchar_t * tl_wcsncpy( wchar_t * source, size_t n );

// make copy of string, with allocation
wchar_t * tl_wcscpy( wchar_t * source );

// Concatenate several strings, with allocation.
wchar_t * tl_wcsnconcat( int n, ...);

// wide-char [v]asprintf with automatic sizing and allocation
// first make sure about swprintf & co

#ifdef __mingw_snwprintf
  #undef swprintf
  #define swprintf __mingw_snwprintf
#endif

#ifdef _vswprintf_p
  #undef vswprintf
  #define vswprintf _vswprintf_p
#endif

// `wide-char' [v]asprint with automatic sizing and allocation
int tl_vwasprintf( wchar_t ** ret, const wchar_t * fmt, va_list ap );
int tl_wasprintf( wchar_t ** ret, const wchar_t * fmt, ... );

// in-place removal of \n and \r at the end of a string
void chomp( wchar_t * str );

// in-place character substitution
void wchar_subst( wchar_t * s, wchar_t oldchar, wchar_t newchar );

// for assembling a list of warnings
void add_to_mess( wchar_t ** mess, wchar_t * fmt, ... );

// convert memory block of given size to utf-16
wchar_t * to_wide_string( BYTE * s, int l );

// read file at one go up to max_size
BYTE * slurp_file( wchar_t * fname, DWORD max_size, DWORD * lread );

// and convert into wide-char string
// the resulting string may be larger than max_size
wchar_t * slurp_file_to_wcs( wchar_t * fname, DWORD max_size );

// capture output via a pipe; may fail if command calls other programs
wchar_t * capture_output( wchar_t * command, BOOL allow_overflow );

// keep temp files used in capturing
int keep_capts;

// capture output via a temp file; quoting tricky
wchar_t * capture_tmp( wchar_t * cmd, BOOL allow_overflow );

// if the command is a single filename, possibly with spaces:
wchar_t * capture_tmp_quoted( wchar_t * cmd, BOOL allow_overflow );

////////// ENVIRONMENT /////////////////////////////

// wrapper for GetEnvironmentVariable
wchar_t * get_env( wchar_t * v );

// wrapper for expanding environment variables
wchar_t * expand_env_vars( wchar_t * s );

////////// REGISTRY /////////////////////////////

// for diagnostic messages:
wchar_t * rootkey_name( HKEY rootkey );
wchar_t * regtype_name( DWORD type );

// struct: string value plus data type (REG_SZ / REG_EXPAND_SZ /REG_MULTI_SZ)
// we can add more fields if and when we need to support non-string data types
typedef struct {
  wchar_t * sval;
  DWORD type;
} RegValueTyped;

// wrappers for getting and setting string values from registry

// existence
int reg_key_exists( HKEY rootkey, wchar_t * regpath );
int reg_value_exists( HKEY rootkey, wchar_t * regpath, wchar_t * valname );

// getting
RegValueTyped
    reg_get_value_raw( HKEY rootkey, wchar_t * regpath, wchar_t * valname );
wchar_t * reg_get_value( HKEY rootkey, wchar_t * regpath, wchar_t * valname );

// setting
// simple special case: environment variable; key exists, no subkeys needed
int reg_set_env_user( wchar_t * name, RegValueTyped value );
// general case
int reg_set_value( HKEY rootkey, wchar_t * regpath,
    wchar_t * name, RegValueTyped value );
// automatic string type
int reg_set_stringvalue( HKEY rootkey, wchar_t * regpath,
    wchar_t * name, wchar_t * value );

// deleting
int reg_del_key( HKEY rootkey, wchar_t * regpath );
// delete registry value
int reg_del_value( HKEY rootkey, wchar_t * regpath, wchar_t * name );

// user path
int modify_user_path( wchar_t * dir, int add );

// get special folder
wchar_t * get_shell_folder( KNOWNFOLDERID rfid );

///////// SUPPORT FOR FILE TYPES; FINDING PROGRAMS ////////

// get the program part of a command; does not include existence check
wchar_t * prog_from_cmd( wchar_t * cmd );

// icon specification (member of progid)
void parse_iconspec( wchar_t * icf, wchar_t **iconfile, UINT * iconindex );

// get command associated with a filetype
wchar_t * get_ftype_cmd( wchar_t * progid );

// find progid data associated with a progid
void get_assoc_data_progid( wchar_t * progid, wchar_t ** p_cmd,
    wchar_t ** p_iconspec, wchar_t ** p_dflt_value );
// find progid data associated with an extension
void get_assoc_data_ext( wchar_t * ext, wchar_t ** p_progid, wchar_t ** p_cmd,
    wchar_t ** p_iconspec, wchar_t ** p_dflt_value );

// find command associated with an extension
wchar_t * get_assoc_cmd_ext( wchar_t * ext );

// find full path of gui executable via application registration
// call only if the executable is not on the process searchpath
wchar_t * find_registered_app( wchar_t * prog );

// create progid registry key.
// tp refers to cmd and can be REG_SZ or REG_EXPAND_SZ.
// for other items, this choice is automatic.
int register_progid( wchar_t * progid, wchar_t * cmd, DWORD tp,
    wchar_t * iconspec, wchar_t * dflt_value );
// remove progid
int remove_progid( wchar_t * progid );

// create secondary association of extension with progid
// using openwithprogids key
void set_2nd_assoc( wchar_t * ext, wchar_t * progid );
// main association
int set_assoc_abs( wchar_t * ext, wchar_t * progid );
int set_assoc( wchar_t * ext, wchar_t * progid, int mode );

// break association of extension with progid
int remove_assoc( wchar_t * ext, wchar_t * progid );

// (un)register a program, optionally with searchpath prefix
int register_prog( wchar_t * path, wchar_t * prefix );
int unregister_prog( wchar_t * filename );

///// FULL PATH OF EXE ////////////////////////////

wchar_t * normalize_path( wchar_t * fname );

// searches prog on process search path and returns full filename,
// or NULL if not found. prog should have an extension
// but no directory component.
extern wchar_t * find_on_path( wchar_t * prog );

// find full path of program by any means
wchar_t * find_program( wchar_t * prog );

// in a command, replace program with possibly quoted full path
wchar_t * normalize_command( wchar_t * command );

/////////// FILES ///////////////////////////////////

#define FILE_EXISTS(f) ( GetFileAttributes(f) != INVALID_FILE_ATTRIBUTES )

__int64 win_filesize( wchar_t * name );

// returns TRUE if either the strings match case-insensitively or
// both files exist and are the same.
// rationale: we do not want to return FALSE just because a resource such as
// a removable drive or network share is temporarily unavailable.
BOOL same_file( wchar_t * f1, wchar_t * f2 );

BOOL is_dir( wchar_t * fname );

void make_writable( wchar_t * fname );

// create directory and ancestors
int mkdir_recursive( wchar_t * dir );
// delete directory and children
int rmdir_recursive( wchar_t * dir );
// delete directory only if empty
int rmdir_ifempty( wchar_t * dirname );

// delete a file; keep trying
int delete_file( wchar_t * path );

// a crude test whether a string may be a url:
// a string containing "://" with at least one character before and after
extern int is_url( wchar_t * s );

// includes allocation
void get_tempfilename( wchar_t * ext, wchar_t ** temp_fname );

// calls make_shortcut_dir_exists for the COM stuff
// int make_shortcut( wchar_t * target, wchar_t * parms, wchar_t * link_loc );

///////////////////// DIAGNOSTICS /////////////////////////

// full path of current executable
// stupidly:
// - _wpgmptr is considered obsolete
// - _get_wpgmptr requires e.g. libmsvcr100.a or msvcr100.dll
// - GetModuleFileName won't tell you how big a buffer is really needed.
wchar_t * get_own_path( void );

// check OS version
/* BOOL os_geq( DWORD maver, DWORD miver ); */

BOOL is_elevated( void );

#ifdef __cplusplus
}
#endif
