package com.dickimawbooks.jmakepdfx;

import java.io.*;
import java.util.*;
import java.awt.event.ActionListener;

import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JToolBar;

public class JpdfxProperties extends Properties
{
   public JpdfxProperties(File propFile, File recentFile,
      boolean loadFile, Jmakepdfx application)
     throws IOException
   {
      super();

      app = application;

      this.propFile = propFile;
      this.recentFile = recentFile;

      recentList = new Vector<String>();

      if (propFile != null && propFile.exists() && loadFile)
      {
         // if propFile exists, load it

         BufferedReader reader = 
            new BufferedReader(new FileReader(propFile));

         load(reader);

         reader.close();
      }
      else
      {
         // set default values

         setDefaults();
      }

      if (recentFile != null && recentFile.exists())
      {
         BufferedReader in = new BufferedReader(new FileReader(recentFile));

         loadRecentFiles(in);

         in.close();
      }
   }

   public JpdfxProperties(Jmakepdfx application)
   {
      super();

      app = application;

      propFile       = null;
      recentFile     = null;

      setDefaults();
   }

   protected void setDefaults()
   {
      setProperty("directory", ".");
      setProperty("toolbarorient", ""+JToolBar.HORIZONTAL);
   }

   public static JpdfxProperties fetchProperties(Jmakepdfx application)
     throws IOException
   {
      File parent = null;

      File settings   = null;
      File recent     = null;

      String home = System.getProperty("user.home");

      boolean loadFile = (home != null);

      if (loadFile)
      {
         // first check if directory ~/.jmakepdfx exists

         parent = new File(home, ".jmakepdfx");

         if (!parent.isDirectory())
         {
            // check if directory ~/jmakepdfx-settings exists

            parent = new File(home, "jmakepdfx-settings");

            if (!parent.isDirectory())
            {
               parent = null;
            }
         }
      }

      if (parent != null)
      {
         settings = new File(parent, baseName);
         recent = new File(parent, recentName);
      }
      else
      {
         loadFile = false;
      }

      return new JpdfxProperties(settings, recent, loadFile, application);
   }

   public void save()
      throws IOException
   {
      if (propFile == null)
      {
         String home = System.getProperty("user.home");

         if (home == null)
         {
            throw new IOException(app.getLabel("error.home_null"));
         }

         File parent;

         if (System.getProperty("os.name").equals("Windows"))
         {
            parent = new File(home, "jmakepdfx-settings");
         }
         else
         {
            parent = new File(home, ".jmakepdfx");
         }

         if (!parent.mkdir())
         {
            throw new IOException(app.getLabelWithValue("error.io.cant_mkdir",
               parent.getAbsolutePath()));
         }

         propFile = new File(parent, baseName);
      }

      String setting = getDefaultDirectorySetting();

      if (setting == null)
      {
         setting = "home";
      }

      if (setting.equals("home"))
      {
         setDefaultDirectory(System.getProperty("user.home"));
      }
      else if (setting.equals("last"))
      {
         setDefaultDirectory(app.getCurrentDirectory());
      }

      PrintWriter out = new PrintWriter(new FileWriter(propFile));

      store(out, "jmakepdfx-gui properties");

      out.close();

      if (recentFile == null)
      {
         recentFile = new File(propFile.getParentFile(), recentName);
      }

      out = new PrintWriter(new FileWriter(recentFile));

      for (int i = 0, n = recentList.size(); i < n; i++)
      {
         out.println(recentList.get(i));
      }

      out.close();

   }

   private void loadRecentFiles(BufferedReader in)
     throws IOException
   {
      String line;

      while ((line = in.readLine()) != null)
      {
         recentList.add(line);
      }
   }

   public void addRecentFile(String fileName)
   {
      recentList.remove(fileName); // just in case it's already in the list
      recentList.add(fileName);
   }

   public void setRecentFiles(JMenu menu, ActionListener listener)
   {
      if (recentList == null) return;

      if (menu.getMenuComponentCount() > 0)
      {
         menu.removeAll();
      }

      int lastIdx = recentList.size()-1;

      int n = Math.min(MAX_RECENT_FILES-1, lastIdx);

      for (int i = 0; i <= n; i++)
      {
         File file = new File(recentList.get(lastIdx-i));
         String num = ""+i;
         JMenuItem item = new JMenuItem(num+": "+file.getName());
         item.setMnemonic(num.charAt(0));
         item.setToolTipText(file.getAbsolutePath());
         item.setActionCommand(num);
         item.addActionListener(listener);

         menu.add(item);
      }
   }

   public String getRecentFileName(int i)
   {
      return recentList.get(recentList.size()-1-i);
   }

   public void clearRecentList()
   {
      recentList.clear();
   }

   public String getDefaultDirectory()
   {
      return getProperty("directory");
   }

   public void setDefaultDirectory(String dir)
   {
      if (dir != null)
      {
         setProperty("directory", dir);
      }
   }

   public String getDefaultDirectorySetting()
   {
      return getProperty("directory.setting");
   }

   public void setDefaultDirectorySetting(String setting)
   {
      if (setting != null)
      {
         setProperty("directory.setting", setting);
      }
   }

   public void setDefaultHomeDir()
   {
      setDefaultDirectorySetting("home");
   }

   public void setDefaultLastDir()
   {
      setDefaultDirectorySetting("last");
   }

   public void setDefaultCustomDir(String dir)
   {
      if (dir == null)
      {
         setDefaultDirectorySetting("last");
      }
      else
      {
         setDefaultDirectorySetting("custom");
         setDefaultDirectory(dir);
      }
   }

   public String getPdfViewer()
   {
      return getProperty("pdfviewer");
   }

   public void setPdfViewer(String path)
   {
      if (path == null)
      {
         setProperty("pdfviewer", "");
      }
      else
      {
         setProperty("pdfviewer", path);
      }
   }

   public int getToolBarOrientation()
   {
       String val = getProperty("toolbarorient");

       if (val == null)
       {
          return JToolBar.HORIZONTAL;
       }

       try
       {
          return Integer.parseInt(val);
       }
       catch (NumberFormatException e)
       {
       }

       return JToolBar.HORIZONTAL;
   }

   public void setToolBarOrientation(int orient)
   {
      setProperty("toolbarorient", ""+orient);
   }

   public String getToolBarPosition()
   {
      String val = getProperty("toolbarpos");

      return (val == null ? "North" : val);
   }

   public void setToolBarPosition(String pos)
   {
      if (pos == null)
      {
         setProperty("toolbarpos", "North");
      }
      else
      {
         setProperty("toolbarpos", pos);
      }
   }

   public String getICCFile()
   {
      return getProperty("iccfile");
   }

   public void setICCFile(String iccfile)
   {
      setProperty("iccfile", iccfile);
   }

   public String getGSApp()
   {
      return getProperty("gs");
   }

   public void setGSApp(String gspath)
   {
      setProperty("gs", gspath);
   } 

   public String getPDFViewer()
   {
      return getProperty("pdfviewer");
   }

   public void setPDFViewer(String path)
   {
      setProperty("pdfviewer", path);
   } 

   public String getProfile()
   {
      String profile = getProperty("profile");

      // default to gray

      if (profile == null)
      {
         profile = "gray";
      }

      if (!(profile.equals("gray") || profile.equals("cmyk")))
      {
         throw new IllegalArgumentException("Invalid profile '"+profile+"'");
      }

      return profile;
   }

   public boolean isUseICC()
   {
      return Boolean.parseBoolean(getProperty("useicc", 
       isGrayProfile() ? "false" : "true"));
   }

   public void setUseICC(boolean useICC)
   {
      setProperty("useicc", useICC ? "true" : "false");
   }

   public boolean isGrayProfile()
   {
      return getProfile().equals("gray");
   }

   public boolean isCMYKProfile()
   {
      return getProfile().equals("cmyk");
   }

   public void setProfile(String profile)
   {
      if (!(profile.equals("gray") || profile.equals("cmyk")))
      {
         throw new IllegalArgumentException("Invalid profile '"+profile+"'");
      }

      setProperty("profile", profile);
   }

   public long getMaxProcessTime()
   {
      String prop = getProperty("timeout");

      long timeout = DEFAULT_TIMEOUT;

      if (prop != null)
      {
         try
         {
            timeout = Long.parseLong(prop);
         }
         catch (NumberFormatException e)
         {
         }
      }

      return timeout;
   }

   public void setMaxProcessTime(long timeout)
   {
      if (timeout <= 0)
      {
         throw new IllegalArgumentException("Illegal process timeout value '"
           + timeout +"'");
      }

      setProperty("timeout", ""+timeout);
   }

   private File propFile;

   private static final String baseName = "jmakepdfx.prop";
   private static String recentName = "recentfiles";

   private static final int MAX_RECENT_FILES=10;

   private Jmakepdfx app;

   private File recentFile;

   private Vector<String> recentList;

   private static final long DEFAULT_TIMEOUT = 1000 * 60 * 180; // 3 hours
}
