## -*-Tcl-*- (install)
 # ###################################################################
 #  Alpha - new Tcl folder configuration
 # 
 #  FILE: "developerUtilities.tcl"
 #                                    created: 10/9/97 {11:22:17 am} 
 #                                last update: 4/7/1999 {10:32:35 am} 
 #  Author: Vince Darley
 #  E-mail: <darley@fas.harvard.edu>
 #    mail: Division of Engineering and Applied Sciences, Harvard University
 #          Oxford Street, Cambridge MA 02138, USA
 #     www: <http://www.fas.harvard.edu/~darley/>
 #  
 # Copyright (c) 1997-1998  Vince Darley, all rights reserved
 # 
 # Reorganisation carried out by Vince Darley with much help from Tom 
 # Fetherston, Johan Linde and suggestions from the Alpha-D mailing list.  
 # Alpha is shareware; please register with the author using the register 
 # button in the about box.
 #  
 # The last 4 procs in this file are copyright (c) Pete Keleher
 # 
 # ###################################################################
 ##

alpha::extension developerUtilities 1.4.1 {
    menu::insert Install items end "(-" removeAllColoursAndHypers removeAllMarks \
      hyperiseUrls hyperiseEmailAddresses colourHeadingsEtc  \
      markAlphaManual markAlphaCommands markAlphaChanges markReadme "(-" \
      ensureDistributionIsUpToDate changeInstallerIcon \
      stuffPackageForDistribution \
      uploadStuffedPackage "(-" updateStuffAndUpload \
      ensureAlphaDistn1UpToDate ensureAlphaDistn2UpToDate \
      ensureAlphaDistn3UpToDate
    menu::insert tclMenu items end compareWithDistribution addToChangesFile
    # where does dropStuff put its stuffed items
    newPref var dropStuffFolder "" Inst
    # default internet location to which we upload stuffed, binhexed packages
    newPref var defaultAlphaUploadSite "" Inst "" remote::site array
    # disk location of first separate Alpha distribution (alpha-lite)
    newPref folder separateAlpha1DistributionFolder "" Inst
    # disk location of second separate Alpha distribution (full version)
    newPref folder separateAlpha2DistributionFolder "" Inst
    # disk location of third separate Alpha distribution 
    # (experimental distribution)
    newPref folder separateAlpha3DistributionFolder "" Inst
    if {[alpha::package exists documentProjects]} {
	llunion elec::DocTemplates 1 \
	  { Tcl "Alpha package" Basic t_package * {extension feature mode menu}}
    }
} maintainer {
    "Vince Darley" darley@fas.harvard.edu <http://www.fas.harvard.edu/~darley/>
} help {
    Marking of help files
    Comparison of file with one from distribution
    Stuffing/uploading/updating distributions
    Document template for packages (requires documentProjects package)	
} uninstall this-file

namespace eval Inst {}

proc Inst::MarkFile {} {
    if {[file extension [win::Current]] == ".tcl"} {
	return [Tcl::MarkFile]
    }
    removeAllMarks
    install::removeAllColoursAndHypers
    install::colourHeadingsEtc
    install::hyperiseUrls
    install::hyperiseEmailAddresses
}

namespace eval install {}

proc install::changeInstallerIcon {} {
    set f [win::Current]
    setFileInfo $f type InSt
    message "Icon changed."
}

proc install::removeAllColoursAndHypers {} {
    # get rid of the old stuff
    catch { removeColorEscapes }
    refresh
}

proc install::colourHeadingsEtc { } {
    catch { install::markAlphaManual }
    catch { install::colorManualMarks }
    
    # search for "something.tcl" and attach appropriate lookup.
    file::searchAndHyperise {\"([-a-zA-Z_+1-9]*(.tcl))\"} \
      {file::hyperOpen \1} 1 3 +1 -1
    # search for "something.tcl" and attach appropriate lookup.
    file::searchAndHyperise "package: (\[^\r\]*)\r" \
      {file::hyperHelpOpen "\1"} 1 4 +9
    # Highlight IMPORTANT bits
    file::searchAndHyperise {IMPORTANT:} { } 0 5
    # make code inserts blue
    set bluestr {^[ \t]*\r\t[^" \t\r][^\r]*\r(\t([ \t]*\r|[ \t]*[^ \t\r]+[^\r]*\r))*[ \t]*\r}
    file::searchAndHyperise $bluestr {} 1 1
    file::searchAndHyperise {CLICK[ A-Z]* INSTALL} "install::installThisPackage" 1 3	
    if {![catch {set inds [search -f 1 -r 1 {Jump to recent changes} 0]}]} {
	set from [lindex $inds 0]
	set to [lindex $inds 1]
	insertColorEscape $from 3
	insertColorEscape $from 15 {gotoMark " Recent Changes:"}
	insertColorEscape $to 12
	insertColorEscape $to 0
    }
    goto [minPos]
}
	

## 
 # ----------------------------------------------------------------------
 #	 
 #	"install::hyperiseUrls" --
 #	
 #	 A useful example of how the above function	can	be used. This
 #	 attaches hypertext	links to all '<http:...	>' strings in a	document.
 #	 This procedure works best on files in Text mode; in other modes the
 #	 colouring schemes can make the links invisible (although they still
 #	 function).
 #	
 # ----------------------------------------------------------------------
 ##
proc install::hyperiseUrls {} {
    file::searchAndHyperise {<((http|mailto|ftp):[^ ]*)>} {icURL "\1"} 1
}

proc install::hyperiseEmailAddresses {} {
    file::searchAndHyperise \
      {<([-_a-zA-Z.]+@[-_a-zA-Z]+(.[-_a-zA-Z]+(.[-_a-zA-Z]+)?)?)>} \
      {icURL "mailto:\1"} 1
}

## 
 # -------------------------------------------------------------------------
 # 
 # "install::ensureDistributionIsUpToDate" --
 # 
 #  Helpful for package developers.  You can keep your current versions in
 #  the Alpha source tree, but use this procedure periodically to backup
 #  the code, and prepare for distribution.  All files in the distribution
 #  hierarchy (up to one level deep) are replaced if a more recent version
 #  can be found anywhere in the Alpha source tree.  You are told which
 #  were replaced, and which couldn't be found.
 #  
 #  The only exception is that if there is a file whose name matches
 #  *install*.tcl in the top level of your distribution, it is ignored.
 #  It is assumed that such a file contains installation scripts for
 #  your package, and that it will not be installed itself.
 #  
 #  Doesn't cope with recursive directories.
 # -------------------------------------------------------------------------
 ##
proc install::ensureDistributionIsUpToDate {} {
    global HOME 
    set win::Current [win::Current]
    if {[file extension ${win::Current}] == ".tcl"} {
	# single file to install
	set distribfiles [list ${win::Current}]
    } else {
	set currD [file dirname ${win::Current}]
	set distribfiles [glob -t TEXT -nocomplain -dir $currD *]
	set distribfiles [lremove -glob $distribfiles "*\[iI\]nstall*.tcl"]
	set distribfiles [lremove $distribfiles [win::Current]]
	eval lappend distribfiles [glob -nocomplain -join -dir $currD * *]
    }	
    set failed ""
    set replaced ""
    foreach ff $distribfiles {
	if {[file isdirectory $ff]} {
	    lappend failed $ff
	    continue
	}
	set looking 1
	set f [file tail $ff]
	if {[catch {file::standardFind $f} to]} {
	    lappend failed $f
	} else {
	    if {[file::replaceSecondIfOlder $to $ff]} {
		lappend replaced $f
	    }
	}
    }
    if {$failed == ""} {set failed "none"}
    if {$replaced == ""} {set replaced "none"}
    if {[catch {alertnote "Replaced $replaced, failed to find $failed."}]} {
	alertnote "Replaced [llength $replaced], failed to find $failed."
    }
}

proc install::ensureAlphaDistn1UpToDate {} {
    global InstmodeVars 
    install::ensureAlphaDistnUpToDate $InstmodeVars(separateAlpha1DistributionFolder)
}

proc install::ensureAlphaDistn2UpToDate {} {
    global InstmodeVars 
    install::ensureAlphaDistnUpToDate $InstmodeVars(separateAlpha2DistributionFolder)
}

proc install::ensureAlphaDistn3UpToDate {} {
    global InstmodeVars 
    install::ensureAlphaDistnUpToDate $InstmodeVars(separateAlpha3DistributionFolder)
}

proc install::ensureAlphaDistnUpToDate {alpha} {
    global HOME install::log InstmodeVars
    set install::log ""
    set d [pwd]
    install::_ensureAlphaDistnUpToDate ${alpha} :
    install::showLog
    install::_recursivelyRebuildIndices ${alpha}:Tcl:
    cd $d
    alertnote "Done"
}

proc install::_recursivelyRebuildIndices {dir} {
    global tcl_platform
    if {[file exists $dir]} {
	cd $dir
	if {$tcl_platform(platform) == "macintosh"} {
	    if {![catch {glob *.tcl}]} {
		catch { auto_mkindex : }
	    }
	    foreach dir [glob -nocomplain :*:] {
		install::_recursivelyRebuildIndices $dir
	    }
	    cd ::
	} else {
	    if {![catch {glob *.tcl}]} {
		catch { auto_mkindex . }
	    }
	    foreach dir [glob -nocomplain */] {
		install::_recursivelyRebuildIndices $dir
	    }
	    cd ..
	}
    }
}

proc install::_ensureAlphaDistnUpToDate {alpha dir} {
    global HOME InstmodeVars
    message "Examining $dir"
    cd $alpha$dir
    set dirs ""
    set files ""
    set all [glob -nocomplain *]
    set havedir 0
    foreach a $all {
	if {[file isdirectory $a]} { 
	    lappend dirs $a
	    set havedir 1
	} else {
	    lappend files $a
	}
    }
    if {!$havedir} {
	# bottom level directory.  Check file-count
	set cdist [llength $all]
	set corig [llength [glob -nocomplain -path ${HOME}${dir} *]]
	if {$cdist != $corig} {
	    install::log "WARNING: FILE-COUNT CHANGED IN $dir"
	}
    }
	
    foreach f $files {
	if {[file exists ${HOME}${dir}$f]} {
	    if {![regexp {^tclIndexx?$} $f]} {
		file::replaceSecondIfOlder ${HOME}${dir}$f \
		  $alpha$dir$f
	    }
	} else {
	    install::log "Warning: file $f was not found."
	}
    }
    foreach d $dirs {
	if {[file exists ${HOME}${dir}$d]} {
	    install::_ensureAlphaDistnUpToDate $alpha ${dir}${d}:
	} else {
	    install::log "WARNING: Original directory '$d' doesn't exist"
	}
    }
}

proc install::stuffPackageForDistribution {{fore 0}} {
    set stuff [install::_getStuffedFile [install::_getDistributionBaseName]]
    # Try and remove the old stuffed version
    if {$stuff != ""} {
	catch {file delete $stuff}
    }
    install::_stuffDistribution [install::_getDistributionBase] $fore
}

proc install::_getDistributionBaseName {} {
    return [file tail [install::_getDistributionBase]]
}
proc install::_getDistributionBase {} {
    # Is it a file or directory?
    set f [win::Current]
    if {[file extension $f] != ".tcl"} {
	return [file dirname $f]
    } else {
	return $f
    }
}

proc install::_stuffDistribution {ff {fore 0}} {
    global file::separator
    if {[file isdirectory $ff] && ![regexp -- "${file::separator}\$" $ff]} {
	append ff ${file::separator}
    }
    # Now stuff new distribution
    app::launchBack DStf
    if {$fore} {
	sendOpenEvent reply 'DStf' $ff
    } else {
	sendOpenEvent noReply 'DStf' $ff
    }
	
    sendQuitEvent 'DStf'
}

proc install::_getStuffedFile {pref} {
    global InstmodeVars
    set files [glob -nocomplain -path [file join $InstmodeVars(dropStuffFolder) ${pref}] *.hqx]
    if {[llength $files] == 1} {
	return [lindex $files 0]
    }
    return ""
}

proc install::uploadStuffedPackage {{ask 1}} {
    set stuff [install::_getStuffedFile [install::_getDistributionBaseName]]
    if {$stuff == ""} {
	alertnote "Sorry, I couldn't find the stuffed distribution."
	error ""
    }
    global InstmodeVars remote::site
    if {$ask} {
	set sitename [dialog::optionMenu "Upload to which site?" \
	  [lsort -ignore [array names remote::site]] \
	  $InstmodeVars(defaultAlphaUploadSite)]
    } else {
	set sitename $InstmodeVars(defaultAlphaUploadSite)
    }
    url::store [set remote::site($sitename)] $stuff
}

proc install::updateStuffAndUpload {} {
    install::ensureDistributionIsUpToDate
    install::stuffPackageForDistribution 1
    install::uploadStuffedPackage 0
    alertnote "Distribution upload complete."
}

proc install::markReadme {} {
    removeAllMarks
    install::removeAllColoursAndHypers
    file::multiSearchAndHyperise "Home Page" \
      {icURL http://alpha.olm.net/} \
      "Quick Start" {edit -r [file join ${HOME} Help "Quick Start"]} \
      "Bug Reports And Debugging" \
      {edit -r [file join ${HOME} Help "Bug Reports And Debugging"]} \
      "Manual" {edit -r [file join ${HOME} Help "Alpha Manual"]} \
      "Click here to update Alpha's list of remote packages via the internet" \
      {package::queryWebForList} \
      "Config->Preferences->International" \
      {dialog::preferences Preferences International}
    
    file::searchAndHyperise "\rHelp" {alphaHelp} 1 3 1
    file::searchAndHyperise "'Changes'" {edit -r [file join ${HOME} Help Changes]} 0 3 1 -1
    file::searchAndHyperise {<get ([^>]+)>} {remote::get \1} 1
    file::searchAndHyperise {[\w ']+ \-\-\-} { } 1 5 0 -4
    install::hyperiseUrls
    install::hyperiseEmailAddresses
    
}

proc compareWithDistribution {} {
    global auto_path
    set w [win::Current]
    set wn [file tail $w]
    foreach dir $auto_path {
	set f [file join ${dir} ${wn}]
	if {[file exists $f]} {
	    if {$f == $w} { alertnote "It's part of the distribution!" ; return }
	    file::openQuietly $f
	    compare::windows
	    return
	}
    }
    alertnote "No distribution file with this name was found."
}
proc addToChangesFile {} {
    global HOME
    file::openQuietly [file join ${HOME} Help Changes]
    setWinInfo read-only 0
    set p [search -f 1 -r 1 -m 0 -s "^ \[\n\r\]" [minPos]]
    goto [lindex $p 0]
    insertText " \r"
    backwardChar
}

proc t_package {name parentdoc subtype } {
    # Possible 'subtypes' are: extension mode menu
    
    # remove trailing mode/menu etc if possible
    regsub -nocase -- $subtype $name "" name
    set t "\r\# $subtype declaration\r"
    append t "alpha::$subtype $name version "
    switch -- $subtype {
	"mode" {
	    append t "${name}Dummy \{extensions\} \{mode-menus\} "
	}
	"menu" {
	    append t "global or modes title/icon "
	}
    }
    append t "\{\r\tstartup script\r\}"
    if {$subtype == "feature" || $subtype == "menu"} {
	append t " \{\r\tactivate script\r\}"
	append t " \{\r\tdeactivate script\r\}"
    }
    append t " maintainer \{\r"
    global user
    append t "\t\{$user(author)\} $user(email) $user(www)\r\}"
    append t " help \{\r\tfile 'name' or description\r\}"
    append t " uninstall script\r"
    switch $subtype {
	"mode" {
	    append t "\# to auto-load\r"
	    append t "proc ${name}Dummy {} {}\r\r"
	    append t "\#  mode preferences  #\r"
	    append t "newPref type name default $name proc , options, sub-opts\r"
	    append t "\r\r"
	}
    }
    return $t
}

#===============================================================================
# Pete's manual-marking routines

proc install::markAlphaManual {} {
    set pos [minPos]
    set labels ""
    while {[string length [set inds [search -f 1 -r 1 {^	  	} $pos]]]} {
	set pos1 [lindex $inds 1]
	set label [getText $pos1 [pos::math [nextLineStart $pos1] - 1]]
	set pos2 [lineStart $pos1]
	if {$label == ""} {set label "-"}
	while {[lsearch $labels $label] != -1} { append label " " }
	setNamedMark $label $pos2 $pos2 $pos2
	lappend labels $label
	set pos [nextLineStart $pos1]
    }
}

proc install::colorManualMarks {} {
    goto [minPos]
    hiliteWord
    endLineSelect
    set from [getPos]
    set to [selEnd]
    insertColorEscape $from 5
    insertColorEscape $from 15
    insertColorEscape $to 12
    insertColorEscape $to 0
    
    foreach mk [getNamedMarks] {
	set name [lindex $mk 0]
	set disp [lindex $mk 2]
	set pos [lindex $mk 3]
	set end [lindex $mk 4]
	
	goto $disp
	hiliteWord
	endLineSelect
	set from [getPos]
	set to [selEnd]
	
	insertColorEscape $from 5
	insertColorEscape $from 15
	insertColorEscape $to 12
	insertColorEscape $to 0
    }
}

#===============================================================================

proc install::markAlphaCommands {} {
    global HOME
    file::openQuietly [file join $HOME Help "Alpha Commands"]
    setWinInfo read-only 0
    install::removeAllColoursAndHypers
    removeAllMarks
    changeMode Tcl
    set pos [minPos]
    while {![catch {search -f 1 -r 1 {^ } $pos} inds]} {
	set pos1 [lindex $inds 1]
	goto $pos1
	hiliteWord
	set label [getSelect]
	set from [getPos]
	set to [selEnd]
	setNamedMark $label $pos1 $from $to
	insertColorEscape $from 1
	insertColorEscape $to 0
	set pos [nextLineStart $pos1]
    }
    select [minPos] [nextLineStart [nextLineStart [nextLineStart [minPos]]]]
    redWord
    changeMode Text
    goto [minPos]
    save
}

#===============================================================================

proc install::markAlphaChanges {} {
    set pos [minPos]
    while {[string length [set inds [search -f 1 -r 1 {^= } $pos]]]} {
	set pos1 [lindex $inds 1]
	goto $pos1
	endLineSelect
	redWord
	set pos [nextLineStart $pos1]
    }
}

#===============================================================================



