/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* GRAPHICS.C --- Augment BGI. */

#ifndef NO_ASM
#pragma inline
#endif
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <graphics.h>
#include <string.h>
#include <values.h>
#include "window.h"

#ifdef NO_SMALL_FONT
#define registerbgifont(P) 1
#endif

/* Initialize graphics. Return 0 on success. */
int init_graphics(void)
{
  int gdriver, gmode;

  if (registerbgidriver(EGAVGA_driver) < 0
   || registerbgifont(small_font) < 0)
    return 1;

  gdriver = VGA;
  gmode = VGAHI;
  initgraph(&gdriver, &gmode, "");

  if (graphresult() != grOk)
    return 2;

  return 0;
}

/* Shut down graphics. */
int close_graphics(void)
{
  closegraph();
  return 0;
}

/* Return round((double)x * num / den), where
   num and den are positive.  Go fast because
   this is the scaling routine for many ops.*/
int scale_int(int x, int num, int den)
{
  assert (num >= 0 && den > 0);

  /* bx = (x < 0) ? -1 : 0
     ax = abs x */
  asm mov ax, x
  asm cwd
  asm mov bx, dx
  asm xor ax, dx
  asm sub ax, dx

  /* ax = abs x * num / den 
     dx = remainder of division */
  asm mul word ptr num
  asm div word ptr den

  /* if (dx >= den/2) ++ax */
  asm shl dx, 1
  asm inc dx
  asm cmp den, dx
  asm adc ax, 0

  /* if (bx == 0xffff) ax = -ax */
  asm xor ax, bx
  asm sub ax, bx

  return _AX;
}

/* Same as above, but num and den may be signed. */
int signed_scale_int(int x, int num, int den)
{
  /* bx = (num < 0) ? -1 : 0 
     num = abs num */
  asm mov ax, num
  asm cwd
  asm mov bx, dx
  asm xor ax, dx
  asm sub ax, dx
  asm mov num, ax

  /* if (den < 0) bx ^= 0xffff
     cx = abs den */
  asm mov ax, den
  asm cwd
  asm xor bx, dx
  asm mov cx, ax
  asm xor cx, dx
  asm sub cx, dx

  /* if (x < 0) bx ^= 0xffff
     ax = abs x */
  asm mov ax, x
  asm cwd
  asm xor bx, dx
  asm xor ax, dx
  asm sub ax, dx

  /* ax = ax * num / cx 
     dx = remainder of division */
  asm mul word ptr num
  asm div cx

  /* if (dx >= den/2) ++ax */
  asm shl dx, 1
  asm inc dx
  asm cmp den, dx
  asm adc ax, 0


  /* if (bx = 0xffff) ax = -ax */
  asm xor ax, bx
  asm sub ax, bx

  return _AX;
}

/* Routines to stack and restore bgi graphic parameters. */

#define SAVE_STACK_SIZE 4

static struct gstate {
  int color;
  struct fillsettingstype fs;
  struct linesettingstype ls;
  struct textsettingstype ts;
  struct viewporttype vs;
} graphics_save_stack[SAVE_STACK_SIZE],
  *gsp = graphics_save_stack;

void push_graphics_state(WINDOW w, int clip_p)
{
  struct gstate *p = gsp++;

  assert(p < graphics_save_stack + SAVE_STACK_SIZE);
  p->color = getcolor();
  getfillsettings(&p->fs);
  getlinesettings(&p->ls);
  gettextsettings(&p->ts);
  getviewsettings(&p->vs);
  setviewport(w->x, w->y, w->x + w->width - 1, w->y + w->height - 1, clip_p);
}

void pop_graphics_state(void)
{
  struct gstate *p = --gsp;
  struct fillsettingstype *fs = &p->fs;
  struct linesettingstype *ls = &p->ls;
  struct textsettingstype *ts = &p->ts;
  struct viewporttype *vs = &p->vs;

  assert(p >= graphics_save_stack);
  setcolor(p->color);
  setfillstyle(fs->pattern, fs->color);
  setlinestyle(ls->linestyle, ls->upattern, ls->thickness);
  settextjustify(ts->horiz, ts->vert);
  settextstyle(ts->font, ts->direction, ts->charsize);
  setviewport(vs->left, vs->top, vs->right, vs->bottom, vs->clip);
}

/* Local abbreviation. */
typedef unsigned char byte;

/* OR the inverse of the source into the destination. */
LOCAL(void) meminvor(byte *dst, byte *src, int n)
{
  while (n-- > 0) *dst++ |= ~*src++;
}

#ifdef NOT_USED

/* Return a copy of the input bitmap that is subdued by
   zeroing out some pixels. */
void *make_subdued_bitmap(void *void_bits, int width, int height)
#define bits ((byte*)void_bits)
{
  int i, n, n_bytes_in_row = (width + 7) >> 3;
  byte mask, *result, *p, *q;

  result = malloc(n_bytes_in_row * height);

  for (i = 0, p = bits, q = result;
       i < height;
       ++i, p += n_bytes_in_row, q += n_bytes_in_row) {
    mask = (i & 1) ? 0xaa : 0x55;
    for (n = n_bytes_in_row; n; --n)
      *q++ = *p++ & mask;
  }
  return result;
}
#undef bits

#endif

/* Accept a bit map and change it into an image that
   can be written to the screen with put-image. 1's
   go to foreground color, 0's to background.
   Return value is malloc()'ed. */
void *make_image(void *void_bits, int width, int height, int fg, int bg)
#define bits ((byte*)void_bits)
{
  int c, n_planes, n_bytes_in_line, bit;
  unsigned size;
  byte *image, *ip, *bp;

  for (c = getmaxcolor(), n_planes = 0; c; c >>= 1, ++n_planes)
    ;
  n_bytes_in_line = (width + 7) >> 3;
  size = 6 + n_bytes_in_line * n_planes * height;
  assert(size == imagesize(1, 1, width, height));
  image = malloc(size);

  ((int*)image)[0] = width - 1;
  ((int*)image)[1] = height - 1;
  for (ip = image+4, bp = bits; height; --height, bp += n_bytes_in_line)
    for (bit = 1 << (n_planes-1); bit; ip += n_bytes_in_line, bit >>= 1) {
      if (fg & bit)
	memcpy(ip, bp, n_bytes_in_line);
      else
	memset(ip, 0, n_bytes_in_line);
      if (bg & bit)
	meminvor(ip, bp, n_bytes_in_line);
      assert(ip + n_bytes_in_line < image + size);
    }

  return (void*)image;
}

#undef bits

/* Return a `pseudo-arctangent' of the direction given by dy and dx,
   i.e. t, 0 <= t <= 8scale, where t is really a scaled slope or inverse 
   slope offset an appropriate amount. */
LOCAL(int) patan2(int dy, int dx, int scale)
{
  /* Lower quadrants are reflections of upper. */
  if (dy < 0) {
    int t = patan2(-dy, dx, scale);
    return t ? 8 * scale - t : 0;
  }

  /* Left top is reflection of right top. */
  if (dx < 0) 
    return 4 * scale - patan2(dy, -dx, scale);

  /* Right top is base case. */
  return (dx < dy) ? 2 * scale - scale_int(dx, scale, dy) : scale_int(dy, scale, dx);
}

/* Draw text with an underscore at hot char position (if there is one). */
void out_hot_textxy(int x, int y, HOT_TEXT text)
{
  int i = text->hot_char;

  outtextxy(x, y, text->str);
  if (i != NO_HOT_CHAR && i != HOT_SPACE && i < strlen(text->str)) {
    /* Write an underscore at the hot char position. */
    char buf[80];
    int j = 0;
    while (text->str[j]) {
      buf[j] = (j == i) ? '_' : ' ';
      ++j;
    }
    buf[j] = '\0';
    outtextxy(x, y + 1, buf);
  }
}

/* Table of arrow head offsets by pseudo-atan2. */
#include "ahdtbl.c"
#define AHOFS_BASE (sizeof ahofs/sizeof ahofs[0]/8)

/* Draw an arrowhead at x1,y1 assuming x0,y0 is
   the tail of the arrow. */
void far arrowhead(int x0, int y0, int x1, int y1)
{
  int dx = x1 - x0, dy = y1 - y0;

  if (dx || dy) {
    int i = patan2(dy, dx, AHOFS_BASE);
    struct ahofs *p = &ahofs[i];
    line(x1 + p->x1, y1 + p->y1, x1, y1);
    line(x1 + p->x2, y1 + p->y2, x1, y1);
  }
}

/* This is embarrassingly slow, but it XORs. */
LOCAL(void) dot(int x, int y)
{
  line(x, y, x, y);
}

/* Draw dot (x,y) at each of 4
   quadrant rotations around (x0,y0). */
#define dot4(x0, y0, x, y) \
  if (mask & 1) dot(x0+x,y0+y); \
  if (mask & 2) dot(x0-y,y0+x); \
  if (mask & 4) dot(x0-x,y0-y); \
  if (mask & 8) dot(x0+y,y0-x)

/* Draw quadrants of a circle touching each dot 
   exactly once and calling line() so XOR_PUT works. 
   Bits of mask decide which quadrants are drawn:
   1st thru 4th in CCW order from x-axis are bits
   0 to 3 respectively. */
void arcs(int x0, int y0, int r, unsigned char mask)
{
  int x, y, e, u, v;

  e = 1 - r;
  u = 1;
  v = e - r;
  for (x = 0, y = r; x < y; ++x) {
    dot4(x0, y0, x, y);
    u += 2;
    if (e < 0) {
      v += 2;
      e += u;
    }
    else {
      v += 4;
      e += v;
      --y;
    }
  }

  e -= x + y;
  u = 1 - y - y;

  for ( ; y > 0; --y) {
    dot4(x0, y0, x, y);
    u += 2;
    if (e > 0) {
      v += 2;
      e += u;
    }
    else {
      v += 4;
      e += v;
      ++x;
    }
  }
}

/* Draw all or part of an oval that fits in the rectangle with diagonal
   (x0,y0)--(x1,y1).  The mask tells which corners are drawn.  The
   edges adjacent to all corners are also drawn.  Bits are mapped as
   follows:
		  ____                   ____
      Bit 0 (1):      \      Bit 1 (2): /
		      |                 |
		  |                         |
      Bit 2 (4):  \___       Bit 3 (8):  ___/

   The value of `r' is the _maximum_ allows radius of corners.  If
   dx or dy is too small, this is shrunk so the respective corners
   are still quarter-circles.  Pixel painting is done with care so 
   each pixel is touched exactly once.  Thus XOR mode writes are ok. */

/* Directional lines to prevent drawing oval points twice. */
LOCAL(void) right(int x0, int x1, int y) { if (x0 <= x1) line(x0, y, x1, y); }
LOCAL(void) up(int x, int y0, int y1) { if (y0 <= y1) line(x, y0, x, y1); }

/* Make far for compatibility with BGI. */
void far oval(int x0, int y0, int x1, int y1, int r, unsigned mask)
{
  int xr, yr, t;

  if (x0 > x1) { t = x0; x0 = x1; x1 = t; }
  if (y0 > y1) { t = y0; y0 = y1; y1 = t; }

  xr = x1 - x0;
  yr = y1 - y0;

  if (((mask & 3) == 3) || ((mask & 12) == 12))
    xr >>= 1;

  if (((mask & 9) == 9) || ((mask & 6) == 6))
    yr >>= 1;

  r = min(r, min(xr, yr));

  if (mask & 1) {
    arcs(x1 - r, y1 - r, r, 1);
    right((mask & 2) ? x0 + r : x0, x1 - r - 1, y1);
    if ((mask & 8) == 0) 
      up(x1, y0, y1 - r);
  }
  if (mask & 2) {
    arcs(x0 + r, y1 - r, r, 2);
    up(x0, (mask & 4) ? y0 + r : y0, y1 - r - 1);
    if ((mask & 1) == 0) 
      right(x0 + r, x1, y1);
  }
  if (mask & 4) {
    arcs(x0 + r, y0 + r, r, 4);
    right(x0 + r + 1, (mask & 8) ? x1 - r : x1, y0);
    if ((mask & 2) == 0) 
      up(x0, y0 + r, y1);
  }
  if (mask & 8) {
    arcs(x1 - r, y0 + r, r, 8);
    up(x1, y0 + r + 1, (mask & 1) ? y1 - r : y1);
    if ((mask & 4) == 0) 
      right(x0, x1 - r, y0);
  }
}

/* Handy arithmetic. */
int max(int x, int y) { return x > y ? x : y; }
int min(int x, int y) { return x < y ? x : y; }
void neg(int *i) { *i = -(*i); }
void swap(int *x, int *y) { int t = *x; *x = *y; *y = t; }
#ifdef NOT_USED
long lmax(long x, long y) { return x > y ? x : y; }
long lsqr(long x) { return x * x; }
#endif
LOCAL(long) lmin(long x, long y) { return x < y ? x : y; }
#pragma warn -rvl
long sqr(int x) { asm { mov ax, x; imul word ptr x; } }
#pragma warn .rvl
long sum_sqr(int x, int y) { return sqr(x) + sqr(y); }

/* Return the greatest integer i s.t. i^2 <= x. */
int isqrt(long x)
{
  int n;
  long t, q, r;

  assert(0 <= x && x <= (0x7fffl * 0x7fffl));

  if (x <= 1)
    return x;

  for (t = x, n = 0; t >>= 2; ++n) /* skip */ ;
  r = x >> n;

  while ((q = x / r) < r)
    r = (r + q) >> 1;

  return r;
}


/* Return floor of min distance from point (x,y) to 
   segment (x0,y0)---(x1, y1).  I.e. use perpendicular
   distance if the perpendicular intersects the segment, 
   else the distance to the nearest endpoint. */
int pt_seg_dist(int x, int y, int x0, int y0, int x1, int y1)
{
  long dx = x1 - x0, dy = y1 - y0;

  return (dx*(x - x1) + dy*(y - y1) < 0 && 0 < dx*(x - x0) + dy*(y - y0)) ?
    labs( dy*(x - x0) + dx*(y0 - y) ) / isqrt(dx * dx + dy * dy) :
    isqrt(lmin(sum_sqr(x - x0, y - y0), sum_sqr(x - x1, y - y1)));
}

LOCAL(int) vh_dist(int c, int d, int c01, int d0, int d1)
{
  if (d < d0)
    return isqrt(sum_sqr(c - c01, d - d0));
  else if (d > d1)
    return isqrt(sum_sqr(c - c01, d - d1));
  else
    return abs(c - c01);
}

/* Return distance from point to nearest 
   point on box with given diagonal. */
int pt_box_dist(int x, int y, int x0, int y0, int x1, int y1)
{
  int t;
  long d, dp;

  if (x0 > x1) { t = x0; x0 = x1; x1 = t; }
  if (y0 > y1) { t = y0; y0 = y1; y1 = t; }

  d = vh_dist(x, y, x0, y0, y1);

  dp = vh_dist(x, y, x1, y0, y1);
  if (dp < d) d = dp;

  dp = vh_dist(y, x, y0, x0, x1);
  if (dp < d) d = dp;

  dp = vh_dist(y, x, y1, x0, x1);
  if (dp < d) d = dp;

  return d;
}

/* Return the shortest distance to any point on the
   given oval or part of oval.  Conventions same as
   for drawing function above. */
int pt_oval_dist(int x, int y, int x0, int y0, int x1, int y1, int r, unsigned mask)
{
  int xr, yr, t;
  long dx0, dy0, dx1, dy1;

  if (x0 > x1) { t = x0; x0 = x1; x1 = t; }
  if (y0 > y1) { t = y0; y0 = y1; y1 = t; }

  xr = x1 - x0;
  yr = y1 - y0;

  if (((mask & 3) == 3) || ((mask & 12) == 12))
    xr >>= 1;

  if (((mask & 9) == 9) || ((mask & 6) == 6))
    yr >>= 1;

  r = min(r, min(xr, yr));

  dx0 = (x0 + r) - x;
  dy0 = (y0 + r) - y;
  dx1 = x - (x1 - r);
  dy1 = y - (y1 - r);

  /* Take distance to circle in corners, else distance to box. */
  if ((mask & 1) && dx1 > 0 && dy1 > 0)
    return abs(r - isqrt(dx1*dx1 + dy1*dy1));

  if ((mask & 2) && dx0 > 0 && dy1 > 0)
    return abs(r - isqrt(dx0*dx0 + dy1*dy1));

  if ((mask & 4) && dx0 > 0 && dy0 > 0)
    return abs(r - isqrt(dx0*dx0 + dy0*dy0));

  if ((mask & 8) && dx1 > 0 && dy0 > 0)
    return abs(r - isqrt(dx1*dx1 + dy0*dy0));

  /* Check right if tr or br corner in oval. */
  r = (mask & 9) ? vh_dist(x, y, x1, y0, y1) : MAXINT;

  /* Check top for tr or tl. */
  if (mask & 3)
    r = min(r, vh_dist(y, x, y1, x0, x1));

  /* Check left for tl or bl. */
  if (mask & 6)
    r = min(r, vh_dist(x, y, x0, y0, y1));

  /* Check bot for bl or br. */
  if (mask & 12)
    r = min(r, vh_dist(y, x, y0, x0, x1));

  return r;
}

/* Return non-0 iff (x,y) is in the box with diagonal x0,y0 x1,y1 */
int in_box_p(int x, int y, int x0, int y0, int x1, int y1)
{
  int t;
  if (x0 > x1) { t = x0; x0 = x1; x1 = t; }
  if (y0 > y1) { t = y0; y0 = y1; y1 = t; }
  return x >= x0 && x <= x1 && y >= y0 && y <= y1;
}

#ifdef NOT_USED

/* Return non-0 iff (x,y) is in the circle 
   with center (x0,y0) and radius r. */
int in_circle_p(int x, int y, int x0, int y0, int r)
{
  return sum_sqr(x - x0, y - y0) < (long)r * r;
}

#endif
