package com.jaspersoft.jasperserver.war.cascade;

import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.common.domain.impl.ExecutionContextImpl;
import com.jaspersoft.jasperserver.api.engine.common.service.EngineService;
import com.jaspersoft.jasperserver.api.engine.common.service.ReportInputControlsInformation;
import com.jaspersoft.jasperserver.api.engine.jasperreports.service.impl.ReportLoadingService;
import com.jaspersoft.jasperserver.api.metadata.common.domain.InputControl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.InputControlsContainer;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Query;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ResourceReference;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.ReportUnit;
import com.jaspersoft.jasperserver.war.cascade.cache.ControlLogicCacheManager;
import com.jaspersoft.jasperserver.war.cascade.cache.SessionCache;
import com.jaspersoft.jasperserver.war.cascade.token.FilterResolver;
import org.apache.commons.collections.OrderedMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;

/**
 * Provides cached access to some methods of EngineService.
 * @author Anton Fomin
 * @version $Id: CachedEngineService.java 24202 2012-06-21 13:10:48Z afomin $
 */
@Service
public class CachedEngineService {

    private static final Log log = LogFactory.getLog(CachedEngineService.class);

    @Resource
    private ControlLogicCacheManager controlLogicCacheManager;

    @Resource
    private CachedRepositoryService cachedRepositoryService;

    @Resource
    protected EngineService engineService;

    @Resource
    private FilterResolver filterResolver;

    @javax.annotation.Resource
    private ReportLoadingService reportLoadingService;

    @Resource(name = "inputControlsCachingEnabled")
    private boolean doCache;

    public ReportInputControlsInformation getReportInputControlsInformation(ExecutionContext exContext, InputControlsContainer container, Map<String, Object> initialParameters) {
        final SessionCache sessionCache = controlLogicCacheManager.getSessionCache();
        ReportInputControlsInformation infos = sessionCache.getCacheInfo(ReportInputControlsInformation.class, container.getURI());
        if (!doCache || infos == null) {
            infos = engineService.getReportInputControlsInformation(exContext, container, initialParameters);
            sessionCache.setCacheInfo(ReportInputControlsInformation.class, container.getURI(), infos);
        }
        return infos;
    }

    public List<InputControl> getInputControls(InputControlsContainer container) throws CascadeResourceNotFoundException {
        final SessionCache sessionCache = controlLogicCacheManager.getSessionCache();
        // Cache Input Controls as List class
        List<InputControl> inputControls = sessionCache.getCacheInfo(List.class, container.getURI());
        if (!doCache || inputControls == null) {
            inputControls = reportLoadingService.getInputControls(ExecutionContextImpl.getRuntimeExecutionContext(), container);
            sessionCache.setCacheInfo(List.class, container.getURI(), inputControls);
        }
        return inputControls;
    }

    public ReportInputControlsInformation getReportInputControlsInformation(ReportUnit reportUnit) {
        return getReportInputControlsInformation(ExecutionContextImpl.getRuntimeExecutionContext(), reportUnit, null);
    }

    public OrderedMap executeQuery(ExecutionContext context, ResourceReference queryReference, String keyColumn, String[] resultColumns,
			ResourceReference defaultDataSourceReference, Map<String, Object> parameterValues, Map<String, Class<?>> parameterTypes, /* Temporary for test */String controlName)
            throws CascadeResourceNotFoundException {

        final Query query = cachedRepositoryService.getResource(Query.class, queryReference);
        final String cacheKey = String.valueOf(filterResolver.getCacheKey(query.getSql(), parameterValues));
        final SessionCache sessionCache = controlLogicCacheManager.getSessionCache();
        OrderedMap results = sessionCache.getCacheInfo(OrderedMap.class, cacheKey);
        if (!doCache || results == null) {
            results = engineService.executeQuery(context != null ? context : ExecutionContextImpl.getRuntimeExecutionContext(),
                    queryReference, keyColumn, resultColumns, defaultDataSourceReference, parameterValues, parameterTypes, false);
            sessionCache.setCacheInfo(OrderedMap.class, cacheKey, results);
            log.debug("Database query \"" + cacheKey + "\"");
        } else {
            log.debug("Cached query \"" + cacheKey + "\"");
        }
        return results;
    }

    public OrderedMap executeQuery(ExecutionContext context, ResourceReference queryReference, String keyColumn, String[] resultColumns,
			ResourceReference defaultDataSourceReference, Map<String, Object> parameterValues, /* Temporary for test */String controlName)
            throws CascadeResourceNotFoundException {
        return executeQuery(context, queryReference, keyColumn, resultColumns, defaultDataSourceReference, parameterValues, null, controlName);
    }


    public ControlLogicCacheManager getControlLogicCacheManager() {
        return controlLogicCacheManager;
    }

    public void setControlLogicCacheManager(ControlLogicCacheManager controlLogicCacheManager) {
        this.controlLogicCacheManager = controlLogicCacheManager;
    }

    public EngineService getEngineService() {
        return engineService;
    }

    public void setEngineService(EngineService engineService) {
        this.engineService = engineService;
    }

    public boolean isDoCache() {
        return doCache;
    }

    public void setDoCache(boolean doCache) {
        this.doCache = doCache;
    }

    public void setCachedRepositoryService(CachedRepositoryService cachedRepositoryService) {
        this.cachedRepositoryService = cachedRepositoryService;
    }
}
