/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.jaspersoft.jasperserver.war.control;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.springframework.context.MessageSource;
import org.springframework.context.ResourceLoaderAware;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.core.io.ResourceLoader;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.mvc.Controller;


public class LogSettingsController implements Controller, ResourceLoaderAware {
    private static Logger log = Logger.getLogger(LogSettingsController.class);

    private Map<Locale, Map<String, String>> loggerDescriptionsByLocale =
            Collections.synchronizedMap(new HashMap<Locale, Map<String, String>>());
    private Map<String, String> loggers = Collections.synchronizedMap(new LinkedHashMap<String, String>());

    private List<String> loggerDescriptionFiles;
    private ResourceLoader resourceLoader;
    private MessageSource messageSource;

    public ModelAndView handleRequest(HttpServletRequest request, HttpServletResponse response) throws Exception {
        String loggerArg = request.getParameter("logger");
        String levelArg = request.getParameter("level");

        initLoggers();
        initLoggerDescriptions();

        if (loggerArg != null && levelArg != null) {
            Logger log = Logger.getLogger(loggerArg);
            log.setLevel(Level.toLevel(levelArg));
            loggers.put(loggerArg, levelArg);
        }

        for (String l : loggers.keySet()) {
            Logger log = Logger.getLogger(l);
            Level level = log.getEffectiveLevel();
            loggers.put(l, level.toString());
        }
        ModelAndView mav = new ModelAndView("modules/administer/logSettings");
        mav.addObject("loggers", loggers);
        mav.addObject("loggerDesc", loggerDescriptionsByLocale.get(LocaleContextHolder.getLocale()));
        return mav;
    }
    
    private void initLoggers() {
        if (loggers.isEmpty()) {
            if (loggerDescriptionFiles == null || loggerDescriptionFiles.isEmpty()) {
                throw new IllegalStateException("No logger description files specified.");
            }

            for (String file : loggerDescriptionFiles) {
                try {
                    String props = loadWebappFile(file);

                    if (props != null) {
                        Properties p = new Properties();
                        p.load(new ByteArrayInputStream(props.getBytes()));

                        for (Object key : p.keySet()) {
                            loggers.put((String) key, null);
                        }
                    }
                } catch (IOException e) {
                    log.warn("problem loading log descriptions", e);
                }
            }
        }
    }

    private void initLoggerDescriptions() {
        Locale locale = LocaleContextHolder.getLocale();

        if (!loggerDescriptionsByLocale.containsKey(locale)) {
            Map<String, String> loggerDescriptions = new LinkedHashMap<String, String>();

            for (String key : loggers.keySet()) {
                loggerDescriptions.put(key, messageSource.getMessage(key, null, "", locale));
            }

            loggerDescriptionsByLocale.put(locale, loggerDescriptions);
        }
    }

    /**
     * Loads a file located somewhere in the webapp folder.
     *
     * @param path the path to the file in the webapp folder.
     *
     * @return the content of the file as a string.
     *
     * @throws IOException exception when loading file.
     */
    private String loadWebappFile(String path) throws IOException {
        // use resource loader to look up something in servlet context
        org.springframework.core.io.Resource test = resourceLoader.getResource(path);
        InputStream is = test.getInputStream();
        // read it into a string and return it
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        byte[] bytes = new byte[1024];
        int n;
        while ((n = is.read(bytes)) > 0) {
            baos.write(bytes, 0, n);
        }
        return baos.toString();
    }

    /* (non-Javadoc)
      * @see org.springframework.context.ResourceLoaderAware#setResourceLoader(org.springframework.core.io.ResourceLoader)
      */
    public void setResourceLoader(ResourceLoader resourceLoader) {
        this.resourceLoader = resourceLoader;
    }

    public void setLoggerDescriptionFiles(List<String> loggerDescriptionFiles) {
        this.loggerDescriptionFiles = loggerDescriptionFiles;
    }

    public void setMessageSource(MessageSource messageSource) {
        this.messageSource = messageSource;
    }
}
