/*
* Copyright (C) 2005 - 2009 Jaspersoft Corporation. All rights  reserved.
* http://www.jaspersoft.com.
*
* Unless you have purchased  a commercial license agreement from Jaspersoft,
* the following license terms  apply:
*
* This program is free software: you can redistribute it and/or  modify
* it under the terms of the GNU Affero General Public License  as
* published by the Free Software Foundation, either version 3 of  the
* License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Affero  General Public License for more details.
*
* You should have received a copy of the GNU Affero General Public  License
* along with this program.&nbsp; If not, see <http://www.gnu.org/licenses/>.
*/
package com.jaspersoft.jasperserver.remote.services;

import com.jaspersoft.jasperserver.api.engine.common.service.ReportExecutionStatusInformation;
import com.jaspersoft.jasperserver.api.engine.common.service.SchedulerReportExecutionStatusSearchCriteria;
import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.OperationResult;
import com.jaspersoft.jasperserver.remote.ServiceException;
import com.jaspersoft.jasperserver.remote.exception.RemoteException;
import com.jaspersoft.jasperserver.remote.exception.ResourceNotFoundException;
import com.jaspersoft.jasperserver.remote.services.impl.ReportOutputResource;
import com.jaspersoft.jasperserver.war.dto.InputControlState;
import com.jaspersoft.jasperserver.war.dto.ReportInputControl;
import net.sf.jasperreports.engine.JasperPrint;

import javax.activation.DataSource;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Facade service to run reports
 *
 * @author Yaroslav.Kovalchyk
 * @version $Id: RunReportService.java 23655 2012-05-08 12:18:11Z lchirita $
 */
public interface RunReportService {

    /**
     * @return map of currently available input attachments
     */
    Map<String, DataSource> getInputAttachments();

    /**
     *
     * @return map of currently available ouput attachments
     */
    Map<String, DataSource> getOutputAttachments();


    /**
     * Return a response.
     * Generated files (one or more) are put in the output attachments map of this context
     *
     * @param reportUnitURI - target report to run URI
     * @param parameters - report parameters
     * @param arguments - report arguments
     * @return result of operation
     * @throws ServiceException - trown in case of unexpected errors
     */
    OperationResult runReport(String reportUnitURI, Map<String, Object> parameters, Map<String, String> arguments) throws ServiceException;

    /**
     * Export the report in a specific format using the specified arguments
     * Generated files (one or more) are put in the output attachments map of this context
     *
     * @param reportUnitURI - target report to run URI
     * @param jasperPrint - JasperPring object
     * @param arguments indicates the final file format, starting/ending pages, etc...
     * @return result of operation
     * @throws ServiceException - trown in case of unexpected errors
     */
    OperationResult exportReport(String reportUnitURI, JasperPrint jasperPrint, Map<String, String> arguments) throws ServiceException;

    /**
     * @return  map of currently available attributes
     */
    Map<String, Object> getAttributes();

    /**
     * All input controls of a report can be retrieved via this method.
     *
     *
     * @param reportUnitURI - URI of the report
     * @param inputControlIds - list of requested input control ID. Empty list or null means "ALL"
     * @param rawParameters
     * @return list of report input controls or null if not found.
     * @throws ResourceNotFoundException - thrown in case of report unit haven't found
     */
    List<ReportInputControl> getInputControlsForReport(String reportUnitURI, Set<String> inputControlIds, Map<String, String[]> rawParameters) throws ResourceNotFoundException;

    /**
     * Input controls states are accessible via this method.
     *
     * @param reportUnitUri - URI of the report unit
     * @param inputControlIds - list of input control ID, values for which are needed. Empty list or null means "ALL"
     * @param parameters - map of raw selected values
     * @return list of input control states
     * @throws ResourceNotFoundException in case if input controls container resource (URI = reportUnitUri) not found
     */
    List<InputControlState> getValuesForInputControls(String reportUnitUri, Set<String> inputControlIds, Map<String, String[]> parameters) throws ResourceNotFoundException;

    /**
     * Possibility to run report and get its output resource via the only call
     *
     *
     * @param reportUnitURI - URI of the report to run
     * @param outputFormat - output format
     * @param ignorePagination - if true, then pagination is ignored
     * @param page - page number, if null - all pages
     * @param transformerKey - transformer key
     * @param rawParameters - map with parameters in raw state(raw mean, that all parameters are strings or arrays of strings)
     * @param avoidCache - if true, then report will be regenerated     @return resource in corresponding output format
     * @param freshData - if true, ignore existing data snapshots
     * @param saveDataSnapshot - if true, save a data snapshot with the current data
     * @throws RemoteException if any error occurs
     */
    ReportOutputResource getReportOutputFromRawParameters(String reportUnitURI, String outputFormat, Boolean ignorePagination, Integer page, String transformerKey,
                                                          Map<String, String[]> rawParameters, Boolean avoidCache, 
                                                          Boolean freshData, Boolean saveDataSnapshot) throws RemoteException;

    /**
     * Report output items (e.g. images) for specific report can be accessed via this method.
     *
     * @param itemName - name of the report output item
     * @param cacheKey - cache key of a generated report
     * @return the report output item resource
     * @throws RemoteException if any error occurs
     */
    public ReportOutputResource getReportItem(String itemName, Integer cacheKey) throws RemoteException;

    /**
     * Search for currently running reports
     *
     * @param searchCriteria - search criteria
     * @return set of currently running report's information
     * @throws RemoteException if any error occurs
     */
    public Set<ReportExecutionStatusInformation> getCurrentlyRunningReports(SchedulerReportExecutionStatusSearchCriteria searchCriteria)
            throws RemoteException;

    /**
     * Report execution cancellation.
     *
     *
     * @param requestId - request ID
     * @throws RemoteException if any error occurs
     */
    public Boolean cancelReportExecution(String requestId) throws RemoteException;

}
