/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.ant;

import java.io.File;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map.Entry;

import org.apache.log4j.Logger;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;

import de.uni_paderborn.fujaba.uml.UMLClass;
import de.uni_paderborn.fujaba.uml.UMLFile;
import de.uni_paderborn.fujaba.uml.UMLPackage;
import de.uni_paderborn.fujaba.uml.UMLProject;
import de.uni_paderborn.fujaba.uml.UMLType;
import de.uni_paderborn.fujaba.uml.UMLTypeList;
import de.upb.tools.sdm.Path;


/**
 * try to repair <code>UMLProject</code> file <code>srcFileName</code> and
 * save it as <code>targetFileName</code>.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.1.2.2 $
 */
public abstract class AbstractRepairUMLProjectTask extends Task
{

   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (AbstractRepairUMLProjectTask.class);

   /**
    * Path from UMLProject to UMLCLASS.
    */
   private final static String ALL_UMLCLASS = "rootPackage.packages*.declares";

   /**
    * the name of the sourcefile .
    */
   private String srcFileName = null;

   /**
    * the name of the target file
    */
   private String targetFileName = null;

   /**
    * the <code>UMLProject</code> instance.
    */
   private UMLProject umlProject = null;


   /**
    * Creates a new AbstractRepairUMLProjectTask object.
    */
   public AbstractRepairUMLProjectTask()
   {
      super();
   }


   /**
    * @see                    org.apache.tools.ant.Task#execute()
    * @throws BuildException
    */
   public void execute() throws BuildException
   {
      validate();

      loadProject();

      repairProject();

      saveProject();
   }


   /**
    * DOCUMENT ME!
    */
   public void validate()
   {
      if (getSrcFileName() == null)
      {
         throw new BuildException (
            "srcFileName is missing !");
      }

      if (getTargetFileName() == null)
      {
         throw new BuildException (
            "targetFileName is missing !");
      }

      File srcFile = new File (getSrcFileName());

      if (!srcFile.exists())
      {
         throw new BuildException ("srcFileName: '"
            + srcFile.getAbsolutePath() + "' does not exist !");
      }

      File targetFile = new File (getTargetFileName());

      if (targetFile.exists())
      {
         throw new BuildException ("targetFileName: '"
            + targetFile.getAbsolutePath() + "' already exists !");
      }
   }


   /**
    * DOCUMENT ME!
    */
   public abstract void loadProject();


   /**
    * DOCUMENT ME!
    */
   public abstract void saveProject();


   /**
    * DOCUMENT ME!
    */
   public void repairProject()
   {

      repairUMLType();

      repairUMLFile();

      repairTypelist();
   }


   private void repairTypelist()
   {
      UMLTypeList typeList = this.umlProject.getTypeList();

      List theList = new LinkedList();

      Iterator iter = typeList.entriesOfTypes();

      while (iter.hasNext())
      {
         Entry element = (Entry) iter.next();

         if (log.isInfoEnabled())
         {
            log.info (this + ".analyze key='" + element.getKey()
               + "' value='" + element.getValue() + "'");
         }

         theList.add (element);

      }

      iter = theList.iterator();
      while (iter.hasNext())
      {
         Entry tmpEntry = (Entry) iter.next();
         String typeName = (String) tmpEntry.getKey();
         UMLType tmpType = (UMLType) tmpEntry.getValue();

         String name =  (tmpType instanceof UMLClass ?  ((UMLClass) tmpType)
            .getFullClassName() : tmpType.getName());

         if (log.isInfoEnabled())
         {
            log.info (this + ".updateing type " + name);
         }

         typeList.removeKeyFromTypes (typeName);

         typeList.addToTypes (tmpType);
      }

      if (log.isInfoEnabled())
      {
         log.info (this + ".finished()");
      }

   }


   /**
    * remove empty <code>UMLFile</code>
    */
   private void repairUMLFile()
   {
      List toBeRemoved = new LinkedList();

      Iterator iter = this.umlProject.iteratorOfFiles();

      while (iter.hasNext())
      {
         UMLFile element = (UMLFile) iter.next();

         if (log.isInfoEnabled())
         {
            log.info (this + ".analyzing " + element);
         }

         if (element.sizeOfContains() == 0)
         {
            if (log.isInfoEnabled())
            {
               log.info (this + ".--> must remove " + element);
            }

            if (!toBeRemoved.contains (element))
            {
               toBeRemoved.add (element);
            }

         }
      }

      iter = toBeRemoved.iterator();

      while (iter.hasNext())
      {
         UMLFile element = (UMLFile) iter.next();

         if (log.isInfoEnabled())
         {
            log.info (this + ".removing " + element + ",fujabaId="
               + element.getID() + ",hashCode=" + element.hashCode());
         }

         element.removeYou();
      }
   }


   private void repairUMLType()
   {

      List toBeRemoved = new LinkedList();

      // analyze declared classes
      analyzeDeclaredClasses (toBeRemoved);

      // analyze types

      analyzeTypes (toBeRemoved);

      // analyze files
      analyzeFiles (toBeRemoved);

      // remove found types
      removeClasses (toBeRemoved);
   }


   /**
    * @param toBeRemoved
    */
   private void removeClasses (List toBeRemoved)
   {
      Iterator iter = toBeRemoved.iterator();

      while (iter.hasNext())
      {
         UMLClass tmpClass = (UMLClass) iter.next();

         if (log.isInfoEnabled())
         {
            log.info (this + ".removing " + tmpClass + ",fujabaId="
               + tmpClass.getID() + ",hashCode="
               + tmpClass.hashCode());
         }

         // if tmpClass has declares, then set declares.package =
         // tmpClass.package
         Iterator declaresIter = tmpClass.iteratorOfDeclares();
         while (declaresIter.hasNext())
         {
            UMLClass declaresClass = (UMLClass) declaresIter.next();

            if (declaresClass.getDeclaredInPackage() == null)
            {
               UMLPackage tmpPackage = tmpClass.getDeclaredInPackage();

               if (tmpPackage == null)
               {
                  // ok , tmpClass has no package , then use rootpackage
                  tmpPackage = this.umlProject.getRootPackage();

                  if (tmpPackage == null)
                  {
                     throw new BuildException (
                        "rootPackage is null ! you have a problem ;-)");
                  }

               }
               if (log.isInfoEnabled())
               {
                  log.info (this + ".setting '" + declaresClass
                     + "'.declaredInPackage = '" + tmpPackage + "'");
               }

               declaresClass.setDeclaredInPackage (tmpPackage);

               // create new file if necessary
               UMLFile tmpFile = declaresClass.getFile();
               if (tmpFile == null)
               {
                  tmpFile = new UMLFile();
                  tmpFile.addToContains (declaresClass);
                  tmpFile.setProject (this.umlProject);
                  tmpFile.setName (declaresClass.getName());
               }

            }
         }

         tmpClass.removeYou();
      }
   }


   /**
    * @param toBeRemoved
    */
   private void analyzeFiles (List toBeRemoved)
   {
      Path pathToUMLClass = new Path (this.umlProject, "files.contains");

      analyzePath (toBeRemoved, pathToUMLClass);
   }


   /**
    * @param toBeRemoved
    */
   private void analyzeTypes (List toBeRemoved)
   {
      Iterator iter = this.umlProject.getTypeList().iteratorOfTypes();
      while (iter.hasNext())
      {
         UMLType tmpType = (UMLType) iter.next();

         if (log.isInfoEnabled())
         {
            log.info (this + ".analyzing '" + tmpType + "'");
         }

         if (tmpType instanceof UMLClass
            && isPhantomClass ((UMLClass) tmpType))
         {
            if (log.isInfoEnabled())
            {
               log.info (this + ".--> must remove " + tmpType);
            }

            addToBeRemoved (toBeRemoved, tmpType);
         }
      }
   }


   /**
    * @param toBeRemoved
    * @param tmpType
    */
   private void addToBeRemoved (List toBeRemoved, UMLType tmpType)
   {
      if (!toBeRemoved.contains (tmpType))
      {
         toBeRemoved.add (tmpType);
      }
   }


   /**
    * @param toBeRemoved
    */
   private void analyzeDeclaredClasses (List toBeRemoved)
   {
      Path pathToUMLClass = new Path (this.umlProject, ALL_UMLCLASS);

      analyzePath (toBeRemoved, pathToUMLClass);
   }


   /**
    * @param toBeRemoved
    * @param pathToUMLClass
    */
   private void analyzePath (List toBeRemoved, Path pathToUMLClass)
   {
      while (pathToUMLClass.hasMoreElements())
      {
         UMLClass tmpClass = (UMLClass) pathToUMLClass.nextElement();

         if (log.isInfoEnabled())
         {
            log.info (this + ".analyzing '" + tmpClass + "'");
         }

         if (isPhantomClass (tmpClass))
         {
            if (log.isInfoEnabled())
            {
               log.info (this + ".--> must remove " + tmpClass);
            }

            addToBeRemoved (toBeRemoved, tmpClass);
         }
      }
   }


   private boolean isPhantomClass (UMLClass tmpClass)
   {
      if (log.isInfoEnabled())
      {
         log.info (this + ".isPhantomClass(" + tmpClass + ")");
         if (tmpClass.getDeclaredInClass() != null)
         {
            log.info (this + ".class '" + tmpClass + "' declared in class '"
               + tmpClass.getDeclaredInClass() + "'");
            log.info (this + ".class '" + tmpClass
               + "' declared in package '"
               + tmpClass.getDeclaredInPackage() + "'");
         }
      }

      if (!tmpClass.getName().startsWith ("_@id"))
      {
         return false;
      }

      // tmpClass has elements ?
      if ( (tmpClass.sizeOfAttrs() + tmpClass.sizeOfMethods()
         + tmpClass.sizeOfRoles() + tmpClass.sizeOfSignals()) > 0)
      {
         return false;
      }

      // tmpclass used as type ?
      if ( (tmpClass.sizeOfRevAttrType() + tmpClass.sizeOfRevParamType() + tmpClass.sizeOfRevResultType()) > 0)
      {
         return false;
      }

      // tmpclass used in inheritance ?
      if (tmpClass.getSuperClass() != null)
      {
         return false;
      }

      return true;
   }


   /**
    * @return   Returns the srcFileName.
    */
   public String getSrcFileName()
   {
      return this.srcFileName;
   }


   /**
    * @param srcFileName  The srcFileName to set.
    */
   public void setSrcFileName (String srcFileName)
   {
      if (this.srcFileName != srcFileName)
      {
         this.srcFileName = srcFileName;
      }
   }


   /**
    * @return   Returns the targetFileName.
    */
   public String getTargetFileName()
   {
      return this.targetFileName;
   }


   /**
    * @param targetFileName  The targetFileName to set.
    */
   public void setTargetFileName (String targetFileName)
   {
      if (this.targetFileName != targetFileName)
      {
         this.targetFileName = targetFileName;
      }
   }


   /**
    * @return   Returns the umlProject.
    */
   public UMLProject getUmlProject()
   {
      return this.umlProject;
   }


   /**
    * @param umlProject  The umlProject to set.
    */
   public void setUmlProject (UMLProject umlProject)
   {
      if (this.umlProject != umlProject)
      {
         this.umlProject = umlProject;
      }
   }
}

/*
 * $Log: AbstractRepairUMLProjectTask.java,v $
 * Revision 1.1.2.2  2006/06/07 08:42:02  lowende
 * Compile warnings removed.
 *
 */
