/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.app;

import java.awt.*;
import java.beans.*;
import java.util.Iterator;

import javax.swing.*;
import javax.swing.event.InternalFrameAdapter;
import javax.swing.event.InternalFrameEvent;
import javax.swing.tree.DefaultMutableTreeNode;

import de.uni_paderborn.fujaba.asg.ASGDiagram;
import de.uni_paderborn.fujaba.fsa.FSALayeredPane;
import de.uni_paderborn.fujaba.fsa.FSAObject;
import de.uni_paderborn.fujaba.fsa.swing.JDiagramRootPane;
import de.uni_paderborn.fujaba.preferences.ColorsPreferences;
import de.uni_paderborn.fujaba.uml.UMLProject;


/**
 * child window for a diagram
 *
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.1 $ $Date: 2004/11/08 12:07:45 $
 */
public class InternalFrame extends JInternalFrame
{
   /**
    * diagram contained in this frame (may be null)
    */
   final ASGDiagram diagram;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   final FrameMain frameMain;


   /**
    * @return   diagram contained in this frame (may be null)
    */
   public ASGDiagram getDiagram()
   {
      return diagram;
   }


   /**
    * Creates a resizable, closable, maximizable, iconifiable
    * <code>InternalFrame</code> with the diagram associated.
    *
    *
    * @param shownDiagram             associated diagram (may be null)
    * @param frameMain     No description provided
    */
   public InternalFrame (ASGDiagram shownDiagram, FrameMain frameMain)
   {
      super(); //shownDiagram != null ? shownDiagram.getName() : "", true, true, true, true);
      this.frameMain = frameMain;

      setTitle (shownDiagram != null ? shownDiagram.getName() : "");
      setResizable (true);
      setClosable (true);
      setMaximizable (true);
      setIconifiable (true);
      this.diagram = shownDiagram;
      if (shownDiagram != null)
      {
         Icon icon = null;
         for (Iterator it = frameMain.iteratorOfDecorators(); it.hasNext() && icon == null; )
         {
            FrameDecorator decorator = (FrameDecorator) it.next();
            try
            {
               Component displayComp = decorator.getTreeCellRenderer().getTreeCellRendererComponent (new JTree(),
                  new DefaultMutableTreeNode (shownDiagram), false, false, false, 0, false);
               if (displayComp instanceof JLabel)
               {
                  icon =  ((JLabel) displayComp).getIcon();
               }
            }
            catch (Exception e)
            {
               //ok, this decorator does not want to render the diagram
            }
         }
         setFrameIcon (icon);

         shownDiagram.addPropertyChangeListener ("name", new DiagramNameListener());
      }
      createDiagramPanel();
      addInternalFrameListener (new ActivationAndCloseListener());
   }


   /**
    * Panel for diagram specific Toolbars
    */
   private JPanel toolbarsPanel;

   /**
    * component that represents the whole shown diagram
    */
   private JComponent diagramComponent;

   /**
    * scoller panel around the diagram
    */
   private JScrollPane scrollerPanel;

   /**
    * root pane of the diagram
    */
   private JDiagramRootPane diagramRootPane;


   /**
    * Get the diagramPanel attribute
    *
    *
    * @return   The diagramPanel value
    */
   public Container getDiagramPanel()
   {
      return getContentPane();
   }


   /**
    * Get the scrollerPanel attribute
    *
    *
    * @return   The scrollerPanel value
    */
   public JScrollPane getScrollerPanel()
   {
      return scrollerPanel;
   }


   /**
    * Get the diagramComponent attribute
    *
    *
    * @return   The diagramComponent value
    */
   public JComponent getDiagramComponent()
   {
      return diagramComponent;
   }


   /**
    * Sets the diagramComponent attribute
    *
    *
    * @param comp             The new diagramComponent value
    */
   protected void setDiagramComponent (JComponent comp)
   {
      if (this.diagramComponent != null)
      {
         getContentPane().remove (this.diagramComponent);
      }
      this.diagramComponent = comp;
      getContentPane().add (comp, BorderLayout.CENTER);
   }


   /**
    * create components in this frame
    */
   private void createDiagramPanel()
   {
      getContentPane().setLayout (new BorderLayout());
      scrollerPanel = new JScrollPane();
      scrollerPanel.setBackground (ColorsPreferences.get().DIAGRAM_BACKGROUND);

      scrollerPanel.getViewport().setBackground (ColorsPreferences.get().DIAGRAM_BACKGROUND);

      scrollerPanel.getViewport().setDoubleBuffered (true);
      scrollerPanel.getViewport().setScrollMode (JViewport.BLIT_SCROLL_MODE);
      scrollerPanel.getVerticalScrollBar().setUnitIncrement (20);
      scrollerPanel.getHorizontalScrollBar().setUnitIncrement (20);

      scrollerPanel.getViewport().add (getDiagramRootPane());

      setDiagramComponent (scrollerPanel);

      toolbarsPanel = new JPanel();
      toolbarsPanel.setLayout (new GridLayout (1, 0));
      getContentPane().add (toolbarsPanel, BorderLayout.WEST);
   }


   /**
    * Get the diagramRootPane attribute
    *
    *
    * @return   The diagramRootPane value
    */
   public JDiagramRootPane getDiagramRootPane()
   {
      if (diagramRootPane == null)
      {
         diagramRootPane = new JDiagramRootPane();
         diagramRootPane.setBackground (ColorsPreferences.get().DIAGRAM_BACKGROUND);
      }
      return diagramRootPane;
   }


   /**
    * Get the FSALayeredPane that is the root of the FSA-Tree for the current
    * diagram
    *
    *
    * @return   The diagramRoot value
    */
   public FSALayeredPane getDiagramRoot()
   {
      FSALayeredPane diagramRoot = null;

      if (diagramRootPane != null)
      {
         JComponent comp = diagramRootPane.getContentPane();
         if (comp != null)
         {
            FSAObject obj = FSAObject.getFSAObjectFromJComponent (comp);

            if (obj != null && obj instanceof FSALayeredPane)
            {
               diagramRoot = (FSALayeredPane) obj;
            }
         }
      }

      return diagramRoot;
   }


   /**
    * Access method for an one to n association.
    *
    *
    * @param toolBar             The object added.
    */
   public void addDiagramToolBar (JToolBar toolBar)
   {
      if (toolBar != null)
      {
         toolBar.setVisible (true);
         toolBar.setOrientation (SwingConstants.VERTICAL);
         toolbarsPanel.add (toolBar);
      }
   }


   /**
    * remove a toolbar from the diagram toolbar area
    *
    *
    * @param toolBar             what to add
    */
   public void removeDiagramToolBar (JToolBar toolBar)
   {
      if (toolBar != null)
      {
         toolBar.setVisible (false);
         toolbarsPanel.remove (toolBar);
      }
   }


   /**
    * remove all JToolBars added by {@link #addDiagramToolBar}(in fact remove
    * all components on the toolbarsPabel)
    */
   public void removeAllDiagramToolBars()
   {
      toolbarsPanel.removeAll();
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.1 $ $Date: 2004/11/08 12:07:45 $
    */
   private class ActivationAndCloseListener extends InternalFrameAdapter
   {
      /**
       * Invoked when an internal frame is activated.
       *
       *
       * @param e             No description provided
       */
      public void internalFrameActivated (InternalFrameEvent e)
      {
         if (!frameMain.isInShowDiagram() && diagram != null && !isIcon())
         {
            frameMain.showDiagram (diagram);
         }
      }


      /**
       * Invoked when an internal frame has been closed.
       *
       *
       * @param e             No description provided
       */
      public void internalFrameClosed (InternalFrameEvent e)
      {
         if (diagram != null)
         {
            if (UMLProject.get().getCurrentDiagram() == diagram)
            {
               frameMain.showDiagram (null);
            }
            frameMain.frames.remove (diagram);
            Iterator iter = diagram.iteratorOfFsaObjects();

            while (iter.hasNext())
            {
               FSAObject fsa = (FSAObject) iter.next();

               fsa.removeYou();
            }
         }
      }


      /**
       * Invoked when an internal frame is iconified.
       *
       *
       * @param e             No description provided
       */
      public void internalFrameIconified (InternalFrameEvent e)
      {
         try
         {
            setMaximum (false);
         }
         catch (PropertyVetoException e1)
         {
            //leave it
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.1 $ $Date: 2004/11/08 12:07:45 $
    */
   private class DiagramNameListener implements PropertyChangeListener
   {
      /**
       * This method gets called when a bound property is changed.
       *
       *
       * @param evt             A PropertyChangeEvent object describing the event source and
       *           the property that has changed.
       */
      public void propertyChange (PropertyChangeEvent evt)
      {
         InternalFrame.this.setTitle (diagram.getName());
      }
   }
}

/*
 * $Log: InternalFrame.java,v $
 * Revision 1.1  2004/11/08 12:07:45  lowende
 * Inner class of FrameMain extracted
 *
 */
