/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.app.action;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.*;
import java.net.URL;
import java.net.URLDecoder;
import java.util.*;

import javax.swing.*;
import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.app.FrameMain;
import de.uni_paderborn.fujaba.app.FujabaApp;
import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.basic.FileStringReader;
import de.uni_paderborn.fujaba.basic.ProcessOutputViewer;
import de.uni_paderborn.fujaba.messages.*;
import de.uni_paderborn.fujaba.metamodel.FFile;
import de.uni_paderborn.fujaba.metamodel.FPackage;
import de.uni_paderborn.fujaba.preferences.GeneralPreferences;
import de.uni_paderborn.fujaba.texteditor.TextEditor;
import de.uni_paderborn.fujaba.uml.UMLClass;
import de.uni_paderborn.fujaba.uml.UMLProject;
import de.uni_paderborn.fujaba.uml.actions.EditTextAction;
import de.uni_paderborn.lib.classloader.UPBClassLoader;
import de.upb.lib.plugins.PluginJarFile;
import de.upb.lib.plugins.PluginManager;


/**
 * Compiles Java Sources.
 *
 * @author    $Author: cschneid $
 * @version   $Revision: 1.44.2.8 $
 */
public class CompileAction extends AbstractAction
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private ProcessOutputViewer myView = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean quietCompile = false;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String outPath = "";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private ActionListener exitAction = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String MESSAGE_CLASS_COMPILE_ERROR = "Compiler error";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String MESSAGE_CLASS_COMPILE_WARNING = "Compiler warning";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String FILE_SUFFIX = ".java";


   /**
    * sets the ProcessOutputViewer to compile a project if no one is set, CompileAction generates
    * a new one
    *
    * @param myView  The new processOutputViewer value
    */
   public void setProcessOutputViewer (ProcessOutputViewer myView)
   {
      this.myView = myView;
   }


   /**
    * gets the ProcessOutputViewer that was used to compile the project
    *
    * @return   ProcessOutputViewer
    */
   public ProcessOutputViewer getProcessOutputViewer()
   {
      return myView;
   }


   /**
    * sets CompileAction to quiet (no ProcessOutputViewer)
    *
    * @param value  The new quiet value
    */
   public void setQuiet (boolean value)
   {
      quietCompile = value;
   }


   /**
    * Sets the outputPath attribute of the CompileAction object
    *
    * @param value  The new outputPath value
    */
   public void setOutputPath (String value)
   {
      outPath = value;
   }


   /**
    * Sets the exitAction attribute of the CompileAction object
    *
    * @param act  The new exitAction value
    */
   public void setExitAction (ActionListener act)
   {
      exitAction = act;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void actionPerformed (ActionEvent e)
   {
      compile (e);
   }


   /**
    * Compile all sources
    *
    * @param e  action event
    * @return   true when compile was successful if e==null, when e is not null returns always
    *         true
    */
   public boolean compile (ActionEvent e)
   {
      FrameMain frameMain = FrameMain.get();
      frameMain.setCursorWait();

      String userCmdLine;
      String expPath = GeneralPreferences.get().getExportFolder();
      String fileSeparator = System.getProperty ("file.separator");
      //Add all Packages to compile
      Iterator iter = UMLProject.get().iteratorOfFiles();
      StringBuffer compileDirs = new StringBuffer();

      while (iter.hasNext())
      {
         FFile curFile = (FFile) iter.next();
         if (!curFile.isCodeGenDenied() && curFile.necessaryToCreateFile())
         {
            FPackage pack = curFile.getFPackage();
            StringBuffer packageDir = new StringBuffer();

            if (pack != null)
            {
               packageDir.append (expPath);

               if (pack.getPackagePath().trim().equals ("."))
               {
                  packageDir.append (fileSeparator + "*.java");
               }
               else
               {
                  packageDir.append (fileSeparator + pack.getPackagePath() + fileSeparator + "*.java");
               }

               if (compileDirs.indexOf (packageDir.toString()) == -1)
               {
                  if (System.getProperty ("os.name").startsWith ("Windows"))
                  {
                     compileDirs.append ("\"" + packageDir.toString() + "\" ");
                  }
                  else
                  {
                     compileDirs.append (packageDir.toString() + " ");
                  }
               }
            }
         }
      }

      boolean win32 = System.getProperty ("os.name").startsWith ("Windows");

      if (getCompiler() == null)
      {
         final String jdkFolder = GeneralPreferences.get().getJDKFolder();
         final String binFolder = jdkFolder.length() > 0 ? jdkFolder + fileSeparator + "bin" + fileSeparator : "";
         userCmdLine = binFolder + "javac";
      }
      else
      {
         userCmdLine = getCompiler().getCompilerCommand();
      }
      if (outPath.length() != 0)
      {
         userCmdLine += " -d ";

         if (win32)
         {
            userCmdLine += "\""; //Support for long file and directory names
         }

         userCmdLine += outPath;

         if (win32)
         {
            userCmdLine += "\""; //Support for long file and directory names
         }
      }

      userCmdLine += " -g -deprecation ";

      //put classpath into environment instead of command line to allow long classpathes
//      userCmdLine += "-classpath ";
//      if (win32)
//      {
//         userCmdLine += "\"" + getClassPath() + "\" "; //Support for long file and directory names
//      }
//      else
//      {
//         userCmdLine += getClassPath() + " ";
//      }
      String classpathEnvironment;
      /*
       *  if (win32)
       *  {
       *  classpathEnvironment = "\"" + getClassPath() + "\"";
       *  }
       *  else
       *  {
       */
      classpathEnvironment = getClassPath();
      //}

      userCmdLine += compileDirs.toString().trim();

      try
      {
         if (myView == null)
         {
            myView = new ProcessOutputViewer (true, e != null);
            myView.addOutputListener (
               new ProcessOutputViewer.OutputListener()
               {
                  public void outputCleared() { }


                  public void outputAppended (String newLine) { }


                  public void outputFinished (String output, int exitValue)
                  {
                     try
                     {
                        handleCompilerOutput (output, exitValue == 0);
                     }
                     catch (RuntimeException e)
                     {
                        e.printStackTrace();
                        ErrorMessage errorMessage = new ErrorMessage ("Error while parsing compiler output: " + e.toString());
                        FrameMain.get().getMessageView().addToMessages (errorMessage);
                     }
                  }

               });
            //myView.setMouseListener (new CompileRunMouseListener());
         }

         myView.getEnvironment().put ("CLASSPATH", classpathEnvironment);

         if (exitAction != null)
         {
            myView.setExitAction (exitAction);
         }

         if (e == null)
         {
            return myView.compileAndWaitForResult (userCmdLine);
         }
         else if (quietCompile)
         {
            myView.quietExecuteCommand (userCmdLine);
         }
         else
         {
            myView.executeCommand (userCmdLine, "Compiler");
         }
      }
      finally
      {
         frameMain.setCursorDefault();
      }
      return true;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param compilerMessage  No description provided
    * @return                 No description provided
    */
   public static Action createGotoSourceAction (final CompilerMessage compilerMessage)
   {
      return
         new AbstractAction ("Show source code")
         {
            /**
             * Invoked when an action occurs.
             *
             * @param e  No description provided
             */
            public void actionPerformed (ActionEvent e)
            {
               UMLClass cls = UMLProject.get().findClass (null, compilerMessage.getClassName());

               TextEditor.Buffer buffer = new EditTextAction().useTextEditor (cls.getFile());
               if (compilerMessage.getLineNumber() > 0)
               {
                  buffer.showLine (compilerMessage.getLineNumber());
               }
            }
         };
   }


   /**
    * Get the classPath attribute of the CompileAction class
    *
    * @return   The classPath value
    */
   public static String getClassPath()
   {
      FrameMain frameMain = FrameMain.get();
      StringBuffer classPath = new StringBuffer();
      String pathSeparator = System.getProperty ("path.separator");
      String fileSeparator = System.getProperty ("file.separator");

      String cp = UPBClassLoader.get (UPBClassLoader.DEFAULT_CLASSLOADER).getClassPath();

      if (cp == null)
      {
         JOptionPane.showMessageDialog (frameMain, "Java classpath system property not set! \n" +
            "Using Fujaba path as default.", "Compile",
            JOptionPane.ERROR_MESSAGE);
         cp = System.getProperty ("user.dir");
      }

      classPath.append (cp + pathSeparator);

      cp = UMLProject.get().getAdditionalClassPath();

      if ( (cp != null) && ! (cp.equals ("")))
      {
         classPath.append (cp + pathSeparator);
      }

      String[] libs = new File ("libs").list (
         new FilenameFilter()
         {
            public boolean accept (File dir, String name)
            {
               return name.endsWith (".jar");
            }
         });
      if (libs != null)
      {
         for (int i = 0; i < libs.length; i++)
         {
            classPath.append ("libs" + fileSeparator + libs[i] + pathSeparator);
         }
      }

      PluginManager pluginManager = FujabaApp.getPluginManager();
      Iterator iter = pluginManager.iteratorOfJarFiles();

      int missingPluginJars = 0;

      while (iter.hasNext())
      {
         PluginJarFile jarFile = (PluginJarFile) iter.next();
         String jarFileName = jarFile.getFileName();
         if (jarFileName.endsWith (".xml"))
         {
            if (jarFile.getProperty() == null)
            {
               jarFileName = null;
            }
            else
            {
               jarFileName = jarFileName.substring (0, jarFileName.lastIndexOf (fileSeparator) + 1)
                  + jarFile.getProperty().getPluginJarFile();
            }
         }
         if (jarFileName != null)
         {
            if (new File (jarFileName).exists())
            {
               classPath.append (jarFileName + pathSeparator);
            }
            else
            {
               missingPluginJars++;
            }
         }
      }

      //when run by webstart the 'system' classloader does not know the actual classpath of webstart
      //this is indicated by loaded plugins but missing their jars
      if (missingPluginJars >= 0)
      { //ok, this is a little bit strange :/

         //so search for all jars near to Fujaba.jar

         //version xml resides in Fujaba.jar or the Fujaba directory
         final URL versionXmlUrl = UPBClassLoader.get (UPBClassLoader.DEFAULT_CLASSLOADER).getResource ("version.xml");
         if (versionXmlUrl != null)
         {
            String urlString = versionXmlUrl.toExternalForm();
            if (urlString.startsWith ("jar:"))
            {
               urlString = urlString.substring ("jar:".length(), urlString.lastIndexOf ('!'));
            }
            if (urlString.startsWith ("file:"))
            {
               try
               {
                  File fujabaDir = new File (URLDecoder.decode (urlString.substring ("file:".length()), "UTF-8"));
                  if (fujabaDir.exists() && !fujabaDir.isDirectory())
                  {
                     fujabaDir = fujabaDir.getParentFile();
                  }
                  if (log.isInfoEnabled())
                  {
                     log.info ("using jars in fujaba directory: " + fujabaDir.getAbsolutePath());
                  }

                  //ok, we have found the fujaba directory now scan for jars
                  appendJarsIn (fujabaDir, classPath, pathSeparator);
               }
               catch (UnsupportedEncodingException e)
               {
                  //should never happen as UTF-8 is always supported
                  e.printStackTrace();
               }
            }
            else
            {
               //somewhere in the net?! - so we can't browse for jars :(
            }
         }
         else
         {
            //version.xml not found - no options then :(
         }
      }
      else
      {
         if (log.isInfoEnabled())
         {
            log.info ("no missing plugin jars");
         }
      }

      classPath.append (GeneralPreferences.get().getExportFolder());

      return classPath.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param dir            No description provided
    * @param classPath      No description provided
    * @param pathSeparator  No description provided
    */
   private static void appendJarsIn (File dir, StringBuffer classPath, String pathSeparator)
   {
      final File[] files = dir.listFiles();
      for (int i = 0; files != null && i < files.length; i++)
      {
         File file = files[i];
         if (file.isDirectory())
         {
            appendJarsIn (file, classPath, pathSeparator);
         }
         else if (file.getName().endsWith (".jar") && !file.getName().startsWith ("RT"))
         {
            final String path = file.getPath();
            classPath.append (path + pathSeparator);
         }
      }
   }


   /**
    * @return       the export path
    * @deprecated   Use GeneralPreferences.get().getExportFolder() instead.
    */
   public static String getExpPath()
   {
      return GeneralPreferences.get().getExportFolder();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static transient Logger log = Logger.getLogger (CompileAction.class);


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param output   No description provided
    * @param success
    */
   static void handleCompilerOutput (String output, boolean success)
   {
      FrameMain.get().getMessageView().deleteMessages (MESSAGE_CLASS_COMPILE_ERROR);
      FrameMain.get().getMessageView().deleteMessages (MESSAGE_CLASS_COMPILE_WARNING);

      List messages = new LinkedList();

      StringTokenizer lineTokenizer = new StringTokenizer (output, "\n");

      String filename = null;
      boolean isWarning = false;
      String text = null;
      String context = null;
      int lineNr = 0;
      int colNr;
      while (lineTokenizer.hasMoreTokens())
      {
         String line = lineTokenizer.nextToken();
         if (line.trim().startsWith ("^") || line.trim().length() == 0)
         {
            colNr = line.indexOf ('^');
            //final line of an error
            if (filename != null)
            {
               messages.add (0, createMessage (filename, isWarning, lineNr, colNr, text, context));
            }
            filename = null;
         }
         line = line.replaceAll ("\\^", "").trim();

         int indexOfJava = line.indexOf (FILE_SUFFIX);
         if (indexOfJava >= 0)
         {
            //first line of an error
            if (filename != null)
            {
               log.error ("Unrecognized compiler output!");
            }
            isWarning = false;
            lineNr = 0;
            text = null;
            context = null;

            filename = line.substring (0, indexOfJava + FILE_SUFFIX.length());
            line = line.substring (indexOfJava + FILE_SUFFIX.length() + 1);

            StringTokenizer colonTokenizer = new StringTokenizer (line, ": ");
            while (colonTokenizer.hasMoreTokens())
            {
               String token = colonTokenizer.nextToken().trim();
               if ("warning".equals (token))
               {
                  if (text != null)
                  {
                     text += " " + token;
                  }
                  else
                  {
                     isWarning = true;
                  }
               }
               else if ("error".equals (token))
               {
                  if (text != null)
                  {
                     text += " " + token;
                  }
               }
               else
               {
                  boolean isLineNr = false;
                  try
                  {
                     if (lineNr <= 0)
                     {
                        lineNr = Integer.valueOf (token).intValue();
                        isLineNr = true;
                     }
                  }
                  catch (NumberFormatException e)
                  {
                  }
                  if (!isLineNr)
                  {
                     if (text == null)
                     {
                        text = token;
                     }
                     else
                     {
                        text += " " + token;
                     }
                  }
               }
            }
         }
         else if ("javac: no source files".equals (line) || "error no sources specified".equals (line))
         {
            messages.add (0, createMessage ("*.java", false, 0, 0, "no java files were found!", null));
            return;
         }
         else
         {
            //context
            if (context == null)
            {
               context = line;
            }
            else
            {
               context += " \n" + line;
            }
         }
      }

      for (Iterator it = messages.iterator(); it.hasNext(); )
      {
         Message message = (Message) it.next();
         FrameMain.get().getMessageView().addToMessages (message);
      }

      if (success)
      {
         FrameMain.get().setStatusLabel ("Compilation successfull");
      }
      else
      {
         FrameMain.get().setStatusLabel ("Compilation failed");
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param filename  No description provided
    * @param warning   No description provided
    * @param lineNr    No description provided
    * @param colNr     No description provided
    * @param text      No description provided
    * @param context   No description provided
    * @return          No description provided
    */
   public static Message createMessage (String filename, boolean warning, int lineNr, int colNr, String text, String context)
   {
      Message message;
      if (warning)
      {
         text = "Compiler warning: \n" + text;
         message = new CompilerWarning (className (filename), lineNr);
         message.setMessageCategory (MESSAGE_CLASS_COMPILE_WARNING);
      }
      else
      {
         text = "Compiler error: \n" + text;
         message = new CompilerErrorMessage (className (filename), lineNr);
         message.setMessageCategory (MESSAGE_CLASS_COMPILE_ERROR);
         FrameMain.get().showMessageView();
      }
      fillMessage (message, filename, warning, lineNr, colNr, text, context);
      return message;
   }


   /**
    * Create a new Message and return it.
    *
    * @param filename  No description provided
    * @param warning   No description provided
    * @param lineNr    No description provided
    * @param colNr     No description provided
    * @param text      No description provided
    * @param context   No description provided
    * @param message   No description provided
    */
   public static void fillMessage (Message message, String filename, boolean warning, int lineNr, int colNr, String text, String context)
   {
      if (lineNr > 0)
      {
         String shortfilename = relativeFileName (filename);
         text += " (" + shortfilename + ":" + lineNr;

         if (colNr > 0)
         {
            text += ":" + colNr;
         }

         String id = getIncrementIdForGeneratedCodeLine (filename, lineNr);
         if (id != null && !"".equals (id))
         {
            ASGElement incr = UMLProject.get().searchID (id);
            if (incr != null)
            {
               message.addToContext (incr);
            }
            else
            {
               //should not happen if files are up to date
               text += " - " + id;
            }
         }
         else
         {
            text += " - not found in model";
         }
         text += ")";
      }

      if (context != null)
      {
         text += ": " + context;
      }
      message.setText (text);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param filename  No description provided
    * @return          No description provided
    */
   private static String relativeFileName (String filename)
   {
      String shortfilename;
      if (new File (filename).exists())
      {
         try
         {
            shortfilename = new File (filename).getCanonicalPath();
            String expPath = new File (GeneralPreferences.get().getExportFolder()).getCanonicalPath();
            if (shortfilename.startsWith (expPath))
            {
               shortfilename = shortfilename.substring (expPath.length() + 1);
            }
         }
         catch (IOException e)
         {
            shortfilename = filename;
         }
      }
      else
      {
         shortfilename = filename;
      }
      return shortfilename;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param filename  No description provided
    * @return          No description provided
    */
   private static String className (String filename)
   {
      if (filename.endsWith (".java"))
      {
         filename = relativeFileName (filename);
         filename = filename.substring (0, filename.length() - 5);
         filename = filename.replaceAll ("/", ".");
         filename = filename.replaceAll ("\\\\", ".");
         return filename;
      }
      else
      {
         return null;
      }
   }


   /**
    * Get the incrementIdForGeneratedCodeLine attribute of the CompileAction class
    *
    * @param file    No description provided
    * @param lineNr  No description provided
    * @return        The incrementIdForGeneratedCodeLine value
    */
   public static String getIncrementIdForGeneratedCodeLine (String file, int lineNr)
   {
      file = file.substring (0, file.length() - 4) + "dlr";
      if (!new File (file).exists())
      {
         file = GeneralPreferences.get().getExportFolder() + File.separatorChar + file;
      }
      String id = new DLRParser().getIncrementId (file, lineNr);
      return id;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: cschneid $
    * @version   $Revision: 1.44.2.8 $ $Date: 2005/12/20 13:16:47 $
    */
   public interface Compiler
   {
      /**
       * Get the compilerCommand attribute of the Compiler object
       *
       * @return   The compilerCommand value
       */
      String getCompilerCommand();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static Compiler compiler = null;


   /**
    * Get the compiler attribute of the CompileAction class
    *
    * @return   The compiler value
    */
   public static Compiler getCompiler()
   {
      return compiler;
   }


   /**
    * Sets the compiler attribute of the CompileAction class
    *
    * @param compiler  The new compiler value
    */
   public static void setCompiler (Compiler compiler)
   {
      CompileAction.compiler = compiler;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: cschneid $
    * @version   $Revision: 1.44.2.8 $
    */
   private static class DLRParser extends FileStringReader
   {

      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private String incrId = null;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private int lineNr;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private int incrLineNr = Integer.MIN_VALUE;


      /**
       * Get the incrementId attribute of the GetIncrementId object
       *
       * @param fileName  No description provided
       * @param lineNr    No description provided
       * @return          The incrementId value
       */
      public String getIncrementId (String fileName, int lineNr)
      {
         setComment ("#");
         this.lineNr = lineNr;
         doFile (fileName);
         return incrId;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param currentLine  No description provided
       */
      protected void doCurrentLine (String currentLine)
      {
         if (!currentLine.startsWith ("file"))
         {
            StringTokenizer st = new StringTokenizer (currentLine, "=");
            String lineNrSt = st.nextToken().trim();
            String incrIdSt = st.nextToken().trim();
            int curLineNr = Integer.parseInt (lineNrSt);
            if ( (curLineNr <= lineNr) &&  (curLineNr > incrLineNr))
            {
               incrId = incrIdSt;
               incrLineNr = curLineNr;
            }
         }
      }
   }
}

/*
 * $Log: CompileAction.java,v $
 * Revision 1.44.2.8  2005/12/20 13:16:47  cschneid
 * More warning/error info, some copy/paste fixes from erik, message listener for JEM (andreas/manuel), new libs
 *
 */
