/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.app.action;

import java.awt.event.ActionEvent;
import java.io.File;
import java.util.Iterator;
import java.util.Map;

import javax.swing.AbstractAction;
import javax.swing.JOptionPane;

import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.app.FrameMain;
import de.uni_paderborn.fujaba.basic.RuntimeExceptionWithContext;
import de.uni_paderborn.fujaba.codegen.CodeGenException;
import de.uni_paderborn.fujaba.codegen.CodeGenFactory;
import de.uni_paderborn.fujaba.coobra.FujabaChangeManager;
import de.uni_paderborn.fujaba.messages.ErrorMessage;
import de.uni_paderborn.fujaba.messages.MessageView;
import de.uni_paderborn.fujaba.metamodel.FPackage;
import de.uni_paderborn.fujaba.preferences.CodeGenPreferences;
import de.uni_paderborn.fujaba.preferences.GeneralPreferences;
import de.uni_paderborn.fujaba.uml.UMLProject;


/**
 * Export the project to java sources.
 *
 * @author    $Author: creckord $
 * @version   $Revision: 1.33.2.9 $
 */
public class ExportFilesAction extends AbstractAction
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (ExportFilesAction.class);

   /**
    * Keeps the state of the ExportFilesAction for other actions
    */
   private boolean success = false;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String MESSAGE_CLASS_EXPORT = "Export";


   /**
    * Returns the state of ExportFilesAction, true if the export was successful, else false
    *
    * @return   The successful value
    */
   public boolean isSuccessful()
   {
      return success;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void actionPerformed (final ActionEvent e)
   {
      MessageView messageView = FrameMain.get().getMessageView();
      messageView.deleteMessages (MESSAGE_CLASS_EXPORT);
      messageView.deleteMessages (CompileAction.MESSAGE_CLASS_COMPILE_ERROR);
      messageView.deleteMessages (CompileAction.MESSAGE_CLASS_COMPILE_WARNING);

      Runnable runnable =
         new Runnable()
         {
            public void run()
            {
               exportFiles (e);
            }
         };
      FujabaChangeManager.executeTransient (runnable);
   }


   /**
    * Export files
    *
    * @param e  Action event that caused the export
    * @return   true when successful
    */
   public boolean exportFiles (ActionEvent e)
   {
      FPackage pkg;
      final boolean askIfFilesShouldBeDeleted = e != null && !"./generated".equals (expPath);
      if (e != null && e.getSource() instanceof FPackage)
      {
         pkg = (FPackage) e.getSource();
      }
      else
      {
         pkg = null;
      }
      return exportFiles (e != null, askIfFilesShouldBeDeleted, pkg);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param deleteExistingFiles        No description provided
    * @param askIfFilesShouldBeDeleted  No description provided
    * @param pkg                        No description provided
    * @return                           No description provided
    */
   public boolean exportFiles (boolean deleteExistingFiles, final boolean askIfFilesShouldBeDeleted, FPackage pkg)
   {
      FrameMain frameMain = FrameMain.get();
      String expPath = GeneralPreferences.get().getExportFolder();
      String tmpTarget = CodeGenFactory.get().getCurrentTarget().getName();

      boolean exceptionOccured = false;
      try
      {
         if (deleteExistingFiles)
         {
            if (!clearExportDirectory (expPath, askIfFilesShouldBeDeleted))
            {
               return false;
            }
         }

         frameMain.setCursorWait();
         success = false;

         //Vector target = JavaPreferences.get().getCodeGenTargetName();
         Map target = CodeGenPreferences.get().getSelectedCodeGenTargetNames();
         Iterator iter = target.keySet().iterator();

         while (iter.hasNext())
         {
            String key = (String) iter.next();
            Boolean bool = (Boolean) target.get (key);
            if (bool.booleanValue())
            {
               CodeGenFactory.get().setCurrentTarget (key);
               long time = System.currentTimeMillis();
               if (pkg != null)
               {
                  CodeGenFactory.get().generateFElement (pkg);
               }
               else
               {
                  CodeGenFactory.get().generateFProject();
               }
               time = System.currentTimeMillis() - time;
               frameMain.setStatusLabel ("Source for target " + key + " generated in " + time / 1000d + "s.");
            }
         }
         success = true;

//         if (e != null)
//         {
//            JOptionPane.showMessageDialog (frameMain, "Java code successfully exported to: \n" + expPath, "Export JAVA", JOptionPane.INFORMATION_MESSAGE);
//         }

         frameMain.setStatusLabel ("Java code created successfully!");
      }
      catch (Exception ex)
      {
         if (! (ex instanceof CodeGenException) || log.isDebugEnabled())
         {
            ex.printStackTrace();
         }

//         JOptionPane.showMessageDialog (frameMain, ex + "\n" +
//            FujabaDebug.getStackTrace (ex, 0, 2),
//            "Warning",
//            JOptionPane.WARNING_MESSAGE);

         FrameMain.get().getMessageView().deleteMessages (MESSAGE_CLASS_EXPORT);
         showErrorMessageForException (ex);
         exceptionOccured = true;
      }
      finally
      {
         CodeGenFactory.get().setCurrentTarget (tmpTarget);
         UMLProject.get().refreshDisplay();
         frameMain.setCursorDefault();
      }
      if (!exceptionOccured)
      {
         FrameMain.get().getMessageView().deleteMessages (MESSAGE_CLASS_EXPORT);
      }
      return success;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param ex  No description provided
    */
   private void showErrorMessageForException (Exception ex)
   {
      if (ex instanceof CodeGenException)
      {
         for (Iterator it =  ((CodeGenException) ex).iteratorOfExceptions(); it.hasNext(); )
         {
            Exception exception = (Exception) it.next();
            showErrorMessageForException (exception);
         }
      }
      else
      {
         ex.printStackTrace();
         ErrorMessage errorMessage = new ErrorMessage();
         errorMessage.setThrowable (ex);
         if (ex instanceof RuntimeExceptionWithContext)
         {
            errorMessage.setText (ex.getMessage());
         }
         else
         {
            errorMessage.setText (ex.toString());
         }
         for (Throwable t = ex; t != null; t = t.getCause())
         {
            if (t instanceof RuntimeExceptionWithContext)
            {
               RuntimeExceptionWithContext sdmException = (RuntimeExceptionWithContext) t;
               errorMessage.addToContext (sdmException.getContext());
            }
         }
         errorMessage.setMessageCategory (MESSAGE_CLASS_EXPORT);
         FrameMain.get().getMessageView().addToMessages (errorMessage);
         FrameMain.get().showMessageView();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param expPath                    No description provided
    * @param askIfFilesShouldBeDeleted  true if a dialog should be shown to ask whether to
    *      delete file s or not
    * @return                           No description provided
    */
   private boolean clearExportDirectory (String expPath, boolean askIfFilesShouldBeDeleted)
   {
      frameMain = FrameMain.get();

      File currDir = new File (expPath);

      this.askIfFilesShouldBeDeleted = askIfFilesShouldBeDeleted;
      this.expPath = expPath;
      foundJava = false;
      choice = 0;
      success = true;

      File[] files = currDir.listFiles();
      if (files == null)
      {
         //output directory does not exist!
         int answer = askIfFilesShouldBeDeleted ? JOptionPane.showConfirmDialog (FrameMain.get(),
            "The export directory (" + currDir.getAbsolutePath() + ") does not exist - create it?", "Export", JOptionPane.YES_NO_OPTION)
            : JOptionPane.YES_OPTION;
         if (answer != JOptionPane.YES_OPTION)
         {
            //do not create directory ?! Then we cannot export!
            return false;
         }
         else
         {
            currDir.mkdirs();
            return true;
         }
      }

      boolean success = deleteFileOrDirectory (currDir, false);

      return success;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient boolean askIfFilesShouldBeDeleted = false;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient FrameMain frameMain = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient String expPath = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient boolean foundJava = false;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient int choice = 0;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param f               No description provided
    * @param includingParam  No description provided
    * @return                No description provided
    */
   private boolean deleteFileOrDirectory (File f, boolean includingParam)
   {
      if (f.isDirectory())
      {
         File[] fileList = f.listFiles();
         for (int i = 0; fileList != null && i < fileList.length; i++)
         {
            deleteFileOrDirectory (fileList[i], true);
            if (foundJava &&  (!success || choice != JOptionPane.OK_OPTION))
            {
               return success;
            }
         }
      }

      if (includingParam)
      {
         if (f.getName().endsWith (".java") || f.getName().endsWith (".class") || f.getName().endsWith (".dlr") || f.getName().endsWith (".tmp"))
         {
            if (askIfFilesShouldBeDeleted && !foundJava)
            {
               choice = JOptionPane.showConfirmDialog (frameMain, "There are already .java, .class or" +
                  " .dlr files in directory \n" + expPath + ". \n It is recommend to remove" +
                  " all of these files to avoid problems while compiling the project. \n\n Do" +
                  " you want to remove all .java, .class and .dlr files ?",
                  "Export Java - Warning", JOptionPane.YES_NO_CANCEL_OPTION,
                  JOptionPane.WARNING_MESSAGE);
               foundJava = true;
            }

            if (askIfFilesShouldBeDeleted && choice == JOptionPane.CANCEL_OPTION)
            {
               success = false;
               return success;
            }
            else if (!askIfFilesShouldBeDeleted || choice == JOptionPane.OK_OPTION)
            {
               if (log.isInfoEnabled())
               {
                  log.info ("deleting file: " + f);
               }
               f.delete();
            }
         }
      }
      return success;
   }
}

/*
 * $Log: ExportFilesAction.java,v $
 * Revision 1.33.2.9  2006/04/27 10:29:20  creckord
 * Fixed some codegen exception issues
 *
 */
