/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.basic;

import java.util.Iterator;
import java.util.StringTokenizer;

import javax.swing.*;

import de.uni_paderborn.fujaba.app.FrameMain;
import de.uni_paderborn.fujaba.app.messages.WarningMessages;
import de.uni_paderborn.fujaba.asg.ASGDiagram;
import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.uml.*;
import de.uni_paderborn.fujaba.views.ViewDiagram;
import de.upb.tools.fca.FHashMap;
import de.upb.tools.fca.FTreeSet;


/**
 * Creates a diagram from templates (for fgrafik and javakara). ToDo: The methods used to generate
 * the diagram items should be seperated in a factory to use them in all edit dialogs to.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.14 $
 */
public class ClassDiagTemplateLoader extends FileStringReader
{

   /**
    * Stores the builded class diagram.
    */
   private FClassDiagram classDiag = null;

   /**
    * Stores the last package name read out of the template file.
    */
   private String fpackage = null;

   /**
    * HashMap storing all loaded FClasses by their name.
    */
   private FHashMap classes = new FHashMap();

   /**
    * TypeList to look up attribute types.
    */
   private FTypeList types = null;


   /**
    * Creates a diagram from the template "Templates/<name>.tpl".
    *
    * @param name    No description provided
    * @param loader  No description provided
    */
   public ClassDiagTemplateLoader (String name, ClassLoader loader)
   {
      this (name, "", loader);
   }


   /**
    * Creates a diagram from the template "<path>
    *
    * /Templates/<name>.tpl".
    *
    * @param name    No description provided
    * @param path    No description provided
    * @param loader  No description provided
    */
   public ClassDiagTemplateLoader (String name, String path, ClassLoader loader)
   {
      FDiagram fgrafikDiagram = UMLProject.get().getFromDiags (name);

      if (fgrafikDiagram != null)
      {
         int pane = JOptionPane.showConfirmDialog (FrameMain.get(),
            name + " diagram quite exists. \n Delete and create new?", "Create "
            + name + " Diagram", JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE);

         if (pane == JOptionPane.YES_OPTION)
         {
            FTreeSet itemsToDelete = new FTreeSet();
            Iterator itemsIter = fgrafikDiagram.iteratorOfElements();

            while (itemsIter.hasNext())
            {
               FElement item = (FElement) itemsIter.next();
               Iterator diagsIter = item.iteratorOfDiagrams();

               // check if the item is contained in another diagram
               boolean found = false;
               while (!found && diagsIter.hasNext())
               {
                  FDiagram tmpDiag = (FDiagram) diagsIter.next();
                  if (tmpDiag != fgrafikDiagram && ! (tmpDiag instanceof ViewDiagram))
                  {
                     found = true;
                  }
               }
               if (!found)
               {
                  itemsToDelete.add (item);
               }
            }

            itemsIter = itemsToDelete.iterator();
            while (itemsIter.hasNext())
            {
                ((FDiagramItem) itemsIter.next()).removeYou();
            }
            fgrafikDiagram.removeYou();

         }
         else
         {
            return;
         }
      }

      FrameMain.get().setStatusLabel ("Please wait...");
      FrameMain.get().setCursorWait();

      setTypes (UMLProject.get().getTypeList());
      setClassDiag (createClassDiagram (name));

      if (getClassDiag() != null)
      {
         setComment ("#");
         doFile (path + "Templates/" + name + ".tpl", loader);
      }

      FrameMain.get().setStatusLabel ("Ready");
      FrameMain.get().setCursorDefault();
   }


   /**
    * Called for every line of the template file. Creates the classes, atributes and assocs.
    *
    * @param currentLine  No description provided
    */
   protected void doCurrentLine (String currentLine)
   {
      StringTokenizer st = new StringTokenizer (currentLine, ",");
      String type = st.nextToken().trim();
      if (type.equals ("package"))
      {
         setFpackage (st.nextToken().trim());
      }
      else if (type.equals ("class"))
      {
         String name = st.nextToken().trim();
         FClass parent = (FClass) getClasses().get (st.nextToken().trim());
         boolean isInterface = false;
         if (st.hasMoreTokens())
         {
            isInterface = st.nextToken().trim().equals ("true");
         }
         FClass clazz = createClass (name, getClassDiag(), parent, getFpackage(), isInterface);
         getClasses().put (name, clazz);
      }
      else if (type.equals ("assoc"))
      {
         String name = st.nextToken();
         FClass rightClass = (FClass) getClasses().get (st.nextToken().trim());
         FClass leftClass = (FClass) getClasses().get (st.nextToken().trim());
         String rightName = st.nextToken().trim();
         String rightKard = st.nextToken().trim();
         String leftName = st.nextToken().trim();
         String leftKard = st.nextToken().trim();
         int assocType = Integer.parseInt (st.nextToken().trim());
         createAssoc (name, rightClass, leftClass, rightName, rightKard, leftName,
            leftKard, assocType);
      }
      else if (type.equals ("attr"))
      {
         String name = st.nextToken().trim();
         String attrType = st.nextToken().trim();
         FClass attrClass = (FClass) getClasses().get (st.nextToken().trim());
         createAttr (name, getTypes().getFromFTypes (attrType), attrClass);
      }
   }

   //From here the generator methods start. Instead of this brute force implementation, a factory should do the job.

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name       No description provided
    * @param left       No description provided
    * @param right      No description provided
    * @param leftName   No description provided
    * @param leftCard   No description provided
    * @param rightName  No description provided
    * @param rightCard  No description provided
    * @param sel        No description provided
    * @return           No description provided
    */
   private FAssoc createAssoc (String name, FClass left, FClass right, String leftName,
                               String leftCard, String rightName, String rightCard, int sel)
   {
      FRole leftRole = new UMLRole();
      leftRole.setCard (new UMLCardinality());
      FRole rightRole = new UMLRole();
      rightRole.setCard (new UMLCardinality());
      FAssoc assoc = new UMLAssoc (name,
         UMLAssoc.LEFTRIGHT, null,
         (UMLRole) leftRole, (UMLRole) rightRole);
      assoc.setDirection (UMLAssoc.LEFTRIGHT);

      if (sel == 2)
      {
         leftRole.setAdornment (FRole.AGGREGATION);
         rightRole.setAdornment (FRole.NONE);
      }
      else if (sel == 4)
      {
         leftRole.setAdornment (FRole.COMPOSITION);
         rightRole.setAdornment (FRole.NONE);
      }
      else if (sel == 5)
      {
         leftRole.setAdornment (FRole.REFERENCE);
         rightRole.setAdornment (FRole.NONE);
      }
      else if (sel == 1)
      {
         leftRole.setAdornment (FRole.NONE);
         rightRole.setAdornment (FRole.AGGREGATION);
      }
      else if (sel == 3)
      {
         leftRole.setAdornment (FRole.NONE);
         rightRole.setAdornment (FRole.COMPOSITION);
      }
      else if (sel == 6)
      {
         leftRole.setAdornment (FRole.NONE);
         rightRole.setAdornment (FRole.REFERENCE);
      }
      else
      {
         leftRole.setAdornment (FRole.NONE);
         rightRole.setAdornment (FRole.NONE);
      }

      //left-side
      leftRole.setName (leftName);
      leftRole.getFCard().setCardString (leftCard);

      /*
       *  Types for visibility
       *  FIncrement.PRIVATE <-default for all fgrafik assocs
       *  FIncrement.PUBLIC
       *  FIncrement.PROTECTED
       *  FIncrement.PACKAGE
       */
      leftRole.setUmlVisibility (FDeclaration.PRIVATE);

      leftRole.setTarget (left);

      //right-side

      rightRole.setName (rightName);
      rightRole.getFCard().setCardString (rightCard);

      rightRole.setUmlVisibility (FDeclaration.PRIVATE);

      rightRole.setTarget (right);

      //add assoc to every diagram or remove, if nesseccary
      Iterator iter = UMLProject.get().iteratorOfDiags();
      FClass leftClass = leftRole.getFTarget();
      FClass rightClass = rightRole.getFTarget();
      FDiagram diag;
      while (iter.hasNext())
      {
         diag = (ASGDiagram) iter.next();
         if (diag.hasInElements (leftClass) || diag.hasInElements (rightClass))
         {
            //----- add assoc to diagram
            diag.addToElements (assoc);
            diag.addToElements (leftClass);
            diag.addToElements (rightClass);
         }
         else if (diag.hasInElements (assoc))
         {
            //----- remove assoc from diagram
            diag.removeFromElements (assoc);
         }
      }

      return assoc;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name   No description provided
    * @param type   No description provided
    * @param clazz  No description provided
    * @return       No description provided
    */
   private FAttr createAttr (String name, FType type, FClass clazz)
   {
      FAttr newAttr = clazz.getFromFAttrs (name);
      if (newAttr != null)
      {
         return newAttr;
      }
      newAttr = new UMLAttr();
      newAttr.setName (name);
      newAttr.setFinal (false);
      newAttr.setStatic (false);
      newAttr.setTransient (false);
      newAttr.setInitialValue ("");

      /*
       *  FTypes are:
       *  FBaseTypes (null, "Initializer", "",      typeList);
       *  FBaseTypes (null, FBaseTypes.BOOLEAN,      "boolean", typeList);
       *  FBaseTypes (null, FBaseTypes.CHARACTER, "char", typeList);
       *  FBaseTypes (null, "String", "String", typeList);
       *  FBaseTypes (null, FBaseTypes.INTEGER,      "int", typeList);
       *  FBaseTypes (null, FBaseTypes.BYTE, "byte", typeList);
       *  FBaseTypes (null, FBaseTypes.SHORT_INTEGER, "short", typeList);
       *  FBaseTypes (null, FBaseTypes.LONG_INTEGER, "long", typeList);
       *  FBaseTypes (null, "Float", "float", typeList);
       *  FBaseTypes (null, "Double", "double", typeList);
       *  or any FClass as reference
       */
      newAttr.setAttrType (type);

      /*
       *  Visibilities are:
       *  umlVisibilityChoice.add ("- private");
       *  umlVisibilityChoice.add ("+ public");
       *  umlVisibilityChoice.add ("# protected");
       *  umlVisibilityChoice.add ("~ package");
       *  umlVisibilityChoice.add ("! user defined");
       *  visibilityChoice.add ("- private");
       *  visibilityChoice.add ("+ public");
       *  visibilityChoice.add ("# protected");
       *  visibilityChoice.add ("~ package");
       */
      newAttr.setVisibility (FDeclaration.PUBLIC); //default values for fgrafik assocs

      newAttr.setParent (clazz);
      return newAttr;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param className    No description provided
    * @param diag         No description provided
    * @param genClass     No description provided
    * @param packageName  No description provided
    * @param interf       No description provided
    * @return             No description provided
    */
   private FClass createClass (String className, FDiagram diag, FClass genClass, String packageName, boolean interf)
   {
      FClass clazz = null;
      FGeneralization gen = null;

      clazz = UMLProject.get().getFromClasses (className);
      if ( (clazz != null) &&  (packageName.equals (clazz.getFDeclaredInPackage().getFullPackageName())))
      {
         return clazz;
      }
      clazz = new UMLClass (className, UMLStereotypeManager.get().getFromStereotypes (UMLStereotypeManager.REFERENCE), UMLProject.get().getTypeList(), null);

      diag.addToElements (clazz);

      clazz.setStatic (false);
      if (interf)
      {
         clazz.addToStereotypes (UMLStereotypeManager.get().getFromStereotypes (UMLStereotypeManager.INTERFACE));
      }
      else
      {
         clazz.removeFromStereotypes (UMLStereotypeManager.get().getFromStereotypes (UMLStereotypeManager.INTERFACE));
      }

      if (packageName == null || packageName.length() == 0)
      {
         clazz.setDeclaredInPackage (UMLProject.get().getDefaultPackage());
      }
      else
      {
         FPackage tmpPackage = UMLProject.get().getNewFromPackages (packageName);
         clazz.setDeclaredInPackage (tmpPackage);
      }

      if (genClass != null)
      {
         gen = new UMLGeneralization ((UMLClass) clazz, (UMLClass) genClass);
         diag.addToElements (gen);

         Iterator diagrams = UMLProject.get().iteratorOfDiags();
         while (diagrams.hasNext())
         {
            FDiagram item = (ASGDiagram) diagrams.next();
            if (item.hasInElements (clazz))
            {
               if (!item.hasInElements (gen))
               {
                  item.addToElements (gen);
               }
            }
         }
      }
      return clazz;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name  No description provided
    * @return      No description provided
    */
   private FClassDiagram createClassDiagram (String name)
   {
      FProject project = UMLProject.get();
      ASGDiagram diagram;
      FClassDiagram newDiag = null;
      boolean found = false;
      boolean done = false;

      try
      {
         Iterator iter = project.iteratorOfDiags();

         while (!found && iter.hasNext())
         {
            diagram = (ASGDiagram) iter.next();
            if ( (diagram instanceof FClassDiagram) &&
                (diagram.getName().equals (name)))
            {
               WarningMessages.warnDoubleIdentifiedDiagram (FrameMain.get(), "class", name);
               found = true;
            }
         }
         if (!found)
         {
            newDiag = new UMLClassDiagram (name, null);
            project.addToDiags (newDiag);
            FrameMain.get().createNewTreeItems();
            FrameMain.get().selectTreeItem (newDiag);
            done = true;
         }
      }
      catch (Exception e)
      {
         e.printStackTrace();
      } // end of catch

      if (done)
      {
         return newDiag;
      }
      else
      {
         return null;
      }
   }

   // access methods

   /**
    * Get the classDiag attribute of the ClassDiagTemplateLoader object
    *
    * @return   The classDiag value
    */
   public FClassDiagram getClassDiag()
   {
      return this.classDiag;
   }


   /**
    * Sets the classDiag attribute of the ClassDiagTemplateLoader object
    *
    * @param classDiag  The new classDiag value
    */
   public void setClassDiag (FClassDiagram classDiag)
   {
      this.classDiag = classDiag;
   }


   /**
    * Get the fpackage attribute of the ClassDiagTemplateLoader object
    *
    * @return   The fpackage value
    */
   public String getFpackage()
   {
      return this.fpackage;
   }


   /**
    * Sets the fpackage attribute of the ClassDiagTemplateLoader object
    *
    * @param fpackage  The new fpackage value
    */
   public void setFpackage (String fpackage)
   {
      this.fpackage = fpackage;
   }


   /**
    * Get the classes attribute of the ClassDiagTemplateLoader object
    *
    * @return   The classes value
    */
   public FHashMap getClasses()
   {
      return this.classes;
   }


   /**
    * Sets the classes attribute of the ClassDiagTemplateLoader object
    *
    * @param classes  The new classes value
    */
   public void setClasses (FHashMap classes)
   {
      this.classes = classes;
   }


   /**
    * Get the types attribute of the ClassDiagTemplateLoader object
    *
    * @return   The types value
    */
   public FTypeList getTypes()
   {
      return this.types;
   }


   /**
    * Sets the types attribute of the ClassDiagTemplateLoader object
    *
    * @param types  The new types value
    */
   public void setTypes (FTypeList types)
   {
      this.types = types;
   }
}

/*
 * $Log: ClassDiagTemplateLoader.java,v $
 * Revision 1.14  2004/11/13 12:04:17  lowende
 * Removed some compile warnings.
 *
 */
