/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.basic;

import java.io.File;
import java.io.FilenameFilter;
import java.util.StringTokenizer;
import java.util.Vector;

import de.uni_paderborn.fujaba.preferences.GeneralPreferences;


/**
 * The class CommandLineParser checks the provided parameter for wildcards and substitutes
 * them with complete filepaths. It handles wildcards with following filetypes as '*.java'
 * and filenames with wildcards in the file extension like 'a.*'. !!! This case will be interpretet
 * as 'a*.*' for now.!!! It contains two inner classes that define FilenameFilters. !!! Maybe
 * this parsing is not necessary under other command line interpreters than LINUX/UNIX !!!
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.19 $
 */
public class CommandLineParser
{
   /**
    * Constant indicating that the provided line is command line
    */
   public static int COMMAND_LINE = 1;

   /**
    * Constant indicating that the provided line is a list of pathes seperated by path.seperator
    */
   public static int PATH_LIST = 2;

   /**
    * Constant indicating no error occured during parsing
    */
   public final static int NO_ERROR = 0;

   /**
    * Constant indicating a wrong wildcard expression was found (like Elevator*)
    */
   public final static int WRONG_EXPRESSION = 1;

   /**
    * Constant indicating that the expression contains no command
    */
   public final static int NO_COMMAND = 2;

   /**
    * Constant indicating that the provided String is empty (no job to do)
    */
   public final static int EMPTY_COMMANDLINE = 4;

   /**
    * Constant indicating that a provided pathname is not valid
    */
   public final static int PATH_NOT_EXISTS = 8;

   /**
    * Contains the current parsing mode (COMMAND_LINE, PATH_LIST);
    */
   private int mode = 0;

   /**
    * Keeps the last parsing error
    */
   private int errorType = NO_ERROR;

   /**
    * The commandLine with substituted wildcards
    */
   private String commandLine = "";

   /**
    * Holds a vecor of File objects with all matching files of the expression
    */
   private Vector allFiles = new Vector();


   /**
    * Constructor for CommandLineParser. Calls parseLine with the provided String.
    *
    * @param line                        The String to be parsed for wildcards.
    * @throws IllegalArgumentException,  if a invalid wildcard expression is found in the provided
    *      String
    */

   public CommandLineParser (String line)
   {
      mode = COMMAND_LINE;
      parseLine (line);
   }


   /**
    * Constructor for class CommandLineParser
    *
    * @param line       No description provided
    * @param selection  No description provided
    * @param type       No description provided
    */
   public CommandLineParser (String line, String selection, int type)
   {
      switch (type)
      {
         case 1: //COMMAND_LINE
            mode = COMMAND_LINE;
            parseLine (line);
            break;
         case 2: //PATH_LIST
            mode = PATH_LIST;
            StringTokenizer token = new StringTokenizer (selection, ",");
            while (token.hasMoreTokens())
            {
               parseLine (createPathListSelection (line, token.nextToken()));
            }
            break;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param line       No description provided
    * @param selection  No description provided
    * @return           No description provided
    */
   private String createPathListSelection (String line, String selection)
   {
      final char fileseparator = System.getProperty ("file.separator").charAt (0); //the fileseperator ('/' for Linux)
      final char pathseparator = System.getProperty ("path.separator").charAt (0); //the pathseperator (':' for Linux)

      int pos = 0;
      int start = 0;
      String pathLS = "";

      if (line.length() == 0)
      {
         return "";
      }

      boolean win32 = System.getProperty ("os.name").startsWith ("Windows");

      while (start < line.length())
      {
         pos = line.indexOf (pathseparator, start);
         if (pos == -1)
         {
            if (win32)
            {
               pathLS += "\"";
            } //Support for long file and directory names

            pathLS += line.substring (start) + fileseparator + selection;

            if (win32)
            {
               pathLS += "\"";
            } //Support for long file and directory names
            return pathLS;
         }

         if (win32)
         {
            pathLS += "\"";
         }

         pathLS += line.substring (start, pos) + fileseparator + selection;

         if (win32)
         {
            pathLS += "\" ";
         }
         else
         {
            pathLS += " ";
         }

         start = pos + 1;
      }
      return pathLS;
   }


   /**
    * Returns the last parsed commandLine.
    *
    * @return   The commandLine value
    */
   public String getCommandLine()
   {
      return commandLine;
   }


   /**
    * Returns a vector of File objects with all matching files
    *
    * @return   The fileVector value
    */
   public Vector getFileVector()
   {
      return allFiles;
   }


   /**
    * Returns an array of File objects with all matching files
    *
    * @return   The fileArray value
    */
   public File[] getFileArray()
   {
      if (!allFiles.isEmpty())
      {
         File[] files = new File[allFiles.size()];
         for (int i = 0; i < allFiles.size(); i++)
         {
            files[i] = (File) allFiles.get (i);
         }
         return files;
      }
      else
      {
         return new File[]{};
      }
   }


   /**
    * Returns the last error
    *
    * @return   The errorState value
    */
   public int getErrorState()
   {
      return errorType;
   }


   /**
    * The provided String will be examined for wildcards. All wildcards found will be substituted
    * with the complete path/filenames for the matching files. If no path is specified, parseLine
    * uses UMLProject.getRootDir(). All other parts of the provided String will be copied.
    *
    * @param line                        The String to be parsed for wildcards.
    * @throws IllegalArgumentException,  if a invalid wildcard expression is found in the provided
    *      String
    */
   public void parseLine (String line)
   {
      final char fileseparator = System.getProperty ("file.separator").charAt (0); //the fileseperator ('/' for Linux)
      String filter; //the fileseperator ('/' for Linux)
      String path; //filter and path components of the wildcard expression
      String expPath = GeneralPreferences.get().getExportFolder() + fileseparator; //if no path is provided scan here for files
      int currPos = -1; //last wildcard position
      int copyPos = 0; //last copy position for all non wildcard expressions
      int start = 0; //starting point of the wildcard expression
      int end = 0; //ending point of the wildcard expression, POINTS ON THE BLANK that follows the expression
      File[] files; //Array of filenames matching the wildcard expression
      boolean longFilename = false; //Support for long filenames

      commandLine = "";

      if (line.length() == 0)
      {
         errorType = EMPTY_COMMANDLINE;
         throw new IllegalArgumentException ("Empty command.");
      }

      while (true)
      {
         currPos = line.indexOf ("*", end);
         if (currPos == -1)
         {
            break;
         } //no * found

         end = line.indexOf (' ', currPos);
         if (end == -1)
         {
            end = line.length();
         }

         //Support for long file and directory names
         if (line.charAt (end - 1) == '"')
         {
            longFilename = true;
            end--;
         }

         //Support for long file and directory names
         if (longFilename)
         {
            start = line.substring (0, currPos).lastIndexOf ('"');
            if (start == -1)
            {
               throw new IllegalArgumentException ("No beginning quotes for long filename");
            }
            start++;
         }
         else
         {
            start = line.substring (0, currPos).lastIndexOf (' ') + 1;
         }

         if (start == 0 && mode == COMMAND_LINE)
         {
            errorType = NO_COMMAND;
            commandLine = "";
            throw new IllegalArgumentException ("No command in expression.");
         }

         if (line.charAt (currPos - 1) == '.')
         { // (.../)Elevator.*
            int pathEnd = line.substring (start, currPos).lastIndexOf (fileseparator);

            if (pathEnd == -1)
            {
               path = expPath;
               filter = line.substring (start, currPos - 1);
            }
            else
            {
               path = line.substring (start, pathEnd + 1);
               filter = line.substring (pathEnd + 1, currPos - 1);
            }

            File myFile = new File (path);
            files = myFile.listFiles (new NameFilter (filter));
         }
         else if (line.length() >  (currPos + 1) && line.charAt (currPos + 1) == '.')
         { // (.../)*.java
            if (line.charAt (currPos - 1) == fileseparator)
            {
               path = line.substring (start, currPos);
            }
            else
            {
               path = expPath;
            }
            filter = line.substring (currPos + 2, end);
            File myFile = new File (path);
            files = myFile.listFiles (new ExtFilter (filter));
         }
         else if (start == end - 1)
         { // *
            path = expPath;
            File myFile = new File (path);
            files = myFile.listFiles();
         }
         else if (line.charAt (currPos - 1) == fileseparator)
         { // .../*
            path = line.substring (start, currPos);
            File myFile = new File (path);
            files = myFile.listFiles();
         }
         else
         { //wildcard expression does not match with the valid models
            errorType = WRONG_EXPRESSION;
            commandLine = "";
            throw new IllegalArgumentException ("Wrong wildcard expression: " + line.substring (start, end));
         }

         //if the filelist is null, the pathname does not denote a existing path
         if (files == null && mode == COMMAND_LINE)
         {
            errorType = PATH_NOT_EXISTS;
            commandLine = "";
            throw new IllegalArgumentException ("Path does not exist: " + path);
         }

         /*
          *  Before copying the other parts of the expression,
          *  jump out of the quotes of a long filename.
          */
         if (longFilename)
         {
            start--;
            end++;
         }

         //copy leading parts of the expression
         if (copyPos < start)
         {
            commandLine += line.substring (copyPos, start);
         }
         copyPos = end;

         //copy substituted filenames to the commandLine
         if (files != null)
         {
            for (int i = 0; i < files.length; i++)
            {
               if (files[i].isFile())
               {
                  if (longFilename)
                  {
                     commandLine += "\"";
                  } //Support for long file and directory names
                  commandLine += files[i].getPath();
                  if (longFilename)
                  {
                     commandLine += "\"";
                  } //Support for long file and directory names
                  commandLine += " ";

                  allFiles.add (files[i]);
               }
            }
         }

      } //while

      //copy bottom parts of the expression
      if (copyPos < line.length())
      {
         commandLine = commandLine + line.substring (copyPos, line.length());
      }
   }


   /**
    * Inner class of CommandLineParser to filter files using their names purposes
    *
    * @author    $Author: schneider $
    * @version   $Revision: 1.19 $
    */
   private static class NameFilter implements FilenameFilter
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      String start;


      /**
       * Constructor for class NameFilter
       *
       * @param start  No description provided
       */
      public NameFilter (String start)
      {
         this.start = start;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param dir   No description provided
       * @param name  No description provided
       * @return      No description provided
       */
      public boolean accept (File dir, String name)
      {
         return name.startsWith (start);
      }
   }


   /**
    * Inner class of CommandLineParser to filter files using their extensions purposes
    *
    * @author    $Author: schneider $
    * @version   $Revision: 1.19 $
    */
   private static class ExtFilter implements FilenameFilter
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      String ext;


      /**
       * Constructor for class ExtFilter
       *
       * @param ext  No description provided
       */
      public ExtFilter (String ext)
      {
         this.ext = ext;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param dir   No description provided
       * @param name  No description provided
       * @return      No description provided
       */
      public boolean accept (File dir, String name)
      {
         return name.endsWith (ext);
      }
   }
}

/*
 * $Log: CommandLineParser.java,v $
 * Revision 1.19  2004/10/20 17:49:27  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
