/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.basic;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import java.util.Iterator;
import java.util.Vector;

import javax.swing.*;
import javax.swing.border.EmptyBorder;
import javax.swing.event.InternalFrameAdapter;
import javax.swing.event.InternalFrameEvent;
import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.app.FrameMain;
import de.uni_paderborn.fujaba.app.action.ExportFilesAction;
import de.uni_paderborn.fujaba.preferences.GeneralPreferences;
import de.upb.tools.fca.FTreeSet;


/**
 * This class represents an internal window. It executes a command via the Process.exec method.
 * Two included threads poll the input and the error stream of the forked process and writes
 * the characters into to text area.
 *
 * @author    $Author: l3_g5 $
 * @version   $Revision: 1.65.2.4 $
 */
public class ProcessOutputViewer extends JInternalFrame
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (ProcessOutputViewer.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static String[] states =
      {"waiting...", "starting...", "running...", "finished"};

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   String exitText = "Exit";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JTextArea normalOutput = new JTextArea();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JButton exit = new JButton (exitText);
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JButton execute = new JButton ("Execute");

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JButton recompile = new JButton ("Again");

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JButton clearHistory = new JButton ("Clear History");
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JButton clearDisplay = new JButton ("Clear Display");
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JLabel processCommandLabel = new JLabel ("Command: ");
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JComboBox commandLineBox = new JComboBox();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JLabel processStateLabel = new JLabel ("State: " + states[0]);
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int maxProcessStates = 4;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int curProcessState = 0;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private FTreeSet processListeners = new FTreeSet();

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private StreamPoller normalOutputPoller = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private StreamPoller errorOutputPoller = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   Process process = null;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private ActionListener exitAction = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JPanel advancedPanel;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private Map getenv()
   {
      //todo: the following is a quick hack to get this running under Linux
      // use
      // return System.getenv();
      // if Java 5 may be used
      Map env = new HashMap();
      LineNumberReader reader = null;
      try
      {
         reader = new LineNumberReader (new FileReader ("/proc/self/environ"));
         try
         {
            String[] lines = reader.readLine().split ("\000");
            for (int i = 0; i < lines.length; i++)
            {
               String line = lines[i];
               int n = line.indexOf ('=');
               env.put (line.substring (0, n), line.substring (n + 1));
            }
         }
         finally
         {
            try
            {
               reader.close();
            }
            catch (IOException ioe)
            {
               /*
                *  ignore
                */
            }
         }
      }
      catch (Exception e)
      {
         // /proc/self/environ did not exist, try env command
         try
         {
            Process process = Runtime.getRuntime().exec ("env");
            reader = new LineNumberReader (new InputStreamReader (process.getInputStream()));
            for (String line;  (line = reader.readLine()) != null; )
            {
               int n = line.indexOf ('=');
               env.put (line.substring (0, n), line.substring (n + 1));
            }
         }
         catch (Exception e1)
         {
            //probably no linux
            return env;
         }
      }
      return env;
   }


   /**
    * store the value for field environment
    */
   private Map environment = getenv();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param properties  No description provided
    * @return            No description provided
    */
   String[] convertEnvironment (Map properties)
   {

      final String[] environment = new String[properties.size()];
      int i = 0;
      for (Iterator it = properties.entrySet().iterator(); it.hasNext(); )
      {
         Map.Entry entry = (Map.Entry) it.next();
         environment[i++] = entry.getKey() + "=" + entry.getValue();
      }
      return environment;
   }


   /**
    * getter for field environment
    *
    * @return   current value of field environment
    */
   public Map getEnvironment()
   {
      return this.environment;
   }


   /**
    * Constructor for class ProcessOutputViewer
    */
   public ProcessOutputViewer()
   {
      this (false, true);
   }


   /**
    * Constructor for class ProcessOutputViewer
    *
    * @param calledByCompileAction  No description provided
    * @param show                   No description provided
    */
   public ProcessOutputViewer (boolean calledByCompileAction, boolean show)
   {
      super();

      this.calledByCompileAction = calledByCompileAction;

      setTitle ("Process Output Viewer");
      setResizable (true);
      setClosable (true);
      setMaximizable (true);
      setIconifiable (true);

      // add window listener in order to stop the internal threads
      addInternalFrameListener (
         new InternalFrameAdapter()
         {
            /**
             * Invoked when an internal frame is in the process of being closed. The close
             * operation can be overridden at this point.
             *
             * @param e  No description provided
             */
            public void internalFrameClosing (InternalFrameEvent e)
            {
               stopThreads();
               dispose();
            }
         });

      // construct the frame
      JPanel container = new JPanel();

      container.setLayout (new BorderLayout (20, 10));

      JPanel exitPanel = new JPanel (new BorderLayout (5, 5));
      exitPanel.add (processStateLabel, BorderLayout.WEST);
      exit.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent e)
            {
               exitButtonPressed();
               if (getExitAction() != null && exit.getText().equals (exitText))
               {
                  getExitAction().actionPerformed (e);
               }
            }
         });

      execute.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent e)
            {
               executeButtonPressed();
            }
         });

      recompile.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent e)
            {
               recompileButtonPressed();
            }
         });

      clearHistory.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent e)
            {
               commandLineBox.removeAllItems();
            }
         });
      clearDisplay.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent e)
            {
               normalOutput.setText (null);
               if (listeners != null)
               {
                  for (Iterator iter = listeners.iterator(); iter.hasNext(); )
                  {
                     OutputListener listener = (OutputListener) iter.next();
                     listener.outputCleared();
                  }
               }
            }
         });

      JButton advanced = new JButton ("more...");

      advanced.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent e)
            {
               advancedPanel.setVisible (!advancedPanel.isVisible());
            }
         });

      Vector history = GeneralPreferences.get().getProcessHistory();
      if (history != null)
      {
         Iterator iter = history.iterator();
         while (iter.hasNext())
         {
            commandLineBox.addItem (iter.next());
         }
      }

      JPanel advancedButtonPanel = new JPanel (new FlowLayout());

      JPanel buttonPanel = new JPanel (new FlowLayout (FlowLayout.RIGHT));

      advancedButtonPanel.add (clearDisplay);
      advancedButtonPanel.add (clearHistory);
      advancedButtonPanel.add (execute);
      buttonPanel.add (recompile);
      buttonPanel.add (advanced);
      buttonPanel.add (exit);

      exitPanel.add (advancedButtonPanel, BorderLayout.EAST);
      exitPanel.add (new JSeparator(), BorderLayout.SOUTH);

      advancedPanel = new JPanel (new BorderLayout (5, 5));
      advancedPanel.setBorder (new EmptyBorder (5, 5, 5, 5));

      JPanel commandPanel = new JPanel (new BorderLayout());
      commandPanel.add (processCommandLabel, BorderLayout.WEST);
      commandPanel.add (commandLineBox, BorderLayout.CENTER);

      advancedPanel.add (commandPanel, BorderLayout.NORTH);
      advancedPanel.add (exitPanel, BorderLayout.SOUTH);

      advancedPanel.setVisible (false);
      JPanel southPanel = new JPanel (new BorderLayout());
      southPanel.add (advancedPanel, BorderLayout.SOUTH);
      southPanel.add (buttonPanel, BorderLayout.CENTER);

      container.add (southPanel, BorderLayout.SOUTH);

      Font nonPropFont = new Font ("Courier", 0, 12);

      JPanel tmpPanel = new JPanel (new BorderLayout());
//      tmpPanel.add (new JLabel ("Output"), BorderLayout.NORTH);

      JScrollPane scrollPane = new JScrollPane (ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS, ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS);

      normalOutput = new AutoScrollTextArea (scrollPane);
      normalOutput.setFont (nonPropFont);
      normalOutput.setEditable (false);
      normalOutput.setWrapStyleWord (true);

      scrollPane.getViewport().add (normalOutput);
      tmpPanel.add (scrollPane, BorderLayout.CENTER);

      container.add (tmpPanel, BorderLayout.CENTER);

      getContentPane().add (container);

      resetProcessState();

      if (show)
      {
         FrameMain.get().getDesktop().add (this);
      }

// calculate size and location of this frame
//      Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
//      Insets insets = Toolkit.getDefaultToolkit().getScreenInsets (getGraphicsConfiguration());
//      Dimension dim = new Dimension();
//      dim.width = Math.min (700, screenSize.width * 3 / 4);
//      dim.height = Math.min (480, screenSize.height * 3 / 4);
//      setSize (dim);
//      setLocation (screenSize.width - insets.right - dim.width, screenSize.height - insets.bottom - dim.height);

      if (getParent() != null)
      {
         Dimension size = new Dimension (Math.min (Math.max ((int) getPreferredSize().getWidth(), 500), getParent().getWidth()),
            Math.min (Math.max ((int) getPreferredSize().getHeight(), 400), getParent().getHeight()));
         setSize (size);
         setLocation (getParent().getWidth() - getWidth(), getParent().getHeight() - getHeight());
      }

   }


   /**
    * Sets the mouseListener attribute of the ProcessOutputViewer object
    *
    * @param mouseListener  The new mouseListener value
    */
   public void setMouseListener (MouseListener mouseListener)
   {
      if (mouseListener != null)
      {
         normalOutput.addMouseListener (mouseListener);
      }
   }


   /**
    * to decide if CompileAction has called POV
    */
   public boolean calledByCompileAction;


   /**
    * Switches the state label and enables the exit button if finished. 'state: ' is added
    * as a prefix.
    */
   public void switchProcessState()
   {
      curProcessState =  (curProcessState + 1) % maxProcessStates;
      processStateLabel.setText ("state: " + states[curProcessState]);
      if (curProcessState == 1)
      {
         exit.setText ("Abort");
         execute.setEnabled (false);
         recompile.setEnabled (false);
         clearHistory.setEnabled (false);
         clearDisplay.setEnabled (false);
         commandLineBox.setEnabled (false);
      }
      if (curProcessState == maxProcessStates - 1)
      {
         exit.setText (exitText);
         execute.setEnabled (true);
         recompile.setEnabled (true);
         clearHistory.setEnabled (true);
         clearDisplay.setEnabled (true);
         commandLineBox.setEnabled (true);
         commandLineBox.setEditable (true);
      }
   }


   /**
    * Resets the state label and the exit/abort button's text.
    */
   public void resetProcessState()
   {
      curProcessState = 3;
      normalOutput.setText ("");
      exit.setText (exitText);
      execute.setEnabled (true);
      recompile.setEnabled (true);
      clearHistory.setEnabled (true);
      commandLineBox.setEnabled (true);
      commandLineBox.setEditable (true);
      processStateLabel.setText ("state: " + states[curProcessState]);
   }


   /**
    * Sets the exitAction attribute of the ProcessOutputViewer object
    *
    * @param act  The new exitAction value
    */
   public void setExitAction (ActionListener act)
   {
      exitAction = act;
      if (act != null)
      {
         exitText = "Continue";
      }
      else
      {
         exitText = "Exit";
      }
      exit.setText (exitText);
   }


   /**
    * Get the exitAction attribute of the ProcessOutputViewer object
    *
    * @return   The exitAction value
    */
   public ActionListener getExitAction()
   {
      return this.exitAction;
   }


   /**
    * Get the commandLine attribute of the ProcessOutputViewer object
    *
    * @return   The commandLine value
    */
   String getCommandLine()
   {
      return (String) commandLineBox.getSelectedItem();
   }


   /**
    * Sets the commandLine attribute of the ProcessOutputViewer object
    *
    * @param commandLine  The new commandLine value
    */
   private void setCommandLine (String commandLine)
   {
      boolean contains = false;

      commandLine = commandLine.trim();

      for (int i = 0; i < commandLineBox.getItemCount(); i++)
      {
         if (commandLine.equals (commandLineBox.getItemAt (i)))
         {
            contains = true;
            commandLineBox.setSelectedItem (commandLine);
         }
      }

      if (!contains)
      {
         commandLineBox.addItem (commandLine);
         commandLineBox.setSelectedItem (commandLine);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param event  No description provided
    */
   void notifyAllProcessListeners (ProcessEvent event)
   {
      Iterator iter = processListeners.iterator();
      ProcessListener tmpListener;
      while (iter.hasNext())
      {
         tmpListener = (ProcessListener) iter.next();
         tmpListener.processFinished (event);
      }
   }


   /**
    * Adds a new listner to the ProcessOutputViewer who is interested to know when the Process
    * is finished.
    *
    * @param newListener  The new listener.
    */
   public void addProcessListener (ProcessListener newListener)
   {
      // just add the new listener to the OrderedSet
      processListeners.add (newListener);
   }


   /**
    * Shows the ProcessOutputViewer and executes the command
    *
    * @param commandLine  No description provided
    */
   public void executeCommand (String commandLine)
   {
      String commandName = commandLine;
      if (commandName != null)
      {
         final int indexOfSpace = commandName.indexOf (' ');
         if (indexOfSpace > 0)
         {
            commandName = commandName.substring (0, indexOfSpace);
         }
         final int indexOfSeperator = commandName.indexOf (File.separatorChar);
         if (indexOfSeperator > 0 && indexOfSeperator < commandName.length() - 1)
         {
            commandName = commandName.substring (indexOfSeperator + 1);
         }
         if (commandName.length() > 25)
         {
            commandName = commandName.substring (0, 22) + "...";
         }
      }
      executeCommand (commandLine, commandName);
   }


   /**
    * Shows the ProcessOutputViewer and executes the command
    *
    * @param commandLine  No description provided
    * @param title        No description provided
    */
   public void executeCommand (String commandLine, String title)
   {
      if (!this.isShowing())
      {
         this.setVisible (true);
         setTitle (title);
      }
      quietExecuteCommand (commandLine);
   }


   /**
    * Executes the command, shows ProcessOutputView only if an error occures
    *
    * @param commandLine  No description provided
    */
   public void quietExecuteCommand (String commandLine)
   {
      if (process != null | commandLine == null)
      {
         return;
      }

      switchProcessState();
      setCommandLine (commandLine);
      switchProcessState();

      Thread processThread = new MyThread (this);
      processThread.start();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param cmdLine  No description provided
    * @return         No description provided
    */
   public boolean compileAndWaitForResult (String cmdLine)
   {
      try
      {
         CommandLineParser parser = new CommandLineParser (cmdLine);
         cmdLine = parser.getCommandLine();
         log.info ("Going to invoke: " + cmdLine);
         process = Runtime.getRuntime().exec (cmdLine, convertEnvironment (getEnvironment()));
         startThreads();
         process.waitFor();
         dispose();
         if (listeners != null)
         {
            for (Iterator iter = listeners.iterator(); iter.hasNext(); )
            {
               OutputListener listener = (OutputListener) iter.next();
               listener.outputFinished (getOutput(), process.exitValue());
            }
         }
         return  (process.exitValue() == 0);
      }
      catch (Exception e)
      {
         e.printStackTrace();
      }
      return false;
   }


   /**
    * Shows if a process is currently running
    *
    * @return   The running value
    */
   public boolean isRunning()
   {
      return  (curProcessState != 3);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: l3_g5 $
    * @version   $Revision: 1.65.2.4 $
    */
   class MyThread extends Thread
   {

      /**
       * Contains the caller (a JPanel), if the caller is not visible during executing a command
       * and an error occures, it will made be visible.
       */
      private JInternalFrame frame;


      /**
       * Constructor for class MyThread
       *
       * @param aframe  No description provided
       */
      public MyThread (JInternalFrame aframe)
      {
         frame = aframe;
      }


      /**
       * Main processing method for the MyThread object
       */
      public void run()
      {
         try
         {
            switchProcessState();

            GeneralPreferences options = GeneralPreferences.get();
            if (options.isCommandLineParsing())
            {
               CommandLineParser myParser = new CommandLineParser (getCommandLine());
               process = Runtime.getRuntime().exec (myParser.getCommandLine(), convertEnvironment (getEnvironment()));
            }
            else
            {
               process = Runtime.getRuntime().exec (getCommandLine(), convertEnvironment (getEnvironment()));
            }

            startThreads();

            process.waitFor();
            notifyAllProcessListeners (new ProcessEvent (process.exitValue()));
            sleep (1000);

            stopThreads();
            switchProcessState();

            if (listeners != null)
            {
               for (Iterator iter = listeners.iterator(); iter.hasNext(); )
               {
                  OutputListener listener = (OutputListener) iter.next();
                  listener.outputFinished (getOutput(), process.exitValue());
               }
            }
            // Changed Process exit code = 0 to  compiled successfullyif it is so
            if (process.exitValue() == 0 && calledByCompileAction &&  (getExitAction() != null))
            {
               normalOutput.append ("Project compiled successfully!" + "\n\n" +
                  "Please be patient while starting Dobs." + "\n\n");

               // Dobs starts automatically
               exitButtonPressed();
               getExitAction().actionPerformed (null);

            }
            else
            {
               normalOutput.append ("Process exit code = " + process.exitValue() + "\n\n");
            }
            process = null;
         }
         catch (Exception except)
         {
            if (!frame.isShowing())
            {
               frame.setVisible (true);
            }
            except.printStackTrace();
            JOptionPane.showMessageDialog (ProcessOutputViewer.this,
               "Error occurred starting process: " + except.toString(),
               "Process",
               JOptionPane.ERROR_MESSAGE);
            stopThreads();
            resetProcessState();
         }
      }
   }


   /**
    * Get the output attribute of the ProcessOutputViewer object
    *
    * @return   The output value
    */
   public String getOutput()
   {
      return normalOutput.getText();
   }


   /**
    * Starts the internal threads.
    */
   void startThreads()
   {
      if (process == null || normalOutputPoller != null || errorOutputPoller != null)
      {
         return;
      }

      normalOutputPoller = new StreamPoller (process.getInputStream(), normalOutput);
      errorOutputPoller = new StreamPoller (process.getErrorStream(), normalOutput);

      new Thread (normalOutputPoller).start();
      new Thread (errorOutputPoller).start();
   }


   /**
    * Stops the internal threads.
    */
   void stopThreads()
   {
      if (normalOutputPoller != null)
      {
         normalOutputPoller.stop();
      }

      if (errorOutputPoller != null)
      {
         if ( (!this.isShowing()))
         {
            this.setVisible (true);
         }
         errorOutputPoller.stop();
      }

      normalOutputPoller = null;
      errorOutputPoller = null;
   }


   /**
    * Disposes the frame in order to close it.
    */
   public void exitButtonPressed()
   {
      if (exit.getText().equals ("Abort"))
      {
         process.destroy();
      }
      else
      {
         stopThreads();

         GeneralPreferences option = GeneralPreferences.get();
         Vector history = new Vector();
         for (int i = 0; i < commandLineBox.getItemCount(); i++)
         {
            history.add (commandLineBox.getItemAt (i));
         }
         option.setProcessHistory (history);
         dispose();
      }
   }


   /**
    * Execute the command line.
    */
   public void executeButtonPressed()
   {
      process = null;
      executeCommand (getCommandLine());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void recompileButtonPressed()
   {
      // clear display
      normalOutput.setText (null);

      // first regenerate if in compile action
      if (isAutoContinueEnabled())
      {
         //we are likely to be a compile process
         ExportFilesAction myexport = new ExportFilesAction();
         myexport.actionPerformed (null);
      }

      // second compile it
      executeButtonPressed();
   }


   /**
    * Get the autoContinueEnabled attribute of the ProcessOutputViewer object
    *
    * @return   The autoContinueEnabled value
    */
   public boolean isAutoContinueEnabled()
   {
      return calledByCompileAction;
   }


   /**
    * The main method allows to start the process output viewer separately
    *
    * @param argv  No description provided
    */
   public static void main (String argv[])
   {
      ProcessOutputViewer pv = new ProcessOutputViewer();
      pv.setCommandLine ("java de.uni_paderborn.fujaba.app.FujabaApp");
      pv.show();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: l3_g5 $
    * @version   $Revision: 1.65.2.4 $ $Date: 2006/06/09 08:40:34 $
    */
   public interface OutputListener
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      public void outputCleared();


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param newLine  No description provided
       */
      public void outputAppended (String newLine);


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param output     No description provided
       * @param exitValue  No description provided
       */
      public void outputFinished (String output, int exitValue);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   Collection listeners;


   /**
    * Access method for a To N-association.
    *
    * @param listener  The object added.
    */
   public void addOutputListener (OutputListener listener)
   {
      if (listeners == null)
      {
         listeners = new LinkedList();
      }
      listeners.add (listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param listener  No description provided
    */
   public void removeOutputListener (OutputListener listener)
   {
      if (listeners != null)
      {
         listeners.remove (listener);
      }
   }


   /**
    * This class polls an input stream and appends the input to the text in a text area. Be careful,
    * if you create a thread out of this class, remind that you have to stop the thread explicitly,
    * because the run method is implemented as a never ending loop.
    *
    * @author    $Author: l3_g5 $
    * @version   $Revision: 1.65.2.4 $
    */
   class StreamPoller implements Runnable
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private InputStreamReader in = null;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private JTextArea out = null;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private boolean stopped = false;


      /**
       * Constructor for class StreamPoller
       *
       * @param in   No description provided
       * @param out  No description provided
       */
      public StreamPoller (InputStream in, JTextArea out)
      {
         this.in = new InputStreamReader (in);
         this.out = out;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      public synchronized void stop()
      {
         stopped = true;
      }


      /**
       * Main processing method for the StreamPoller object
       */
      public void run()
      {
         StringBuffer buf = new StringBuffer();
         try
         {
            while (!stopped)
            {
               char errorText[] = new char[4000];
               int len = in.read (errorText, 0, 4000);
               if (len > 0)
               {
                  String text = new String (errorText, 0, len);
                  out.append (text);
                  buf.append (text);
               }
               else if (len < 0)
               {
                  //eof
                  return;
               }
               // inform listeners if new line was added
               int index = buf.indexOf ("\n");
               while (index > 0)
               {
                  String newLine = buf.substring (0, index);
                  buf.delete (0, index + 1);
                  index = buf.indexOf ("\n");

                  if (listeners != null)
                  {
                     for (Iterator iter = listeners.iterator(); iter.hasNext(); )
                     {
                        OutputListener listener = (OutputListener) iter.next();
                        listener.outputAppended (newLine);
                     }
                  }
               }
            }
         }
         catch (IOException except)
         {
            except.printStackTrace();
         }
      }
   }


   /**
    * TextArea which scrolls down when something is appended. The JScrollPane holding this Object
    * needs to be passed to the Constructor.
    *
    * @author    $Author: l3_g5 $
    * @version   $Revision: 1.65.2.4 $
    */
   class AutoScrollTextArea extends JTextArea
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private JScrollPane scrollPane;


      /**
       * Constructor
       *
       * @param scrollPane  scrollPane to which this Component gets added to
       */
      public AutoScrollTextArea (JScrollPane scrollPane)
      {
         this.scrollPane = scrollPane;
      }


      /**
       * overridden from JTextArea so we scroll down when this method is called
       *
       * @param str
       * @see        JTextArea#append
       */
      public void append (String str)
      {
         super.append (str);
         Adjustable a = scrollPane.getVerticalScrollBar();
         a.setValue (a.getMaximum());
      }
   }
}

/*
 * $Log: ProcessOutputViewer.java,v $
 * Revision 1.65.2.4  2006/06/09 08:40:34  l3_g5
 * one more way to copy environment
 *
 */
