/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import java.util.*;

import de.uni_paderborn.fujaba.metamodel.*;


/**
 * Class AbstractCPPGenVisitor
 *
 * @author    $Author: mtt $
 * @version   $Revision: 1.39.2.1 $
 */
public abstract class AbstractCPPGenVisitor
    extends OOGenVisitor
{
   /**
    * Default Constructor
    *
    * @param extender  No description provided
    */
   public AbstractCPPGenVisitor (String extender)
   {
      super (extender);

      SDM_ENSURE_STRING = "CppSDM::ensure";

      SDM_EXCEPTION_STRING = "CppSDMException";

      POINTER_POSTFIX = "*";

      INTERFACE_STRING = "";

      NULL_STRING = "NULL";

      EXTENDS_STRING = ":";

      IMPLEMENTS_STRING = "";

      INTERRUPTED_EXCEPTION_STRING = "InterruptedException";

      ILLEGALDATA_EXCEPTION_STRING = "IllegalDataException";

      THIS_STRING = "this->";

      FINAL_STRING = "final";

      TRANSIENT_STRING = "";

      //don't use for class declarations
      ABSTRACT_STRING = "virtual";

      SYNCHRONIZED_STRING = "";

      NATIVE_STRING = "native";
   }

   //
   // begin assoc template section
   //

   /**
    * Get the referenceTemplateName attribute of the AbstractCPPGenVisitor object
    *
    * @return   The referenceTemplateName value
    */
   public String getReferenceTemplateName()
   {
      return "Templates/ReferenceTemplate.CPP.tpl";
   }


   /**
    * Get the assocTemplateName attribute of the AbstractCPPGenVisitor object
    *
    * @return   The assocTemplateName value
    */
   public String getAssocTemplateName()
   {
      return "Templates/AssocTemplateFCA.CPP.tpl";
   }

   //
   // end assoc template section
   //

   /**
    * get the full qualified containername for type <code>containerType</code>
    *
    * @param containerType  the <code>containerType</code>
    * @param bound          true, if assoc is bound
    * @return               the full qualified containername
    */
   public String getContainerName (OOCollectionEnum containerType, boolean bound)
   {
      StringBuffer containerName = new StringBuffer();

      containerName.append ("std::");

      if (OOCollectionEnum.ITERATOR.equals (containerType))
      {
         containerName.append ("Iterator");
      }
      if (OOCollectionEnum.LIST_ITERATOR.equals (containerType))
      {
         containerName.append ("ListIterator");
      }
      if (OOCollectionEnum.KEY_VALUE_PAIR.equals (containerType))
      {
         containerName.append ("MapEntry");
      }

      if (OOCollectionEnum.LIST.equals (containerType))
      {
         containerName.append ("list");
      }
      else if (OOCollectionEnum.HASHTABLE.equals (containerType))
      {
         containerName.append ("set");
      }
      else if (OOCollectionEnum.SET.equals (containerType))
      {
         containerName.append ("set");
      }
      else if (OOCollectionEnum.MAP.equals (containerType))
      {
         containerName.append ("map");
      }
      else if (OOCollectionEnum.MULTI_MAP.equals (containerType))
      {
         containerName.append ("duplicatedMap");
      }
      else if (OOCollectionEnum.TREESET.equals (containerType))
      {
         containerName.append ("set");
      }
      else if (OOCollectionEnum.SORTED_MAP.equals (containerType))
      {
         containerName.append ("map");
      }
      else if (OOCollectionEnum.SORTED_MULTI_MAP.equals (containerType))
      {
         containerName.append ("duplicatedMap");
      }

      return containerName.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void initProgLangTypes()
   {
      addToProgLangTypes (FBaseTypes.INITIALIZER, "");
      addToProgLangTypes (FBaseTypes.BOOLEAN, "bool");
      addToProgLangTypes (FBaseTypes.CHARACTER, "char");
      addToProgLangTypes (FBaseTypes.STRING, "string");
      addToProgLangTypes (FBaseTypes.INTEGER, "int");
      addToProgLangTypes (FBaseTypes.BYTE, "byte");
      addToProgLangTypes (FBaseTypes.SHORT_INTEGER, "short");
      addToProgLangTypes (FBaseTypes.LONG_INTEGER, "long");
      addToProgLangTypes (FBaseTypes.FLOAT, "float");
      addToProgLangTypes (FBaseTypes.DOUBLE, "double");
      addToProgLangTypes (FBaseTypes.VOID, "void");
      addToProgLangTypes (FBaseTypes.CONSTRUCTOR, "");

      addToProgLangTypes ("BooleanArray", "bool[]");
      addToProgLangTypes ("CharacterArray", "char[]");
      addToProgLangTypes ("StringArray", "string[]");
      addToProgLangTypes ("IntegerArray", "int[]");
      addToProgLangTypes ("ByteArray", "byte[]");
      addToProgLangTypes ("ShortIntegerArray", "short[]");
      addToProgLangTypes ("LongIntegerArray", "long[]");
      addToProgLangTypes ("FloatArray", "float[]");
      addToProgLangTypes ("DoubleArray", "double[]");
      addToProgLangTypes ("ObjectArray", "Object[]");

      // fix me
      addToProgLangTypes ("FWIterator", "iterator");

   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void initVisibilityStrings()
   {
      visibilityStrings = new String[]{
         "private",
         "public",
         "protected",
         "/*package*/",
         "/*user*/"
         };
   }


   /**
    * Get the methodName attribute of the OOGenVisitor object
    *
    * @param methodName  No description provided
    * @return            The methodName value
    */
   public String getMethodName (OOMethod methodName)
   {
      // int type = methodName.getType().getNr();
      String name = methodName.getName();

      StringBuffer result = new StringBuffer();

      if (methodName.getType() == OOMethodType.DEFAULT_METHOD)
      {
         if (methodName == OOMethod.STRING_COMPARE_METHOD)
         {
            return "strcmp";
         }
         return methodName.getName();
      }
      else if (methodName.getType() == OOMethodType.GET_METHOD)
      {
         result.append ("get");
      }
      else if (methodName.getType() == OOMethodType.IS_METHOD)
      {
         result.append ("is");
      }
      else if (methodName.getType() == OOMethodType.SET_METHOD)
      {
         result.append ("set");
      }
      else if (methodName.getType() == OOMethodType.HAS_IN_METHOD)
      {
         result.append ("hasIn");
      }
      else if (methodName.getType() == OOMethodType.ELEMENTS_OF_METHOD)
      {
         result.append ("elementsOf");
      }
      else if (methodName.getType() == OOMethodType.ITERATOR_OF_METHOD)
      {
         if (!"".equals (name))
         {
            result.append ("iteratorOf");
         }
         else
         {
            result.append ("iterator");
         }
      }
      else if (methodName.getType() == OOMethodType.OLD_STYLE_ELEMENTS)
      {
         result.append (name);
         result.append ("Elements");
         return result.toString();
      }
      else if (methodName.getType() == OOMethodType.REMOVE_YOU_METHOD)
      {
         return "removeYou";
      }
      else if (methodName.getType() == OOMethodType.ADD_METHOD)
      {
         if (!"".equals (name))
         {
            result.append ("addTo");
         }
         else
         {
            result.append ("add");
         }
      }
      else if (methodName.getType() == OOMethodType.REMOVE_METHOD)
      {
         if (!"".equals (name))
         {
            result.append ("removeFrom");
         }
         else
         {
            result.append ("remove");
         }
      }
      else if (methodName.getType() == OOMethodType.GET_FROM_METHOD)
      {
         result.append ("getFrom");
      }
      else if (methodName.getType() == OOMethodType.SIZE_OF_METHOD)
      {
         result.append ("sizeOf");
      }
      else if (methodName.getType() == OOMethodType.REMOVE_ALL_FROM_METHOD)
      {
         result.append ("removeAllFrom");

         // ordered, sorted assoc
      }
      else if (methodName.getType() == OOMethodType.INDEX_OF_METHOD)
      {
         result.append ("indexOf");
      }
      else if (methodName.getType() == OOMethodType.LAST_INDEX_OF_METHOD)
      {
         result.append ("lastIndexOf");
      }
      else if (methodName.getType() == OOMethodType.IS_BEFORE_OF_METHOD)
      {
         result.append ("isBeforeOf");
      }
      else if (methodName.getType() == OOMethodType.IS_AFTER_OF_METHOD)
      {
         result.append ("isAfterOf");
      }
      else if (methodName.getType() == OOMethodType.GET_FIRST_OF_METHOD)
      {
         result.append ("getFirstOf");
      }
      else if (methodName.getType() == OOMethodType.GET_LAST_OF_METHOD)
      {
         result.append ("getLastOf");
      }
      else if (methodName.getType() == OOMethodType.GET_NEXT_OF_METHOD)
      {
         result.append ("getNextOf");
      }
      else if (methodName.getType() == OOMethodType.GET_NEXT_INDEX_OF_METHOD)
      {
         result.append ("getNextIndexOf");
      }
      else if (methodName.getType() == OOMethodType.GET_PREVIOUS_OF_METHOD)
      {
         result.append ("getPreviousOf");
      }
      else if (methodName.getType() == OOMethodType.GET_PREVIOUS_INDEX_OF_METHOD)
      {
         result.append ("getPreviousIndexOf");
      }
      else if (methodName.getType() == OOMethodType.ADD_BEFORE_OF_METHOD)
      {
         result.append ("addBeforeOf");
      }
      else if (methodName.getType() == OOMethodType.ADD_AFTER_OF_METHOD)
      {
         result.append ("addAfterOf");

         // Qualified assoc methods.
      }
      else if (methodName.getType() == OOMethodType.GET_VALUE_FROM_METHOD)
      {
         result.append ("getValueFrom");
      }
      else if (methodName.getType() == OOMethodType.GET_KEY_FROM_METHOD)
      {
         result.append ("getKeyFrom");
      }
      else if (methodName.getType() == OOMethodType.HAS_VALUE_IN_METHOD)
      {
         result.append ("hasValueIn");
      }
      else if (methodName.getType() == OOMethodType.HAS_KEY_IN_METHOD)
      {
         result.append ("hasKeyIn");
      }
      else if (methodName.getType() == OOMethodType.HAS_ENTRY_IN_METHOD)
      {
         result.append ("hasEntryIn");
      }
      else if (methodName.getType() == OOMethodType.KEYS_OF_METHOD)
      {
         result.append ("keysOf");
      }
      else if (methodName.getType() == OOMethodType.REMOVE_VALUE_FROM_METHOD)
      {
         result.append ("removeValueFrom");
      }
      else if (methodName.getType() == OOMethodType.REMOVE_KEY_FROM_METHOD)
      {
         result.append ("removeKeyFrom");
      }
      else if (methodName.getType() == OOMethodType.REMOVE_ENTRY_FROM_METHOD)
      {
         result.append ("removeEntryFrom");
      }
      else if (methodName.getType() == OOMethodType.ENTRIES_OF_METHOD)
      {
         result.append ("entriesOf");
      }
      else if (methodName.getType() == OOMethodType.GET_AT_METHOD)
      {
         result.append ("get").append (CGU.upFirstChar (name)).append ("At");
         return result.toString();
      }
      else if (methodName.getType() == OOMethodType.IS_EMPTY_METHOD)
      {
         result.append ("is" + CGU.upFirstChar (name) + "Empty");
         return result.toString();
      }
      else if (methodName.getType() == OOMethodType.EQUALS_METHOD)
      {
         return EQUALS_METHOD_STRING;
      }
      else if (methodName.getType() == OOMethodType.ADD_TO_SET_METHOD)
      {
         return ADD_TO_SET_METHOD_STRING;
      }
      else if (methodName.getType() == OOMethodType.SET_CONTAINS_METHOD)
      {
         return SET_CONTAINS_METHOD_STRING;
      }

      result.append (CGU.upFirstChar (name));

      return result.toString();
   }

   //
   // begin visit OOStatement section
   //

   /**
    * Get the sourceCode attribute of the AbstractCPPGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public final String getSourceCode (OOThrowStatement statement)
   {
      StringBuffer result = new StringBuffer();

      result.append (createIndentString());
      result.append ("throw ");
      result.append (statement.getMyException().getSourceCode (this));

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public String getSourceCode (OOTypeCastExpr expression)
   {
      StringBuffer result = new StringBuffer();
      result.append ("dynamic_cast<");

      result.append (getOOTypeAsString (expression.getTypeName()));
      //if the type isn't a BaseType then it is a normal class.
      if (this.getFromProgLangTypes (getOOTypeAsString (expression.getTypeName())) == null)
      {
         result.append ("*");
      }
      result.append (">");
      result.append (" ( ");
      result.append (expression.getCastSource().getSourceCode (this));
      result.append (" ) ");
      return result.toString();
   }

   //
   // end visit OOStatement section
   //

   //
   // begin visit OOExpression section
   //

   /**
    * Get the sourceCode attribute of the AbstractCPPGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public String getSourceCode (OOObjectOfTypeExpr expression)
   {
      StringBuffer result = new StringBuffer();

      result.append ("dynamic_cast<");
      result.append (getOOTypeAsString (expression.getTypeName()));

      //if the type isn't a BaseType then it is a normal class.
      if (this.getFromProgLangTypes (getOOTypeAsString (expression.getTypeName())) == null)
      {
         result.append ("*");
      }

      result.append (">(");
      result.append (getVarName (expression.getObjectName()));
      result.append (")");

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the AbstractCPPGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public final String getSourceCode (OOClassOfVariableExpr expression)
   {
      return this + ".getSourceCode(" + expression + ") is empty !";
   }


   /**
    * Get the sourceCode attribute of the AbstractCPPGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public final String getSourceCode (OOFWIteratorOfExpr expression)
   {
      StringBuffer result = new StringBuffer();

      result.append (getVarName (expression.getObjectName()));
      result.append (".begin ()");

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the AbstractCPPGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public final String getSourceCode (OOFWIteratorGetExpr expression)
   {
      StringBuffer result = new StringBuffer();

      result.append ("*");
      result.append (getVarName (expression.getObjectName()));

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the AbstractCPPGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public String getSourceCode (OONewArrayExpr expression)
   {
      StringBuffer result = new StringBuffer();

      Iterator iter = expression.iteratorOfParameter();

      result.append ("new ");
      result.append (getOOTypeAsString (expression.getElementType()));
      result.append ("[");
      if (!iter.hasNext())
      {
         result.append (expression.getSizeExpr().getSourceCode (this));
      }
      result.append ("] ");

      if (iter.hasNext())
      {
         result.append ("{");

         while (iter.hasNext())
         {
            OOExpression tmpParameter = (OOExpression) iter.next();
            result.append (tmpParameter.getSourceCode (this));
            if (iter.hasNext())
            {
               result.append (", ");
            }
         }

         result.append ("}");
      }

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the AbstractCPPGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public final String getSourceCode (OOFWIteratorHasNextExpr expression)
   {
      StringBuffer result = new StringBuffer();

      result.append (getVarName (expression.getIterName()));
      result.append (" != ");

      if (expression.isUseAccessMethod())
      {
         //	 result.append ( START_BLOCK_COMMENTARY + " isUseAccessMethod undefined " + END_BLOCK_COMMENTARY ) ;
         //	 result.append ( downFirstChar (expression.getObjectName().getName ())) ;
         if (expression.getObjectName() != null)
         {
            result.append (getVarName (expression.getObjectName()));
            result.append ("->");
         }

         result.append ("end");
         result.append (CGU.upFirstChar (getMethodName (new OOMethod (getOOTypeAsString (expression.getElemType()),
            OOMethodType.ITERATOR_OF_METHOD))));
         result.append (" ()");
      }
      else
      {
         result.append (CGU.downFirstChar (getVarName (expression.getObjectName())) + "->end ()");
      }

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the AbstractCPPGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public final String getSourceCode (OOFWIteratorGotoNextExpr expression)
   {
      return getVarName (expression.getIterName()) + "++";
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public String getSourceCode (OOCallAttrExpr expression)
   {
      StringBuffer result = new StringBuffer();

      if ( (expression.getObjectName() != null) &&  (getVarName (expression.getObjectName()) != null))
      {
         result.append (getVarName (expression.getObjectName()));
         result.append ("->");
      }

      if ( (expression.getAttrName() != null) &&  (getVarName (expression.getAttrName()) != null))
      {
         result.append (getVarName (expression.getAttrName()));
      }
      else
      {
         result.append (" /* oops, no attr name */ ");
      }

      // append next call, if available
      if (expression.getNextCall() != null)
      {
         result.append ("->");
         result.append (expression.getNextCall().getSourceCode (this));
      }

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public String getSourceCode (OOCallMethodExpr expression)
   {
      StringBuffer result = new StringBuffer();

      if ( (expression.getObjectName() != null) &&  (getVarName (expression.getObjectName()) != null))
      {
         result.append (getVarName (expression.getObjectName()));
         if (!getMethodName (expression.getMethodName()).equals (EQUALS_METHOD_STRING))
         {
            result.append ("->");
         }
      }

      if (getMethodName (expression.getMethodName()).equals (EQUALS_METHOD_STRING))
      {
         result.append (" == ");
      }

      if (!getMethodName (expression.getMethodName()).equals (EQUALS_METHOD_STRING))
      {
         result.append (getMethodName (expression.getMethodName()));
         result.append (" ");
         result.append (OPEN_NORM_BRACKET);
      }

      Iterator iter = expression.iteratorOfParameter();
      OOExpression tmpParameter = null;
      while (iter.hasNext())
      {
         tmpParameter = (OOExpression) iter.next();
         result.append (tmpParameter.getSourceCode (this));
         if (iter.hasNext())
         {
            result.append (LIST_SEPARATOR);
            result.append (" ");
         }
      }

      if (!getMethodName (expression.getMethodName()).equals (EQUALS_METHOD_STRING))
      {
         result.append (CLOSE_NORM_BRACKET);
         // append next call, if available
         if (expression.getNextCall() != null)
         {
            if (!getMethodName ( ((OOCallMethodExpr) expression.getNextCall()).getMethodName()).equals (EQUALS_METHOD_STRING))
            {
               result.append ("->");
            }
            result.append (expression.getNextCall().getSourceCode (this));
         }
      }
      return result.toString();
   }


   //
   // end visit OOExpression section
   //

   /**
    * Get the uMLTypeAsString attribute of the OOGenVisitor object
    *
    * @param theType    No description provided
    * @param isPointer  No description provided
    * @return           The uMLTypeAsString value
    */
   public String getTypeAsString (FType theType, boolean isPointer)
   {
      String typeString = null;
      if (theType instanceof FBaseTypes)
      {
         typeString = getProgLangType ((FBaseTypes) theType);
      }
      else if (theType instanceof FArray)
      {
         typeString = getTypeAsString ( ((FArray) theType).getFArrayType(), false) + "[]";
      }
      else
      {
         if (theType instanceof FClass)
         {
            FClass theClass = (FClass) theType;
            typeString = theClass.getName();
            theClass = theClass.getFDeclaredInClass();
            while (theClass != null)
            {
               typeString = theClass.getName() + "->" + typeString;
               theClass = theClass.getFDeclaredInClass();
            }
         }
         else
         {
            typeString = theType.getName();
         }
         if (isPointer)
         {
            typeString += POINTER_POSTFIX;
         }
      }

      return typeString;
   }


   /**
    * Get the oOTypeAsString attribute of the AbstractCPPGenVisitor object
    *
    * @param theType  No description provided
    * @param isBound  No description provided
    * @return         The oOTypeAsString value
    */
   public String getOOTypeAsString (OOContainerType theType, boolean isBound)
   {
      StringBuffer result = new StringBuffer();
      int i;
      int length;

      if (theType.getCollectionType() != null)
      {
         result.append (getTypeNameAsString (getContainerName (theType.getCollectionType())));

         // add template parameter
         result.append ("<");
         length = theType.getParameter().length;
         for (i = 0; i < length; i++)
         {
            result.append (getOOTypeAsString (theType.getParameter()[i]));
            if (i + 1 < length)
            {
               result.append (",");
            }
         }

         result.append (">");
      }

      return result.toString();
   }

   //
   // generate section
   //
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param thePackage  No description provided
    * @return            No description provided
    */
   public String generateImportPackage (FPackage thePackage)
   {
      return "";
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theClass  No description provided
    * @return          No description provided
    */
   public String generateImportClass (FClass theClass)
   {
      return "";
   }

   //
   // create section
   //

   //
   // append section
   //

   //
   // get identifier, keywords, literals etc.
   //

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      super.removeYou();
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "AbstractGenVisitor[]";
   }
}

/*
 * $Log: AbstractCPPGenVisitor.java,v $
 * Revision 1.39.2.1  2006/03/08 11:16:36  mtt
 * removed final from getSourceCode(OOObjectOfTypeExpr)
 *
 */
