/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import java.io.*;
import java.util.StringTokenizer;
import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.metamodel.FFile;


/**
 * Class CodeGenVisitor
 *
 * <h2>Associations</h2>
 *
 * <pre>
 *                 -------- 0..1        hasVisitors        0..1
 * CodeGenStrategy | name |------------------------------------- CodeGenVisitor
 *                 -------- codeGenStrategy      codeGenVisitor
 *
 *                 0..1                        0..1
 * CodeGenVisitor ---------------------------------- CodeGenStrategy
 *                 currentVisitor   currentStrategy
 * </pre>
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.44.2.1 $
 */
public class CodeGenVisitor
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (CodeGenVisitor.class);

   // visitor types
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String JAVA_VISITOR = "Java";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String EIFFEL_VISITOR = "Eiffel";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String GNU_CPP_CLASS_VISITOR = "GnuCPPClass";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String GNU_CPP_HEADER_VISITOR = "GnuCPPHeader";

   // extender for duplicate files
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TMP_FILE_EXTENDER = "tmp";

   // constants for code generation
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String START_BLOCK_COMMENTARY = "/**";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String END_BLOCK_COMMENTARY = " */";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String LINE_IN_BLOCK_COMMENTARY = " * ";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String LINE_COMMENTARY = "//";

   // indent string for code generation
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String INDENT_STRING = "   ";


   /**
    * Default Constructor
    */
   public CodeGenVisitor() { }


   /**
    * Default Constructor
    *
    * @param extender  No description provided
    */
   public CodeGenVisitor (String extender)
   {
      setExtender (extender);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param path  No description provided
    * @param name  No description provided
    */
   public final void createFiles (String path, String name)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".createFiles(" + path + "," + name + ")");
         }
         if (log.isInfoEnabled())
         {
            log.info (this + ".creating:" + path + File.separator +
               name + "." + getExtender());
         }
      }

      // create missing directories
      createDirectory (path);

      // create files
      setCurrentFile (createFile (path, name + "." + getExtender()));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void initBuffer()
   {
      // create buffer
      setFirstFragment (null);

      // reset indentNr
      setIndentNr (0);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param path         No description provided
    * @param name         No description provided
    * @param createFiles  No description provided
    */
   public void initVisitor (String path, String name, boolean createFiles)
   {
      if (createFiles)
      {
         createFiles (path, name);
      }
      setPath (path);
      setName (name);

      initBuffer();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void writeFiles()
   {
      if (log.isDebugEnabled())
      {
         log.debug (this + ".writeFiles ()");
      }

      writeFragmentsToFile (getCurrentFile());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param path  No description provided
    */
   public final void createDirectory (String path)
   {
      // create missing directories
      File file = new File (path);
      if (!file.exists())
      {
         file.mkdirs();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param path  No description provided
    * @param name  No description provided
    * @return      No description provided
    */
   public final File createFile (String path, String name)
   {
      File newFile = new File (path, name);

      if (newFile.exists())
      {
         File tmpFile = new File (path, name + "." + TMP_FILE_EXTENDER);
         tmpFile.delete();
         newFile.renameTo (tmpFile);
         newFile = new File (path, name);
      }

      return newFile;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theBuffer  No description provided
    * @param theFile    No description provided
    * @return           No description provided
    */
   public final boolean writeBufferToFile (StringBuffer theBuffer, File theFile)
   {
      boolean success = true;

      FileWriter theWriter = null;

      try
      {
         theWriter = new FileWriter (theFile);
         success = writeToStream (theBuffer.toString(), theWriter);
         theWriter.flush();
         theWriter.close();
      }
      catch (IOException exception)
      {
         log.error (this + ".writeBufferToFile(" + theBuffer + "," + theFile + ") failed !");
         log.error ("exception=" + exception);

         success = false;
      }

      return success;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param text       No description provided
    * @param theWriter  No description provided
    * @return           No description provided
    */
   public final boolean writeToStream (String text, Writer theWriter)
   {
      boolean success = true;

      try
      {
         theWriter.write (text, 0, text.length());
      }
      catch (IOException exception)
      {
         log.error (this + ".writeToStream(" + text + ") failed !");
         log.error ("exception=" + exception);

         success = false;
      }

      return success;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param fragment   No description provided
    * @param theWriter  No description provided
    * @return           No description provided
    */
   public final boolean writeFragmentToStream (CodeGenFragment fragment, Writer theWriter)
   {
      return writeToStream (fragment.getSourceCode(), theWriter);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theWriter  No description provided
    * @return           No description provided
    */
   public boolean writeFragmentsToStream (Writer theWriter)
   {
      CodeGenFragment fragment = getFirstFragment();
      boolean success = true;
      while (success && fragment != null)
      {
         success = writeFragmentToStream (fragment, theWriter);
         fragment = fragment.getNextFragment();
      }
      return success;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theFile  No description provided
    * @return         No description provided
    */
   public final boolean writeFragmentsToFile (File theFile)
   {
      boolean success = true;

      FileWriter theWriter = null;

      try
      {
         theWriter = new FileWriter (theFile);
         success = writeFragmentsToStream (theWriter);
         theWriter.flush();
         theWriter.close();
      }
      catch (IOException exception)
      {
         log.error (this + ".writeFragmentsToFile(" + theFile + ") failed !");
         log.error ("exception=" + exception);

         success = false;
      }

      return success;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param code  No description provided
    */
   public void append (StringBuffer code)
   {
      getCurrentBuffer().append (code);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param code  No description provided
    */
   public void append (String code)
   {
      getCurrentBuffer().append (code);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private CodeGenFragment firstFragment = null;


   /**
    * Get the firstFragment attribute of the CodeGenVisitor object
    *
    * @return   The firstFragment value
    */
   public CodeGenFragment getFirstFragment()
   {
      return this.firstFragment;
   }


   /**
    * Sets the firstFragment attribute of the CodeGenVisitor object
    *
    * @param fragment  The new firstFragment value
    */
   public void setFirstFragment (CodeGenFragment fragment)
   {
      this.firstFragment = fragment;
      if (fragment != null)
      {
         while (fragment.getNextFragment() != null)
         {
            fragment = fragment.getNextFragment();
         }
      }
      this.currentFragment = fragment;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private CodeGenFragment currentFragment = null;


   /**
    * Get the currentFragment attribute of the CodeGenVisitor object
    *
    * @return   The currentFragment value
    */
   public CodeGenFragment getCurrentFragment()
   {
      return this.currentFragment;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param fragment  No description provided
    */
   public void appendFragment (CodeGenFragment fragment)
   {
      if (currentFragment != null)
      {
         currentFragment.setNextFragment (fragment);
         currentFragment = fragment;
      }
      else
      {
         setFirstFragment (fragment);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name  No description provided
    * @return      No description provided
    */
   public TextFragment appendTextFragment (String name)
   {
      TextFragment textFragment = new TextFragment (name);
      appendFragment (textFragment);
      return textFragment;
   }


   /**
    * Get the commentary attribute of the CodeGenVisitor object
    *
    * @param text  No description provided
    * @return      The commentary value
    */
   public boolean isCommentary (String text)
   {
      // this regular expression describes valid commentaries in Java
      String regExpCommentary = "((\\s*/\\*(.|\\s)*\\*/\\s*)|(\\s*//.*\\s*))*";

      return text.matches (regExpCommentary);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param text  No description provided
    * @return      No description provided
    */
   public String removeIndentFromCommentary (String text)
   {
      StringTokenizer toki = new StringTokenizer (text, "\n");
      StringBuffer result = new StringBuffer();
      String currentToken = toki.nextToken();

      int pos = currentToken.indexOf (START_BLOCK_COMMENTARY);

      result.append (currentToken.substring (pos));
      result.append ("\n");

      while (toki.hasMoreTokens())
      {
         currentToken = toki.nextToken();

         result.append (currentToken.substring (pos));
         result.append ("\n");
      }

      return result.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param text  No description provided
    * @return      No description provided
    */
   public String createCommentary (String text)
   {
      // test, if text is correct comment for target language
      if (!isCommentary (text))
      {
         StringTokenizer tokenizer = new StringTokenizer (text, "\n");
         StringBuffer comment = new StringBuffer();

         comment.append (START_BLOCK_COMMENTARY);
         comment.append ("\n");

         while (tokenizer.hasMoreTokens())
         {
            String currentToken = tokenizer.nextToken();

            comment.append (LINE_IN_BLOCK_COMMENTARY);
            comment.append (currentToken);
            comment.append ("\n");
         }

         comment.append (END_BLOCK_COMMENTARY);
         comment.append ("\n");

         text = comment.toString();
      }

      text = indentText (text);

      // check if last character is a new line character
      if (!text.endsWith ("\n"))
      {
         text = text + "\n";
      }

      return text;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param text  No description provided
    * @return      No description provided
    */
   public String createLineCommentary (String text)
   {
      return LINE_COMMENTARY + " " + text;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param text          No description provided
    * @param indentString  No description provided
    * @return              No description provided
    */
   public String indentText (String text, String indentString)
   {
      StringTokenizer tokenizer = new StringTokenizer (text, "\n");
      StringBuffer indentBuffer = new StringBuffer();
      String currentToken = null;

      while (tokenizer.hasMoreTokens())
      {
         currentToken = tokenizer.nextToken();

         indentBuffer.append (indentString);
         indentBuffer.append (currentToken);

         if (tokenizer.hasMoreTokens())
         {
            indentBuffer.append ("\n");
         }
      }

      return indentBuffer.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param indentText  No description provided
    * @param nr          No description provided
    * @return            No description provided
    */
   public String createIndentString (String indentText, int nr)
   {
      StringBuffer indent = new StringBuffer();
      int i;

      for (i = 0; i < nr; i++)
      {
         indent.append (indentText);
      }

      return indent.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private File currentFile;


   /**
    * Get the value of currentFile.
    *
    * @return   Value of currentFile.
    */
   public final File getCurrentFile()
   {
      return this.currentFile;
   }


   /**
    * Set the value of currentFile.
    *
    * @param currentFile  Value to assign to currentFile.
    */
   public final void setCurrentFile (File currentFile)
   {
      if (this.currentFile != currentFile)
      {
         this.currentFile = currentFile;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public final String createIndentString()
   {
      return createIndentString (INDENT_STRING, getIndentNr());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param text  No description provided
    * @return      No description provided
    */
   public final String indentText (String text)
   {
      return indentText (text, createIndentString());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final void newLine()
   {
      getCurrentBuffer().append ("\n");
   }


   /**
    * Get the fileName attribute of the OOGenVisitor object
    *
    * @param theFile  No description provided
    * @return         The fileName value
    */
   public String getFileName (FFile theFile)
   {
      String name = theFile.getName();
      int pos = name.lastIndexOf (".");

      if (pos > -1)
      {
         return name.substring (0, pos - 1);
      }
      else
      {
         return name;
      }
   }


   /**
    * UMLAttribute : 'indentNr : int '
    */
   private int indentNr;


   /**
    * Get the value of indentNr.
    *
    * @return   Value of indentNr.
    */
   public final int getIndentNr()
   {
      return this.indentNr;
   }


   /**
    * Set the value of indentNr.
    *
    * @param indentNr  Value to assign to indentNr.
    */
   public final void setIndentNr (int indentNr)
   {
      if (this.indentNr != indentNr)
      {
         this.indentNr = indentNr;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final void incIndentNr()
   {
      this.indentNr++;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final void decIndentNr()
   {
      this.indentNr--;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String extender;


   /**
    * Get the value of extender.
    *
    * @return   Value of extender.
    */
   public final String getExtender()
   {
      return this.extender;
   }


   /**
    * Set the value of extender.
    *
    * @param extender  Value to assign to extender.
    */
   public final void setExtender (String extender)
   {
      if ( (this.extender == null && extender != null) ||
          (this.extender != null && !this.extender.equals (extender)))
      {
         this.extender = extender;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String path;


   /**
    * Get the value of path.
    *
    * @return   Value of path.
    */
   public String getPath()
   {
      return this.path;
   }


   /**
    * Set the value of path.
    *
    * @param path  Value to assign to path.
    */
   public void setPath (String path)
   {
      if (path == null || !path.equals (this.path))
      {
         this.path = path;
      }
   }


   /**
    * <pre>
    *                 0..1                        0..1
    * CodeGenVisitor ---------------------------------- CodeGenStrategy
    *                 currentVisitor   currentStrategy
    * </pre>
    */
   private CodeGenStrategy currentStrategy;


   /**
    * @param value  The new currentStrategy value
    * @return       No description provided
    * @see          #currentStrategy
    */
   public boolean setCurrentStrategy (CodeGenStrategy value)
   {
      if (this.currentStrategy != value)
      {
         if (this.currentStrategy != null)
         {
            CodeGenStrategy oldValue = this.currentStrategy;
            this.currentStrategy = null;
            oldValue.setCurrentVisitor (null);
         }
         this.currentStrategy = value;
         if (value != null)
         {
            this.currentStrategy.setCurrentVisitor (this);
         }

         return true;
      }

      return false;
   }


   /**
    * @return   The currentStrategy value
    * @see      #currentStrategy
    */
   public CodeGenStrategy getCurrentStrategy()
   {
      return this.currentStrategy;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String name;


   /**
    * Get the value of name.
    *
    * @return   Value of name.
    */
   public String getName()
   {
      return this.name;
   }


   /**
    * Set the value of name.
    *
    * @param name  Value to assign to name.
    */
   public void setName (String name)
   {
      if (name == null || !name.equals (this.name))
      {
         this.name = name;
      }
   }


   /**
    * <pre>
    *                 -------- 0..1        hasVisitors        0..1
    * CodeGenStrategy | name |------------------------------------- CodeGenVisitor
    *                 -------- codeGenStrategy      codeGenVisitor
    * </pre>
    */
   private CodeGenStrategy codeGenStrategy;


   /**
    * @param value  The new codeGenStrategy value
    * @return       No description provided
    * @see          #codeGenStrategy
    */
   public boolean setCodeGenStrategy (CodeGenStrategy value)
   {
      boolean changed = false;
      if (this.codeGenStrategy != value)
      {
         if (this.codeGenStrategy != null)
         {
            CodeGenStrategy oldValue = this.codeGenStrategy;
            this.codeGenStrategy = null;
            oldValue.removeFromCodeGenVisitor (this);
         }
         this.codeGenStrategy = value;
         if (value != null)
         {
            value.addToCodeGenVisitor (this);
         }
         changed = true;
      }
      return changed;
   }


   /**
    * @return   The codeGenStrategy value
    * @see      #codeGenStrategy
    */
   public CodeGenStrategy getCodeGenStrategy()
   {
      return this.codeGenStrategy;
   }


   /**
    * Get the value of currentBuffer.
    *
    * @return   Value of currentBuffer.
    */
   public final StringBuffer getCurrentBuffer()
   {
      CodeGenFragment currentFragment = getCurrentFragment();
      if (currentFragment instanceof TextFragment)
      {
         return  ((TextFragment) currentFragment).getCodeBuffer();
      }
      TextFragment textFragment = appendTextFragment (null);
      return textFragment.getCodeBuffer();
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "CodeGenVisitor[]";
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      CodeGenStrategy tmpCodeGenStrategy = getCodeGenStrategy();
      if (tmpCodeGenStrategy != null)
      {
         setCodeGenStrategy (null);
      }
   }
}

/*
 * $Log: CodeGenVisitor.java,v $
 * Revision 1.44.2.1  2005/09/30 18:56:54  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
