/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import java.util.*;
import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.basic.BasicIncrement;
import de.uni_paderborn.fujaba.basic.Utility;
import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.preferences.JavaPreferences;
import de.uni_paderborn.fujaba.uml.*;
import de.upb.tools.fca.FEmptyIterator;


/**
 * Class OOGenVisitor
 *
 * @author    $Author: fklar $
 * @version   $Revision: 1.98.2.2 $
 */
public abstract class OOGenVisitor extends CodeGenVisitor
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (OOGenVisitor.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TMP_FILE_EXTENDER = "tmp";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String EMPTY_STRING = "";

   // Identifier
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String CLASS_STRING = "class";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String INTERFACE_STRING = "interface";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String NULL_STRING = "null";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String EXTENDS_STRING = "extends";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String IMPLEMENTS_STRING = "implements";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String START_BLOCK = "{";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String END_BLOCK = "}";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String START_IF_BLOCK = "{";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String START_METHOD_BLOCK = "{";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String START_METHOD_BODY = "";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String END_METHOD_BLOCK = "}";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String OPEN_NORM_BRACKET = "(";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String CLOSE_NORM_BRACKET = ")";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String END_OF_STATEMENT = ";";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String END_OF_LINE = "\n";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String END_OF_STATEMENT_LINE = END_OF_STATEMENT + END_OF_LINE;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String LIST_SEPARATOR = ",";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String POINTER_POSTFIX = "";

   // sdm methods
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String SDM_ENSURE_STRING = "JavaSDM.ensure";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String SDM_EXCEPTION_STRING = "JavaSDMException";

   // unit assert methods
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String UNIT_ASSERT_STRING = "junit.framework.Assert.assertTrue";

   // exceptions
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String INTERRUPTED_EXCEPTION_STRING = "InterruptedException";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String ILLEGALDATA_EXCEPTION_STRING = "de.uni_paderborn.lib.resource.IllegalDataException";

   // assignments
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String ASSIGNMENT_OP_STRING = "=";

   // boolean
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String FALSE_STRING = "false";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String TRUE_STRING = "true";

   // control flow
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String IF_STRING = "if";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String WHILE_STRING = "while";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String TRY_STRING = "try";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String CATCH_STRING = "catch";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String FINALLY_STRING = "finally";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String ELSE_STRING = "else";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String DO_STRING = "do";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String BREAK_STRING = "break";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String FOR_STRING = "for";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String SWITCH_STRING = "switch";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String CASE_STRING = "case";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String THIS_STRING = "this.";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String RETURN_STRING = "return";

   // infix op
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String AND_OP_STRING = "&&";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String OR_OP_STRING = "||";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String EQUAL_OP_STRING = "==";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String NOT_EQUAL_OP_STRING = "!=";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String LESS_OP_STRING = "<";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String GREATER_OP_STRING = ">";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String LESS_EQUAL_OP_STRING = "<=";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String GREATER_EQUAL_OP_STRING = ">=";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String ADD_OP_STRING = "+";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String SUB_OP_STRING = "-";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String MUL_OP_STRING = "*";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String DIV_OP_STRING = "/";

   // prefix ops
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String NOT_OP_STRING = "!";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String INCR_OP_STRING = "++";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String DECR_OP_STRING = "--";

   // uml
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String F_STATIC_STRING = "static";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String FINAL_STRING = "final";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String TRANSIENT_STRING = "transient";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String ABSTRACT_STRING = "abstract";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String SYNCHRONIZED_STRING = "synchronized";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String NATIVE_STRING = "native";

   // method names
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String EQUALS_METHOD_STRING = "equals";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String COMPARE_TO_METHOD_STRING = "compareTo";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String ADD_TO_SET_METHOD_STRING = "add";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String SET_CONTAINS_METHOD_STRING = "contains";

   // It might be that there are different types of ordered associations in the future
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int NO_ORDER = 0;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int ORDER = 1;


   /**
    * Default Constructor
    */
   public OOGenVisitor()
   {
      initVisibilityStrings();
      initProgLangTypes();
   }


   /**
    * Default Constructor
    *
    * @param extender  No description provided
    */
   public OOGenVisitor (String extender)
   {
      this();
      setExtender (extender);
   }

   //
   // begin assoc template section
   //

   /**
    * Get the referenceTemplateName attribute of the CodeGenVisitor object
    *
    * @return   The referenceTemplateName value
    */
   public abstract String getReferenceTemplateName();


   /**
    * Get the assocTemplateName attribute of the CodeGenVisitor object
    *
    * @return   The assocTemplateName value
    */
   public abstract String getAssocTemplateName();

   //
   // end assoc template section
   //

   //
   // abstract method section
   //

   /**
    * get the full qualified containername for type <code>containerType</code>
    *
    * @param containerType  the <code>containerType</code>
    * @param bound          true, if assoc is bound
    * @return               the full qualified containername
    */
   public abstract String getContainerName (OOCollectionEnum containerType, boolean bound);


   /**
    * Get the containerName attribute of the CodeGenVisitor object
    *
    * @param containerType  No description provided
    * @return               The containerName value
    */
   public final String getContainerName (OOCollectionEnum containerType)
   {
      return getContainerName (containerType, false);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public abstract void generateClassEndBlock();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public abstract void generateClassStartBlock();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theFile  No description provided
    */
   public abstract void generatePackageBegin (FFile theFile);


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theFile  No description provided
    */
   public abstract void generatePackageEnd (FFile theFile);


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param thePackage  No description provided
    * @return            No description provided
    */
   public abstract String generateImportPackage (FPackage thePackage);


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theClass  No description provided
    * @return          No description provided
    */
   public abstract String generateImportClass (FClass theClass);


   /**
    * Get the attrDeclName attribute of the OOGenVisitor object
    *
    * @param theAttr  No description provided
    * @return         The attrDeclName value
    */
   public abstract String getAttrDeclName (FAttr theAttr);


   /**
    * Get the methodDeclName attribute of the OOGenVisitor object
    *
    * @param theMethod  No description provided
    * @return           The methodDeclName value
    */
   public abstract String getMethodDeclName (FMethod theMethod);

   //
   // begin of method name section
   //

   /**
    * Get the methodName attribute of the OOGenVisitor object
    *
    * @param methodName  No description provided
    * @return            The methodName value
    */
   public String getMethodName (OOMethod methodName)
   {
      // int type = methodName.getType().getNr();
      String name = methodName.getName();

      StringBuffer result = new StringBuffer();

      if (methodName.getType() == OOMethodType.DEFAULT_METHOD)
      {
         return methodName.getName();
      }
      else if (methodName.getType() == OOMethodType.GET_METHOD)
      {
         result.append ("get");
      }
      else if (methodName.getType() == OOMethodType.IS_METHOD)
      {
         result.append ("is");
      }
      else if (methodName.getType() == OOMethodType.SET_METHOD)
      {
         result.append ("set");
      }
      else if (methodName.getType() == OOMethodType.HAS_IN_METHOD)
      {
         if (!"".equals (name))
         {
            result.append ("hasIn");
         }
         else
         {
            result.append ("contains");
         }
      }
      else if (methodName.getType() == OOMethodType.ELEMENTS_OF_METHOD)
      {
         result.append ("elementsOf");
      }
      else if (methodName.getType() == OOMethodType.ITERATOR_OF_METHOD)
      {
         if (!"".equals (name))
         {
            result.append ("iteratorOf");
         }
         else
         {
            result.append ("iterator");
         }
      }
      else if (methodName.getType() == OOMethodType.OLD_STYLE_ELEMENTS)
      {
         result.append (name);
         result.append ("Elements");
         return result.toString();
      }
      else if (methodName.getType() == OOMethodType.REMOVE_YOU_METHOD)
      {
         return "removeYou";
      }
      else if (methodName.getType() == OOMethodType.ADD_METHOD)
      {
         if (!"".equals (name))
         {
            result.append ("addTo");
         }
         else
         {
            result.append ("add");
         }
      }
      else if (methodName.getType() == OOMethodType.REMOVE_METHOD)
      {
         if (!"".equals (name))
         {
            result.append ("removeFrom");
         }
         else
         {
            result.append ("remove");
         }
      }
      else if (methodName.getType() == OOMethodType.GET_FROM_METHOD)
      {
         if (!"".equals (name))
         {
            result.append ("getFrom");
         }
         else
         {
            result.append ("get");
         }
      }
      else if (methodName.getType() == OOMethodType.SIZE_OF_METHOD)
      {
         if (!"".equals (name))
         {
            result.append ("sizeOf");
         }
         else
         {
            result.append ("size");
         }
      }
      else if (methodName.getType() == OOMethodType.REMOVE_ALL_FROM_METHOD)
      {
         if (!"".equals (name))
         {
            result.append ("removeAllFrom");
         }
         else
         {
            result.append ("clear");
         }

         // ordered, sorted assoc
      }
      else if (methodName.getType() == OOMethodType.INDEX_OF_METHOD)
      {
         result.append ("indexOf");
      }
      else if (methodName.getType() == OOMethodType.LAST_INDEX_OF_METHOD)
      {
         result.append ("lastIndexOf");
      }
      else if (methodName.getType() == OOMethodType.IS_BEFORE_OF_METHOD)
      {
         result.append ("isBeforeOf");
      }
      else if (methodName.getType() == OOMethodType.IS_AFTER_OF_METHOD)
      {
         result.append ("isAfterOf");
      }
      else if (methodName.getType() == OOMethodType.GET_FIRST_OF_METHOD)
      {
         result.append ("getFirst");
         if (!"".equals (name))
         {
            result.append ("Of");
         }
      }
      else if (methodName.getType() == OOMethodType.GET_LAST_OF_METHOD)
      {
         result.append ("getLast");
         if (!"".equals (name))
         {
            result.append ("Of");
         }
      }
      else if (methodName.getType() == OOMethodType.GET_NEXT_OF_METHOD)
      {
         result.append ("getNextOf");
      }
      else if (methodName.getType() == OOMethodType.GET_NEXT_INDEX_OF_METHOD)
      {
         result.append ("getNextIndexOf");
      }
      else if (methodName.getType() == OOMethodType.GET_PREVIOUS_OF_METHOD)
      {
         result.append ("getPreviousOf");
      }
      else if (methodName.getType() == OOMethodType.GET_PREVIOUS_INDEX_OF_METHOD)
      {
         result.append ("getPreviousIndexOf");
      }
      else if (methodName.getType() == OOMethodType.ADD_BEFORE_OF_METHOD)
      {
         result.append ("addBeforeOf");
      }
      else if (methodName.getType() == OOMethodType.ADD_AFTER_OF_METHOD)
      {
         result.append ("addAfterOf");

         // Qualified assoc methods.
      }
      else if (methodName.getType() == OOMethodType.GET_VALUE_FROM_METHOD)
      {
         result.append ("getValueFrom");
      }
      else if (methodName.getType() == OOMethodType.GET_KEY_FROM_METHOD)
      {
         result.append ("getKeyFrom");
      }
      else if (methodName.getType() == OOMethodType.HAS_VALUE_IN_METHOD)
      {
         if (!"".equals (name))
         {
            result.append ("hasValueIn");
         }
         else
         {
            result.append ("containsValue");
         }
      }
      else if (methodName.getType() == OOMethodType.HAS_KEY_IN_METHOD)
      {
         if (!"".equals (name))
         {
            result.append ("hasKeyIn");
         }
         else
         {
            result.append ("containsKey");
         }
      }
      else if (methodName.getType() == OOMethodType.HAS_ENTRY_IN_METHOD)
      {
         result.append ("hasEntryIn");
      }
      else if (methodName.getType() == OOMethodType.KEYS_OF_METHOD)
      {
         if (!"".equals (name))
         {
            result.append ("keysOf");
         }
         else
         {
            result.append ("keySet ().iterator");
         }
      }
      else if (methodName.getType() == OOMethodType.REMOVE_VALUE_FROM_METHOD)
      {
         result.append ("removeValueFrom");
      }
      else if (methodName.getType() == OOMethodType.REMOVE_KEY_FROM_METHOD)
      {
         if (!"".equals (name))
         {
            result.append ("removeKeyFrom");
         }
         else
         {
            result.append ("remove");
         }
      }
      else if (methodName.getType() == OOMethodType.REMOVE_ENTRY_FROM_METHOD)
      {
         result.append ("removeEntryFrom");
      }
      else if (methodName.getType() == OOMethodType.ENTRIES_OF_METHOD)
      {
         if (!"".equals (name))
         {
            result.append ("entriesOf");
         }
         else
         {
            result.append ("entrySet ().iterator");
         }
      }
      else if (methodName.getType() == OOMethodType.GET_AT_METHOD)
      {
         result.append ("get").append (CGU.upFirstChar (name)).append ("At");
         return result.toString();
      }
      else if (methodName.getType() == OOMethodType.IS_EMPTY_METHOD)
      {
         result.append ("is" + CGU.upFirstChar (name) + "Empty");
         return result.toString();
      }
      else if (methodName.getType() == OOMethodType.EQUALS_METHOD)
      {
         return EQUALS_METHOD_STRING;
      }
      else if (methodName.getType() == OOMethodType.ADD_TO_SET_METHOD)
      {
         return ADD_TO_SET_METHOD_STRING;
      }
      else if (methodName.getType() == OOMethodType.SET_CONTAINS_METHOD)
      {
         return SET_CONTAINS_METHOD_STRING;
      }

      result.append (CGU.upFirstChar (name));

      return result.toString();
   }

   //
   // end of method name section
   //

   //
   // generate section
   //

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param blockType  No description provided
    */
   public void generateOpenBlock (String blockType)
   {
      getCurrentBuffer().append (indentText (blockType));
      newLine();
      incIndentNr();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param blockType  No description provided
    */
   public void generateCloseBlock (String blockType)
   {
      decIndentNr();
      getCurrentBuffer().append (indentText (blockType));
      newLine();
   }

   //
   // create section
   //

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theClass  No description provided
    * @return          No description provided
    */
   public String createClassInterfaceModifier (FClass theClass)
   {
      StringBuffer modifier = new StringBuffer();

      if (log.isDebugEnabled())
      {
         log.debug ("getVisibilityString(" + theClass +
            "," + theClass.getVisibility() + ")=" +
            getVisibilityString (theClass.getVisibility()));
      }

      modifier.append (getVisibilityString (theClass.getVisibility()));

      if (theClass.isStatic())
      {
         modifier.append (" ");
         modifier.append (F_STATIC_STRING);
      }

      if (theClass.isFinal())
      {
         modifier.append (" ");
         modifier.append (FINAL_STRING);
      }

      return modifier.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theClass  No description provided
    * @return          No description provided
    */
   public String createInterfaceDeclaration (FClass theClass)
   {
      StringBuffer interfaceDeclString = new StringBuffer();

      interfaceDeclString.append (createClassInterfaceModifier (theClass));
      interfaceDeclString.append (" ");
      interfaceDeclString.append (INTERFACE_STRING);
      interfaceDeclString.append (" ");
      interfaceDeclString.append (getInterfaceName (theClass));

      interfaceDeclString.append (createInterfaceExtendsDeclaration (theClass));

      return interfaceDeclString.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theClass  No description provided
    * @return          No description provided
    */
   public String createClassDeclaration (FClass theClass)
   {
      StringBuffer classDeclString = new StringBuffer();

      if (createClassInterfaceModifier (theClass) != null)
      {
         classDeclString.append (createClassInterfaceModifier (theClass));
         classDeclString.append (" ");
      }

      // following cannot be in createFClassInterfaceModifier as it would then also
      // affect interfaces
      if (theClass.isAbstract())
      {
         classDeclString.append (ABSTRACT_STRING);
         classDeclString.append (" ");
      }

      if (theClass.hasKeyInStereotypes ("final"))
      {
         classDeclString.append ("final ");
      }

      if (theClass.hasKeyInStereotypes ("strictfp"))
      {
         classDeclString.append ("strictfp ");
      }

      classDeclString.append (CLASS_STRING);
      classDeclString.append (" ");
      classDeclString.append (getClassName (theClass));

      classDeclString.append (createClassExtendsDeclaration (theClass));

      return classDeclString.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theClass  No description provided
    * @return          No description provided
    */
   public String createClassExtendsDeclaration (FClass theClass)
   {
      StringBuffer declString = new StringBuffer();

      // currently only single inheritence is supported
      FClass superClass = theClass.getFSuperClass();

      if (superClass != null)
      {
         declString.append ("\n");
         declString.append (EXTENDS_STRING);
         declString.append (" ");
         declString.append (getClassName (superClass));
      }

      Enumeration parentEnum = theClass.iteratorOfSuperClasses();
      boolean firstTime = true;
      FClass tmpClass = null;
      FStereotype interfaceType = UMLStereotypeManager.get().getFromStereotypes (UMLStereotypeManager.INTERFACE);

      while (parentEnum.hasMoreElements())
      {
         tmpClass = (FClass) parentEnum.nextElement();

         if (tmpClass.hasInStereotypes (interfaceType))
         {
            if (!firstTime)
            {
               declString.append (", ");
            }
            else
            {
               declString.append ("\n");
               declString.append (IMPLEMENTS_STRING);
               declString.append (" ");
               firstTime = false;
            }

            declString.append (getClassName (tmpClass));
         }
      }

      return declString.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theClass  No description provided
    * @return          No description provided
    */
   public String createInterfaceExtendsDeclaration (FClass theClass)
   {
      StringBuffer declString = new StringBuffer();

      Enumeration parentEnum = theClass.iteratorOfSuperClasses();
      FClass tmpClass = null;
      boolean firstTime = true;

      while (parentEnum.hasMoreElements())
      {
         tmpClass = (FClass) parentEnum.nextElement();

         if (!firstTime)
         {
            declString.append (", ");
         }
         else
         {
            declString.append ("\n");
            declString.append (EXTENDS_STRING);
            declString.append (" ");
            firstTime = false;
         }

         declString.append (getClassName (tmpClass));
      }

      return declString.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theCommentary  No description provided
    * @return               No description provided
    */
   public String createClassCommentary (FCommentary theCommentary)
   {
      return createCommentary (theCommentary.getText());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theCommentary  No description provided
    * @return               No description provided
    */
   public String createAttrCommentary (FCommentary theCommentary)
   {
      return createCommentary (theCommentary.getText());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theCommentary  No description provided
    * @return               No description provided
    */
   public String createMethodCommentary (FCommentary theCommentary)
   {
      return createCommentary (theCommentary.getText());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theAttr  No description provided
    * @return         No description provided
    */
   public String createAttrDeclaration (FAttr theAttr)
   {
      StringBuffer declString = new StringBuffer();

      int visibility = theAttr.getVisibility();
      if (theAttr.isNeedsAccessMethods() && !theAttr.isParsed())
      {
         visibility = FDeclaration.PRIVATE;
      }
      declString.append (getAttrDeclVisibilityString (visibility));
      declString.append (" ");
      if (theAttr.isStatic())
      {
         declString.append (F_STATIC_STRING);
         declString.append (" ");
      }

      if (theAttr.isFinal())
      {
         declString.append (FINAL_STRING);
         declString.append (" ");
      }

      if (theAttr.isTransient())
      {
         declString.append (TRANSIENT_STRING);
         declString.append (" ");
      }

      if (theAttr.hasKeyInStereotypes ("volatile"))
      {
         declString.append ("volatile ");
      }

      declString.append (getTypeAsString (theAttr.getFAttrType(), false));
      declString.append (" ");
      declString.append (getAttrDeclName (theAttr));

      if (!Utility.isNullOrEmpty (theAttr.getInitialValue()))
      {
         declString.append (" = " + theAttr.getInitialValue());
      }

      declString.append (";");
      declString.append ("\n\n");
      return indentText (declString.toString());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theMethod  No description provided
    * @return           No description provided
    */
   public String createMethodDeclaration (FMethod theMethod)
   {
      String declString = null;

      FType returnType = theMethod.getFResultType();
      if (returnType == null)
      {
         log.error ("Method has no Return Type: " + theMethod.getName());
      }
      if (returnType != null && returnType.getName().equals (FBaseTypes.INITIALIZER))
      {
         declString = createInitializerDeclaration (theMethod);
      }
      else
      {
         declString = createNormalFMethodDeclaration (theMethod);
      }

      return declString;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theMethod  No description provided
    * @return           No description provided
    */
   public String createInitializerDeclaration (FMethod theMethod)
   {
      StringBuffer declString = new StringBuffer();

      // Initializers are always static
      declString.append (F_STATIC_STRING);
      declString.append (" ");

      // and have nothing else
      return indentText (declString.toString());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theMethod  No description provided
    * @return           No description provided
    */
   public String createNormalFMethodDeclaration (FMethod theMethod)
   {
      StringBuffer declString = new StringBuffer();

      if (getMethodDeclVisibilityString (theMethod.getVisibility()) != null)
      {
         declString.append (getMethodDeclVisibilityString (theMethod.getVisibility()));
         declString.append (" ");
      }

      if (theMethod.isAbstract())
      {
         declString.append (ABSTRACT_STRING);
         declString.append (" ");
      }

      if (theMethod.isUmlNative())
      { //will be replaced with stereotypoe

         declString.append (NATIVE_STRING);
         declString.append (" ");
      }

      if (theMethod.isSynchronized())
      {
         declString.append (SYNCHRONIZED_STRING);
         declString.append (" ");
      }

      if (theMethod.isStatic())
      {
         declString.append (F_STATIC_STRING);
         declString.append (" ");
      }

      if (theMethod.isFinal())
      {
         declString.append (FINAL_STRING);
         declString.append (" ");
      }

      // append result type
      if (theMethod.getFResultType() != null)
      {
         declString.append (getTypeAsString (theMethod.getFResultType(), theMethod.isResultTypeIsPointer()));
         declString.append (" ");
      }
      else
      {
         declString.append ("result_type_is_missing");
      }

      // append methodname
      declString.append (getMethodDeclName (theMethod));

      // generate parameters
      declString.append (OPEN_NORM_BRACKET);
      declString.append (createMethodParamList (theMethod));
      declString.append (CLOSE_NORM_BRACKET);

      // append throw types
      if (theMethod.sizeOfThrowsTypes() > 0)
      {
         boolean firstThrow = true;
         Iterator throwsIter = theMethod.iteratorOfThrowsTypes();
         FType myThrowType = null;
         while (throwsIter.hasNext())
         {
            if (firstThrow)
            {
               declString.append (" throws ");
               firstThrow = false;
            }
            else
            {
               declString.append (", ");
            }
            myThrowType = (FType) throwsIter.next();
            declString.append (myThrowType.getName());
         }
      }

      return indentText (declString.toString());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param blockType  No description provided
    * @return           No description provided
    */
   public String createOpenBlock (String blockType)
   {
      String result = "\n" + indentText (blockType) + "\n";
      incIndentNr();
      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param blockType  No description provided
    * @return           No description provided
    */
   public String createCloseBlock (String blockType)
   {
      decIndentNr();
      String result = "\n" + indentText (blockType) + "\n";
      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theMethod  No description provided
    * @return           No description provided
    */
   public String createMethodParamList (FMethod theMethod)
   {
      Iterator iter = theMethod.iteratorOfParam();
      FParam param;
      StringBuffer declString = new StringBuffer();
      while (iter.hasNext())
      {
         param = (FParam) iter.next();
         declString.append (createMethodParam (param));
         if (iter.hasNext())
         {
            declString.append (LIST_SEPARATOR);
            declString.append (" ");
         }
      }

      return declString.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theParam  No description provided
    * @return          No description provided
    */
   public String createMethodParam (FParam theParam)
   {
      StringBuffer declString = new StringBuffer();
      if (theParam.getFParamType() == null)
      {

         if (log.isInfoEnabled())
         {
            log.info ("ERROR: parameter in method found without a type !");
         }
         if (theParam.getFRevParam().getFParent() != null)
         {
            if (log.isInfoEnabled())
            {
               log.info ("\tClass    : " +
                  theParam.getFRevParam().getFParent().
                  getID() + ": " +
                  theParam.getFRevParam().getFParent().
                  getName());
            }
         }
         else
         {
            if (log.isInfoEnabled())
            {
               log.info ("\tClass    : NULL !!!");
            }
         }
         if (log.isInfoEnabled())
         {
            log.info ("\tMethod   : " +
               theParam.getFRevParam().getID() +
               ": " + theParam.getFRevParam().getName());
         }
         if (log.isInfoEnabled())
         {
            log.info ("\tParameter: " + theParam.getID() +
               ": " + theParam.getName());
         }
      }
      else
      {
         declString.append (getTypeAsString (theParam.getFParamType(),
            theParam.isPointer()));
         declString.append (" ");
         declString.append (theParam.getName());
      }

      return declString.toString();
   }

   //
   // begin visit OOStatement section
   //

   // abstract methods first
   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public abstract String getSourceCode (OOThrowStatement statement);


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOExprStatement statement)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".getSourceCode(" + statement + ")");
         }
      }

      StringBuffer result = new StringBuffer();

      String exprCode = statement.getExpr().getSourceCode (this);

      if (exprCode == null)
      {
         return null;
      }

      result.append (createIndentString());
      result.append (exprCode);
      //      result.append ( " " ) ;
      result.append (END_OF_STATEMENT);

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOIfStatement statement)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".getSourceCode(" + statement + ")");
         }
      }

      StringBuffer result = new StringBuffer();
      result.append (createIndentString());
      result.append (IF_STRING);
      result.append (" ");
      result.append (OPEN_NORM_BRACKET);
      result.append (" ");
      if (statement.getCondition() != null)
      {
         result.append (statement.getCondition().getSourceCode (this));
      }
      else
      {
         result.append (START_BLOCK_COMMENTARY +
            " oops, condition is missing " + END_BLOCK_COMMENTARY);
      }

      result.append (" ");
      result.append (CLOSE_NORM_BRACKET);
      //      result.append (appendStatement (statement));
      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOWhileStatement statement)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".getSourceCode(" + statement + ")");
         }
      }

      StringBuffer result = new StringBuffer();
      result.append (createIndentString());
      result.append (WHILE_STRING);
      result.append (" ");
      result.append (OPEN_NORM_BRACKET);
      result.append (" ");
      if (statement.getCondition() != null)
      {
         result.append (statement.getCondition().getSourceCode (this));
      }
      else
      {
         result.append (START_BLOCK_COMMENTARY +
            " oops, condition is missing " + END_BLOCK_COMMENTARY);
      }

      result.append (" ");
      result.append (CLOSE_NORM_BRACKET);

      if (statement.isFoot())
      {
         result.append (" ");
         result.append (END_OF_STATEMENT);
      }

      //      result.append (appendStatement (statement));
      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOForStatement statement)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".getSourceCode(" + statement + ")");
         }
      }

      StringBuffer result = new StringBuffer();
      result.append (createIndentString());
      result.append (FOR_STRING);
      result.append (" ");
      result.append (OPEN_NORM_BRACKET);
      result.append (" ");

      if (statement.getInitExpr() != null)
      {
         result.append (statement.getInitExpr().getSourceCode (this));
      }
      else
      {
         result.append (START_BLOCK_COMMENTARY +
            " oops, initExpr is missing " + END_BLOCK_COMMENTARY);
      }

      result.append (END_OF_STATEMENT);

      if (statement.getEndExpr() != null)
      {
         result.append (statement.getEndExpr().getSourceCode (this));
      }
      else
      {
         result.append (START_BLOCK_COMMENTARY +
            " oops, endExpr is missing " + END_BLOCK_COMMENTARY);
      }

      result.append (END_OF_STATEMENT);

      if (statement.getStepExpr() != null)
      {
         result.append (statement.getStepExpr().getSourceCode (this));
      }
      else
      {
         result.append (START_BLOCK_COMMENTARY +
            " oops, stepExpr is missing " + END_BLOCK_COMMENTARY);
      }

      result.append (" ");
      result.append (CLOSE_NORM_BRACKET);

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOElseStatement statement)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".getSourceCode(" + statement + ")");
         }
      }

      StringBuffer result = new StringBuffer();
      result.append (createIndentString());
      result.append (ELSE_STRING);
      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOLineCommentStatement statement)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".getSourceCode(" + statement + ")");
         }
      }

      StringTokenizer toki =
         new StringTokenizer (statement.getComment(), "\n");
      StringBuffer result = new StringBuffer();
      if (toki.countTokens() == 1)
      {
         // create line comment
         result.append (createIndentString());
         result.append (createLineCommentary (statement.getComment()));
      }
      else if (toki.countTokens() > 1)
      {
         // create block comment


         result.append (indentText
             (createCommentary (statement.getComment()),
            createIndentString()));
      }

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOLocalVarDeclStatement statement)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".getSourceCode(" + statement + ")");
         }
      }

      StringBuffer result = new StringBuffer();
      result.append (createIndentString());

      if (!statement.isDeclared())
      {
         result.append (getOOTypeAsString (statement.getVarType()));
         result.append (" ");
      }

      result.append (getVarName (statement.getObjectName()));
      OOExpression initExpr = statement.getInitExpr();
      if (initExpr != null)
      {
         result.append (" ");
         result.append (ASSIGNMENT_OP_STRING);
         result.append (" ");
         result.append (initExpr.getSourceCode (this));
      }

      result.append (" ");
      result.append (END_OF_STATEMENT);
      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOStartBlockStatement statement)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".getSourceCode(" + statement + ")");
         }
      }

      String result = createIndentString() + START_BLOCK;
      incIndentNr();
      return result;
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOEndBlockStatement statement)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".getSourceCode(" + statement + ")");
         }
      }

      decIndentNr();
      StringBuffer result = new StringBuffer();
      result.append (createIndentString());
      result.append (END_BLOCK);
      if (statement.getText() != null)
      {
         result.append (" ");
         result.append (createLineCommentary (statement.getText()));
      }

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOSDMEnsureStatement statement)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".getSourceCode(" + statement + ")");
         }
      }

      StringBuffer result = new StringBuffer();
      result.append (createIndentString());
      if (statement instanceof OOUnitAssertStatement)
      {
         result.append (UNIT_ASSERT_STRING);
      }
      else
      {
         result.append (SDM_ENSURE_STRING);
      }
      result.append (" ");
      result.append (OPEN_NORM_BRACKET);
      result.append (" ");
      if (statement instanceof OOUnitAssertStatement)
      {
         result.append ("\"" +  ((OOUnitAssertStatement) statement).getDescription() + "\", ");
      }
      if (statement.getCondition() != null)
      {
         result.append (statement.getCondition().getSourceCode (this));
      }
      else if (statement.getConditionText() != null)
      {
         result.append (statement.getConditionText());
      }
      else
      {
         result.append (START_BLOCK_COMMENTARY +
            " oops, condition is missing " + END_BLOCK_COMMENTARY);
      }

      result.append (" ");
      result.append (CLOSE_NORM_BRACKET);
      result.append (" ");
      result.append (END_OF_STATEMENT);
      //         result.append (appendStatement (statement));
      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOTryStatement statement)
   {
      return createIndentString() + TRY_STRING;
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOReturnStatement statement)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".getSourceCode(" + statement + ")");
         }
      }

      StringBuffer result = new StringBuffer();

      result.append (createIndentString());
      result.append (RETURN_STRING);
      if (statement.getReturnValue() != null)
      {
         result.append (" ");
         result.append (OPEN_NORM_BRACKET);
         result.append (statement.getReturnValue().getSourceCode (this));
         result.append (CLOSE_NORM_BRACKET);
         result.append (" ");
      }
      else if (statement.getReturnValueName() != null &&
         ! (statement.getReturnValueName().equals ("")))
      {
         result.append (" ");
         result.append (OPEN_NORM_BRACKET);
         result.append (statement.getReturnValueName());
         result.append (CLOSE_NORM_BRACKET);
         result.append (" ");
      }
      result.append (END_OF_STATEMENT);

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOCatchStatement statement)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".getSourceCode(" + statement + ")");
         }
      }

      StringBuffer result = new StringBuffer();
      result.append (createIndentString());
      result.append (CATCH_STRING);
      result.append (" ");
      result.append (OPEN_NORM_BRACKET);
      result.append (" ");
      result.append (statement.getExceptionType().getSourceCode (this));
      result.append (" ");
      result.append (getVarName (statement.getExceptionName()));
      result.append (" ");
      result.append (CLOSE_NORM_BRACKET);
      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOStartMethodBodyStatement statement)
   {
      return null; // createIndentString () + "" ; // appendStatement (statement);
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOBeginMethodBodyStatement statement)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".getSourceCode(" + statement + ")");
         }
      }

      String result = createIndentString() + START_METHOD_BLOCK;
      incIndentNr();
      return result;
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOEndMethodBodyStatement statement)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".getSourceCode(" + statement + ")");
         }
      }

      decIndentNr();
      return createIndentString() + END_METHOD_BLOCK;
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOEmptyLineStatement statement)
   {
      return createIndentString() + " "; //  LINE_COMMENTARY ; // + appendStatement ( statement) ;
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OODoStatement statement)
   {
      return createIndentString() + DO_STRING;
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOMemoStatement statement)
   {
      return createIndentString() + statement.getMemo();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOBreakStatement statement)
   {
      return createIndentString() + BREAK_STRING + END_OF_STATEMENT;
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public final String getSourceCode (OOSwitchCaseStatement statement)
   {
      StringBuffer result = new StringBuffer();
      result.append (createIndentString() + SWITCH_STRING + OPEN_NORM_BRACKET + statement.getVariable().getFirstName() + CLOSE_NORM_BRACKET + START_BLOCK + END_OF_LINE);
      incIndentNr();
      Iterator iter = statement.getCases();
      while (iter.hasNext())
      {
         OOCaseStatement caseStatement = (OOCaseStatement) iter.next();
         result.append (caseStatement.getSourceCode (this));
      }
      decIndentNr();
      result.append (createIndentString() + END_BLOCK);

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param caseStatement  No description provided
    * @return               The sourceCode value
    */
   public final String getSourceCode (OOCaseStatement caseStatement)
   {
      StringBuffer result = new StringBuffer();
      Iterator iter = caseStatement.getValues();
      while (iter.hasNext())
      {
         String value = (String) iter.next();
         result.append (createIndentString() + CASE_STRING + " " + value + ":" + END_OF_LINE);
      }
      Vector vector = caseStatement.getStatements();
      for (int i = 0; i < vector.size(); i++)
      {
         OOStatement statement = (OOStatement) vector.get (i);
         result.append (createIndentString() + statement.getSourceCode (this) + END_OF_LINE);
      }
      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public final String getSourceCode (OOStatement statement)
   {
      return this + ".getSourceCode(" + statement + ") is empty !";
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param statement  No description provided
    * @return           No description provided
    */
   public final String appendStatement (OOStatement statement)
   {
      return LINE_COMMENTARY + " " + statement;
   }

   //
   // end visit OOStatement section
   //

   //
   // begin visit OOExpression section
   //

   // abstract method first
   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   abstract String getSourceCode (OOObjectOfTypeExpr expression);


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   abstract String getSourceCode (OOClassOfVariableExpr expression);


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   abstract String getSourceCode (OOFWIteratorOfExpr expression);


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   abstract String getSourceCode (OOFWIteratorGetExpr expression);


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   abstract String getSourceCode (OOFWIteratorHasNextExpr expression);


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   abstract String getSourceCode (OOFWIteratorGotoNextExpr expression);


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public abstract String getSourceCode (OONewArrayExpr expression);


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public final String getSourceCode (OONewObjectExpr expression)
   {
      StringBuffer result = new StringBuffer();

      result.append ("new ");
      result.append (getOOTypeAsString (expression.getVarType()));
      result.append (" (");

      Iterator iter = expression.iteratorOfParameter();

      while (iter.hasNext())
      {
         OOExpression tmpParameter = (OOExpression) iter.next();
         result.append (tmpParameter.getSourceCode (this));
         if (iter.hasNext())
         {
            result.append (LIST_SEPARATOR);
            result.append (" ");
         }
      }

      result.append (" )");

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public String getSourceCode (OOPrefixExpr expression)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".getSourceCode(" + expression + ")");
         }
      }

      // don't enclose variables in brackets - looks ugly
      boolean needBrackets = ! (expression.getMyExpr() instanceof OOVariable);

      StringBuffer result = new StringBuffer();

      if (expression.getMyOp() == OOPrefixOp.NOT_OP)
      {
         result.append (NOT_OP_STRING);
      }
      else if (expression.getMyOp() == OOPrefixOp.INCR_OP)
      {
         result.append (INCR_OP_STRING);
      }
      else if (expression.getMyOp() == OOPrefixOp.DECR_OP)
      {
         result.append (DECR_OP_STRING);
      }
      else
      {
         result.append (START_BLOCK_COMMENTARY).append (" unsupported op ").
            append (expression.getMyOp()).append (END_BLOCK_COMMENTARY);
      }

      if (needBrackets)
      {
         result.append (OPEN_NORM_BRACKET);
      }

      result.append (expression.getMyExpr().getSourceCode (this));

      if (needBrackets)
      {
         result.append (CLOSE_NORM_BRACKET);
      }

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public String getSourceCode (OOInfixExprLeft expression)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".getSourceCode(" + expression + ")");
         }
      }

      StringBuffer result = new StringBuffer();
      boolean needBrackets = expression.getMyExp() instanceof OOInfixExprLeft;
      OOInfixExprRight tmpRight = null;
      if (needBrackets)
      {
         result.append (OPEN_NORM_BRACKET);
      }

      if (expression.getMyExp() != null)
      {
         result.append (expression.getMyExp().getSourceCode (this));
      }
      else
      {
         result.append (START_BLOCK_COMMENTARY +
            " oops, getMyExp() is null ! " + END_BLOCK_COMMENTARY);
      }

      if (needBrackets)
      {
         result.append (CLOSE_NORM_BRACKET);
      }

      tmpRight = expression.getRightExp();
      while (tmpRight != null)
      {
         needBrackets = tmpRight.getMyExp() instanceof OOInfixExprLeft;
         result.append (" ");
         result.append (getSourceCode (tmpRight.getMyOp()));
         result.append (" ");
         if (needBrackets)
         {
            result.append (OPEN_NORM_BRACKET);
         }

         result.append (tmpRight.getMyExp().getSourceCode (this));
         if (needBrackets)
         {
            result.append (CLOSE_NORM_BRACKET);
         }

         tmpRight = tmpRight.getNextExpression();
      }

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public String getSourceCode (OOIdentifierExpr expression)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".getSourceCode(" + expression + ")");
         }
      }

      String result = null;
      if (expression == OOIdentifierExpr.NULL_IDENTIFIER)
      {
         result = NULL_STRING;
      }
      else if (expression == OOIdentifierExpr.FALSE_IDENTIFIER)
      {
         result = FALSE_STRING;
      }
      else if (expression == OOIdentifierExpr.TRUE_IDENTIFIER)
      {
         result = TRUE_STRING;
      }
      else
      {
         result = getVarName (expression.getVariableName());
      }

      return result;
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public String getSourceCode (OOTypeCastExpr expression)
   {
      StringBuffer result = new StringBuffer();
      result.append (OPEN_NORM_BRACKET);
      result.append (getOOTypeAsString (expression.getTypeName(), true));
      result.append (CLOSE_NORM_BRACKET);
      result.append (" ");
      result.append (expression.getCastSource().getSourceCode (this));
      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public String getSourceCode (OOAssignExpr expression)
   {
      StringBuffer result = new StringBuffer();

      result.append (expression.getLeftExpr().getSourceCode (this));
      result.append (" ");
      result.append (ASSIGNMENT_OP_STRING);
      result.append (" ");
      result.append (expression.getRightExpr().getSourceCode (this));

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public String getSourceCode (OOCallAttrExpr expression)
   {
      StringBuffer result = new StringBuffer();

      if ( (expression.getObjectName() != null) &&  (getVarName (expression.getObjectName()) != null))
      {
         result.append (getVarName (expression.getObjectName()));
         result.append (".");
      }

      if ( (expression.getAttrName() != null) &&  (getVarName (expression.getAttrName()) != null))
      {
         result.append (getVarName (expression.getAttrName()));
      }
      else
      {
         result.append (" /* oops, no attr name */ ");
      }

      // append next call, if available
      if (expression.getNextCall() != null)
      {
         result.append (".");
         result.append (expression.getNextCall().getSourceCode (this));
      }

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public String getSourceCode (OOCallMethodExpr expression)
   {
      StringBuffer result = new StringBuffer();

      if ( (expression.getObjectName() != null) &&  (getVarName (expression.getObjectName()) != null))
      {
         result.append (getVarName (expression.getObjectName()));
         result.append (".");
      }

      result.append (getMethodName (expression.getMethodName()));
      result.append (" ");
      result.append (OPEN_NORM_BRACKET);
      Iterator iter = expression.iteratorOfParameter();
      OOExpression tmpParameter = null;
      while (iter.hasNext())
      {
         tmpParameter = (OOExpression) iter.next();
         result.append (tmpParameter.getSourceCode (this));
         if (iter.hasNext())
         {
            result.append (LIST_SEPARATOR);
            result.append (" ");
         }
      }

      result.append (CLOSE_NORM_BRACKET);

      // append next call, if available
      if (expression.getNextCall() != null)
      {
         result.append (".");
         result.append (expression.getNextCall().getSourceCode (this));
      }

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public String getSourceCode (OOExceptionExpr expression)
   {
      if (expression == OOExceptionExpr.SDM_EXCEPTION)
      {
         return SDM_EXCEPTION_STRING;
      }
      else if (expression == OOExceptionExpr.INTERRUPTED_EXCEPTION)
      {
         return INTERRUPTED_EXCEPTION_STRING;
      }
      else if (expression == OOExceptionExpr.ILLEGALDATA_EXCEPTION)
      {
         return ILLEGALDATA_EXCEPTION_STRING;
      }
      else
      {
         return getOOTypeAsString (expression.getExceptionType());
      }
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param var  No description provided
    * @return     The sourceCode value
    */
   public String getSourceCode (OOVariable var)
   {
      return getVarName (var);
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public final String getSourceCode (OOStringExpr expression)
   {
      return expression.getCondition();
   }


   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public final String getSourceCode (OOExpression expression)
   {
      return this + ".getSourceCode(" + expression + ") is empty !";
   }

   //
   // end visit OOExpression section
   //

   /**
    * Get the sourceCode attribute of the OOGenVisitor object
    *
    * @param theOp  No description provided
    * @return       The sourceCode value
    */
   public String getSourceCode (OOInfixOp theOp)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".getSourceCode(" + theOp + ")");
         }
      }

      String result = null;
      if (OOInfixOp.AND_OP.equals (theOp))
      {
         result = AND_OP_STRING;
      }
      else if (OOInfixOp.OR_OP.equals (theOp))
      {
         result = OR_OP_STRING;
      }
      else if (OOInfixOp.EQUAL_OP.equals (theOp))
      {
         result = EQUAL_OP_STRING;
      }
      else if (OOInfixOp.NOT_EQUAL_OP.equals (theOp))
      {
         result = NOT_EQUAL_OP_STRING;
      }
      else if (OOInfixOp.LESS_OP.equals (theOp))
      {
         result = LESS_OP_STRING;
      }
      else if (OOInfixOp.GREATER_OP.equals (theOp))
      {
         result = GREATER_OP_STRING;
      }
      else if (OOInfixOp.LESS_EQUAL_OP.equals (theOp))
      {
         result = LESS_EQUAL_OP_STRING;
      }
      else if (OOInfixOp.GREATER_EQUAL_OP.equals (theOp))
      {
         result = GREATER_EQUAL_OP_STRING;
      }
      else if (OOInfixOp.ADD_OP.equals (theOp))
      {
         result = ADD_OP_STRING;
      }
      else if (OOInfixOp.SUB_OP.equals (theOp))
      {
         result = SUB_OP_STRING;
      }
      else if (OOInfixOp.MUL_OP.equals (theOp))
      {
         result = MUL_OP_STRING;
      }
      else if (OOInfixOp.DIV_OP.equals (theOp))
      {
         result = DIV_OP_STRING;
      }
      else
      {
         result = "unkown op " + theOp;
      }

      return result;
   }

   //
   // default append section
   //
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param comment  No description provided
    */
   public void appendClassCommentary (String comment)
   {
      getCurrentBuffer().append (comment);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param declString  No description provided
    */
   public void appendClassDeclaration (String declString)
   {
      getCurrentBuffer().append (declString);
      newLine();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param comment  No description provided
    */
   public void appendAttrCommentary (String comment)
   {
      getCurrentBuffer().append (comment);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param declString  No description provided
    */
   public void appendAttrDeclaration (String declString)
   {
      getCurrentBuffer().append (declString);
      newLine();
      newLine();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param comment  No description provided
    */
   public void appendMethodCommentary (String comment)
   {
      getCurrentBuffer().append (comment);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param declString  No description provided
    */
   public void appendMethodDeclaration (String declString)
   {
      getCurrentBuffer().append (declString);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param body  No description provided
    */
   public void appendMethodBody (String body)
   {
      getCurrentBuffer().append (body != null ? body : END_OF_STATEMENT);
   }

   //
   // get identifier, keywords, literals etc.
   //

   /**
    * Get the varName attribute of the OOGenVisitor object
    *
    * @param type  No description provided
    * @return      The varName value
    */
   public String getVarName (OOVariableType type)
   {
      return getVarName (OO.variable (type));
   }


   /**
    * Get the varName attribute of the OOGenVisitor object
    *
    * @param variableName  No description provided
    * @return              The varName value
    */
   public String getVarName (OOVariable variableName)
   {
      String fujabaPrefix = JavaPreferences.get().getInternalVariablePrefix();
      String result;

      if (OOVariableType.iFujabaOrig == variableName.getType())
      {
         result = variableName.getFirstName();
      }
      else if (OOVariableType.iFujabaSuccess == variableName.getType())
      {
         result = fujabaPrefix + "Success";
      }
      else if (OOVariableType.iFujabaAtEnd == variableName.getType())
      {
         result = fujabaPrefix + "AtEnd";
      }
      else if (OOVariableType.iFujabaNextStep == variableName.getType())
      {
         result = fujabaPrefix + "NextStep";
      }
      else if (OOVariableType.iFujabaReturnParam == variableName.getType())
      {
         result = fujabaPrefix + "ReturnParam";
      }
      else if (OOVariableType.iFujabaEnum == variableName.getType())
      {
         result = fujabaPrefix + "Enum";
      }
      else if (OOVariableType.iFujabaOptional == variableName.getType())
      {
         result = fujabaPrefix + "Optional";
      }
      else if (OOVariableType.iFujabaFirstBinding == variableName.getType())
      {
         result = fujabaPrefix + "FirstBinding";
      }
      else if (OOVariableType.iBreakExceptionVariable == variableName.getType())
      {
         result = fujabaPrefix + "InternalException";
      }
      else if (OOVariableType.iBreakExceptionText == variableName.getType())
      {
         result =
            CGU.upFirstChar (getVarName (OOVariableType.iBreakExceptionVariable)).toString();
      }
      else if (OOVariableType.iFujabaTmpObject == variableName.getType())
      {
         result = fujabaPrefix + "TmpObject" + variableName.getFirstName();
      }
      else if (OOVariableType.iFujabaGoodObject == variableName.getType())
      {
         result = fujabaPrefix + "GoodObject";
      }
      else if (OOVariableType.iFujabaIter == variableName.getType())
      {
         result = fujabaPrefix + "Iter" + variableName.getFirstName();
      }
      else if (OOVariableType.iFujabaFor == variableName.getType())
      {
         result = fujabaPrefix + CGU.upFirstChar (variableName.getFirstName()) +
            "For" + CGU.upFirstChar (variableName.getSecondName());
      }
      else if (OOVariableType.iFujabaUpperBound == variableName.getType())
      {
         result = fujabaPrefix + "UpperBound_" + CGU.upFirstChar (variableName.getFirstName());
      }

      else
      {
         throw new
            RuntimeException
             ("Don't know how to handle the internal variable kind : " + variableName);
      }

      return result;
   }


   /**
    * Get the uMLAttrDeclVisibilityString attribute of the OOGenVisitor object
    *
    * @param visibility  No description provided
    * @return            The uMLAttrDeclVisibilityString value
    */
   public String getAttrDeclVisibilityString (int visibility)
   {
      return getVisibilityString (visibility);
   }


   /**
    * Get the uMLMethodDeclVisibilityString attribute of the OOGenVisitor object
    *
    * @param visibility  No description provided
    * @return            The uMLMethodDeclVisibilityString value
    */
   public String getMethodDeclVisibilityString (int visibility)
   {
      return getVisibilityString (visibility);
   }


   /**
    * Get the interfaceName attribute of the OOGenVisitor object
    *
    * @param theClass  No description provided
    * @return          The interfaceName value
    */
   public String getInterfaceName (FClass theClass)
   {
      return theClass.getName();
   }


   /**
    * Get the className attribute of the OOGenVisitor object
    *
    * @param theClass  No description provided
    * @return          The className value
    */
   public String getClassName (FClass theClass)
   {
      return theClass.getName();
   }


   /**
    * Get the oOTypeAsString attribute of the OOGenVisitor object
    *
    * @param theType  No description provided
    * @return         The oOTypeAsString value
    */
   public final String getOOTypeAsString (OOTypeInterface theType)
   {
      return getOOTypeAsString (theType, false);
   }


   /**
    * Get the oOTypeAsString attribute of the OOGenVisitor object
    *
    * @param theType  No description provided
    * @param isBound  No description provided
    * @return         The oOTypeAsString value
    */
   public final String getOOTypeAsString (OOTypeInterface theType, boolean isBound)
   {
      if (theType instanceof OOContainerType)
      {
         return getOOTypeAsString ((OOContainerType) theType, isBound);
      }
      else if (theType instanceof OOType)
      {
         return getOOTypeAsString ((OOType) theType, isBound);
      }
      else
      {
         throw new Error (this + ".getOOTypeAsString(" + theType + "," + isBound + ")");
      }
   }


   /**
    * Get the oOTypeAsString attribute of the OOGenVisitor object
    *
    * @param theType  No description provided
    * @param isBound  No description provided
    * @return         The oOTypeAsString value
    */
   public String getOOTypeAsString (OOContainerType theType, boolean isBound)
   {
      return getTypeNameAsString (getContainerName (theType.getCollectionType(),
         isBound),
         theType.isPointer());
   }


   /**
    * Get the oOTypeAsString attribute of the OOGenVisitor object
    *
    * @param theType  No description provided
    * @return         The oOTypeAsString value
    */
   public final String getOOTypeAsString (OOType theType)
   {
      return getOOTypeAsString (theType, false);
   }


   /**
    * Get the oOTypeAsString attribute of the OOGenVisitor object
    *
    * @param theType  No description provided
    * @param isBound  No description provided
    * @return         The oOTypeAsString value
    */
   public String getOOTypeAsString (OOType theType, boolean isBound)
   {
      if (theType.getUmlType() != OOType.EMPTY_TYPE)
      {
         return getTypeAsString (theType.getUmlType(), theType.isPointer());
      }
      else if (theType.getTypeName() != null)
      {
         return getTypeNameAsString (theType.getTypeName(), theType.isPointer());
      }
      else
      {
         return START_BLOCK_COMMENTARY + " illegal type : " +
            theType + " " + END_BLOCK_COMMENTARY;
      }
   }


   /**
    * Get the uMLTypeAsString attribute of the OOGenVisitor object
    *
    * @param theType  No description provided
    * @return         The uMLTypeAsString value
    */
   public String getTypeAsString (FType theType)
   {
      return getTypeAsString (theType, false);
   }


   /**
    * Get the uMLTypeAsString attribute of the OOGenVisitor object
    *
    * @param theType    No description provided
    * @param isPointer  No description provided
    * @return           The uMLTypeAsString value
    */
   public String getTypeAsString (FType theType, boolean isPointer)
   {
      String typeString = null;
      if (theType instanceof FBaseTypes)
      {
         typeString = getProgLangType ((FBaseTypes) theType);
      }
      else if (theType instanceof FArray)
      {
         typeString = getTypeAsString ( ((FArray) theType).getFArrayType(), false) + "[]";
      }
      else
      {
         if (theType instanceof FClass)
         {
            FClass theClass = (FClass) theType;
            typeString = theClass.getName();
            theClass = theClass.getFDeclaredInClass();
            while (theClass != null)
            {
               typeString = theClass.getName() + "." + typeString;
               theClass = theClass.getFDeclaredInClass();
            }
         }
         else
         {
            typeString = theType != null ? theType.getName() : null;
         }
         if (isPointer)
         {
            typeString += POINTER_POSTFIX;
         }
      }

      return typeString;
   }


   /**
    * Get the uMLTypeNameAsString attribute of the OOGenVisitor object
    *
    * @param typeName  No description provided
    * @return          The uMLTypeNameAsString value
    */
   public String getTypeNameAsString (String typeName)
   {
      return getTypeNameAsString (typeName, false);
   }


   /**
    * Get the uMLTypeNameAsString attribute of the OOGenVisitor object
    *
    * @param typeName   No description provided
    * @param isPointer  No description provided
    * @return           The uMLTypeNameAsString value
    */
   public String getTypeNameAsString (String typeName, boolean isPointer)
   {
      String newTypeName = getFromProgLangTypes (typeName);
      // typeName is basetype
      if (newTypeName != null)
      {
         return newTypeName;
      }

      // ensure type exists
      //FIXME: Codegen shouldn't add something to model
      //       hope this breaks nothing
      boolean created = false;
      FType tmpType = UMLProject.get().findType (null, typeName);
      if (tmpType == null)
      {
         tmpType = UMLProject.get().getOrNewType (null, typeName, false);
         created = true;
      }
      if (tmpType instanceof FClass)
      {
         newTypeName =  ((FClass) tmpType).getFullClassName();
      }
      else
      {
         newTypeName = tmpType.getName();
      }
      if (created && tmpType instanceof BasicIncrement)
      {
          ((BasicIncrement) tmpType).removeYou();
      }
      return newTypeName +
          (isPointer ? POINTER_POSTFIX : "");
   }


   /**
    * Get the operationText attribute of the OOGenVisitor object
    *
    * @param operation  No description provided
    * @return           The operationText value
    * @deprecated       seems not to be used, otherwise remove deprecation 07.10.03 AZ
    */
   public final String getOperationText (int operation)
   {
      switch (operation)
      {
         case UMLAttrExprPair.EQUAL:
            return  (EQUAL_OP_STRING);
         case UMLAttrExprPair.NOTEQUAL:
            return  (NOT_EQUAL_OP_STRING);
         case UMLAttrExprPair.LESS:
            return  (LESS_OP_STRING);
         case UMLAttrExprPair.GREATER:
            return  (GREATER_OP_STRING);
         case UMLAttrExprPair.LESSEQUAL:
            return  (LESS_EQUAL_OP_STRING);
         case UMLAttrExprPair.GREATEREQUAL:
            return  (GREATER_EQUAL_OP_STRING);
         default:
            throw new Error ("unsupported operation : " + operation);
      }
   }


   /**
    * FAttribute : 'progLangTypes : TreeMap (String,String) '
    */
   private transient TreeMap progLangTypes = null;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected abstract void initProgLangTypes();


   /**
    * Get the progLangType attribute of the CodeGenVisitor object
    *
    * @param baseType  No description provided
    * @return          The progLangType value
    */
   public final String getProgLangType (FBaseTypes baseType)
   {
      return getFromProgLangTypes (baseType.getName());
   }


   /**
    * @param object  No description provided
    * @return        No description provided
    * @see           #progLangTypes
    */
   protected final boolean hasInProgLangTypes (String object)
   {
      if (progLangTypes == null)
      {
         return false;
      }
      else
      {
         return progLangTypes.containsValue (object);
      }
   }


   /**
    * @param key  No description provided
    * @return     No description provided
    * @see        #progLangTypes
    */
   protected final boolean hasKeyInProgLangTypes (String key)
   {
      if (progLangTypes == null)
      {
         return false;
      }
      else
      {
         return progLangTypes.containsKey (key);
      }
   }


   /**
    * @return   No description provided
    * @see      #progLangTypes
    */
   protected final Iterator iteratorOfProgLangTypes()
   {
      if (progLangTypes == null)
      {
         return FEmptyIterator.get();
      }
      else
      {
         return progLangTypes.values().iterator();
      }
   }


   /**
    * @return   No description provided
    * @see      #progLangTypes
    */
   protected final Iterator keysOfProgLangTypes()
   {
      if (progLangTypes == null)
      {
         return FEmptyIterator.get();
      }
      else
      {
         return progLangTypes.keySet().iterator();
      }
   }


   /**
    * @return   No description provided
    * @see      #progLangTypes
    */
   protected final Iterator entriesOfProgLangTypes()
   {
      if (progLangTypes == null)
      {
         return FEmptyIterator.get();
      }
      else
      {
         return progLangTypes.entrySet().iterator();
      }
   }


   /**
    * @param key  No description provided
    * @return     The fromProgLangTypes value
    * @see        #progLangTypes
    */
   protected final String getFromProgLangTypes (String key)
   {
      String elem;

      if (this.hasKeyInProgLangTypes (key))
      {
         elem = (String) this.progLangTypes.get (key);
      }
      else
      {
         elem = null;
      }

      return elem;
   }


   /**
    * @param key   The object added.
    * @param elem  The object added.
    * @see         #progLangTypes
    */
   protected final void addToProgLangTypes (String key, String elem)
   {
      if (key != null && elem != null)
      {
         if (progLangTypes == null)
         {
            progLangTypes = new TreeMap();
         }

         this.progLangTypes.put (key, elem);
      }
   }


   /**
    * @param key  No description provided
    * @see        #progLangTypes
    */
   protected final void removeKeyFromProgLangTypes (String key)
   {
      if (this.hasKeyInProgLangTypes (key))
      {
         this.progLangTypes.remove (key);
      }
   }


   /**
    * @see   #progLangTypes
    */
   protected final void removeAllFromProgLangTypes()
   {
      Iterator iter = keysOfProgLangTypes();
      String tmpKey = null;
      while (iter.hasNext())
      {
         tmpKey = (String) iter.next();
         iter.remove();
         removeKeyFromProgLangTypes (tmpKey);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected String[] visibilityStrings = null;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public abstract void initVisibilityStrings();


   /**
    * Get the visibilityString attribute of the CodeGenVisitor object
    *
    * @param visibility  No description provided
    * @return            The visibilityString value
    */
   public final String getVisibilityString (int visibility)
   {
      return visibilityStrings[visibility];
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      super.removeYou();
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "OOGenVisitor[]";
   }
}

/*
 * $Log: OOGenVisitor.java,v $
 * Revision 1.98.2.2  2005/10/10 10:48:09  fklar
 * changed log-message-type from error to debug
 *
 */
