/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import java.util.Iterator;
import java.util.Set;
import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.uml.*;
import de.upb.tools.fca.FEmptyIterator;
import de.upb.tools.fca.FHashSet;


/**
 * @author    $Author: cschneid $
 * @version   $Revision: 1.46 $
 */
public class UMLMethodOOHandler extends OOGenStrategyHandler
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (UMLMethodOOHandler.class);


   /**
    * Default Constructor
    */
   public UMLMethodOOHandler()
   {
      super();
   }


   /**
    * Get the responsible attribute of the UMLMethodOOHandler object
    *
    * @param incr  No description provided
    * @return      The responsible value
    */
   public boolean isResponsible (FElement incr)
   {
      return  (incr instanceof FMethod);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public final boolean needToken()
   {
      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param incr       No description provided
    * @param prevToken  No description provided
    * @param param      No description provided
    * @return           No description provided
    */
   public OOGenToken generateSourceCode (FElement incr, OOGenToken prevToken, Object param[])
   {
      FMethod theMethod = (FMethod) incr;

      if (log.isDebugEnabled())
      {
         log.debug (this + ".generateSourceCode(theMethod=" + theMethod + ")");
      }

      OOGenStrategyClient client = (OOGenStrategyClient) getClientOfChain();

      // generate comment
      FCommentary fCommentary = theMethod.getFComment();
      if ( (fCommentary == null) &&  (theMethod.getFRevSpec() != null))
      {
         fCommentary =  ((UMLStartActivity) theMethod.getFRevSpec()).getFComment();
      }
      if (fCommentary != null)
      {
         String commentaryText;
         if (theMethod.isParsed())
         {
            commentaryText = fCommentary.getText();
         }
         else
         {
            commentaryText = client.createFMethodCommentary (fCommentary);
         }
         client.appendFMethodCommentary (commentaryText);
      }

      // generate declaration
      String umlMethodDecl = client.createFMethodDeclaration (theMethod);
      client.appendFMethodDeclaration (umlMethodDecl);

      // generate body
      if (generateBody (theMethod))
      {
         String umlMethodBody = createFMethodBody (theMethod);

         client.appendFMethodBody (umlMethodBody);
      }
      else
      {
         client.appendFMethodBody (null);
      }

      // has no tokens
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theMethod  No description provided
    * @return           No description provided
    */
   public String createFMethodBody (FMethod theMethod)
   {
      if (log.isDebugEnabled())
      {
         log.debug (this + ".createFMethodBody(" + theMethod + ")");
      }

      StringBuffer theBody = new StringBuffer();

      if (theMethod.getMethodBody() != null)
      {
         if (theMethod.isParsed())
         {
            theBody.append (theMethod.getMethodBody());
         }
         else
         {
            OOGenVisitor current = (OOGenVisitor) getClientOfChain().getCurrentVisitor();
            theBody.append ("\n");
            theBody.append (current.getSourceCode (OO.beginMethodBody()));

            theBody.append ("\n");
            theBody.append (current.indentText (theMethod.getMethodBody()));

            theBody.append ("\n");
            theBody.append (current.getSourceCode (OO.endMethodBody()));
         }
      }
      else if (theMethod.getFRevSpec() != null
         &&  ((UMLStartActivity) theMethod.getFRevSpec()).getActivityDiagram() != null)
      {
         theBody.append ("\n");
         theBody.append (generateCode ( ((UMLStartActivity) theMethod.getFRevSpec()).getActivityDiagram()));
      }
      else if (theMethod.isSignal())
      {
         OOGenToken token = new OOGenToken();
         token.addToStatement (OO.beginMethodBody());
         token.addToStatement (OO.varDecl (UMLBaseTypes.INTEGER, "priority", null));

         OOVariable priority = OO.variable ("priority");
         token.addToStatement (OO.ifStat (OO.infixOp (OO.call ("myFReactive", "getThread"), OOInfixOp.EQUAL_OP, OO.call ("Thread",
            "currentThread"))));
         token.addToStatement (OO.startBlock());
         token.addToStatement (OO.assignStat (priority, new OOStringExpr ("FReactive.INTERNAL_EVENT")));
         token.addToStatement (OO.endBlock());
         token.addToStatement (new OOElseStatement());
         token.addToStatement (OO.startBlock());
         token.addToStatement (OO.assignStat (priority, new OOStringExpr ("FReactive.EXTERNAL_EVENT")));
         token.addToStatement (OO.endBlock());

         OOVariable event = OO.variable ("FEvent event");
         OOExpression tmpParameters = new OOStringExpr ("\"" + theMethod.getName() + "\"");
         OONewObjectExpr tmpExpression = OO.newObject ("FEvent", tmpParameters);
         OOStatement tmpAssignment = OO.assignStat (event, tmpExpression);
         token.addToStatement (tmpAssignment);
         event = OO.variable ("event");

         Iterator iter = theMethod.iteratorOfParam();
         while (iter.hasNext())
         {
            UMLParam param = (UMLParam) iter.next();
            String paramName = param.getName();
            String paramType = param.getFParamType().getName();

            //HOTFIX: primitive types are not handled correctly
            OOGenVisitor visit =  ((OOGenStrategyClient) getClientOfChain()).getCurrentOOGenVisitor();
            String correctParamType = visit.getTypeAsString (param.getFParamType());
            if (!paramType.equals (correctParamType))
            {
               // exp = OO.newObject(param.getParamType(), new
               // OOStringExpr(paramName));
               paramName = "new " + paramType + " (" + paramName + ")";
            }

            OOMethod tmpMethod = OO.method ("Args", OOMethodType.ADD_METHOD);
            tmpParameters = new OOStringExpr (paramType + ".class" + ", " + paramName);
            OOExpression tmpCallMethodExpr = OO.call (event, tmpMethod, tmpParameters);
            token.addToStatement (new OOExprStatement (tmpCallMethodExpr));
         }

         OOVariable queue = OO.variable ("myFReactive");
         OOMethod tmpMethod = OO.method ("enqueueEvent");
         tmpParameters = new OOStringExpr ("event, priority");
         OOExpression tmpCallMethodExpr = OO.call (queue, tmpMethod, tmpParameters);
         token.addToStatement (new OOExprStatement (tmpCallMethodExpr));
         tmpMethod = OO.method ("notifyMe");
         tmpCallMethodExpr = OO.call (queue, tmpMethod, null);
         token.addToStatement (new OOExprStatement (tmpCallMethodExpr));

         token.addToStatement (OO.endMethodBody());

         theBody.append ("\n");
         theBody.append (token.getSourceCode());
      }
      else
      {
         OOGenVisitor current = (OOGenVisitor) getClientOfChain().getCurrentVisitor();
         theBody.append ("\n");
         theBody.append (current.getSourceCode (OO.beginMethodBody()));
         theBody.append ("\n");
         theBody.append (current.getSourceCode (OO.endMethodBody()));
      }

      processMethodBodyReviewers (theBody, theMethod);

      return theBody.toString();
   }


   /**
    * Determines whether to generate a body for an FMethod or not.
    *
    * @param theMethod  the method to exermine
    * @return           true if generating a body is necessary, otherwise false.
    */
   protected boolean generateBody (FMethod theMethod)
   {
      final FStereotype interfaceType = UMLStereotypeManager.get().getFromStereotypes (UMLStereotypeManager.INTERFACE);

      return ! (theMethod.isAbstract() || interfaceType.hasInIncrements (theMethod.getFParent()));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param activityDiagram  No description provided
    * @return                 No description provided
    */
   public StringBuffer generateCode (UMLActivityDiagram activityDiagram)
   {
      if (log.isDebugEnabled())
      {
         log.debug (this + ".generateCode(activityDiagram=" + activityDiagram + ")");
      }

      StringBuffer buf = new StringBuffer();

      OOGenToken token = new OOGenToken();
      getClientOfChain().generateSourceCodeFor (activityDiagram, token, null);

      while (token != null)
      {
         buf.append (token.getSourceCode());
         token = token.getNext();
      }

      return buf;
   } // generateCode


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "UMLMethodOOHandler[]";
   }


   /**
    * Let all reviewers modify the method body (no order guaranteed)
    *
    * @param methodBody  the standard method body
    * @param method      the method the body belongs to
    */
   private void processMethodBodyReviewers (StringBuffer methodBody, FMethod method)
   {
      Iterator it = iteratorOfReviewers();
      while (it.hasNext())
      {
         MethodBodyReviewer reviewer = (MethodBodyReviewer) it.next();
         reviewer.review (methodBody, method, this);
      }
   }


   /**
    * All reviewers are stored here
    */
   private static Set reviewers;


   /**
    * Interface that reviewers must implement
    *
    * @author    $Author: cschneid $
    * @version   $Revision: 1.46 $
    */
   public interface MethodBodyReviewer
   {
      /**
       * This method is called whenever a method body was generated to give the Reviewer the
       * possibility to change the code.
       *
       * @param methodBody  current method body to modify
       * @param method      method this body will belong to
       * @param handler     the handler that processes the code generation
       */
      public void review (StringBuffer methodBody, FMethod method, UMLMethodOOHandler handler);
   }


   /**
    * Add a reviewer to reviewers. All reviewers may modify code generated for all methods
    *
    * @param reviewer  the new reviewer
    */
   public static void addToReviewers (MethodBodyReviewer reviewer)
   {
      if (reviewers == null)
      {
         reviewers = new FHashSet();
      }
      reviewers.add (reviewer);
   }


   /**
    * Iterator through all reviewers
    *
    * @return   the iterator
    */
   public static Iterator iteratorOfReviewers()
   {
      return reviewers == null ? FEmptyIterator.get() : reviewers.iterator();
   }


   /**
    * Remove a reviewer from the list of reviewers.
    *
    * @param reviewer  the reviewer to be remove
    */
   public static void removeFromReviewers (MethodBodyReviewer reviewer)
   {
      if (reviewers != null)
      {
         reviewers.remove (reviewer);
      }
   }
}

/*
 * $Log: UMLMethodOOHandler.java,v $
 * Revision 1.46  2005/01/27 22:16:51  cschneid
 * fixed FReactive codegen
 *
 */
