/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen.digester.rules;

import java.lang.reflect.Method;
import java.util.Stack;

import org.apache.log4j.Logger;
import org.xml.sax.Attributes;

import de.uni_paderborn.fujaba.codegen.digester.XDigester;
import de.uni_paderborn.fujaba.codegen.digester.XRule;


/**
 * Rule implementation that calls a method on the (top-1) (parent) object, passing the top
 * object (child) as an argument. It is commonly used to establish parent-child relationships.
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.9.2.1 $ $Date: 2005/09/30 18:57:16 $
 */
public class XAddToRule extends XRule
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (XAddToRule.class);


   /**
    * Construct a "set next" rule with the specified method name.
    *
    * @param digester    The associated Digester
    * @param methodName  Method name of the parent method to call
    * @param className   No description provided
    */
   public XAddToRule (XDigester digester, String methodName,
                      String className)
   {
      super (digester);
      this.methodName = methodName;
      this.className = className;
   }


   // ----------------------------------------------------- Instance Variables


   /**
    * name of attribute which contains methodname
    */
   protected String methodName = null;

   /**
    * name of attribute which contains classname
    */
   protected String className = null;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Stack methodValueStack;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Stack classValueStack;
   // --------------------------------------------------------- Public Methods

   /**
    * Process the start of this element.
    *
    * @param attributes  The attribute list for this element
    * @throws Exception  Exception description not provided
    */
   public void begin (Attributes attributes) throws Exception
   {
      String methodValue = attributes.getValue (methodName);
      String classValue = attributes.getValue (className);

      //remember them for end()
      if (methodValueStack == null)
      {
         methodValueStack = new Stack();
      }
      methodValueStack.push (methodValue);
      if (classValueStack == null)
      {
         classValueStack = new Stack();
      }
      classValueStack.push (classValue);
   }


   /**
    * Process the end of this element.
    *
    * @throws Exception  Exception description not provided
    */
   public void end() throws Exception
   {
      String methodValue = (String) methodValueStack.pop();
      String classValue = (String) classValueStack.pop();

      // Identify the objects to be used
      Object child = digester.peek (0);
      Object parent = digester.peek (1);

      // Call the specified method
      Class classNames[] = new Class[1];
      if (className != null)
      {
         classNames[0] = digester.getClassLoader().loadClass (classValue);
      }
      else
      {
         classNames[0] = child.getClass();
      }
      Method method = null;
      while (method == null && classNames[0] != null)
      {
         try
         {
            method = parent.getClass().getMethod (methodValue, classNames);
         }
         catch (NoSuchMethodException exception)
         {
            method = null;

            // failed , try again with superclass
            classNames[0] = classNames[0].getSuperclass();
         }
      }
      if (digester.getDebug() >= 1)
      {
         digester.log ("Call " + parent.getClass().getName() + "." +
            methodValue + "(" + child + ")");
      }

      if (method == null)
      {
         if (log.isInfoEnabled())
         {
            log.info ("method is null");
         }
      }
      method.invoke (parent, new Object[]
         {
         child
         }
         );

   }


   /**
    * Clean up after parsing is complete.
    *
    * @throws Exception  Exception description not provided
    */
   public void finish() throws Exception
   {

      methodName = null;
      className = null;

   }

}

/*
 * $Log: XAddToRule.java,v $
 * Revision 1.9.2.1  2005/09/30 18:57:16  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
