/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.coobra;

import java.lang.reflect.InvocationTargetException;
import java.util.Set;

import javax.swing.SwingUtilities;

import de.tu_bs.coobra.ObjectChange;
import de.tu_bs.coobra.errors.ErrorHandlerAdapter;
import de.tu_bs.coobra.errors.ServerIdentifierChangedException;
import de.tu_bs.coobra.errors.UndoRedoException;
import de.tu_bs.xmlreflect.util.errors.AccessorImplementationException;
import de.tu_bs.xmlreflect.util.errors.AccessorInvocationException;
import de.uni_paderborn.fujaba.app.FrameMain;
import de.uni_paderborn.fujaba.messages.Warning;
import de.uni_paderborn.fujaba.metamodel.FElement;
import de.upb.tools.fca.FHashSet;


/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author$
 * @version   $Revision$ $Date$
 */
public class FujabaErrorHandler extends ErrorHandlerAdapter
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   Set reportedErrors = new FHashSet();

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   boolean ignoredReportedErrors = false;


   /**
    * Called when an exception occures
    *
    * @param exception  exception that must be handled (throw it if you don't know how
    *      to handle)
    */
   public void handle (final RuntimeException exception)
   {
      if (exception != null)
      {
         if (exception instanceof ServerIdentifierChangedException)
         {
            throw exception;
         }
         Throwable realException = exception;
         FElement element = null;
         while (realException.getCause() != null &&  (
            realException instanceof InvocationTargetException
            || realException instanceof AccessorImplementationException
            || realException instanceof AccessorInvocationException
            || realException instanceof UndoRedoException
            || realException.getClass().equals (RuntimeException.class)
            ))
         {
            if (realException instanceof UndoRedoException)
            {
               UndoRedoException urEx = (UndoRedoException) realException;
               ObjectChange change = urEx.getChange();
               if (change != null && change.getAffectedObject() instanceof FElement)
               {
                  element = (FElement) change.getAffectedObject();
               }
            }
            realException = realException.getCause();
         }

         if (realException.getMessage() == null || !reportedErrors.contains (realException.toString()))
         {
            realException.printStackTrace();
            final Throwable exceptionToPrint = realException;
            final FElement referredElement = element;
            SwingUtilities.invokeLater (
               new Runnable()
               {
                  /**
                   * @see   Thread#run()
                   */
                  public void run()
                  {
                     Warning warning = new Warning ("Problem in loading/undo/redo:" + exceptionToPrint.toString());
                     if (referredElement != null)
                     {
                        warning.addToContext (referredElement);
                     }

                     warning.setMessageCategory ("Loading/Undo/Redo");
                     warning.setThrowable (exceptionToPrint);
                     FrameMain.get().getMessageView().addToMessages (warning);
                  }
               });
            reportedErrors.add (realException.toString());
         }
         else
         {
            if (!ignoredReportedErrors)
            {
               FujabaChangeManager.log.error ("Additional errors of same kind, will not be reported!");
               ignoredReportedErrors = true;
            }
         }
      }
   }


   /**
    * Called when an error with a description occures
    *
    * @param errorMessage  description of the error (throw new RuntimeException(errorMessage)
    *      if you don't know how to handle)
    */
   public void handle (final String errorMessage)
   {
      FujabaChangeManager.log.error (errorMessage);
      SwingUtilities.invokeLater (
         new Runnable()
         {
            /**
             * @see   Thread#run()
             */
            public void run()
            {
               FrameMain.get().setStatusLabel (errorMessage);
            }
         });
   }
}

/*
 * $Log$
 * Revision 1.1.2.3  2006/04/27 14:42:15  lowende
 * Some compile warnings removed.
 *
 */
