/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.coobra.actions;

import java.awt.event.ActionEvent;
import java.util.*;

import javax.swing.*;
import org.apache.log4j.Logger;

import de.tu_bs.coobra.ObjectChange;
import de.tu_bs.coobra.ObjectChangeUtils;
import de.uni_paderborn.fujaba.coobra.FujabaChangeManager;
import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.uml.UMLActivityDiagram;
import de.uni_paderborn.fujaba.uml.UMLStoryPattern;
import de.upb.lib.userinterface.UserInterfaceManager;
import de.upb.tools.fca.*;


/**
 * Copy some ASGElements (changes to them) via CoObRA
 *
 * @author    $Author: cschneid $
 * @version   $Revision: 1.16.2.1 $
 */
public class CopyAction extends AbstractAction
{
   /**
    * Instances of this class store data for copying object with CoObRA
    *
    * @author    $Author: cschneid $
    * @version   $Revision: 1.16.2.1 $
    */
   public static class CopyData
   {
      /**
       * list of selected objects upon copy
       */
      private List selectedObjects = new FLinkedList();
      /**
       * object that were copied (may be more than the selected ones)
       */
      private Set copyObjects = new FHashSet();
      /**
       * the set of changes that should be copied on paste
       */
      private Set changesToBeCopied = new FTreeSet (ObjectChangeUtils.get().reverseComparator);


      /**
       * @return   the set of changes that should be copied on paste
       */
      public Set getChangesToBeCopied()
      {
         return changesToBeCopied;
      }


      /**
       * @return   object that were copied (may be more than the selected ones)
       */
      public Set getCopyObjects()
      {
         return copyObjects;
      }


      /**
       * @return   list of selected objects upon copy (add custom objects here)
       */
      public List getSelectedObjects()
      {
         return selectedObjects;
      }
   }


   /**
    * @return   the CopyData that is the VMs clipboard
    */
   public static CopyData getClipboard()
   {
      if (clipboard == null)
      {
         clipboard = new CopyData();
      }
      return clipboard;
   }


   /**
    * what has been copied
    */
   private static CopyData clipboard;
   /**
    * (possibly) ignored values when determining which changes to copy
    */
   private Set possiblyIgnoredValues;
   /**
    * a map of compartment links: <br>
    * either ("fieldName", null) to copy all objects that link to another copied object via
    * fieldName <br>
    * or (ObjectClass.class, "fieldName") to copy all objects that link to another copied object
    * of class ObjectClass via fieldName
    */
   private Map compartmentLinks;
   /**
    * data for copying
    */
   private CopyData copyData;


   /**
    * @return   the data for copying the objects
    */
   public CopyData getCopyData()
   {
      return copyData;
   }


   /**
    * Create a copy action with a specific data storage
    *
    * @param copyData  where object to be copied reside (selectedObjects) and the data for
    *                 copying will be entered
    */
   public CopyAction (CopyData copyData)
   {
      if (copyData == null)
      {
         throw new NullPointerException();
      }
      this.copyData = copyData;
   }


   /**
    * Defines an <code>Action</code> object with a default description string and default icon.
    */
   public CopyAction()
   {
      this (getClipboard());
   }


   /**
    * initialize sets
    */
   private void init()
   {
      newValueToChangesMap = null;
      if (possiblyIgnoredValues == null)
      {
         possiblyIgnoredValues = new FHashSet();
      }
      if (clipboard != null)
      {
         if (compartmentLinks == null)
         {
            compartmentLinks = new FHashMap();
            //common
            compartmentLinks.put ("parent", null);
            //roles
            compartmentLinks.put ("revRightRole", null);
            compartmentLinks.put ("revLeftRole", null);
            // cardinality
            compartmentLinks.put ("revCard", null);
            //generalization
            compartmentLinks.put (FGeneralization.SUBCLASS_PROPERTY, null);
            //methods
            compartmentLinks.put ("storyMethod", null);
            compartmentLinks.put (FParam.REV_PARAM_PROPERTY, null);
            //stories
            compartmentLinks.put ("spec", null);
            //files
            compartmentLinks.put ("contains", null);
            //activity diags
            compartmentLinks.put (UMLActivityDiagram.class, "diagrams");
            compartmentLinks.put (UMLStoryPattern.class, "diagrams");
            compartmentLinks.put ("revAttrs", null);
            compartmentLinks.put ("revStoryPattern", null);
            compartmentLinks.put ("uMLAction", null);
            compartmentLinks.put ("revGuard", null);
            compartmentLinks.put ("activity", null);
            compartmentLinks.put ("diag", null);
            //for UMLCollabStat
            compartmentLinks.put ("myPattern", null);
            //for ASGUnparseInformation
            compartmentLinks.put ("aSGElement", null);
            compartmentLinks.put ("unparseInformations", null);
            compartmentLinks.put ("aSGInformation", null);
            //compartmentLinks.put("information", null);
            //for FChapter
            compartmentLinks.put ("revTitle", null);
            // constraints
            //compartmentLinks.put (UMLConstraint.class, "constraints");

         }
      }
      getCopyData().getCopyObjects().clear();
      possiblyIgnoredValues.clear();
      getCopyData().getChangesToBeCopied().clear();
   }


   /**
    * Invoked when the action occurs.
    *
    * @param e  event that caused the action
    */
   public void actionPerformed (ActionEvent e)
   {
      getCopyData().getSelectedObjects().clear();

      Object source = e.getSource();
      if (source instanceof Iterator)
      {
         Iterator it = (Iterator) source;
         while (it.hasNext())
         {
            Object obj = it.next();
            if (obj instanceof FElement)
            {
               getCopyData().getSelectedObjects().add (obj);
            }
         }
      }
      else if (source instanceof FElement)
      {
         getCopyData().getSelectedObjects().add (source);
      }
      { //workaround for methods/attrs (classes are selected, too)

         Iterator it = getCopyData().getSelectedObjects().iterator();
         while (it.hasNext())
         {
            Object obj = it.next();
            if (obj instanceof FMethod)
            {
               getCopyData().getSelectedObjects().remove ( ((FMethod) obj).getFParent());
            }
            else if (obj instanceof FAttr)
            {
               getCopyData().getSelectedObjects().remove ( ((FAttr) obj).getFParent());
            }
         }
      }

      copy();
   }


   /**
    * map from object to all changes having the key as new value
    */
   FDuplicatedHashMap newValueToChangesMap;

   /**
    * Logger
    */
   private final static Logger L = Logger.getLogger (CopyAction.class);


   /**
    * complete the CopyData with the copyObjects and copyChanges data. Objects are expected
    * to be in the selectedObjects set.
    */
   public void copy()
   {
      try
      {
         init();
         if (!getCopyData().getSelectedObjects().isEmpty())
         {
            Iterator it = getCopyData().getSelectedObjects().iterator();
            while (it.hasNext())
            {
               FElement element = (FElement) it.next();
               L.debug ("copy " + element + ":");
               addToCopyObjects (element);
            }

            //do not ignore what we copy
            possiblyIgnoredValues.removeAll (getCopyData().getCopyObjects());

            it = getCopyData().getCopyObjects().iterator();
            while (it.hasNext())
            {
               Object obj = it.next();
               L.debug ("I will copy " + obj + " (" + obj.getClass() + ")");
            }
            it = possiblyIgnoredValues.iterator();
            while (it.hasNext())
            {
               Object obj = it.next();
               L.debug ("I will ignore value " + obj + " (" + obj.getClass() + ")");
            }

            //todo: when copying gets slow find another way to get the interesting changes:
            for (int chosenIterator = 0; chosenIterator < 2; ++chosenIterator)
            {
               //do it with two different iterators
               if (chosenIterator == 0)
               {
                  it = FujabaChangeManager.getVMRepository().iteratorOfChangesBase();
               }
               else
               {
                  it = FujabaChangeManager.getVMRepository().iteratorOfChanges();
               }

               while (it.hasNext())
               {
                  ObjectChange change = (ObjectChange) it.next();
                  if (getCopyData().getCopyObjects().contains (change.getAffectedObject()))
                  {
                     if (!possiblyIgnoredValues.contains (change.getNewValue()) &&
                        !possiblyIgnoredValues.contains (change.getOldValue()))
                     {
                        getCopyData().getChangesToBeCopied().add (change);
                     }
                     else
                     {
                        L.debug ("ignored: " + change);
                     }
                  }
               }
            }

            if (getCopyData() == getClipboard())
            {
               //we now have the changes that should be copied on paste
               UserInterfaceManager.get().getFromActions ("coobra.paste").setEnabled (true);
            }
            //clear unneeded data
            possiblyIgnoredValues.clear();
         }
      }
      finally
      {
         newValueToChangesMap = null;
      }
   }


   /**
    * Adds an element to the getCopyData().getCopyObjects() set. Recursively processes the
    * added object to find compartments and ignored values.
    *
    * @param element  what to copy
    */
   private void addToCopyObjects (FElement element)
   {
      if (!getCopyData().getCopyObjects().contains (element))
      {
         getCopyData().getCopyObjects().add (element);
         addReferencedObjectsFromChanges (element);
         //addReferencedObjectsFromChanges (FujabaChangeManager.getVMRepository().iteratorOfChanges(), element);
      }
   }


   /**
    * add all objects referenced in changes that have <code>element</code> as new value
    *
    * @param element  ASGElement that was already added
    */
   private void addReferencedObjectsFromChanges (FElement element)
   {
      if (element != null)
      {
         if (newValueToChangesMap == null)
         {
            newValueToChangesMap =
               new FDuplicatedHashMap()
               {
                  /**
                   * @return   a new collection that will be used to store values
                   */
                  protected Collection createValueCollection()
                  {
                     return new HashSet();
                  }
               };
            putChangesIntoHashMap (FujabaChangeManager.getVMRepository().iteratorOfChangesBase());
            putChangesIntoHashMap (FujabaChangeManager.getVMRepository().iteratorOfChanges());
         }

         final Collection values = newValueToChangesMap.values (element);
         if (values != null)
         {
            Iterator it = values.iterator();
            while (it.hasNext())
            {
               ObjectChange change = (ObjectChange) it.next();
               { //if (change.getNewValue() == element)

                  Class newValueClass = change.getNewValue() != null ?
                     change.getNewValue().getClass() : null;
                  if (compartmentLinks.containsKey (change.getFieldName()) ||
                     change.getFieldName().equals (compartmentLinks.get (newValueClass)))
                  {
                     addToCopyObjects ((FElement) change.getAffectedObject());
                  }
                  else if (change.getKey() == null &&
                     change.getTypeOfChange() == ObjectChange.CHANGE_TYPE_ALTER)
                  {
                     if (!possiblyIgnoredValues.contains (change.getAffectedObject()))
                     {
                        //log.info("Possibly ignoring " + change.getAffectedObject()
                        //   + " ("+ change.getAffectedObject().getClass().getName() + "."
                        //   + change.getFieldName() +")" );
                        possiblyIgnoredValues.add (change.getAffectedObject());
                        L.debug ("possibly ignoring " + change.getAffectedObject() + "(" + change.getAffectedObject().getClass() + ") cause of field " + change.getFieldName());
                     }
                  }
               }
            }
         }
      }
   }


   /**
    * adds changes to {@link #newValueToChangesMap}
    *
    * @param it  iterator through changes
    */
   private void putChangesIntoHashMap (Iterator it)
   {
      while (it.hasNext())
      {
         ObjectChange change = (ObjectChange) it.next();
         if (change.getNewValue() != null)
         {
            newValueToChangesMap.put (change.getNewValue(), change);
         }
      }
   }
}

/*
 * $Log: CopyAction.java,v $
 * Revision 1.16.2.1  2005/12/20 13:16:49  cschneid
 * More warning/error info, some copy/paste fixes from erik, message listener for JEM (andreas/manuel), new libs
 *
 */
