/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.coobra.actions;

import java.awt.event.ActionEvent;
import java.util.Iterator;

import javax.swing.AbstractAction;

import de.tu_bs.coobra.LocalRepository;
import de.tu_bs.coobra.ObjectChange;
import de.tu_bs.coobra.errors.SomeChangesFailedException;
import de.uni_paderborn.fujaba.app.FrameMain;
import de.uni_paderborn.fujaba.app.action.UpdateProjectTreeAction;
import de.uni_paderborn.fujaba.asg.*;
import de.uni_paderborn.fujaba.coobra.FujabaChangeManager;
import de.uni_paderborn.fujaba.messages.*;
import de.uni_paderborn.fujaba.uml.UMLProject;


/**
 * Update the local repository from the server
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.24 $
 */
public class UpdateAction extends AbstractAction implements LocalRepository.ProgressCallback
{

   /**
    * dialog for displaying progress
    */
   ProgressDialog progress = new ProgressDialog (FrameMain.get());
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String MESSAGE_CLASS_MERGE_CONFLICT = "Merge conflict";


   /**
    * called when update/restore/store progress
    *
    * @param percent  progress percentage
    */
   public void progress (int percent)
   {
      if (progress != null)
      {
         //progress.setVisible( percent > 0 && percent < 100 );
         progress.setPercentage (percent);
      }
   }


   /**
    * called when action should be performed
    *
    * @param e
    */
   public void actionPerformed (ActionEvent e)
   {
      try
      {
         if (FujabaChangeManager.getVMRepository() != null && FujabaChangeManager.getVMRepository().isConnected())
         {
               new Thread()
               {
                  /**
                   * @see   Runnable#run()
                   */
                  public void run()
                  {
                     while (!progress.isVisible())
                     {
                        try
                        {
                           Thread.sleep (10);
                        }
                        catch (InterruptedException e)
                        {
                        }
                     }
                     try
                     {
                        update();
                     }
                     finally
                     {
                        progress.hide();
                     }
                  }
               }.start();
            progress.show();

            UMLProject.get().refreshDisplay();
            new UpdateProjectTreeAction().actionPerformed (null);

            FrameMain.get().setStatusLabel ("update succeeded");
         }
         else
         {
            FrameMain.get().setStatusLabel ("Not connected to a server!");
         }
      }
      catch (RuntimeException ex)
      {
         FrameMain.get().setStatusLabel ("update failed: " + ex.getMessage());
         throw ex;
      }
   }


   /**
    * perform the update
    */
   void update()
   {
      MessageView messageView = FrameMain.get().getMessageView();
      messageView.deleteMessages (MESSAGE_CLASS_MERGE_CONFLICT);
      try
      {
         LocalRepository repository = FujabaChangeManager.getVMRepository();
         if (repository != null)
         {
            repository.update (this);
            UMLProject.get().setSaved (!repository.isUndoPossible());
         }
      }
      catch (SomeChangesFailedException ex)
      {
         boolean layoutConflicts = false;
         Iterator it = ex.iteratorOfFailedChanges();
         while (it.hasNext())
         {
            ObjectChange change = (ObjectChange) it.next();
            if (change.getAffectedObject() instanceof ASGUnparseInformation || change.getAffectedObject() instanceof ASGInformation)
            {
               layoutConflicts = true;
            }
            else
            {
               String text;
               switch (change.getTypeOfChange())
               {
                  case ObjectChange.CHANGE_TYPE_ALTER:
                     text = "changing field '" + change.getFieldName() +
                        "' of object '" + change.getAffectedObject() +
                        "' from '" + change.getOldValue() +
                        "' to '" + change.getNewValue() + "'";
                     break;
                  default:
                     text = change.toString();
               }
               Object reason = ex.getReason (change);
               if (reason != null)
               {
                  if (! (reason instanceof Exception) ||
                     ! ((Exception) reason).getMessage().startsWith ("Current field value does not match to redo change"))
                  {
                     text += "\nreason:" + reason + "";
                  }
               }
               Message message = new Warning ("Conflict while updating: " + text);
               message.setMessageCategory (MESSAGE_CLASS_MERGE_CONFLICT);
               message.addToContext ((ASGElement) change.getAffectedObject());
               messageView.addToMessages (message);
            }
         }
         if (layoutConflicts)
         {
            messageView.addToMessages (new Message ("some layout conflicts"));
         }
         if (ex.iteratorOfFailedChanges().hasNext())
         {
            messageView.addToMessages (new Message ("run update again to clear update messages"));
            FrameMain.get().showMessageView();
         }
      }
   }
}

/*
 * $Log: UpdateAction.java,v $
 * Revision 1.24  2005/02/24 15:54:51  lowende
 * Removed some compile warnings.
 *
 */
