/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa;

import java.awt.*;
import java.util.*;

import javax.swing.*;

import de.uni_paderborn.fujaba.fsa.unparse.LogicUnparseInterface;


/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: fklar $
 * @version   $Revision: 1.21.2.1 $
 */
public abstract class FSAContainer extends FSAObject
{
   /**
    * Constructor for class FSAContainer
    *
    * @param incr      No description provided
    * @param propName  No description provided
    */
   public FSAContainer (LogicUnparseInterface incr, String propName)
   {
      super (incr, propName);
   }


   /**
    * Constructor for class FSAContainer
    *
    * @param incr      No description provided
    * @param propName  No description provided
    * @param parent    No description provided
    */
   public FSAContainer (LogicUnparseInterface incr, String propName, JComponent parent)
   {
      super (incr, propName, parent);
   }


   /**
    * Constructor for class FSAContainer
    *
    * @param incr                 No description provided
    * @param propName             No description provided
    * @param parent               No description provided
    * @param transientProperties  No description provided
    */
   public FSAContainer (LogicUnparseInterface incr, String propName, JComponent parent, boolean transientProperties)
   {
      super (incr, propName, parent, transientProperties);
   }


   /**
    * Constructor for class FSAContainer
    *
    * @param incr          No description provided
    * @param propName      No description provided
    * @param parent        No description provided
    * @param myJComponent  No description provided
    */
   public FSAContainer (LogicUnparseInterface incr,
                        String propName,
                        JComponent parent,
                        JComponent myJComponent)
   {
      super (incr, propName, parent, myJComponent);
   }


   /**
    * Get the defaultUpdaterClass attribute of the FSAContainer object
    *
    * @return   The defaultUpdaterClass value
    */
   public Class getDefaultUpdaterClass()
   {
      return null;
   }


   /**
    * Get the defaultAttrName attribute of the FSAContainer object
    *
    * @return   The defaultAttrName value
    */
   public String getDefaultAttrName()
   {
      return null;
   }


   /**
    * Get the childConnector attribute of the FSAContainer object
    *
    * @return   The childConnector value
    */
   public JComponent getChildConnector()
   {
      return getJComponent();
   }


   /**
    * Access method for an one to n association.
    *
    * @param object  The object added.
    */
   public void addToChildren (FSAObject object)
   {
      object.setParentOfJComponent (getChildConnector());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param object  No description provided
    */
   public void removeFromChildren (FSAObject object)
   {
      object.setParentOfJComponent (null);
   }


   /**
    * Get the children attribute of the FSAContainer object
    *
    * @return   The children value
    */
   public FSAObject[] getChildren()
   {
      Vector children = new Vector();

      Iterator childIter = iteratorOfChildren();
      while (childIter.hasNext())
      {
         children.addElement (childIter.next());
      }
      if (children.size() > 0)
      {
         FSAObject[] fsaChildren = new FSAObject[children.size()];
         return (FSAObject[]) children.toArray (fsaChildren);
      }
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfChildren()
   {
      FSAObject[] children = getChildren();

      if (children != null)
      {
         return children.length;
      }
      return 0;
   }


   /**
    * Get the fromChildren attribute of the FSAContainer object
    *
    * @param pos  No description provided
    * @return     The fromChildren value
    */
   public FSAObject getFromChildren (int pos)
   {
      Iterator childIter = iteratorOfChildren();
      for (int i = 0; i < pos && childIter.hasNext(); i++)
      {
         childIter.next();
      }
      if (childIter.hasNext())
      {
         return (FSAObject) childIter.next();
      }
      throw new IndexOutOfBoundsException ("FSAContainer::getChild (int): " + pos);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfChildren()
   {
      return
         new Iterator()
         {
            private JComponent connector;
            private Stack stack;
            private FSAObject next;
            {
               stack = new Stack();
               connector = getChildConnector();
               if (connector != null)
               {
                  stack.push (connector);
               }

               next = getNext();
            }


            public boolean hasNext()
            {
               return  (next != null);
            }


            public Object next() throws NoSuchElementException
            {
               Object value = next;
               if (next == null)
               {
                  throw new NoSuchElementException();
               }
               else
               {
                  next = getNext();
               }

               return value;
            }


            public void remove() throws UnsupportedOperationException
            {
               throw new UnsupportedOperationException();
            }


            private FSAObject getNext()
            {
               while (!stack.empty())
               {
                  Component comp = (Component) stack.pop();
                  if (comp != null && comp != connector && comp instanceof JComponent)
                  {
                     Object child =  ((JComponent) comp).
                        getClientProperty (FSAObject.JCOMPONENT_CLIENT_PROPERTY);

                     if (child != null && child instanceof FSAObject &&
                        child != FSAContainer.this)
                     {
                        return (FSAObject) child;
                     }
                  }

                  if (comp instanceof Container)
                  {
                     Container container = (Container) comp;
                     for (int i = container.getComponentCount() - 1; i > -1; i--)
                     {
                        Component child = container.getComponent (i);
                        stack.push (child);
                     }
                  }
               }
               return null;
            }
         };
   }


   /**
    * Get the fromChildren attribute of the FSAContainer object
    *
    * @param p  No description provided
    * @return   The fromChildren value
    */
   public FSAObject[] getFromChildren (Point p)
   {
      Vector childrenAt = new Vector();
      Iterator childIter = iteratorOfChildren();
      while (childIter.hasNext())
      {
         FSAObject child = (FSAObject) childIter.next();
         JComponent comp = child.getJComponent();
         if (comp != null && comp.contains (p))
         {
            childrenAt.addElement (child);
         }
      }
      if (childrenAt.size() > 0)
      {
         return (FSAObject[]) childrenAt.toArray();
      }

      return null;
   }


   /**
    * Get the fromChildren attribute of the FSAContainer object
    *
    * @param rect      No description provided
    * @param complete  No description provided
    * @return          The fromChildren value
    */
   public FSAObject[] getFromChildren (Rectangle rect, boolean complete)
   {
      Vector childrenIn = new Vector();
      Iterator childIter = iteratorOfChildren();
      while (childIter.hasNext())
      {
         FSAObject child = (FSAObject) childIter.next();
         JComponent comp = child.getJComponent();

         if (comp != null)
         {
            Rectangle bounds = comp.getBounds();
            if ( (complete && rect.contains (bounds)) ||
                (!complete && rect.intersects (bounds)))
            {
               childrenIn.addElement (child);
            }
         }
      }
      if (childrenIn.size() > 0)
      {
         return (FSAObject[]) childrenIn.toArray();
      }

      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param object  No description provided
    * @return        No description provided
    */
   public boolean hasInChildren (FSAObject object)
   {
      if (object == null)
      {
         return false;
      }

      JComponent connector = getChildConnector();
      Container comp = object.getJComponent();

      while (comp != null)
      {
         comp = comp.getParent();

         if (comp == connector)
         {
            return true;
         }
      }

      return false;
   }


   /**
    * Access method for an one to n association.
    *
    * @param comp  The object added.
    */
   public void addToComponents (Component comp)
   {
      if (comp != null)
      {
         Container parent = comp.getParent();
         if (parent != null)
         {
            parent.remove (comp);
         }

         getChildConnector().add (comp);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param comp  No description provided
    */
   public void removeFromComponents (Component comp)
   {
      Container connector = getChildConnector();

      if (connector != null)
      {
         getChildConnector().remove (comp);
      }
   }


   /**
    * Get the components attribute of the FSAContainer object
    *
    * @return   The components value
    */
   public Component[] getComponents()
   {
      Container connector = getChildConnector();

      if (connector == null)
      {
         return null;
      }

      return connector.getComponents();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfComponents()
   {
      Container connector = getChildConnector();

      if (connector == null)
      {
         return 0;
      }

      return connector.getComponentCount();
   }


   /**
    * Get the fromComponents attribute of the FSAContainer object
    *
    * @param pos  No description provided
    * @return     The fromComponents value
    */
   public Component getFromComponents (int pos)
   {
      Container connector = getChildConnector();

      if (connector != null)
      {
         return connector.getComponent (pos);
      }

      throw new IndexOutOfBoundsException ("FSAContainer::getFromComponents (int): " + pos);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfComponents()
   {
      return
         new Iterator()
         {
            private JComponent connector;
            private int pos;
            private int size;
            {
               connector = getChildConnector();
               pos = -1;
               size =  (connector == null) ? 0 : connector.getComponentCount();
            }


            public boolean hasNext()
            {
               return  (pos < size - 1);
            }


            public Object next() throws NoSuchElementException
            {
               if (!hasNext())
               {
                  throw new NoSuchElementException();
               }
               else
               {
                  pos++;
               }

               return connector.getComponent (pos);
            }


            public void remove() throws UnsupportedOperationException
            {
               throw new UnsupportedOperationException();
            }
         };
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param comp  No description provided
    * @return      No description provided
    */
   public boolean hasInComponents (Component comp)
   {
      if (comp == null)
      {
         return false;
      }

      JComponent connector = getChildConnector();

      return  (comp.getParent() == connector);
   }


   /**
    * Sets the bold attribute of the FSAContainer object
    *
    * @param bold  The new bold value
    */
   public void setBold (boolean bold)
   {
      super.setBold (bold);

      Iterator iter = iteratorOfChildren();
      while (iter.hasNext())
      {
         FSAObject child = (FSAObject) iter.next();
         child.setBold (bold);
      } // end of while ()
   }


   /**
    * Sets the italic attribute of the FSAContainer object
    *
    * @param italic  The new italic value
    */
   public void setItalic (boolean italic)
   {
      super.setItalic (italic);

      Iterator iter = iteratorOfChildren();
      while (iter.hasNext())
      {
         FSAObject child = (FSAObject) iter.next();
         child.setItalic (italic);
      } // end of while ()
   }


   /**
    * Sets the opaque attribute of the FSAContainer object
    *
    * @param opaque  The new opaque value
    */
   public void setOpaque (boolean opaque)
   {
      super.setOpaque (opaque);

      Iterator iter = iteratorOfChildren();
      while (iter.hasNext())
      {
         FSAObject child = (FSAObject) iter.next();
         child.setOpaque (opaque);
      } // end of while ()
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      FSAObject[] children = getChildren();
      if (children != null)
      {
         for (int i = 0; i < children.length; i++)
         {
            children[i].removeYou();
         }
      }
      super.removeYou();
   }
}

/*
 * $Log: FSAContainer.java,v $
 * Revision 1.21.2.1  2005/05/30 15:31:59  fklar
 * introduced new constructor that takes transientProperties as argument
 *
 */
