/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa;

import java.lang.reflect.Method;
import java.util.Iterator;
import java.util.TreeMap;

import de.uni_paderborn.fujaba.basic.Utility;
import de.upb.tools.fca.FEmptyIterator;


/**
 * cache FSAObject.derivePropertyMethods
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.15 $
 */
public class FSADerivePropertyMethodsCache
{
   /**
    * the Singleton instance
    */
   private static FSADerivePropertyMethodsCache theInstance = null;


   /**
    * Default Constructor (private)
    */
   private FSADerivePropertyMethodsCache() { }


   /**
    * get the Singleton instance
    *
    * @return   the Singleton instance
    */
   public static FSADerivePropertyMethodsCache get()
   {
      if (theInstance == null)
      {
         theInstance = new FSADerivePropertyMethodsCache();
      }

      return theInstance;
   }


   /**
    * Get the propertyMethods attribute of the FSADerivePropertyMethodsCache object
    *
    * @param propertyName  No description provided
    * @param obj           No description provided
    * @return              The propertyMethods value
    */
   public Method[] getPropertyMethods (String propertyName, Object obj)
   {

      Class objClass = obj.getClass();

      return getPropertyMethods (propertyName, objClass);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static transient String COLON = ":";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static transient String GET_PREFIX = "get";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static transient String IS_PREFIX = "is";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static transient String HAS_PREFIX = "has";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static transient String SET_PREFIX = "set";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static transient String ITERATOROF_PREFIX = "iteratorOf";


   /**
    * Get the propertyMethods attribute of the FSADerivePropertyMethodsCache object
    *
    * @param propertyName  No description provided
    * @param objClass      No description provided
    * @return              The propertyMethods value
    */
   public Method[] getPropertyMethods (String propertyName, Class objClass)
   {
      String className = objClass.getName();
      StringBuffer buffer = new StringBuffer (className).append (COLON).append (propertyName);
      String key = buffer.toString();
      Method result[];
      result = getFromMethodCache (key);

      if (result == null)
      {
         String propFirstUp = Utility.upFirstChar (propertyName);
         boolean found = false;

         Method[] methods = getFromMethodCache (className);
         if (methods == null)
         {
            methods = objClass.getMethods();
            addToMethodCache (className, methods);
         } // end of if ()

         Method boolGetMethod = null;
         Method exactNameMethod = null;
         Method iterGetMethod = null;

         Method getMethod = null;
         Method setMethod = null;

         buffer.delete (0, buffer.length());
         String getProp = buffer.append (GET_PREFIX).append (propFirstUp).toString();
         String hasProp = buffer.replace (0, 3, HAS_PREFIX).toString();
         String setProp = buffer.replace (0, 3, SET_PREFIX).toString();
         String isProp = buffer.replace (0, 3, IS_PREFIX).toString();
         String iterProp = buffer.replace (0, 2, ITERATOROF_PREFIX).toString();

         // this workaround is needed because getMethod(String name, Class[] signature)
         // only works if you know the exact types of the parameters
         for (int i = 0; i < methods.length && getMethod == null; i++)
         {
            Class returnType = methods[i].getReturnType();
            Class[] parameters = methods[i].getParameterTypes();
            String methodName = methods[i].getName();

            if ( (returnType != Void.TYPE
               && parameters.length == 0
               && methodName.equals (getProp))
               )
            {
               getMethod = methods[i];
            }
            else if (boolGetMethod == null
               && returnType == Boolean.TYPE
               && parameters.length == 0
               &&  (methodName.equals (isProp)
               || methodName.equals (hasProp))
               )
            {
               boolGetMethod = methods[i];
            }
            else if (iterGetMethod == null
               && returnType.equals (Iterator.class)
               && parameters.length == 0
               &&  (methodName.equals (iterProp))
               )
            {
               iterGetMethod = methods[i];
            }
            else if (exactNameMethod == null
               &&  ( (parameters.length == 0
               && returnType != Void.TYPE) ||
                (parameters.length == 1
               &&  (returnType == Void.TYPE ||
               returnType == Boolean.TYPE ||
               returnType == parameters[0])
               ))
               && methodName.equals (propertyName)
               )
            {
               exactNameMethod = methods[i];
            }
         }

         if (getMethod == null)
         {
            getMethod = boolGetMethod;
         }
         if (getMethod == null)
         {
            getMethod = iterGetMethod;
         }

         found =  (getMethod != null);

         if (found)
         {
            try
            {
               setMethod = objClass.getMethod (setProp,
                  new Class[]
                  {
                  getMethod.getReturnType()
                  }
                  );
            }
            catch (NoSuchMethodException e)
            {
            }
         }
         else if (exactNameMethod != null)
         {
            if (exactNameMethod.getParameterTypes().length == 0)
            {
               getMethod = exactNameMethod;
            }
            else
            {
               setMethod = exactNameMethod;
            }
         }

         result = new Method[2];
         result[0] = getMethod;
         result[1] = setMethod;

         addToMethodCache (key, result);
      }

      return result;
   }


   /**
    * UMLAttribute : 'methodCache : TreeMap (umlIncr.getClass().getName():<name>,Method[])
    * '
    */
   private TreeMap methodCache = null;


   /**
    * @param key  No description provided
    * @return     No description provided
    * @see        #methodCache
    */
   private boolean hasKeyInMethodCache (String key)
   {
      if (methodCache == null)
      {
         return false;
      }
      else
      {
         return methodCache.containsKey (key);
      }
   }


   /**
    * @return   No description provided
    * @see      #methodCache
    */
   private Iterator iteratorOfMethodCache()
   {
      if (methodCache == null)
      {
         return FEmptyIterator.get();
      }
      else
      {
         return methodCache.values().iterator();
      }
   }


   /**
    * @return   No description provided
    * @see      #methodCache
    */
   private Iterator keysOfMethodCache()
   {
      if (methodCache == null)
      {
         return FEmptyIterator.get();
      }
      else
      {
         return methodCache.keySet().iterator();
      }
   }


   /**
    * @return   No description provided
    * @see      #methodCache
    */
   private Iterator entriesOfMethodCache()
   {
      if (methodCache == null)
      {
         return FEmptyIterator.get();
      }
      else
      {
         return methodCache.entrySet().iterator();
      }
   }


   /**
    * @param key  No description provided
    * @return     The fromMethodCache value
    * @see        #methodCache
    */
   private Method[] getFromMethodCache (String key)
   {
      Method[] elem;

      if (this.hasKeyInMethodCache (key))
      {
         elem = (Method[]) this.methodCache.get (key);
      }
      else
      {
         elem = null;
      }

      return elem;
   }


   /**
    * @param key   The object added.
    * @param elem  The object added.
    * @see         #methodCache
    */
   private void addToMethodCache (String key, Method elem[])
   {
      if (key != null && elem != null)
      {
         if (methodCache == null)
         {
            methodCache = new TreeMap();
         }

         this.methodCache.put (key, elem);
      }
   }


   /**
    * @param key  No description provided
    * @see        #methodCache
    */
   private void removeKeyFromMethodCache (String key)
   {
      if (this.hasKeyInMethodCache (key))
      {
         this.methodCache.remove (key);
      }
   }


   /**
    * @see   #methodCache
    */
   private void removeAllFromMethodCache()
   {
      Iterator iter = keysOfMethodCache();
      String tmpKey = null;
      while (iter.hasNext())
      {
         tmpKey = (String) iter.next();
         iter.remove();
         removeKeyFromMethodCache (tmpKey);
      }
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "FSADerivePropertyMethodsCache";
   }

}

/*
 * $Log: FSADerivePropertyMethodsCache.java,v $
 * Revision 1.15  2004/11/03 10:17:52  lowende
 * Javadoc warnings removed.
 *
 */
