/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa;

import java.awt.*;
import java.util.Iterator;
import java.util.LinkedList;

import javax.swing.*;
import javax.swing.border.BevelBorder;

import org.apache.log4j.Logger;
import org.apache.log4j.Priority;

import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.fsa.actions.AddBendAction;
import de.uni_paderborn.fujaba.fsa.swing.*;
import de.uni_paderborn.fujaba.fsa.unparse.LogicUnparseInterface;
import de.uni_paderborn.fujaba.metamodel.FElement;
import de.uni_paderborn.fujaba.uml.UMLAssoc;


/**
 * Class FSAPolyLine
 *
 * @author    $Author: creckord $
 * @version   $Revision: 1.25.2.6 $
 */
public class FSAPolyLine
    extends FSABendLine
{
   public final static String ARRANGE_LOOP_BEND = "arrangeLoopBend";

   /**
    * log4j logging
    */
   private static Logger logger = Logger.getLogger (FSAPolyLine.class);


   /**
    * Constructor for class FSAPolyLine
    *
    * @param incr      No description provided
    * @param propName  No description provided
    */
   public FSAPolyLine (LogicUnparseInterface incr, String propName)
   {
      super (incr, propName);
   }


   /**
    * Constructor for class FSAObject
    *
    * @param incr          No description provided
    * @param propName      No description provided
    * @param parent        No description provided
    * @param myJComponent  No description provided
    */
   public FSAPolyLine (LogicUnparseInterface incr,
                       String propName,
                       JComponent parent,
                       JComponent myJComponent)
   {
      super (incr, propName, parent, myJComponent);
   }


   /**
    * Constructor for class FSAPolyLine
    *
    * @param incr      No description provided
    * @param propName  No description provided
    * @param parent    No description provided
    */
   public FSAPolyLine (LogicUnparseInterface incr, String propName, JComponent parent)
   {
      super (incr, propName, parent);
      if (incr instanceof UMLAssoc && "items".equals ( ((UMLAssoc) incr).getName()))
      {
         getJComponent().setBorder (new BevelBorder (BevelBorder.RAISED));
      }
   }


   /**
    * create empty JPolyLine
    *
    * @return   No description provided
    */
   protected JComponent createJComponent()
   {
      JPolyLine line = new JPolyLine (new JBend (new Point (0, 0)), new JBend (new Point (0, 0)));
      line.setBackground (FSAObject.COLOR_BACKGROUND);
      line.setForeground (FSAObject.COLOR_FOREGROUND);
      return  (line);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void arrangeLoopBends()
   {
      JPolyLine polyLine = (JPolyLine) getJComponent();

      FSABend start = getStartBend();
      FSABend end = getEndBend();
      if ( (start instanceof FSAGrab) &&  (end instanceof FSAGrab))
      {
         FSAObject startObj =  ((FSAGrab) start).getTarget();
         FSAObject endObj =  ((FSAGrab) end).getTarget();
         if ( (startObj != null) &&  (startObj.equals (endObj)))
         {
            Point point = new Point (startObj.getLocation());
            point.translate (-30, 10);
            FSABend bend = AddBendAction.newBend (this, point);
            bend.addUnparseInformation (ARRANGE_LOOP_BEND, "true");
            polyLine.insertInBends (1, (JBend) bend.getJComponent());
            point.translate (0, -40);
            bend = AddBendAction.newBend (this, point);
            bend.addUnparseInformation (ARRANGE_LOOP_BEND, "true");
            polyLine.insertInBends (2, (JBend) bend.getJComponent());
            point.translate (40, 0);
            bend = AddBendAction.newBend (this, point);
            bend.addUnparseInformation (ARRANGE_LOOP_BEND, "true");
            polyLine.insertInBends (3, (JBend) bend.getJComponent());
         }
         else
         {
            JBendLine line = (JBendLine) getJComponent();
            if (line != null)
            {
               for (Iterator iter = line.iteratorOfBends(); iter.hasNext(); )
               {
                  JBend bend = (JBend) iter.next();
                  FSABend fsaBend = (FSABend) FSAObject.getFSAObjectFromJComponent (bend);
                  if (fsaBend.getUnparseInformation (ARRANGE_LOOP_BEND) != null)
                  {
                     removeBend (bend);
                  }
               }
            }
         }
      }
   }


   /**
    * Sets the startBend attribute of the FSAPolyLine object
    *
    * @param bend  The new startBend value
    */
   public void setStartBend (FSABend bend)
   {
      super.setStartBend (bend);
      if ( ((JPolyLine) getJComponent()).sizeOfBends() == 2)
      {
         arrangeLoopBends();
      }
   }


   /**
    * Sets the endBend attribute of the FSAPolyLine object
    *
    * @param bend  The new endBend value
    */
   public void setEndBend (FSABend bend)
   {
      super.setEndBend (bend);
      if ( ((JPolyLine) getJComponent()).sizeOfBends() == 2)
      {
         arrangeLoopBends();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      JBendLine line = (JBendLine) getJComponent();
      line.removeYou();
      super.removeYou();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param force  No description provided
    */
   public void removeYouWithBends (boolean force)
   {
      JBendLine line = (JBendLine) getJComponent();
      if (line != null)
      {
         LinkedList bends = null;
         Iterator iter = line.iteratorOfBends();
         while (iter.hasNext())
         {
            JBend bend = (JBend) iter.next();

            if (bend != null)
            {
               if (bends == null)
               {
                  bends = new LinkedList();
               }
               bends.add (bend);
            }
         }
         removeYou();
         if (bends != null)
         {
            iter = bends.iterator();
            while (iter.hasNext())
            {
               JBend bend = (JBend) iter.next();
               if (force || bend.sizeOfLines() == 0)
               {
                  removeBend (bend);
               }
            }
         }
      }
      else
      {
         removeYou();
      }
   }


   protected void removeBend (JBend bend)
   {
      FSAObject fsa = FSAObject.getFSAObjectFromJComponent (bend);
      if (fsa != null)
      {
         fsa.removeYou();
      }
      bend.removeAllFromLines();
      Container parent = bend.getParent();
      if (parent != null)
      {
         parent.remove (bend);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void oldInitLocation()
   {
      LogicUnparseInterface incr = getLogic();
      if (incr instanceof ASGElement)
      {
         ASGElement asgElement = (ASGElement) incr;
         int i = 1;
         FElement asgParent = getFSAQualifier();
         //getParent() != null && getParent().getLogic() instanceof ASGElement ? (ASGElement) getParent().getLogic() : null;
         Point point = asgElement.getPointFromUnparseInformation ((ASGElement) asgParent, "bend" + i);
         //for loading old fprs:
         if (point == null)
         {
            point = asgElement.getPointFromUnparseInformation (asgElement, "bend" + i);
         }

         while (point != null)
         {
            FSABend bend = AddBendAction.newBend (this, point);
             ((JPolyLine) getJComponent()).insertInBends (i, (JBend) bend.getJComponent());

            i++;
            point = asgElement.getPointFromUnparseInformation ((ASGElement) asgParent, "bend" + i);
            //for loading old fprs:
            if (point == null)
            {
               point = asgElement.getPointFromUnparseInformation (asgElement, "bend" + i);
            }
         }
      }
   }


   /**
    * load formerly saved additional bends and add them to this polyline
    */
   protected void applyProperties()
   {
      super.applyProperties();

      int i = 1;
      Point bendLocation = getPointFromUnparseInformation (BEND_LOCATION + i);
      while (bendLocation != null)
      {
         FSABend bend = AddBendAction.newBend (this, bendLocation);
          ((JPolyLine) getJComponent()).insertInBends (i, (JBend) bend.getJComponent());

         if (logger.isEnabledFor (Priority.DEBUG))
         {
            logger.debug ("Bend #" + i + " (" + bendLocation.x + "/" + bendLocation.y +
               ") restored from unparse-information " + "of polyline with ID " +
               this.getID());
         }

         i++;
         bendLocation = getPointFromUnparseInformation (BEND_LOCATION + i);
      }

      // try to load old fpr bend information
      if (i == 1)
      {
         oldInitLocation();
      }

      this.numberOfAdditionalBends =  ((JPolyLine) getJComponent()).sizeOfBends() - 2;
   }


   /**
    * Saves the following additional properties:
    *   location-properties of all additional bends placed on this
    *   FSAPolyLine object.
    */
   protected void saveAdditionalFSAProperties()
   {
      saveAdditionalBendLocation();
      saveLayoutManagerLocations();
   }


   /**
    * Number of bends that were added to unparse-information
    * is stored in here. So we can remove the same number of bends
    * we stored in clearAdditionalBendLocation().
    */
   private int numberOfAdditionalBends;


   /**
    * Saves location of all additional bends that are placed on this polyline.
    */
   public void saveAdditionalBendLocation()
   {
      // save all additional bends
      JPolyLine jPolyLine = (JPolyLine) getJComponent();
      this.numberOfAdditionalBends = jPolyLine.sizeOfBends() - 2;
      for (int i = 1; i <= this.numberOfAdditionalBends; i++)
      {
         JBend bend = jPolyLine.getFromBends (i);
         Point bendLocation = bend.getLocation();
         addPointToUnparseInformation (BEND_LOCATION + i, bendLocation);

         if (logger.isEnabledFor (Priority.DEBUG))
         {
            logger.debug ("Bend #" + i + " (" + bendLocation.x + "/" + bendLocation.y +
               ") added to unparse-information " + "of polyline with ID " + this.getID());
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String LAYOUT_MANAGER_LOCATION = "ldc_";


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void saveLayoutManagerLocations()
   {
      JPolyLine jPolyLine = (JPolyLine) getJComponent();
      if (jPolyLine.getLayout() instanceof DecoratorLayout)
      {
         DecoratorLayout layout = (DecoratorLayout) jPolyLine.getLayout();
         Iterator iter = this.iteratorOfChildren();
         while (iter.hasNext())
         {
            FSAObject fsa = (FSAObject) iter.next();
            LogicUnparseInterface lui = this.getLogic();
            ASGElement incr = (ASGElement) lui;
            DecoratorLayout.Constraints constraints = layout.getConstraintsFromLayoutComponents (fsa.getJComponent());
            if (constraints instanceof LineDecoratorConstraints)
            {
               LineDecoratorConstraints ldc = (LineDecoratorConstraints) constraints;
               Point ldcPosition = new Point ((int)  (1000 * ldc.getPosition()), (int)  (1000 * ldc.getGap()));
               addPointToUnparseInformation (LAYOUT_MANAGER_LOCATION + incr.getName() + "_" + fsa.getPropertyName(), ldcPosition);
            }
         }
      }
   }


   /**
    * Removes location of all additional bends stored in this polyline.
    */
   public void clearAdditionalBendLocation()
   {
      ////////////////////////////////////////////////////////
      // To be shure all additional bends are removed,
      // we may remove at least 50 bends.
      //this.numberOfAdditionalBends = Math.max(50, this.numberOfAdditionalBends);
      ////////////////////////////////////////////////////////

      // remove all formerly saved bends
      for (int i = 1; i <= this.numberOfAdditionalBends; i++)
      {
         if (removePointFromUnparseInformation (BEND_LOCATION + i))
         {
            if (logger.isEnabledFor (Priority.DEBUG))
            {
               logger.debug ("Bend #" + i + " removed from unparse-information " +
                  "of polyline with ID " + this.getID());
            }
         }
      }
      this.numberOfAdditionalBends = 0;
   }
}

/*
 * $Log: FSAPolyLine.java,v $
 * Revision 1.25.2.6  2006/06/07 09:13:31  creckord
 * - UMLTransitionGuard can be null instead of UMLTransitionGuard.NONE
 * - old "repair assoc" code removed (access methods are not kept, so repairing them is no longer needed)
 * - loop bends for assocs are removed when loop is removed
 *
 */
