/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.*;
import javax.swing.border.EmptyBorder;
import javax.swing.plaf.ComponentUI;

import de.uni_paderborn.fujaba.fsa.FSAObject;


/**
 * UI that displays an Arrow on the grab
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.16 $
 */
public class ArrowGrabUI extends GrabUI
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static String ARROW = "de.uni_paderborn.fujaba.fsa.swing.ArrowGrabUI:Arrow";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JBend bend;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JArrowHead arrow;


   /**
    * Get the bend attribute of the ArrowGrabUI object
    *
    * @return   The bend value
    */
   public JBend getBend()
   {
      return bend;
   }


   /**
    * Sets the bend attribute of the ArrowGrabUI object
    *
    * @param bend  The new bend value
    * @return      No description provided
    */
   public boolean setBend (JBend bend)
   {
      if (bend != this.bend)
      {
         if (this.bend != null)
         {
            this.bend.setUI (null);
         }
         this.bend = bend;
         if (bend != null)
         {
            if (bend.getUI() != this)
            {
               bend.setUI (this);
            }
         }
         return true;
      }
      return false;
   }


   /**
    * Get the arrow attribute of the ArrowGrabUI object
    *
    * @return   The arrow value
    */
   public JArrowHead getArrow()
   {
      return arrow;
   }


   /**
    * Sets the arrow attribute of the ArrowGrabUI object
    *
    * @param arrow  The new arrow value
    * @return       No description provided
    */
   public boolean setArrow (JArrowHead arrow)
   {
      if (arrow != this.arrow)
      {
         if (this.arrow != null)
         {
            uninstallComponents();
         }
         this.arrow = arrow;
         if (arrow != null)
         {
            configureArrow();
         }

         if (bend != null)
         {
            bend.putClientProperty (ARROW, arrow);

            if (arrow != null)
            {
               installComponents();
            }

            bend.revalidate();
            bend.repaint();
         }
         return true;
      }
      return false;
   }


   //========================
   // begin UI Initialization
   //

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private PropertyChangeListener propertyChangeListener = null;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param c  No description provided
    * @return   No description provided
    */
   public static ComponentUI createUI (JComponent c)
   {
      return new ArrowGrabUI();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param c  No description provided
    */
   public void installUI (JComponent c)
   {
      setBend ((JBend) c);
      bend.setLayout (createLayoutManager());

      installDefaults();

      installComponents();
      installListeners();

      bend.setRequestFocusEnabled (true);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param c  No description provided
    */
   public void uninstallUI (JComponent c)
   {
      bend.setLayout (null);

      uninstallComponents();
      uninstallListeners();
      uninstallDefaults();

      propertyChangeListener = null;
      bend = null;
   }


   /**
    * Installs the default colors, default font, default renderer, and default editor into
    * the JComboBox.
    */
   protected void installDefaults()
   {
      bend.setOpaque (true);
      bend.setForeground (FSAObject.COLOR_FOREGROUND);
      bend.setBackground (FSAObject.COLOR_BACKGROUND);
      bend.setBorder (new EmptyBorder (0, 0, 0, 0));
      if (arrow == null)
      {
         setArrow (createArrow (new JSimpleArrowHead()));
         arrow.setBorder (new EmptyBorder (0, 0, 0, 0));
      }
      bend.putClientProperty (ARROW, arrow);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void installListeners()
   {
      if ( (propertyChangeListener = createPropertyChangeListener()) != null)
      {
         bend.addPropertyChangeListener (propertyChangeListener);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void uninstallDefaults()
   {
      bend.putClientProperty (ARROW, null);
      setArrow (null);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void uninstallListeners()
   {
      if (propertyChangeListener != null)
      {
         bend.removePropertyChangeListener (propertyChangeListener);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   protected PropertyChangeListener createPropertyChangeListener()
   {
      return new PropertyChangeHandler();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   protected LayoutManager createLayoutManager()
   {
      return new BendLayoutManager();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param arrowHead  No description provided
    * @return           No description provided
    */
   protected JArrowHead createArrow (JArrowHead arrowHead)
   {
      if (bend.getClientProperty (ARROW) != null && bend.getClientProperty (ARROW) instanceof JArrowHead)
      {
         return (JArrowHead) bend.getClientProperty (ARROW);
      }
      else
      {
         return (JArrowHead) arrowHead.clone();
      }
   }

   //
   // end UI Initialization
   //======================


   //======================
   // begin Inner classes
   //

   /**
    * This listener watches for bound properties that have changed in the JComboBox. It looks
    * for the model and editor being swapped-out and updates appropriately. It also looks for
    * changes in the editable, enabled, and maximumRowCount properties. This inner class is
    * marked &quot;public&quot; due to a compiler bug. This class should be treated as a &quot;protected&quot;
    * inner class. Instantiate it only within subclasses of <FooUI>.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.16 $
    */
   public class PropertyChangeHandler implements PropertyChangeListener
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void propertyChange (PropertyChangeEvent e)
      {
         String propertyName = e.getPropertyName();
         if (propertyName.equals ("foreground"))
         {
            if (arrow != null)
            {
               arrow.setForeground (bend.getForeground());
            }
            bend.repaint();
         }
         else if (propertyName.equals ("background"))
         {
            if (arrow != null)
            {
               arrow.setBackground (bend.getBackground());
            }
            bend.repaint();
         }
         else if (propertyName.equals ("opaque"))
         {
            if (arrow != null)
            {
               arrow.setOpaque (bend.isOpaque());
            }
            bend.repaint();
         }
         else if (propertyName.equals (JComponent.TOOL_TIP_TEXT_KEY))
         {
            updateToolTipTextForChildren();
         }
         else if (propertyName.equals ("orientation"))
         {
            if (arrow != null)
            {
               Direction orientation =  ((JGrab) bend).getOrientation().reverse();

               arrow.setDirection (orientation);
               bend.resetPoint();
            }
            bend.repaint();
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   void updateToolTipTextForChildren()
   {
      Component[] children = bend.getComponents();
      for (int i = 0; i < children.length; ++i)
      {
         if (children[i] instanceof JComponent)
         {
             ((JComponent) children[i]).setToolTipText (bend.getToolTipText());
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.16 $
    */
   public class BendLayoutManager implements LayoutManager
   {
      /**
       * Access method for an one to n association.
       *
       * @param name  The object added.
       * @param comp  The object added.
       */
      public void addLayoutComponent (String name, Component comp)
      {
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param comp  No description provided
       */
      public void removeLayoutComponent (Component comp)
      {
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param parent  No description provided
       * @return        No description provided
       */
      public Dimension preferredLayoutSize (Container parent)
      {
         return parent.getPreferredSize();
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param parent  No description provided
       * @return        No description provided
       */
      public Dimension minimumLayoutSize (Container parent)
      {
         return parent.getMinimumSize();
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param parent  No description provided
       */
      public void layoutContainer (Container parent)
      {
         Rectangle cvb;

         if (arrow != null)
         {
            cvb = rectangleForCurrentValue();
            arrow.setBounds (cvb);
         }
      }
   }

   //
   // end Inner classes
   //====================


   //===============================
   // begin Sub-Component Management
   //

   /**
    * The editor and arrow button are added to the JComboBox here.
    */
   protected void installComponents()
   {
      configureArrow();
      bend.add (arrow);
   }


   /**
    * The editor and/or arrow button are removed from the JComboBox here. This method calls
    * removeAll() on the JComboBox just to make sure that everything gets removed.
    */
   protected void uninstallComponents()
   {
      bend.removeAll(); // Just to be safe.

   }


   /**
    * Configures the editor by setting its font and adding listeners.
    */
   protected void configureArrow()
   {
      arrow.setBorder (new EmptyBorder (0, 0, 0, 0));

      if (bend != null)
      {
         arrow.setOpaque (bend.isOpaque());
         arrow.setForeground (bend.getForeground());
         arrow.setBackground (bend.getBackground());

         if (bend instanceof JGrab)
         {
            Direction orientation =  ((JGrab) bend).getOrientation().reverse();

            arrow.setDirection (orientation);
         }
      }
   }

   //
   // end Sub-Component Management
   //===============================


   //================================
   // begin ComboBoxUI Implementation
   //

   /**
    * Determines if the JBend is focus traversable. If the JComboBox is editable this returns
    * false, otherwise it returns true.
    *
    * @param c  No description provided
    * @return   The focusTraversable value
    */
   public boolean isFocusTraversable (JBend c)
   {
      return false;
   }

   //
   // end ComboBoxUI Implementation
   //==============================


   //=================================
   // begin ComponentUI Implementation

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param c  No description provided
    * @return   No description provided
    */
   protected Point calculatePoint (JComponent c)
   {
      if (c != bend || arrow == null || !bend.isVisible())
      {
         return super.calculatePoint (c);
      }

      Point result = bend.getLocation();
      Point arrowPoint = arrow.getBasePoint();

      result.x += arrowPoint.x;
      result.y += arrowPoint.y;

      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param g  No description provided
    * @param c  No description provided
    */
   public void paint (Graphics g, JComponent c)
   {
      if (c.isOpaque())
      {
         Color t = g.getColor();
         g.setColor (c.getBackground());
         Dimension size = c.getSize();
         g.fillRect (0, 0, size.width, size.height);
         g.setColor (t);
      }
   }


   /**
    * Get the preferredSize attribute of the ArrowGrabUI object
    *
    * @param c  No description provided
    * @return   The preferredSize value
    */
   public Dimension getPreferredSize (JComponent c)
   {
      Insets insets = getInsets();
      Dimension size = arrow.getPreferredSize();

      size.width += insets.left + insets.right;
      size.height += insets.top + insets.bottom;

      return size;
   }


   /**
    * Get the minimumSize attribute of the ArrowGrabUI object
    *
    * @param c  No description provided
    * @return   The minimumSize value
    */
   public Dimension getMinimumSize (JComponent c)
   {
      Insets insets = getInsets();
      Dimension size = arrow.getMinimumSize();

      size.width += insets.left + insets.right;
      size.height += insets.top + insets.bottom;

      return size;
   }


   /**
    * Get the maximumSize attribute of the ArrowGrabUI object
    *
    * @param c  No description provided
    * @return   The maximumSize value
    */
   public Dimension getMaximumSize (JComponent c)
   {
      Dimension size = getPreferredSize (c);
      size.width = Short.MAX_VALUE;
      return size;
   }

   //
   // end ComponentUI Implementation
   //===============================


   //======================
   // begin Utility Methods
   //

   /**
    * Returns the area that is reserved for drawing the currently selected item.
    *
    * @return   No description provided
    */
   protected Rectangle rectangleForCurrentValue()
   {
      int width = bend.getWidth();
      int height = bend.getHeight();
      Insets insets = getInsets();
      return new Rectangle (insets.left, insets.top,
         width -  (insets.left + insets.right),
         height -  (insets.top + insets.bottom));
   }


   /**
    * Gets the insets from the JComboBox.
    *
    * @return   The insets value
    */
   protected Insets getInsets()
   {
      return bend.getInsets();
   }

   //
   // end Utility Methods
   //====================
}

/*
 * $Log: ArrowGrabUI.java,v $
 * Revision 1.16  2004/11/03 10:17:56  lowende
 * Javadoc warnings removed.
 *
 */
