/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.util.*;

import javax.swing.*;
import javax.swing.event.*;
import de.uni_paderborn.fujaba.packagediagrams.PackageDiagram;
import de.uni_paderborn.fujaba.uml.*;

import de.upb.tools.fca.FHashMap;


/**
 * comboBoxModel for UMLVisibility
 *
 * @author    $Author: cschneid $
 * @version   $Revision: 1.19.2.2 $
 */
public class DiagramKindComboBoxModel
    implements ComboBoxModel
{

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private EventListenerList listenerList = null;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static Vector diagramKindComboBoxIconPath = new Vector();


   /**
    * Query icon path for a given diagram kind registered with {@link #registerDiagramKind}.
    *
    * @param diagramKind  index returned by {@link #registerDiagramKind}
    * @return             path to an icon
    */
   public static String getIconPath (int diagramKind)
   {
      return (String) diagramKindComboBoxIconPath.get (diagramKind);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static Vector diagramKindComboBoxIconName = new Vector();


   /**
    * Query name for a given diagram kind registered with {@link #registerDiagramKind}.
    *
    * @param diagramKind  index returned by {@link #registerDiagramKind}
    * @return             name of the diagram kind
    */
   public static String getName (int diagramKind)
   {
      return (String) diagramKindComboBoxIconName.get (diagramKind);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Integer selectedItem = null;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static Map diagClassToKindNo = new FHashMap();


   /**
    * Query index of a diagram kind.
    * Attention: indices depend on order of registration - don't use them for persistency
    *
    * @param diagramClass  class of the diagram
    * @return              index returned by {@link #registerDiagramKind}
    */
   public static int getDiagramKindIndex (Class diagramClass)
   {
      Object result = diagClassToKindNo.get (diagramClass);
      return result != null ?  ((Integer) result).intValue() : -1;
   }


   /**
    * Query class of a diagram kind.
    *
    * @param index  index returned by {@link #registerDiagramKind}
    * @return       class of the diagram
    */
   public static Class getDiagramKindClass (int index)
   {
      for (Iterator it = diagClassToKindNo.entrySet().iterator(); it.hasNext(); )
      {
         Map.Entry entry = (Map.Entry) it.next();
         if ( ((Integer) entry.getValue()).intValue() == index)
         {
            return (Class) entry.getKey();
         }
      }
      return null;
   }

   static
   {
      registerDiagramKind (PackageDiagram.class,
         "de/uni_paderborn/fujaba/packagediagrams/images/packagediagram.gif",
         "package");
      //FIX ME: indices have changed as usecase diagrams have been removed here!
      registerDiagramKind (UMLClassDiagram.class,
         "de/uni_paderborn/fujaba/packagediagrams/images/classdiagram.gif",
         "class");
      registerDiagramKind (UMLStatechart.class,
         "de/uni_paderborn/fujaba/packagediagrams/images/statechart.gif",
         "statechart");
      registerDiagramKind (UMLActivityDiagram.class,
         "de/uni_paderborn/fujaba/packagediagrams/images/storydiagram.gif",
         "story");
   }


   /**
    * Default Constructor
    */
   public DiagramKindComboBoxModel()
   {
      super();
      listenerList = new EventListenerList();
   }


   /**
    * Register a diagram kind that can be used with DiagramUsage.
    *
    * @param diagramClass  class of the diagram
    * @param icon          path to an icon displayed for the diagram kind
    * @param name          short name for the diagram kind
    * @return              index of the newly registed kind (may changed after restarting fujaba!)
    */
   public static int registerDiagramKind (Class diagramClass, String icon, String name)
   {
      int index = diagramKindComboBoxIconPath.size();
      diagClassToKindNo.put (diagramClass, new Integer (index));
      diagramKindComboBoxIconPath.add (icon);
      diagramKindComboBoxIconName.add (name);
      return index;
   }


   /**
    * Get the selectedItem attribute of the DiagramKindComboBoxModel object
    *
    * @return   The selectedItem value
    */
   public Object getSelectedItem()
   {
      return this.selectedItem;
   }


   /**
    * Sets the selectedItem attribute of the DiagramKindComboBoxModel object
    *
    * @param value  The new selectedItem value
    */
   public void setSelectedItem (Object value)
   {

      if (this.selectedItem != value)
      {
         this.selectedItem = (Integer) value;
         fireContentsChanged (this, -1, -1);
      }
   }


   /**
    * Get the elementAt attribute of the DiagramKindComboBoxModel object
    *
    * @param index  No description provided
    * @return       The elementAt value
    */
   public Object getElementAt (int index)
   {
      return new Integer (index); // this.typeCache.elementAt ( index )  ;

   }


   /**
    * Get the size attribute of the DiagramKindComboBoxModel object
    *
    * @return   The size value
    */
   public int getSize()
   {
      return diagramKindComboBoxIconPath.size();
   }


   /**
    * Add a listener to the list that's notified each time a change to the data model occurs.
    *
    * @param l  the ListDataListener
    */
   public void addListDataListener (ListDataListener l)
   {
      listenerList.add (ListDataListener.class, l);
   }


   /**
    * Remove a listener from the list that's notified each time a change to the data model
    * occurs.
    *
    * @param l  the ListDataListener
    */
   public void removeListDataListener (ListDataListener l)
   {
      listenerList.remove (ListDataListener.class, l);
   }


   /**
    * Call this method <b>after</b> one or more elements of the list change (uml-assocs are
    * created or deleted). The changed elements are specified by a closed interval index0,
    * index1, i.e. the range that includes both index0 and index1. Note that index0 need not
    * be less than or equal to index1.
    *
    * @param source  The ListModel that changed, typically "this".
    * @param index0  One end of the new interval.
    * @param index1  The other end of the new interval.
    * @see           EventListenerList
    * @see           DefaultListModel
    */
   protected void fireContentsChanged (Object source, int index0, int index1)
   {
      Object[] listeners = listenerList.getListenerList();
      ListDataEvent e = null;

      for (int i = listeners.length - 2; i >= 0; i -= 2)
      {
         if (listeners[i] == ListDataListener.class)
         {
            if (e == null)
            {
               e = new ListDataEvent (this, ListDataEvent.CONTENTS_CHANGED, index0, index1);
            }
             ((ListDataListener) listeners[i + 1]).contentsChanged (e);
         }
      }
   }


   /**
    * Call this method <b>after</b> one or more elements are added to the model (uml-assocs
    * are created). The new elements are specified by a closed interval index0, index1, i.e.
    * the range that includes both index0 and index1. Note that index0 need not be less than
    * or equal to index1.
    *
    * @param source  The ListModel that changed, typically "this".
    * @param index0  One end of the new interval.
    * @param index1  The other end of the new interval.
    * @see           EventListenerList
    * @see           DefaultListModel
    */
   protected void fireIntervalAdded (Object source, int index0, int index1)
   {
      Object[] listeners = listenerList.getListenerList();
      ListDataEvent e = null;

      for (int i = listeners.length - 2; i >= 0; i -= 2)
      {
         if (listeners[i] == ListDataListener.class)
         {
            if (e == null)
            {
               e = new ListDataEvent (this, ListDataEvent.INTERVAL_ADDED, index0, index1);
            }
             ((ListDataListener) listeners[i + 1]).intervalAdded (e);
         }
      }
   }


   /**
    * Call this method <b>after</b> one or more elements are removed from the model (uml-assocs
    * are removed). The new elements are specified by a closed interval index0, index1, i.e.
    * the range that includes both index0 and index1. Note that index0 need not be less than
    * or equal to index1.
    *
    * @param source  The ListModel that changed, typically "this".
    * @param index0  One end of the new interval.
    * @param index1  The other end of the new interval.
    * @see           EventListenerList
    * @see           DefaultListModel
    */
   protected void fireIntervalRemoved (Object source, int index0, int index1)
   {
      Object[] listeners = listenerList.getListenerList();
      ListDataEvent e = null;

      for (int i = listeners.length - 2; i >= 0; i -= 2)
      {
         if (listeners[i] == ListDataListener.class)
         {
            if (e == null)
            {
               e = new ListDataEvent (this, ListDataEvent.INTERVAL_REMOVED, index0, index1);
            }
             ((ListDataListener) listeners[i + 1]).intervalRemoved (e);
         }
      }
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "DiagramKindComboBoxModel";
   }
}

/*
 * $Log: DiagramKindComboBoxModel.java,v $
 * Revision 1.19.2.2  2005/07/11 09:04:34  cschneid
 * package diagrams support adding diagram kinds (for usecase diagrams), usages assoc pushed up to ASGDiagram, scrollbar hotfix for collabstat-editor
 *
 */
