/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;

import javax.swing.*;

import de.uni_paderborn.fujaba.fsa.listener.AscendDescendMouseHandler;
import de.uni_paderborn.fujaba.fsa.listener.EditableComponentListener;


/**
 * show with JLabel , edit with ...
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.13 $
 */
public abstract class EditableComponent extends JPanel
{
   //FIX ME: EditableComponent doesn't deal with Borders/Insets correctly

   /**
    * Default Constructor
    */
   public EditableComponent()
   {
      super();
      setLayout (OneElementLayouter.get());
      setBorder (BorderFactory.createEmptyBorder());
      AscendDescendMouseHandler.addMouseListener (this, EditableComponentListener.get());
   }


   /**
    * the "renderer" component
    */
   private JComponent component = null;


   /**
    * Get the component attribute of the EditableComponent object
    *
    * @return   The component value
    */
   protected JComponent getComponent()
   {
      return this.component;
   }


   /**
    * Sets the component attribute of the EditableComponent object
    *
    * @param component  The new component value
    */
   protected void setComponent (JComponent component)
   {
      if (this.component != component)
      {
         this.component = component;
      } // end of if ()
   }


   /**
    * UMLAttribute : 'editing : Boolean '
    */
   private boolean editing;


   /**
    * Get the value of editing.
    *
    * @return   Value of editing.
    */
   public boolean isEditing()
   {
      return this.editing;
   }


   /**
    * Set the value of editing.
    *
    * @param editing  Value to assign to editing.
    */
   public void setEditing (boolean editing)
   {
      if (this.editing != editing)
      {
         this.editing = editing;
      }
   }


   /**
    * class saves last edited instance only one instance of AbstractJComboBoxLabel with editing=true
    * is allowed
    */
   private static EditableComponent lastEditedInstance = null;


   /**
    * Get the value of lastEditedInstance.
    *
    * @return   Value of lastEditedInstance.
    */
   public static EditableComponent getLastEditedInstance()
   {
      return lastEditedInstance;
   }


   /**
    * Set the value of lastEditedInstance.
    *
    * @param lastEdited  The new lastEditedInstance value
    */
   public static void setLastEditedInstance (EditableComponent lastEdited)
   {
      if (lastEditedInstance != lastEdited)
      {
         lastEditedInstance = lastEdited;
      }
   }


   /**
    * switch from "renderer" mode to "edit" mode remove rendererComponent from this and add
    * editorComponent to this
    */
   public void setEditMode()
   {
      if (this.isEditing() == false)
      {
         if (getLastEditedInstance() != null)
         {
            getLastEditedInstance().setDisplayMode();
         }

         JComponent component = EditorManager.get().
            getEditor (getClass().getName(), getEditorClassName());

         enableEditor (component);

         this.setEditing (true);
         setLastEditedInstance (this);

         this.revalidate();
         this.repaint();
      }
   }


   /**
    * Get the preferredSize attribute of the EditableComponent object
    *
    * @return   The preferredSize value
    */
   public Dimension getPreferredSize()
   {
      Dimension textDim = getComponent().getPreferredSize();
      if (this.isEditing())
      {
         JComponent component = EditorManager.get().
            getEditor (getClass().getName(), getEditorClassName());

         Dimension dim = component.getPreferredSize();
         dim.height = textDim.height;
         return dim;
      } // end of if ()
      return textDim;
   }


   /**
    * switch from "editor" mode to "renderer" mode remove editorComponent from this and add
    * rendererComponent to this
    */
   public void setDisplayMode()
   {
      if (this.isEditing())
      {
         JComponent component = EditorManager.get().
            getEditor (getClass().getName(), getEditorClassName());

         disableEditor (component);

         this.setEditing (false);

         this.revalidate();
         this.repaint();
      }
   }


   /**
    * UMLAttribute : 'value : Object '
    */
   private Object value;


   /**
    * Get the value of value.
    *
    * @return   Value of value.
    */
   public Object getValue()
   {
      return this.value;
   }


   /**
    * Get the text attribute of the EditableComponent object
    *
    * @return   The text value
    */
   public String getText()
   {
      if (value != null)
      {
         return value.toString();
      } // end of if ()
      return "";
   }


   /**
    * Set the value of value.
    *
    * @param value  Value to assign to value.
    */
   public void setValue (Object value)
   {
      if ( (this.value == null && value != null) ||
          (this.value != null && !this.value.equals (value)))
      {
         Object oldValue = this.value;

         this.value = value;

         updateJLabel();

         if (this == getLastEditedInstance())
         {
            JComponent component = EditorManager.get().
               getEditor (getClass().getName(), getEditorClassName());

            updateEditor (component);

         } // end of if ()

         firePropertyChange ("value", oldValue, this.value);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param component  No description provided
    */
   protected void enableEditor (JComponent component)
   {
      remove (getComponent());
      add (component);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param component  No description provided
    */
   protected void disableEditor (JComponent component)
   {
      remove (component);
      add (getComponent());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param component  No description provided
    */
   protected abstract void updateEditor (JComponent component);


   /**
    * Get the editorClassName attribute of the EditableComponent object
    *
    * @return   The editorClassName value
    */
   protected abstract String getEditorClassName();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected abstract void updateJLabel();
}

/*
 * $Log: EditableComponent.java,v $
 * Revision 1.13  2004/10/20 17:49:44  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
