/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.net.URL;

import javax.swing.*;

import de.uni_paderborn.fujaba.app.FrameMain;
import de.uni_paderborn.fujaba.fsa.FSALayeredPane;
import de.uni_paderborn.lib.basic.ImageResourceManager;


/**
 * The Java Enterprise Grub Gang - shortly called J-EGG - serves to make possible problems
 * that can occur in the design phase of a software project explicit. Imagine a large software
 * design and suddenly you notice that behind nearly every part of that design something hides
 * you did not expect. Getting rid of these things is not easy ...<p>
 *
 * <strong>Please not that JEgg requires FSA (the new Fujaba display structures) to run and
 * one active diagram (what type does not matter).</strong> <p>
 *
 * Should you discover any bugs in this code, please do not remove them. :)
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.17 $
 */
public class JEgg extends JComponent implements Runnable
{
   /**
    * Creates a new JEgg and initializes it. Used by <code>start()</code>.
    */
   private JEgg()
   {
      /*
       *  Load the images, if not loaded yet
       *  and initialize the width and height variables
       */
      getIcons();

      setSize (width, height);

      addMouseListener (new EggDeathMouseListener());

      /*
       *  Calculate the initial values of direction, moveState and directionDelta
       *  randomly.
       */
      direction = (int)  (Math.random() * 8);
      moveState = (int)  (Math.random() * 3);

      if (Math.random() < 0.5)
      {
         directionDelta = -directionDelta;
      }
   }


   /**
    * Counts the instances of Bugs on the diagram.
    */
   private static int instanceCount = 0;


   /**
    * Ensures that the icons required for a JEgg are loaded. The loaded icons are put into
    * the static variables <code>icon</code> and <code>deadIcon</code>.
    *
    * @throws RuntimeException  if the icons could not be loaded.
    */
   private void getIcons()
   {
      if (icons == null)
      {
         ImageIcon myIcons[][] = new ImageIcon[8][3];
         ImageResourceManager manager = ImageResourceManager.get();

         /*
          *  There are 8 different angles having 3 different states
          */
         for (int i = 0; i < 8; i++)
         {
            for (int k = 0; k < 3; k++)
            {
               URL url = ClassLoader.getSystemClassLoader().getResource ("de/uni_paderborn/fujaba/app/images/jegg-" + i + "-" + k + ".gif");

               if (url != null)
               {
                  myIcons[i][k] = manager.getImageIcon (url);

                  /*
                   *  Calculate the maximum dimensions
                   */
                  if (myIcons[i][k] != null)
                  {
                     if (width < myIcons[i][k].getIconWidth())
                     {
                        width = myIcons[i][k].getIconWidth();
                     }

                     if (height < myIcons[i][k].getIconHeight())
                     {
                        height = myIcons[i][k].getIconHeight();
                     }
                  }
               }

               if (myIcons[i][k] == null)
               {
                  throw new RuntimeException ("Could not find egg icon " + i + "-" + k);
               }
            }
         }

         URL url = ClassLoader.getSystemClassLoader().getResource ("de/uni_paderborn/fujaba/app/images/jegg-x-x.gif");

         if (url != null)
         {
            deadIcon = manager.getImageIcon (url);
         }

         if (deadIcon == null)
         {
            throw new RuntimeException ("Could not find egg icon x-x");
         }

         /*
          *  If no exception occured yet, set the static variable.
          *  This ensures, that not a partial filled icon array
          *  is returned
          */
         icons = myIcons;
      }
   }


   /**
    * Returns the preferred size of a JEgg
    *
    * @return   The preferredSize value
    */
   public Dimension getPreferredSize()
   {
      return new Dimension (width, height);
   }


   /**
    * Paints this JEgg according to its state
    *
    * @param g  No description provided
    */
   public void paintComponent (Graphics g)
   {
      if (alive)
      {
         icons[direction][Math.abs (moveState)].paintIcon (this, g, 0, 0);
      }
      else
      {
         deadIcon.paintIcon (this, g, 0, 0);
      }
   }


   /**
    * The icons of the JEgg. To ensure that this array is set-up correctly call <code>getIcons()</code>
    * . The first dimension defines the angle of the JEgg (the variable <code>direction</code>
    * should be used as index for this dimension). This version of JEgg uses 8 different values,
    * representing the angles 30�, 60�, 120�, 150�, 210�, 240�, 300� and 330�.
    * The second dimension defines a move-state of the JEgg. There are currently three move-states.
    * (the variable <code>moveState</code> should be used as index for this dimension).
    */
   private static ImageIcon icons[][];

   /**
    * The icon for the JEgg that will be displayed when <code>alive</code> is false. To ensure
    * that this field is set-up correctly call <code>getIcons()</code>.
    */
   private static ImageIcon deadIcon;

   /**
    * The maximum width of all icons. Calculated by <code>getIcons()</code>.
    */
   private static int width = 0;

   /**
    * The maximum height of all icons. Calculated by <code>getIcons()</code>.
    */
   private static int height = 0;

   /**
    * The direction this JEgg currently has. First index for <code>icons</code> and index for
    * the <code>direction(X|Y)diff</code> arrays.
    */
   private int direction = 0;

   /**
    * The direction in which this JEgg may turn.
    */
   private int directionDelta = 1;

   /**
    * The move state this JEgg currently has. Second index for <code>icons</code>.
    */
   private int moveState = 0;

   /**
    * This array converts an angle as defined in <code>direction</code> into the position difference
    * in pixels of the x-axis that is used when a JEgg makes one step. <code>direction</code>
    * should be used as index for this.
    */
   private final static int[] DIRECTION_X_DIFF = {5, 10, 10, 5, -5, -10, -10, -5};

   /**
    * This array converts an angle as defined in <code>direction</code> into the position difference
    * in pixels of the y-axis that is used when a JEgg makes one step. <code>direction</code>
    * should be used as index for this.
    */
   private final static int[] DIRECTION_Y_DIFF = {-10, -5, 5, 10, 10, 5, -5, -10};

   /**
    * Determines whether this JEgg is still alive or whether it has already been hit by a death-bringing
    * mouse click
    */
   boolean alive = true;


   /**
    * Determines whether this JEgg is still alive or whether it has already been hit by a death-bringing
    * mouse click
    *
    * @return   The alive value
    */
   public boolean isAlive()
   {
      return alive;
   }


   /**
    * Run method for the Thread of this JEgg, as defined by the <code>Runnable</code> interface.
    */
   public void run()
   {
      try
      {
         Thread.sleep (500);
      }
      catch (InterruptedException e)
      {
      }

      synchronized (getParent())
      {
         instanceCount++;
      }

      Cursor oldCursor = getCursor();

      setCursor (new Cursor (Cursor.CROSSHAIR_CURSOR));

      /*
       *  Main loop. Executed every 80 ms
       */
      while (alive)
      {
         try
         {
            Thread.sleep (80);
         }
         catch (InterruptedException e)
         {
         }

         /*
          *  moveState rotates from 2 to 1 to 0 and from there back to 2.
          */
         moveState--;

         if (moveState < 0)
         {
            moveState = 2;
         }

         /*
          *  Check whether this JEgg is visible or concealed by
          *  some element in the view.
          *
          *  If concealed or the JEgg collided with another JEgg or the JEgg has reached a border of the view or in 5% of all other cases:
          *  - Give this JEgg a new direction and try again in the next round
          *
          *  Otherwise:
          *  - Move one step in the current direction
          */
         int concealed = isConcealed();

         if (concealed == CONCEALED || concealed == COLLISION_WITH_JEGG || Math.random() < 0.05)
         {
            changeDirection();
         }
         else
         {
            int newX = getX() + DIRECTION_X_DIFF[direction];
            int newY = getY() + DIRECTION_Y_DIFF[direction];

            if (newX < 0 || newX + getWidth() > getContainerWidth() || newY < 0 || newY + getHeight() > getContainerHeight())
            {
               changeDirection();
            }
            else
            {
               setLocation (newX, newY);
            }
         }
      }

      /*
       *  When the main loop has been left,
       *  this JEgg was killed. Show the new image
       *  and go into the background
       */
      setCursor (oldCursor);

      toBack();

      repaint();

      synchronized (getParent())
      {
         instanceCount--;
         if (instanceCount == 0)
         {
            JOptionPane.showMessageDialog (getParent(), "Your diagram is bug free", "Congratulations!", JOptionPane.INFORMATION_MESSAGE);
         }

      }

      try
      {
         /*
          *  Wait for 15 seconds until removing dead bug
          */
         Thread.sleep (1000L * 15L);
      }
      catch (InterruptedException e)
      {
      }

      removeYou();
   }


   /**
    * Returns the width of the container this component is contained in
    *
    * @return   the width of the container this component is contained in or -1 if this component
    *      has no container.
    */
   private int getContainerWidth()
   {
      Container parent = getParent();

      if (parent != null)
      {
         return parent.getWidth();
      }
      else
      {
         return -1;
      }
   }


   /**
    * Returns the height of the container this component is contained in
    *
    * @return   the height of the container this component is contained in or -1 if this component
    *      has no container.
    */
   private int getContainerHeight()
   {
      Container parent = getParent();

      if (parent != null)
      {
         return parent.getHeight();
      }
      else
      {
         return -1;
      }
   }


   /**
    * Checks whether this component is concealed by any other component in the view (with the
    * exception of JLines, because they are not really useful for hiding :). If this component
    * is completely concealed, <code>CONCEALED</code> is returned. If this component shares
    * its place with another JEgg, <code>COLLISION_WITH_JEGG</code> is returned. Otherwise
    * <code>NOT_CONCEALED</code> is returned.
    *
    * @return   The concealed value
    */
   private int isConcealed()
   {
      Container parent = getParent();

      if (parent != null)
      {
         Component component = parent.getComponentAt (getX() + getWidth() / 2, getY() + getHeight() / 2);

         if (component instanceof JEgg && component != this &&  ((JEgg) component).isAlive())
         {
            return COLLISION_WITH_JEGG;
         }

         if (component != parent && component != this && component != null && ! (component instanceof JLine))
         {
            if (component.getX() <= getX() && component.getY() <= getY()
               && component.getX() + component.getWidth() > getX() + getWidth()
               && component.getY() + component.getHeight() > getY() + getHeight())
            {
               return CONCEALED;
            }
            else
            {
               return NOT_CONCEALED;
            }

         }
         else
         {
            return NOT_CONCEALED;
         }
      }
      else
      {
         return NOT_CONCEALED;
      }
   }


   /**
    * Return value for <code>isConcealed</code>. The JEgg is visible.
    */
   private final static int NOT_CONCEALED = 0;

   /**
    * Return value for <code>isConcealed</code>. The JEgg is not visible.
    */
   private final static int CONCEALED = 1;

   /**
    * Return value for <code>isConcealed</code>. The JEgg shares the same position with another
    * JEgg.
    */
   private final static int COLLISION_WITH_JEGG = 2;


   /**
    * Calculates a new direction for this JEgg. This is done by adding the value of <code>directionDelta</code>
    * to <code>direction</code>. In 30% of all cases <code>directionDelta</code> is automatically
    * inverted.
    */
   private void changeDirection()
   {
      if (Math.random() < 0.3)
      {
         directionDelta = -directionDelta;
      }

      int newDirection =  (direction + directionDelta) % 8;

      if (newDirection < 0)
      {
         newDirection += 8;
      }

      direction = newDirection;

      repaint();
   }


   /**
    * Remove this component from its container
    */
   private void removeYou()
   {
      Container c = getParent();

      if (c != null)
      {
         c.remove (this);
         c.repaint();
      }
   }


   /**
    * Puts this component behind all other components of the container
    */
   private void toBack()
   {
      Container c = getParent();

      if (c != null)
      {
         c.remove (this);

         c.add (this);
      }
   }


   /**
    * Starts JEgg. Note that JEgg requires FSA to run.
    */
   public static void start()
   {
      FSALayeredPane fsaPane = FrameMain.get().getCurrentInternalFrame().getDiagramRoot();

      if (fsaPane != null)
      {
         JLayeredPane pane = (JLayeredPane) fsaPane.getJComponent();

         for (int i = 0; i < 10; i++)
         {
            JEgg jegg = new JEgg();

            pane.add (jegg);

            jegg.setLocation ((int)  (Math.random() *  (pane.getWidth() - jegg.getWidth())), (int)  (Math.random() *  (pane.getHeight() - jegg.getHeight())));

            new Thread (jegg).start();
         }
      }
   }


   /**
    * This key listener listens for a triple click and adds the <code>EggKeyListener</code>
    * to a given component if a triple click has been found
    *
    * @author    $Author: schneider $
    * @version   $Revision: 1.17 $
    */
   public static class EggMouseListener extends MouseAdapter
   {
      /**
       * Adds the <code>EggKeyListener</code> to the Component specified in the constructor
       * if a triple click was found
       *
       * @param e  No description provided
       */
      public void mouseClicked (MouseEvent e)
      {
         if (e.getClickCount() == 3)
         {
            JEgg.start();
         }
      }
   }


   /**
    * Listens for death-bringing mouse clicks
    *
    * @author    $Author: schneider $
    * @version   $Revision: 1.17 $
    */
   private class EggDeathMouseListener extends MouseAdapter
   {
      /**
       * Kills the JEgg
       *
       * @param e  No description provided
       */
      public void mouseClicked (MouseEvent e)
      {
         alive = false;
      }
   }
}

/*
 * $Log: JEgg.java,v $
 * Revision 1.17  2004/10/20 17:49:46  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
