/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.Dimension;
import java.awt.event.KeyListener;

import javax.swing.JComponent;
import javax.swing.JEditorPane;
import javax.swing.border.LineBorder;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.Document;

import de.uni_paderborn.fujaba.fsa.listener.AscendDescendMouseHandler;
import de.uni_paderborn.fujaba.gui.CompletionKeyListener;
import de.uni_paderborn.fujaba.preferences.ColorsPreferences;


/**
 * show value with JLabel , edit with JTextField
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.15.2.1 $
 */
public class JTextEditor extends EditableComponent implements DocumentListener
{
   /**
    * Default constructor, creates a new JTextFieldLabel with normal string editing.
    */
   public JTextEditor()
   {
      super();
      JEditorPane pane = new JEditorPane();
      add (pane);

      setComponent (pane);

      getEditorPane().setEditorKit (new EvenPlainerEditorKit());

      getEditorPane().setContentType ("text/plain");
      getEditorPane().setEditable (false);
      getEditorPane().getDocument().addDocumentListener (this);

      AscendDescendMouseHandler.registerADMouseInputHandler (getEditorPane());
   }


   /**
    * internal used attribute to store the border of this editor.
    */
   private LineBorder lineBorder = null;


   /**
    * set the text of the editor.
    *
    * @param text  The new text value
    */
   public void setText (String text)
   {
      getEditorPane().setText (text);
   }


   /**
    * get the text of the editor as String object.
    *
    * @return   The text value
    */
   public String getText()
   {
      return getEditorPane().getText();
   }


   /**
    * set the document of the editor.
    *
    * @param doc  The new document value
    */
   public void setDocument (Document doc)
   {
      getEditorPane().setDocument (doc);
   }


   /**
    * get the Document of the editor.
    *
    * @return   The document value
    */
   public Document getDocument()
   {
      return getEditorPane().getDocument();
   }


   /**
    * get the internal JEditorPane object of this editor.
    *
    * @return   The editorPane value
    */
   public JEditorPane getEditorPane()
   {
      return (JEditorPane) getComponent();
   }


   /**
    * get the preferred size of this editor.
    *
    * @return   The preferredSize value
    */
   public Dimension getPreferredSize()
   {
      if (isEditing())
      {
         return getComponent (0).getPreferredSize();
      } // end of if ()
      return super.getPreferredSize();
   }


   /**
    * overwrites the updateEditor method of EditableComponent. This method is empty.
    *
    * @param component  No description provided
    */
   protected void updateEditor (JComponent component)
   {
   }


   /**
    * overwrites the enableEditor method of EditableComponent. This method enables the internal
    * stored editor for editing.
    *
    * @param component  No description provided
    */
   protected void enableEditor (JComponent component)
   {
      if (component instanceof JEditorPane)
      {
         if (lineBorder == null)
         {
            lineBorder = new LineBorder (ColorsPreferences.get().EDITOR_BORDER, 1);
         } // end of if ()

         JEditorPane editor = getEditorPane();

         boolean done = false;
         KeyListener[] keyListeners = this.getKeyListeners();
         for (int i = 0; i < keyListeners.length && !done; i++)
         {
            if (keyListeners[i] instanceof CompletionKeyListener)
            {
               KeyListener completionListener = keyListeners[i];
               // add to component if not already known
               KeyListener[] oldKeyListeners = editor.getKeyListeners();
               for (int j = 0; j < oldKeyListeners.length && !done; j++)
               {
                  if (oldKeyListeners[j].equals (completionListener))
                  {
                     // is already there nothing to be done
                     done = true;
                  }
               }

               if (!done)
               {
                  // add it
                  editor.addKeyListener (completionListener);
               }
            }
         }
         editor.setEditable (true);

         editor.setBorder (lineBorder);

         AscendDescendMouseHandler.unregisterADMouseInputHandler (editor);

         if (!editor.hasFocus())
         {
            editor.requestFocus();
         }
      } // end of if ()
   }


   /**
    * overwrites the disableEditor method of EditableComponent. This method disables the internal
    * stored editor for editing.
    *
    * @param component  No description provided
    */
   protected void disableEditor (JComponent component)
   {
      if (component instanceof JEditorPane)
      {
         JEditorPane editor = getEditorPane();
         editor.setEditable (false);

         editor.setBorder (null);

         AscendDescendMouseHandler.registerADMouseInputHandler (editor);

         if (editor.hasFocus())
         {
            requestFocus();
         }
      }
   }


   /**
    * overwrites the getEditorClassName method of EditableComponent. This method returns the
    * class name of the preferred used editor of this component.
    *
    * @return   JEditorPane.class.getName()
    */
   protected String getEditorClassName()
   {
      return JEditorPane.class.getName();
   }


   /**
    * overwrites the updateJLabel method of EditableComponent. This method is empty.
    */
   protected void updateJLabel()
   {
   }


   /**
    * overwrites the setVisible method of Component.
    *
    * @param vis  The new visible value
    */
   public void setVisible (boolean vis)
   {
      super.setVisible (vis);
      getEditorPane().setVisible (vis);
   }


   /**
    * internal used method to update the display.
    */
   private void refresh()
   {
      JComponent parent = (JComponent) getParent();
      if (parent != null)
      {
         while (parent != null && parent.getParent() instanceof JComponent)
         {
            parent.setPreferredSize (null);
            parent = (JComponent) parent.getParent();
         }

         if (parent != null)
         {
            parent.setPreferredSize (null);
            parent.validate();
         }
      }
   }


   /**
    * implements the changedUpdate method of the DocumentListener interface.
    *
    * @param event  No description provided
    */
   public void changedUpdate (DocumentEvent event)
   {
      refresh();
   }


   /**
    * implements the insertUpdate method of the DocumentListener interface.
    *
    * @param event  No description provided
    */
   public void insertUpdate (DocumentEvent event)
   {
      refresh();
   }


   /**
    * implements the removeUpdate method of the DocumentListener interface.
    *
    * @param event  No description provided
    */
   public void removeUpdate (DocumentEvent event)
   {
      refresh();
   }

}

/*
 * $Log: JTextEditor.java,v $
 * Revision 1.15.2.1  2005/08/23 08:24:52  lowende
 * Removed compile warnings.
 *
 */
