/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;

import javax.swing.*;
import javax.swing.plaf.ComponentUI;

import org.apache.log4j.Logger;


/**
 * Base UI for JLines. New UIs for JLines must extend this class or subclasses of this class.
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.10.2.1 $
 */
public abstract class LineUI extends ComponentUI
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (LineUI.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static Stroke PLAIN_STROKE = new BasicStroke();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param c  No description provided
    * @param x  No description provided
    * @param y  No description provided
    * @return   No description provided
    */
   public boolean contains (JComponent c, int x, int y)
   {
      JLine line = (JLine) c;
      return line.getDistance (x + line.getX(), y + line.getY()) < line.getDelta();
   }


   /**
    * Returns the optimal size of the given JLine. In this case, simply the size of getPreferredBounds()
    * is returned.
    *
    * @param c  A JComponent for which the preferred size should be determined. Must be a JLine
    *      for this class.
    * @return   The preferredSize value
    */
   public Dimension getPreferredSize (JComponent c)
   {
      return getPreferredBounds (c).getSize();
   }


   /**
    * Returns the preferred bounds for the given JLine, considering the preferred line bounds
    * as given by getPreferredLineBounds and the child components of the JLine.
    *
    * @param c  No description provided
    * @return   The preferredBounds value
    * @see      #getPreferredLineBounds
    */
   public Rectangle getPreferredBounds (JComponent c)
   {
      boolean debug =  (c instanceof JPolyLine.LineSegment);
      debug = false;
      if (debug)
      {
         if (log.isInfoEnabled())
         {
            log.info ("getPreferredBounds():");
         }
      }

      JLine line = (JLine) c;
      Rectangle lineBounds = getPreferredLineBounds (line);
      if (debug)
      {
         if (log.isInfoEnabled())
         {
            log.info ("\tline: " + lineBounds);
         }
      }

      if (c.getComponentCount() > 0)
      {
         LayoutManager layout = c.getLayout();
         Rectangle layoutBounds =  (layout != null && layout instanceof DecoratorLayout) ?
             ((DecoratorLayout) layout).preferredLayoutBounds (c) :
            new Rectangle (layout.preferredLayoutSize (c));

         if (debug)
         {
            if (log.isInfoEnabled())
            {
               log.info ("\tlayout: " + layoutBounds);
            }
         }
         Point oldLoc = line.getLocation();
         layoutBounds.translate (oldLoc.x, oldLoc.y);
         if (debug)
         {
            if (log.isInfoEnabled())
            {
               log.info ("\tlayout: " + layoutBounds);
            }
         }

         lineBounds.add (layoutBounds);
      }
      if (debug)
      {
         if (log.isInfoEnabled())
         {
            log.info ("total: " + lineBounds);
         }
      }

      /*
       *  Point loc=bounds.getLocation();
       */
      return lineBounds;
   }


   /**
    * Returns the preferred bounds of the given JLine, based on its start and end points.
    *
    * @param c  A JComponent for which the preferred bounds should be determined. Must be a
    *      JLine for this class.
    * @return   The preferredLineBounds value
    */
   public abstract Rectangle getPreferredLineBounds (JLine c);


   /**
    * Paints the given compontent, and especially the line using paintLine()
    *
    * @param c  A JComponent that should be drawn. Must be a JLine for this class.
    * @param g  No description provided
    */
   public void paint (Graphics g, JComponent c)
   {
      super.paint (g, c);
      JLine jLine = (JLine) c;
      Stroke stroke = jLine.getStroke();
      Stroke oldStroke = null;

      if (g instanceof Graphics2D)
      {
         Graphics2D g2d = (Graphics2D) g;

         oldStroke = g2d.getStroke();

         if (stroke == null)
         {
            stroke = PLAIN_STROKE;
         }
         g2d.setStroke (stroke);

         paintLine (g, jLine);
         g2d.setStroke (oldStroke);
      }
      else
      {
         paintLine (g, jLine);
      }
   }


   /**
    * Interface for drawing the line of the JLine. Called by paint()
    *
    * @param g  No description provided
    * @param c  No description provided
    */
   public abstract void paintLine (Graphics g, JLine c);

}

/*
 * $Log: LineUI.java,v $
 * Revision 1.10.2.1  2005/09/30 18:57:07  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
