/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.unparse;

import java.beans.*;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import java.util.*;

import javax.swing.JComponent;

import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.fsa.FSAObject;
import de.uni_paderborn.fujaba.fsa.listener.CascadedPropertyChangeSupport;
import de.upb.tools.fca.FEmptyIterator;
import de.upb.tools.pcs.CollectionChangeEvent;
import de.upb.tools.sdm.Path;


/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: burmi $
 * @version   $Revision: 1.40.2.1 $
 */
public class UnparseManager
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (UnparseManager.class);


   /**
    * Constructor for class UnparseManager
    */
   private UnparseManager() { }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static transient UnparseManager singleton = null;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public static synchronized UnparseManager get()
   {
      if (singleton == null)
      {
         singleton = new UnparseManager();
      }
      return singleton;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient TreeMap unparseModules = new TreeMap();


   /**
    * Get the unparseModule attribute of the UnparseManager object
    *
    * @param iface  No description provided
    * @return       The unparseModule value
    */
   public final synchronized UnparseInterface getUnparseModule (LogicUnparseInterface iface)
   {
      return getUnparseModuleImpl (iface);
   }


   /**
    * Get the unparseModule attribute of the UnparseManager object
    *
    * @param className  No description provided
    * @param loader     No description provided
    * @return           The unparseModule value
    */
   public final synchronized UnparseInterface getUnparseModule (String className, ClassLoader loader)
   {
      return getUnparseModuleImpl (className, loader);
   }



   /**
    * Get the unparseModuleImpl attribute of the UnparseManager object
    *
    * @param iface  No description provided
    * @return       The unparseModuleImpl value
    */
   protected UnparseInterface getUnparseModuleImpl (LogicUnparseInterface iface)
   {
      return getUnparseModuleImpl (iface.getUnparseModuleName(), iface.getClass().getClassLoader());
   }


   /**
    * Get the unparseModuleImpl attribute of the UnparseManager object
    *
    * @param className  No description provided
    * @param loader     No description provided
    * @return           The unparseModuleImpl value
    */
   protected UnparseInterface getUnparseModuleImpl (String className, ClassLoader loader)
   {
      UnparseInterface unparseModule = (UnparseInterface) unparseModules.get (className);

      if (unparseModule == null)
      {
         try
         {
            if (log.isDebugEnabled())
            {
               log.debug ("Loading " + className);
            }
            unparseModule = (UnparseInterface) Class.forName (className, true, loader).newInstance();
            unparseModules.put (className, unparseModule);
         }
         catch (ClassNotFoundException e1)
         {
            log.error ("Could not find the UnparseModule " + className);
            if (log.isDebugEnabled())
            {
               log.debug ("searched for: " + className);
            }
            e1.printStackTrace();
         }
         catch (IllegalAccessException e2)
         {
            e2.printStackTrace();
         }
         catch (InstantiationException e3)
         {
            e3.printStackTrace();
         }
      }

      return unparseModule;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient StringBuffer moduleNameStringBuffer;


   /**
    * Get the unparseModuleName attribute of the UnparseManager object
    *
    * @param iface  No description provided
    * @return       The unparseModuleName value
    */
   public String getUnparseModuleName (LogicUnparseInterface iface)
   {
      String className = iface.getClass().getName();

      int packageClassSeparator = className.lastIndexOf ('.');

      if (moduleNameStringBuffer == null)
      {
         moduleNameStringBuffer = new StringBuffer();
      }
      else
      {
         moduleNameStringBuffer.delete (0, moduleNameStringBuffer.length());
      }

      //the following operations convert a qualified classname of qual.ified.pack.age.name.ClassName
      //to qual.ified.pack.age.name.unparse.UMClassName
      moduleNameStringBuffer.append (className);
      moduleNameStringBuffer.insert (packageClassSeparator + 1, "unparse.UM"); //insert this between package- and classname
      return moduleNameStringBuffer.toString();
   }


   /**
    * Creates a <code>FSAObject</code> for the given logic object.
    *
    * @param incr  the logic object to unparse
    * @return      the FSAObject that represents the logic object (a new one if there was none
    *      before or the old one if one existed already)
    * @see         #unparse(de.uni_paderborn.fujaba.fsa.unparse.LogicUnparseInterface, de.uni_paderborn.fujaba.fsa.FSAObject)
    */
   public FSAObject unparse (LogicUnparseInterface incr)
   {
      FSAInterface iface = incr.getFSAInterface();
      if (iface != null)
      {
         FSAObject obj = unparse (incr, iface.getMainFsaFromFsaObjects (null));
         return obj;
      }
      return null;
   }


   /**
    * Creates a <code>FSAObject</code> for the given logic object. <p>
    *
    * If one already exists no changes are made and the old FSAObject is returned. Otherwise
    * the appropriate UnparseModule is retrieved and its <code>create</code>-Method is called
    * to create a new FSAObject. After successful creation unparsing of child objects of the
    * logic object is triggered as defined by the result set of <code>getChildProperties</code>
    * . At last partner links are checked and perhaps triggered for unparsing as defined by
    * the result set of <code>getPartnerProperties</code>.
    *
    * @param incr       the logic object to unparse
    * @param fsaObject  the Parent into which the logic object should be unparsed
    * @return           the FSAObject that represents the logic object (a new one if there
    *      was none before or the old one if one existed already)
    * @see              #getUnparseModule(LogicUnparseInterface)
    * @see              #isConsistent
    * @see              de.uni_paderborn.fujaba.fsa.unparse.UnparseInterface#create
    * @see              de.uni_paderborn.fujaba.fsa.unparse.UnparseInterface#getChildProperties
    * @see              de.uni_paderborn.fujaba.fsa.unparse.UnparseInterface#getPartnerProperties
    */
   public final synchronized FSAObject unparse (LogicUnparseInterface incr, FSAObject fsaObject)
   {
      return unparseImpl (incr, fsaObject);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param incr       No description provided
    * @param fsaObject  No description provided
    * @return           No description provided
    */
   protected FSAObject unparseImpl (LogicUnparseInterface incr, FSAObject fsaObject)
   {
      FSAObject result = null;
      if (incr != null &&
         incr.getFSAInterface() != null &&
         incr.getFSAInterface().isUnparse())
      {
         FSAInterface iface = incr.getFSAInterface();

         UnparseInterface unparseModule = getUnparseModuleImpl (incr);

         if (unparseModule != null)
         {
            if (iface.hasInFsaObjects (fsaObject))
            {
               result = fsaObject;
            }
            else
            {
               result = iface.getFromFsaObjects (FSAObject.getQualifiedName (fsaObject, unparseModule.getMainFsaName()));

               if (result == null)
               {
                  boolean consistent = observeConsistency (incr, fsaObject);
                  if (consistent)
                  {
                     if (fsaObject != null && fsaObject.getJComponent() != null)
                     {
                        JComponent comp = fsaObject.getJComponent();
                        synchronized (comp.getTreeLock())
                        {
                           result = unparseModule.create (fsaObject, incr);
                           if (result != null)
                           {
                              JComponent jComponent = result.getJComponent();
                              boolean vis = jComponent.isVisible();
                              jComponent.setVisible (false);
                              initialize (unparseModule, incr, result);
                              jComponent.setVisible (vis);
                           } // end of if ()
                        }
                     }
                     else
                     {
                        result = unparseModule.create (fsaObject, incr);
                        if (result != null)
                        {
                           JComponent jComponent = result.getJComponent();
                           boolean vis = jComponent.isVisible();
                           jComponent.setVisible (false);
                           initialize (unparseModule, incr, result);
                           jComponent.setVisible (vis);
                        } // end of if ()
                     } // else
                  }
               }
            }
         }
      }
      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param unparseModule  No description provided
    * @param incr           No description provided
    * @param fsaObject      No description provided
    */
   void initialize (UnparseInterface unparseModule, LogicUnparseInterface incr, FSAObject fsaObject)
   {
      LinkedHashSet properties = new LinkedHashSet();
      unparseModule.getChildProperties (properties);
      FSAInterface iface = incr.getFSAInterface();

      Iterator iter = properties.iterator();
      while (iter.hasNext())
      {
         String prop = (String) iter.next();

         if (prop.indexOf (".") > -1)
         {
            CascadedPropertyChangeSupport cpcs = new CascadedPropertyChangeSupport (prop, incr);
            cpcs.addPropertyChangeListener (prop, iface);

            String base = prop.substring (0, prop.indexOf ("."));
            String name = prop.substring (prop.indexOf ("." + 1));
            Path basePath = new Path (incr, base);
            while (basePath.hasNext())
            {
               Object source = basePath.next();
               Path elements = new Path (source, name);
               while (elements.hasNext())
               {
                  PropertyChangeEvent evt = new PropertyChangeEvent (source, prop, null, elements.next());

                  String id = getFsaID (fsaObject, unparseModule, prop);
                  processAddEvent (incr, evt, id, unparseModule);
               }
            }
         }
         else
         {
            incr.getPropertyChangeSupport().addPropertyChangeListener (prop, iface);
            Path elements = new Path (incr, prop);
            while (elements.hasNext())
            {
               Object obj = elements.next();
               if (obj != null)
               {
                  PropertyChangeEvent evt = new PropertyChangeEvent (incr, prop, null, obj);

                  String id = getFsaID (fsaObject, unparseModule, prop);
                  processAddEvent (incr, evt, id, unparseModule);
               }
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param iface  No description provided
    * @param event  No description provided
    */
   public void propertyChange (FSAInterface iface, PropertyChangeEvent event)
   {
      synchronized (queuedPropertyEvents)
      {
         if (pausePropertyProcessing)
         {
            Object[] array = new Object[2];
            array[0] = iface;
            array[1] = event;

            queuedPropertyEvents.add (array);
            return;
         }
      }

      LogicUnparseInterface incr = iface.getLogic();
      if (iface.sizeOfFsaObjects() > 0)
      {
         Iterator iter = iface.entriesOfFsaObjects();
         UnparseInterface unparseModule = getUnparseModuleImpl (incr);
         String mainFsaName = unparseModule.getMainFsaName();
         String mainFsaPart = "." + mainFsaName;

         while (iter.hasNext())
         {
            Map.Entry entry = (Map.Entry) iter.next();
            String key = (String) entry.getKey();
            if (key.endsWith (mainFsaPart))
            {
               processEvent (incr, event,
                  getFsaID ((FSAObject) entry.getValue(), unparseModule, event.getPropertyName()),
                  unparseModule);
            }
         }
      }
   }


   /**
    * Get the fsaID attribute of the UnparseManager object
    *
    * @param fsaObject      No description provided
    * @param unparseModule  No description provided
    * @param property       No description provided
    * @return               The fsaID value
    */
   private String getFsaID (FSAObject fsaObject, UnparseInterface unparseModule, String property)
   {
      String mainFsaName = unparseModule.getMainFsaName();
      String fsaId;
      if (mainFsaName.equals (unparseModule.getContainerForProperty (property)))
      {
         String key = fsaObject.getQualifiedName();
         fsaId = key.substring (0, key.indexOf ('.'));
      }
      else
      {
         /*
          *  FIX ME: This only allows for unparse containers being direct children
          *  of main components. Normally a lookup through all descendants
          *  would be necessary
          */
         fsaId = fsaObject.getID();
      }
      return fsaId;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private LinkedList queuedPropertyEvents = new LinkedList();

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean pausePropertyProcessing = false;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void pausePropertyProcessing()
   {
      synchronized (queuedPropertyEvents)
      {
         if (pausePropertyProcessing == false)
         {
            pausePropertyProcessing = true;
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void resumePropertyProcessing()
   {
      synchronized (queuedPropertyEvents)
      {
         if (pausePropertyProcessing == true)
         {
            pausePropertyProcessing = false;
            Iterator iter = queuedPropertyEvents.iterator();

            while (iter.hasNext())
            {
               Object[] array = (Object[]) iter.next();
               FSAInterface iface = (FSAInterface) array[0];
               PropertyChangeEvent evt = (PropertyChangeEvent) array[1];
               propertyChange (iface, evt);
               iter.remove();
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param iface          No description provided
    * @param event          No description provided
    * @param fsaId          No description provided
    * @param unparseModule  No description provided
    */
   protected final void processEvent (LogicUnparseInterface iface, PropertyChangeEvent event,
                                      String fsaId, UnparseInterface unparseModule)
   {
      if (event instanceof CollectionChangeEvent)
      {
         CollectionChangeEvent collectionChangeEvent = (CollectionChangeEvent) event;

         if (CollectionChangeEvent.isAddEvent (collectionChangeEvent.getType()))
         {
            processAddEvent (iface, event, fsaId, unparseModule);
         }
         else if (CollectionChangeEvent.isRemoveEvent (collectionChangeEvent.getType()))
         {
            processRemoveEvent (iface, event, fsaId, unparseModule);
         }
         else
         {
            /*
             *  CHANGED
             */
            processChangeEvent (iface, event, fsaId, unparseModule);
         }
      }
      else
      {
         Object oldValue = event.getOldValue();
         Object newValue = event.getNewValue();
         if (oldValue != null && newValue == null)
         {
            processRemoveEvent (iface, event, fsaId, unparseModule);
         }
         else if (newValue != null && oldValue == null)
         {
            processAddEvent (iface, event, fsaId, unparseModule);
         }
         else if (oldValue != null && newValue != null)
         {
            processChangeEvent (iface, event, fsaId, unparseModule);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param iface          No description provided
    * @param event          No description provided
    * @param fsaId          No description provided
    * @param unparseModule  No description provided
    */
   protected final synchronized void processAddEvent (LogicUnparseInterface iface, PropertyChangeEvent event,
                                                      String fsaId, UnparseInterface unparseModule)
   {
      processAddEventImpl (iface, event, fsaId, unparseModule);
   }


   /**
    * method processAddEvent.
    *
    * @param iface          No description provided
    * @param event          No description provided
    * @param fsaId          No description provided
    * @param unparseModule  No description provided
    */
   protected void processAddEventImpl (LogicUnparseInterface iface,
                                       PropertyChangeEvent event,
                                       String fsaId,
                                       UnparseInterface unparseModule)
   {
      LogicUnparseInterface addedObject = (LogicUnparseInterface) event.getNewValue();

      FSAInterface fsaIface = iface.getFSAInterface();

      Object tmpValue = unparseModule.getContainerForProperty (event.getPropertyName());

      FSAObject containerObject = fsaIface.getFromFsaObjects (fsaId + "." + tmpValue);

      if (addedObject.getFSAInterface() != null && addedObject.getFSAInterface().isUnparse())
      {
         unparseImpl (addedObject, containerObject);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param iface          No description provided
    * @param event          No description provided
    * @param fsaId          No description provided
    * @param unparseModule  No description provided
    */
   protected final synchronized void processRemoveEvent (LogicUnparseInterface iface, PropertyChangeEvent event,
                                                         String fsaId, UnparseInterface unparseModule)
   {
      processRemoveEventImpl (iface, event, fsaId, unparseModule);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param iface          No description provided
    * @param event          No description provided
    * @param fsaId          No description provided
    * @param unparseModule  No description provided
    */
   protected void processRemoveEventImpl (LogicUnparseInterface iface, PropertyChangeEvent event,
                                          String fsaId, UnparseInterface unparseModule)
   {
      LogicUnparseInterface removedObject = (LogicUnparseInterface) event.getOldValue();
      if (removedObject == null)
      {
         return;
      }

      FSAInterface fsaIface = iface.getFSAInterface();
      UnparseInterface removedUnparseModule = getUnparseModuleImpl (removedObject);

      FSAObject containerObject = fsaIface.getFromFsaObjects (fsaId + "." +
         unparseModule.getContainerForProperty (event.getPropertyName()));

      if (containerObject != null)
      {
         String id = containerObject.getID();
         containerObject = removedObject.getFSAInterface().getFromFsaObjects (id + "." + removedUnparseModule.getMainFsaName());

         if (containerObject != null)
         {
            removedUnparseModule.remove (containerObject, iface);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param iface          No description provided
    * @param event          No description provided
    * @param fsaId          No description provided
    * @param unparseModule  No description provided
    */
   protected final synchronized void processChangeEvent (LogicUnparseInterface iface, PropertyChangeEvent event,
                                                         String fsaId, UnparseInterface unparseModule)
   {
      processChangeEventImpl (iface, event, fsaId, unparseModule);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param iface          No description provided
    * @param event          No description provided
    * @param fsaId          No description provided
    * @param unparseModule  No description provided
    */
   protected void processChangeEventImpl (LogicUnparseInterface iface, PropertyChangeEvent event,
                                          String fsaId, UnparseInterface unparseModule)
   {
      processRemoveEventImpl (iface, event, fsaId, unparseModule);
      processAddEventImpl (iface, event, fsaId, unparseModule);
   }


   /**
    * Get the consistent attribute of the UnparseManager object
    *
    * @param incr  No description provided
    * @return      The consistent value
    */
   public final synchronized boolean isConsistent (LogicUnparseInterface incr)
   {
      return isConsistentImpl (incr);
   }


   /**
    * Get the consistentImpl attribute of the UnparseManager object
    *
    * @param incr  No description provided
    * @return      The consistentImpl value
    */
   protected boolean isConsistentImpl (LogicUnparseInterface incr)
   {
      boolean result = false;

      if (incr != null)
      {
         FSAInterface fsaIface = incr.getFSAInterface();

         if (fsaIface != null && fsaIface.isUnparse())
         {
            UnparseInterface unparseModule = getUnparseModuleImpl (incr);

            Set partners = new HashSet();
            unparseModule.getPartnerProperties (partners);

            result = true;

            Path elements;
            boolean tmpResult;

            Iterator iter = partners.iterator();
            while (iter.hasNext())
            {
               String prop = (String) iter.next();

               tmpResult = false;
               elements = new Path (incr, prop);

               while (elements.hasNext() && tmpResult == false)
               {
                  Object next = elements.next();
                  if (next instanceof LogicUnparseInterface)
                  {
                     // FIX ME: consistency checking of "next" has to be done recursively here
                     tmpResult = true;
                  }
               }

               if (!tmpResult)
               {
                  result = false;
               }
            } // while
         }
      }
      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param incr           No description provided
    * @param unparseTarget  No description provided
    * @return               No description provided
    */
   protected boolean observeConsistency (LogicUnparseInterface incr, FSAObject unparseTarget)
   {
      boolean result = isConsistentImpl (incr);
      if (incr != null)
      {
         FSAInterface iface = incr.getFSAInterface();

         if (iface != null)
         {
            if (!result && iface.isUnparse())
            {
               UnparseInterface unparseModule = getUnparseModuleImpl (incr);

               Set partners = new HashSet();
               unparseModule.getPartnerProperties (partners);

               Iterator iter = partners.iterator();
               while (iter.hasNext())
               {
                  String prop = (String) iter.next();

                  // FIX ME: dependance graphs should be built and observed here

                  if (prop.indexOf (".") > -1 && !hasKeyInCascadedProperties (incr, prop))
                  {
                     CascadedPropertyChangeSupport cpcs = new CascadedPropertyChangeSupport (prop, incr);
                     addToCascadedProperties (incr, prop, cpcs);
                     cpcs.addPropertyChangeListener (prop, getConsistencyPCL());
                  }
                  else
                  {
                     PropertyChangeSupport support = incr.getPropertyChangeSupport();
                     support.removePropertyChangeListener (prop, getConsistencyPCL());
                     support.addPropertyChangeListener (prop, getConsistencyPCL());
                  } // else
               } // while

               if (unparseTarget != null)
               {
                  addToDeferredUnparseTargets (incr, unparseTarget);
               }
            }
            iface.setConsistent (result);
         }
      }
      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param incr  No description provided
    */
   protected void unobserveConsistency (LogicUnparseInterface incr)
   {
      UnparseInterface unparseModule = getUnparseModuleImpl (incr);
      PropertyChangeSupport support = incr.getPropertyChangeSupport();

      Set partners = new HashSet();
      unparseModule.getPartnerProperties (partners);
      Iterator iter = partners.iterator();
      while (iter.hasNext())
      {
         String prop = (String) iter.next();
         support.removePropertyChangeListener (prop, getConsistencyPCL());
      } // while

      removeKeyFromCascadedProperties (incr);
      removeKeyFromDeferredUnparseTargets (incr);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private PropertyChangeListener consistencyPCL = null;


   /**
    * Get the consistencyPCL attribute of the UnparseManager object
    *
    * @return   The consistencyPCL value
    */
   protected PropertyChangeListener getConsistencyPCL()
   {
      if (this.consistencyPCL == null)
      {
         this.consistencyPCL = new ConsistencyPCL();
      }
      return this.consistencyPCL;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private WeakHashMap cascadedProperties;


   /**
    * Access method for an one to n association.
    *
    * @param value     The object added.
    * @param property  The object added.
    * @param cpcs      The object added.
    * @return          No description provided
    */
   protected boolean addToCascadedProperties (LogicUnparseInterface value, String property, CascadedPropertyChangeSupport cpcs)
   {
      boolean changed = false;
      if (value != null && property != null && cpcs != null)
      {
         if (cascadedProperties == null)
         {
            cascadedProperties = new WeakHashMap();
         }
         Map propMap = (Map) cascadedProperties.get (value);
         if (propMap == null)
         {
            propMap = new HashMap();
            cascadedProperties.put (value, propMap);
         }
         WeakReference ref = new WeakReference (cpcs);
         WeakReference old = (WeakReference) propMap.put (property, ref);
         changed =  (cpcs != old.get());
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasKeyInCascadedProperties (LogicUnparseInterface value)
   {
      return  (value != null && cascadedProperties != null && cascadedProperties.get (value) != null);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value     No description provided
    * @param property  No description provided
    * @return          No description provided
    */
   public boolean hasKeyInCascadedProperties (LogicUnparseInterface value, String property)
   {
      boolean result = false;
      if (value != null && property != null && cascadedProperties != null)
      {
         Map propMap = (Map) cascadedProperties.get (value);
         if (propMap != null)
         {
            WeakReference ref = (WeakReference) propMap.get (property);
            result =  (ref != null && ref.get() != null);
         }
      }
      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value     No description provided
    * @param property  No description provided
    * @param support   No description provided
    * @return          No description provided
    */
   public boolean hasInCascadedProperties (LogicUnparseInterface value, String property, CascadedPropertyChangeSupport support)
   {
      return  (support != null && getFromCascadedProperties (value, property) == support);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public Iterator iteratorOfCascadedProperties (LogicUnparseInterface value)
   {
      if (value != null && cascadedProperties != null)
      {
         Map propMap = (Map) cascadedProperties.get (value);
         if (propMap != null)
         {
            return new WeakKey.ReferenceIterator (propMap.values().iterator());
         }
      }
      return FEmptyIterator.get();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public Iterator keysOfCascadedProperties (LogicUnparseInterface value)
   {
      if (value != null && cascadedProperties != null)
      {
         Map propMap = (Map) cascadedProperties.get (value);
         if (propMap != null)
         {
            return new WeakKey.ReferenceIterator (propMap.keySet().iterator());
         }
      }
      return FEmptyIterator.get();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public Iterator entriesOfCascadedProperties (LogicUnparseInterface value)
   {
      if (value != null && cascadedProperties != null)
      {
         Map propMap = (Map) cascadedProperties.get (value);
         if (propMap != null)
         {
            return new WeakKey.ReferenceIterator (propMap.entrySet().iterator());
         }
      }
      return FEmptyIterator.get();
   }


   /**
    * Get the fromCascadedProperties attribute of the UnparseManager object
    *
    * @param value  No description provided
    * @param prop   No description provided
    * @return       The fromCascadedProperties value
    */
   public CascadedPropertyChangeSupport getFromCascadedProperties (LogicUnparseInterface value, String prop)
   {
      CascadedPropertyChangeSupport result = null;
      if (value != null && prop != null && cascadedProperties != null)
      {
         Map propMap = (Map) cascadedProperties.get (value);
         if (propMap != null)
         {
            WeakReference ref = (WeakReference) propMap.get (prop);
            if (ref != null)
            {
               result = (CascadedPropertyChangeSupport) ref.get();
            }
         }
      }
      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @param prop   No description provided
    * @return       No description provided
    */
   protected boolean removeKeyFromCascadedProperties (LogicUnparseInterface value, String prop)
   {
      boolean changed = false;
      if (value != null && prop != null && cascadedProperties != null)
      {
         Map propMap = (Map) cascadedProperties.get (value);
         if (propMap != null)
         {
            WeakReference ref = (WeakReference) propMap.remove (prop);
            if (ref != null)
            {
               CascadedPropertyChangeSupport support = (CascadedPropertyChangeSupport) ref.get();
               changed =  (support != null);

               if (changed)
               {
                  support.removeYou();
                  ref.clear();
               }

               if (propMap.size() == 0)
               {
                  cascadedProperties.remove (value);
               }
            }
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   protected boolean removeKeyFromCascadedProperties (LogicUnparseInterface value)
   {
      boolean changed = false;
      if (value != null && cascadedProperties != null)
      {
         Map propMap = (Map) cascadedProperties.remove (value);
         if (propMap != null && propMap.size() > 0)
         {
            Iterator iter = propMap.values().iterator();
            while (iter.hasNext())
            {
               WeakReference ref = (WeakReference) iter.next();
               CascadedPropertyChangeSupport support = (CascadedPropertyChangeSupport) ref.get();
               boolean tmpChanged =  (support != null);

               if (tmpChanged)
               {
                  changed = true;
                  support.removeYou();
                  ref.clear();
               }
            } // while

            propMap.clear();
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private WeakHashMap deferredUnparseTargets;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private ReferenceQueue deferredUnparseTargetsGCQueue;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void clearDeferredUnparseTargetsGCQueue()
   {
      if (deferredUnparseTargetsGCQueue != null)
      {
         for (WeakKey value = (WeakKey) deferredUnparseTargetsGCQueue.poll(); value != null;
            value = (WeakKey) deferredUnparseTargetsGCQueue.poll())
         {
            Set set = value.getSet();
            if (set != null)
            {
               set.remove (value);
               if (set.size() == 0)
               {
                  deferredUnparseTargets.values().remove (set);
               }
            }
         } // for
      }
   }


   /**
    * Access method for an one to n association.
    *
    * @param incr    The object added.
    * @param target  The object added.
    * @return        No description provided
    */
   protected boolean addToDeferredUnparseTargets (LogicUnparseInterface incr, FSAObject target)
   {
      boolean changed = false;
      clearDeferredUnparseTargetsGCQueue();
      if (incr != null && target != null)
      {
         if (this.deferredUnparseTargets == null)
         {
            this.deferredUnparseTargets = new WeakHashMap();
            this.deferredUnparseTargetsGCQueue = new ReferenceQueue();
         }
         Set set = (Set) this.deferredUnparseTargets.get (incr);
         if (set == null)
         {
            set = new HashSet();
            this.deferredUnparseTargets.put (incr, set);
         }
         changed = set.add (WeakKey.create (target, set, deferredUnparseTargetsGCQueue));
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasKeyInDeferredUnparseTargets (LogicUnparseInterface value)
   {
      return  ( (this.deferredUnparseTargets != null) &&
          (value != null) &&
         this.deferredUnparseTargets.get (value) != null);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param incr    No description provided
    * @param target  No description provided
    * @return        No description provided
    */
   public boolean hasInDeferredUnparseTargets (LogicUnparseInterface incr, FSAObject target)
   {
      boolean result = false;
      if ( (this.deferredUnparseTargets != null) &&
          (incr != null) &&  (target != null))
      {
         Set set = (Set) this.deferredUnparseTargets.get (incr);
         if (set != null)
         {
            result = set.contains (target);
         }
      }
      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public Iterator iteratorOfDeferredUnparseTargets (LogicUnparseInterface value)
   {
      if ( (this.deferredUnparseTargets != null) &&
          (value != null))
      {
         Set set = (Set) this.deferredUnparseTargets.get (value);
         if (set != null)
         {
            return new WeakKey.ReferenceIterator (set.iterator());
         }
      }
      return FEmptyIterator.get();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator keysOfDeferredUnparseTargets()
   {
      if (this.deferredUnparseTargets != null)
      {
         return new WeakKey.ReferenceIterator (this.deferredUnparseTargets.keySet().iterator());
      }
      return FEmptyIterator.get();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   protected boolean removeKeyFromDeferredUnparseTargets (LogicUnparseInterface value)
   {
      boolean changed = false;
      clearDeferredUnparseTargetsGCQueue();

      if ( (this.deferredUnparseTargets != null) &&  (value != null))
      {
         Set old = (Set) this.deferredUnparseTargets.remove (value);
         if (old != null && old.size() > 0)
         {
            old.clear();
            changed = true;
         }
      }
      clearDeferredUnparseTargetsGCQueue();

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param incr    No description provided
    * @param target  No description provided
    * @return        No description provided
    */
   protected boolean removeFromDeferredUnparseTargets (LogicUnparseInterface incr, FSAObject target)
   {
      boolean changed = false;
      clearDeferredUnparseTargetsGCQueue();

      if ( (this.deferredUnparseTargets != null) &&  (incr != null) &&  (target != null))
      {
         Set set = (Set) this.deferredUnparseTargets.get (incr);
         if (set != null)
         {
            changed = set.remove (target);
            if (changed && set.size() == 0)
            {
               this.deferredUnparseTargets.remove (incr);
            }
         }
      }
      clearDeferredUnparseTargetsGCQueue();
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param target  No description provided
    * @return        No description provided
    */
   protected boolean removeFromDeferredUnparseTargets (FSAObject target)
   {
      boolean changed = false;
      clearDeferredUnparseTargetsGCQueue();

      if ( (this.deferredUnparseTargets != null) &&  (target != null))
      {
         Iterator iter = this.deferredUnparseTargets.keySet().iterator();
         while (iter.hasNext())
         {
            Map.Entry entry = (Map.Entry) iter.next();
            Set set = (Set) entry.getValue();
            boolean tmpChanged = set.remove (target);
            if (tmpChanged)
            {
               changed = true;
               if (set.size() == 0)
               {
                  iter.remove();
               }
            }
         } // while
      }
      clearDeferredUnparseTargetsGCQueue();
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void removeAllFromDeferredUnparseTargets()
   {
      if (deferredUnparseTargets != null && deferredUnparseTargets.size() > 0)
      {
         deferredUnparseTargets.clear();
      }
      clearDeferredUnparseTargetsGCQueue();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: burmi $
    * @version   $Revision: 1.40.2.1 $
    */
   private class ConsistencyPCL implements PropertyChangeListener
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param iface  No description provided
       * @return       No description provided
       */
      private boolean updateConsistency (LogicUnparseInterface iface)
      {
         boolean result = isConsistent (iface);
         if (result)
         {
            try
            {
               Iterator iter = iteratorOfDeferredUnparseTargets (iface);
               while (iter.hasNext())
               {
                  FSAObject target = (FSAObject) iter.next();
                  unparseImpl (iface, target);
               } // while
            }
            finally
            {
               unobserveConsistency (iface);
            } // end of finally
         }
         return result;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param pe  No description provided
       */
      public void propertyChange (PropertyChangeEvent pe)
      {
         Object newValue = pe.getNewValue();

         if (
         /*
          *  (implied) newValue != null &&
          */
            newValue instanceof LogicUnparseInterface)
         {
            Object source = pe.getSource();

            LogicUnparseInterface iface = (LogicUnparseInterface) source;
            synchronized (UnparseManager.this)
            {
               updateConsistency (iface);
            }
         }
      }
   }
}

/*
 * $Log: UnparseManager.java,v $
 * Revision 1.40.2.1  2005/04/23 23:06:43  burmi
 * removed NullPointerException
 *
 */
