/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.update;

import java.beans.PropertyChangeListener;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.*;

import de.uni_paderborn.fujaba.basic.Utility;
import de.upb.tools.fca.FEmptyIterator;


/**
 * helper class for listener registration. allows registration of listeners to objects independent
 * of object type, as long as the object has fitting add-/remove<Whatever>Listener-Methods
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.13 $
 */
public class ListenerHelper
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static ListenerHelper singleton = null;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public static ListenerHelper get()
   {
      if (singleton == null)
      {
         singleton = new ListenerHelper();
      }

      return singleton;
   }


   /**
    * Constructor for class ListenerHelper
    */
   private ListenerHelper() { }


   /**
    * Get the listenerSupported attribute of the ListenerHelper object
    *
    * @param target    No description provided
    * @param listener  No description provided
    * @return          The listenerSupported value
    */
   public boolean isListenerSupported (Object target, EventListener listener)
   {
      String listenerName = getListenerName (listener.getClass());
      return isListenerSupported (target, listener, listenerName);
   }


   /**
    * Get the listenerSupported attribute of the ListenerHelper object
    *
    * @param target        No description provided
    * @param listener      No description provided
    * @param listenerName  No description provided
    * @return              The listenerSupported value
    */
   public boolean isListenerSupported (Object target, EventListener listener, String listenerName)
   {
      Class objClass = target.getClass();
      Method[] methods = getListenerMethods (listenerName, objClass);

      Method addMethod = methods[0];

      return  ( (addMethod != null) &&
          (addMethod.getParameterTypes()[0].isInstance (listener)));
   }


   /**
    * Get the propertyChangeListenerSupported attribute of the ListenerHelper object
    *
    * @param target  No description provided
    * @return        The propertyChangeListenerSupported value
    */
   public boolean isPropertyChangeListenerSupported (Object target)
   {
      Method[] methods = getListenerMethods ("propertyChange", PropertyChangeListener.class);
      Method addMethod = methods[0];

      return  (addMethod != null);
   }


   /**
    * Access method for an one to n association.
    *
    * @param target    The object added.
    * @param listener  The object added.
    */
   public void addListener (Object target, EventListener listener)
   {
      String listenerName = getListenerName (listener.getClass());

      addListener (target, listener, listenerName);
   }


   /**
    * Access method for an one to n association.
    *
    * @param target        The object added.
    * @param listener      The object added.
    * @param listenerName  The object added.
    */
   public void addListener (Object target, EventListener listener, String listenerName)
   {
      Class objClass = target.getClass();
      Method[] methods = getListenerMethods (listenerName, objClass);

      Method addMethod = methods[0];

      if (addMethod == null)
      {
         throw new IllegalArgumentException ("Unable to add a listener of type " + listenerName + " to instance of class " + objClass);
      }
      else if (!addMethod.getParameterTypes()[0].isInstance (listener))
      {
         throw new IllegalArgumentException ("Listener of Type " + listener.getClass().getName() +
            " does not fit type " + addMethod.getParameterTypes()[0].getName() +
            " for \"" + listenerName + "\"");
      }

      try
      {
         addMethod.invoke (target, new Object[]
            {
            listener
            }
            );
      }
      catch (InvocationTargetException ite)
      {
         throw new IllegalArgumentException ("Unable to add a listener of type " + listenerName + " to instance of class " + objClass +
            "\n Reason: " + ite.getTargetException().getClass().getName() + ": " + ite.getTargetException().getMessage());
      }
      catch (Exception e)
      {
         throw new IllegalArgumentException ("Unable to add a listener of type " + listenerName + " to instance of class " + objClass +
            "\n Reason: " + e.getClass().getName() + ": " + e.getMessage());
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param target    No description provided
    * @param listener  No description provided
    */
   public void removeListener (Object target, EventListener listener)
   {
      String listenerName = getListenerName (listener.getClass());

      removeListener (target, listener, listenerName);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param target        No description provided
    * @param listener      No description provided
    * @param listenerName  No description provided
    */
   public void removeListener (Object target, EventListener listener, String listenerName)
   {
      Class objClass = target.getClass();
      Method[] methods = getListenerMethods (listenerName, objClass);

      Method removeMethod = methods[1];

      if (removeMethod == null)
      {
         throw new IllegalArgumentException ("Unable to remove a listener of type " + listenerName + " from instance of class " + objClass);
      }
      else if (!removeMethod.getParameterTypes()[0].isInstance (listener))
      {
         throw new IllegalArgumentException ("Listener of Type " + listener.getClass().getName() +
            " does not fit type " + removeMethod.getParameterTypes()[0].getName() +
            " for \"" + listenerName + "\"");
      }

      try
      {
         removeMethod.invoke (target, new Object[]
            {
            listener
            }
            );
      }
      catch (InvocationTargetException ite)
      {
         throw new IllegalArgumentException ("Unable to remove a listener of type " + listenerName + " from instance of class " + objClass +
            "\n Reason: " + ite.getTargetException().getClass().getName() + ": " + ite.getTargetException().getMessage());
      }
      catch (Exception e)
      {
         throw new IllegalArgumentException ("Unable to add a listener of type " + listenerName + " to instance of class " + objClass +
            "\n Reason: " + e.getClass().getName() + ": " + e.getMessage());
      }
   }


   /**
    * Access method for an one to n association.
    *
    * @param target    The object added.
    * @param listener  The object added.
    */
   public void addPropertyChangeListener (Object target, PropertyChangeListener listener)
   {
      Class objClass = target.getClass();
      Method[] methods = getListenerMethods ("propertyChange", objClass);

      Method addMethod = methods[0];

      if (addMethod == null)
      {
         throw new IllegalArgumentException ("Unable to add PropertyChangeListeners to instances of class " + objClass);
      }

      try
      {
         addMethod.invoke (target, new Object[]
            {
            listener
            }
            );
      }
      catch (InvocationTargetException ite)
      {
         throw new IllegalArgumentException ("Unable to add listener to instance of class " + objClass +
            "\n Reason: " + ite.getTargetException().getClass().getName() + ": " + ite.getTargetException().getMessage());
      }
      catch (Exception e)
      {
         throw new IllegalArgumentException ("Unable to add listener to instance of class " + objClass +
            "\n Reason: " + e.getClass().getName() + ": " + e.getMessage());
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param target    No description provided
    * @param listener  No description provided
    */
   public void removePropertyChangeListener (Object target, PropertyChangeListener listener)
   {
      Class objClass = target.getClass();
      Method[] methods = getListenerMethods ("propertyChange", objClass);

      Method removeMethod = methods[1];

      if (removeMethod == null)
      {
         throw new IllegalArgumentException ("Unable to remove PropertyChangeListeners from instances of class " + objClass);
      }

      try
      {
         removeMethod.invoke (target, new Object[]
            {
            listener
            }
            );
      }
      catch (InvocationTargetException ite)
      {
         throw new IllegalArgumentException ("Unable to remove listener from instance of class " + objClass +
            "\n Reason: " + ite.getTargetException().getClass().getName() + ": " + ite.getTargetException().getMessage());
      }
      catch (Exception e)
      {
         throw new IllegalArgumentException ("Unable to add listener to instance of class " + objClass +
            "\n Reason: " + e.getClass().getName() + ": " + e.getMessage());
      }
   }


   /**
    * Access method for an one to n association.
    *
    * @param target    The object added.
    * @param property  The object added.
    * @param listener  The object added.
    */
   public void addPropertyChangeListener (Object target, String property, PropertyChangeListener listener)
   {
      Class objClass = target.getClass();
      Method[] methods = getListenerMethods ("propertyChange", objClass);

      if (methods.length < 4)
      {
         throw new IllegalArgumentException ("Unable to add PropertyChangeListeners for single properties to instances of class " + objClass);
      }

      Method addMethod = methods[2];

      if (addMethod == null)
      {
         throw new IllegalArgumentException ("Unable to add PropertyChangeListeners for single properties to instances of class " + objClass);
      }

      try
      {
         addMethod.invoke (target, new Object[]
            {
            property, listener
            }
            );
      }
      catch (InvocationTargetException ite)
      {
         throw new IllegalArgumentException ("Unable to add listener to instance of class " + objClass +
            "\n Reason: " + ite.getTargetException().getClass().getName() + ": " + ite.getTargetException().getMessage());
      }
      catch (Exception e)
      {
         throw new IllegalArgumentException ("Unable to add listener to instance of class " + objClass +
            "\n Reason: " + e.getClass().getName() + ": " + e.getMessage());
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param target    No description provided
    * @param property  No description provided
    * @param listener  No description provided
    */
   public void removePropertyChangeListener (Object target, String property, PropertyChangeListener listener)
   {
      Class objClass = target.getClass();
      Method[] methods = getListenerMethods ("propertyChange", objClass);

      if (methods.length < 4)
      {
         throw new IllegalArgumentException ("Unable to remove PropertyChangeListeners for single properties from instances of class " + objClass);
      }

      Method removeMethod = methods[3];

      if (removeMethod == null)
      {
         throw new IllegalArgumentException ("Unable to remove PropertyChangeListeners for single properties from instances of class " + objClass);
      }

      try
      {
         removeMethod.invoke (target, new Object[]
            {
            property, listener
            }
            );
      }
      catch (InvocationTargetException ite)
      {
         throw new IllegalArgumentException ("Unable to remove listener from instance of class " + objClass +
            "\n Reason: " + ite.getTargetException().getClass().getName() + ": " + ite.getTargetException().getMessage());
      }
      catch (Exception e)
      {
         throw new IllegalArgumentException ("Unable to add listener to instance of class " + objClass +
            "\n Reason: " + e.getClass().getName() + ": " + e.getMessage());
      }
   }


   /**
    * Get the listenerName attribute of the ListenerHelper object
    *
    * @param clazz  No description provided
    * @return       The listenerName value
    */
   public String getListenerName (Class clazz)
   {
      String result = null;

      if (clazz.getName().endsWith ("Listener") && EventListener.class.isAssignableFrom (clazz))
      {
         result = clazz.getName();

         int start = result.lastIndexOf (".");
         start = Math.max (start, result.lastIndexOf ("$"));
         start++;

         result = result.substring (start, result.length() - 8);
      }
      else
      {
         Class[] interfaces = clazz.getInterfaces();

         for (int i = 0; i < interfaces.length && result == null; i++)
         {
            result = getListenerName (interfaces[i]);
         }

         if (result == null && clazz.getDeclaringClass() != null)
         {
            result = getListenerName (clazz.getDeclaringClass());
         }
      }

      if (result != null)
      {
         result = Utility.downFirstChar (result);
      }

      return result;
   }



   /**
    * Get the listenerMethods attribute of the ListenerHelper object
    *
    * @param listenerName  No description provided
    * @param objClass      No description provided
    * @return              The listenerMethods value
    */
   public Method[] getListenerMethods (String listenerName, Class objClass)
   {
      String key = objClass.getName() + ":" + listenerName;
      Method result[];
      result = getFromMethodCache (key);

      if (result == null)
      {
         String propFirstUp = Utility.upFirstChar (listenerName) + "Listener";

         Method[] methods = objClass.getMethods();

         Method removeMethod = null;
         Method addMethod = null;
         Method namedPropertyAddMethod = null;
         Method namedPropertyRemoveMethod = null;

         // this workaround is needed because getMethod(String name, Class[] signature)
         // only works if you know the exact types of the parameters
         for (int i = 0; i < methods.length &&
             (addMethod == null || removeMethod == null ||
             ("propertyChange".equals (listenerName) &&
             (namedPropertyAddMethod == null || namedPropertyRemoveMethod == null)));
            i++)
         {
            if (methods[i].getName().equals ("remove" + propFirstUp))
            {
               if (methods[i].getParameterTypes().length == 1)
               {
                  removeMethod = methods[i];
               }
               else if (namedPropertyRemoveMethod == null
                  && "propertyChange".equals (listenerName)
                  && methods[i].getParameterTypes().length == 2
                  && methods[i].getParameterTypes()[0].isAssignableFrom (String.class)
                  && methods[i].getParameterTypes()[1].isAssignableFrom (PropertyChangeListener.class))
               {
                  namedPropertyRemoveMethod = methods[i];
               }
            }
            else if (methods[i].getName().equals ("add" + propFirstUp))
            {
               if (methods[i].getParameterTypes().length == 1)
               {
                  addMethod = methods[i];
               }
               else if (namedPropertyAddMethod == null
                  && "propertyChange".equals (listenerName)
                  && methods[i].getParameterTypes().length == 2
                  && methods[i].getParameterTypes()[0].isAssignableFrom (String.class)
                  && methods[i].getParameterTypes()[1].isAssignableFrom (PropertyChangeListener.class))
               {
                  namedPropertyAddMethod = methods[i];
               }
            }
         }

         if ("propertyChange".equals (listenerName))
         {
            result = new Method[4];
            result[2] = namedPropertyAddMethod;
            result[3] = namedPropertyRemoveMethod;
         }
         else
         {
            result = new Method[2];
         }

         result[0] = removeMethod;
         result[1] = addMethod;

         addToMethodCache (key, result);
      }

      return result;
   }


   /**
    * UMLAttribute : 'methodCache : TreeMap (umlIncr.getClass().getName():<name>,Method[])
    * '
    */
   private TreeMap methodCache = null;


   /**
    * @param key  No description provided
    * @return     No description provided
    * @see        #methodCache
    */
   private boolean hasKeyInMethodCache (String key)
   {
      if (methodCache == null)
      {
         return false;
      }
      else
      {
         return methodCache.containsKey (key);
      }
   }


   /**
    * @return   No description provided
    * @see      #methodCache
    */
   private Iterator iteratorOfMethodCache()
   {
      if (methodCache == null)
      {
         return FEmptyIterator.get();
      }
      else
      {
         return methodCache.values().iterator();
      }
   }


   /**
    * @return   No description provided
    * @see      #methodCache
    */
   private Iterator keysOfMethodCache()
   {
      if (methodCache == null)
      {
         return FEmptyIterator.get();
      }
      else
      {
         return methodCache.keySet().iterator();
      }
   }


   /**
    * @return   No description provided
    * @see      #methodCache
    */
   private Iterator entriesOfMethodCache()
   {
      if (methodCache == null)
      {
         return FEmptyIterator.get();
      }
      else
      {
         return methodCache.entrySet().iterator();
      }
   }


   /**
    * @param key  No description provided
    * @return     The fromMethodCache value
    * @see        #methodCache
    */
   private Method[] getFromMethodCache (String key)
   {
      Method[] elem;

      if (this.hasKeyInMethodCache (key))
      {
         elem = (Method[]) this.methodCache.get (key);
      }
      else
      {
         elem = null;
      }

      return elem;
   }


   /**
    * @param key   The object added.
    * @param elem  The object added.
    * @see         #methodCache
    */
   private void addToMethodCache (String key, Method elem[])
   {
      if (key != null && elem != null)
      {
         if (methodCache == null)
         {
            methodCache = new TreeMap();
         }

         this.methodCache.put (key, elem);
      }
   }


   /**
    * @param key  No description provided
    * @see        #methodCache
    */
   private void removeKeyFromMethodCache (String key)
   {
      if (this.hasKeyInMethodCache (key))
      {
         this.methodCache.remove (key);
      }
   }


   /**
    * @see   #methodCache
    */
   private void removeAllFromMethodCache()
   {
      Iterator iter = keysOfMethodCache();
      String tmpKey = null;
      while (iter.hasNext())
      {
         tmpKey = (String) iter.next();
         iter.remove();
         removeKeyFromMethodCache (tmpKey);
      }
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "ListenerHelper";
   }
}

/*
 * $Log: ListenerHelper.java,v $
 * Revision 1.13  2004/11/03 10:18:02  lowende
 * Javadoc warnings removed.
 *
 */
