/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.gui;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.util.*;

import javax.swing.*;
import javax.swing.border.EtchedBorder;
import javax.swing.border.TitledBorder;

import de.uni_paderborn.fujaba.app.FrameMain;
import de.uni_paderborn.fujaba.app.messages.WarningMessages;
import de.uni_paderborn.fujaba.asg.ASGDiagram;
import de.uni_paderborn.fujaba.gui.comp.FujabaDialog;
import de.uni_paderborn.fujaba.gui.comp.JglListModel;
import de.uni_paderborn.fujaba.uml.*;


/**
 * A dialog that allows modifying the content of a specified UMLClassDiagram.
 * Add / remove UMLClass objects to / from the UMLClassDiagram is supported,
 * as well as renaming the UMLClassDiagram.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.29.2.3 $
 */
public class EditClassDiagDialog extends FujabaDialog
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JTextField diagramNameTextField = new JTextField();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JList classesInDiagramList = new JList();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JList availableClassesList = new JList();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JButton buttonAdd = new JButton ("<--");
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JButton buttonDelete = new JButton ("-->");
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JButton buttonAddContext = new JButton ("<--");
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JButton buttonDeleteContext = new JButton ("-->");

   /*
    *  Data-Stuff
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLClassDiagram classDiag = null;

   /*
    *  help-things
    */
   /**
    * Contains the UMLClass objects that are displayed in the UMLClassDiagram edited by
    * this dialog. The key is the full qualified UMLClass name. The value is the
    * UMLClass object.
    */
   private TreeMap displayedClasses = new TreeMap();

   /**
    * Contains the UMLClass objects that are availableClasses and can be displayed
    * in the UMLClassDiagram that is edited by this dialog.
    * The key is the full qualified UMLClass name. The value is the
    * UMLClass object.
    */
   private TreeMap availableClasses = new TreeMap();

   /**
    * Contains the UMLClass objects that where contained in the
    * UMLClassDiagram before modifications have been made with this dialog.
    * In conjunction with attribute 'displayedClasses' the classes that
    * should be newly added to the classdiagram and the ones that should
    * be removed from the classdiagram can be computed.
    */
   private Set initiallyDisplayedClasses = new HashSet();

   // ######################################################################

   /**
    * use this constructor, if the user want to edit a selected class
    *
    * @param frame  No description provided
    * @param diag   No description provided
    */
   public EditClassDiagDialog (JFrame frame, UMLClassDiagram diag)
   {
      super (frame, "Classdiagram Editor", true);
      this.guiInit();
      this.classDiag = diag;
      this.unparse();
   }


   /**
    * use this constructor, if the user want to edit a new class
    *
    * @param frame  No description provided
    */
   public EditClassDiagDialog (JFrame frame)
   {
      super (frame, "Classdiagram Editor", true);
      this.guiInit();
      this.classDiag = null;
      this.unparse();
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final void guiInit()
   {
      JPanel panel = new JPanel();
      panel.setLayout (new BorderLayout());
      panel.add (this.guiNamePanel(), BorderLayout.NORTH);
      panel.add (this.guiWorkPanel(), BorderLayout.CENTER);
      panel.add (this.guiPanelOkCancelHelp(), BorderLayout.SOUTH);

      // adjust selection-mode of JList objects
      this.classesInDiagramList.setSelectionMode (ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
      this.availableClassesList.setSelectionMode (ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);

      this.getContentPane().add (panel);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private final JPanel guiNamePanel()
   {
      JPanel panel = new JPanel();

      //----- set gridbaglayout to the panel
      GridBagLayout gridbag = new GridBagLayout();
      GridBagConstraints constraints = new GridBagConstraints();

      constraints.gridheight = 1;
      constraints.insets = new Insets (5, 5, 5, 5);
      constraints.anchor = GridBagConstraints.WEST;
      constraints.fill = GridBagConstraints.HORIZONTAL;
      constraints.weightx = 1.0;
      constraints.weighty = 0.0;

      panel.setLayout (gridbag);
      panel.setBorder (new TitledBorder (new EtchedBorder(), "Diagram Name"));
      panel.add (diagramNameTextField, constraints);

      return panel;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private final JPanel guiWorkPanel()
   {
      JPanel panel = new JPanel();
      JPanel tmpPanel;

      //----- set gridbaglayout to the panel
      GridBagLayout gridbag = new GridBagLayout();
      GridBagConstraints constraints = new GridBagConstraints();
      panel.setLayout (gridbag);

      //----- some pre-settings to the constraints
      constraints.gridwidth = 1;
      constraints.gridheight = 1;
      constraints.fill = GridBagConstraints.BOTH;
      constraints.ipadx = 5;
      constraints.ipady = 5;
      constraints.insets = new Insets (5, 0, 5, 5);
      constraints.anchor = GridBagConstraints.NORTHWEST;
      constraints.weightx = 1.0;
      constraints.weighty = 1.0;

      panel.add (guiLeftSide(), constraints);

      constraints.fill = GridBagConstraints.HORIZONTAL;
      constraints.anchor = GridBagConstraints.CENTER;
      constraints.insets = new Insets (5, 5, 5, 5);
      constraints.weightx = 0.0;
      constraints.weighty = 0.0;
      tmpPanel = this.guiMiddle();
      gridbag.setConstraints (tmpPanel, constraints);
      panel.add (tmpPanel);

      constraints.anchor = GridBagConstraints.NORTHWEST;
      constraints.gridwidth = GridBagConstraints.REMAINDER;
      constraints.fill = GridBagConstraints.BOTH;
      constraints.insets = new Insets (5, 5, 5, 0);
      constraints.weightx = 1.0;
      constraints.weighty = 1.0;
      panel.add (guiRightSide(), constraints);

      return panel;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private final JPanel guiLeftSide()
   {
      //----- add all gui-stuff to the panel
      JPanel leftPanel = new JPanel (new BorderLayout());
      JScrollPane sPane = new JScrollPane (this.classesInDiagramList);

      // add and delete context buttons
      leftPanel.setBorder (new TitledBorder (new EtchedBorder(), "classes of diagram"));
      leftPanel.add (sPane, BorderLayout.CENTER);
      leftPanel.setPreferredSize (new Dimension (275, 425));

      return leftPanel;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private final JPanel guiRightSide()
   {
      //----- add all gui-stuff to the panel
      JPanel rightPanel = new JPanel (new BorderLayout());

      JScrollPane sPane = new JScrollPane (this.availableClassesList);

      // add and delete context buttons
      rightPanel.setBorder (new TitledBorder (new EtchedBorder(), "available classes"));
      rightPanel.add (sPane, BorderLayout.CENTER);
      rightPanel.setPreferredSize (new Dimension (275, 425));

      return rightPanel;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private final JPanel guiMiddle()
   {
      GridLayout grid = new GridLayout (2, 1);
      JPanel contextPanel = new JPanel();

      // add and delete context buttons
      contextPanel.setLayout (grid);
      contextPanel.setBorder (new TitledBorder (new EtchedBorder(), "Context"));
      contextPanel.add (buttonAddContext);
      contextPanel.add (buttonDeleteContext);

      //----- set gridbaglayout to the panel
      GridBagLayout gridbag = new GridBagLayout();
      GridBagConstraints constraints = new GridBagConstraints();
      //----- some pre-settings to the constraints
      constraints.gridwidth = GridBagConstraints.REMAINDER;
      constraints.gridheight = 1;
      constraints.fill = GridBagConstraints.HORIZONTAL;
      constraints.anchor = GridBagConstraints.CENTER;
      constraints.weightx = 0.0;
      constraints.weighty = 0.0;
      constraints.insets = new Insets (0, 5, 0, 5);

      // action listeners for context buttons
      this.buttonAddContext.addActionListener (new ButtonAddContextActionListener());
      this.buttonDeleteContext.addActionListener (new ButtonDeleteContextActionListener());

      JPanel panel = new JPanel();

      panel.setLayout (gridbag);
      this.buttonAdd.addActionListener (new ButtonAddActionListener());
      panel.add (this.buttonAdd, constraints);
      this.buttonDelete.addActionListener (new ButtonDeleteActionListener());
      panel.add (this.buttonDelete, constraints);

      constraints.insets = new Insets (10, 0, 0, 0);
      panel.add (contextPanel, constraints);
      return panel;
   }

   // ######################################################################

   /**
    * UML-AST to dialog
    */
   public void unparse()
   {
      //this.unparsePackageTree();
      if (classDiag != null)
      {
         diagramNameTextField.setText (classDiag.getName());
      } // end of if ()

      this.fillDisplayedClasses();
      this.fillAvailableClasses (UMLProject.get().getRootPackage());
      this.updateGuiList();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void updateGuiList()
   {
      JglListModel displayedClassesListModel = new JglListModel (this.displayedClasses);
      displayedClassesListModel.setUseKey (true);
      this.classesInDiagramList.setModel (displayedClassesListModel);

      JglListModel availableClassesListModel = new JglListModel (this.availableClasses);
      availableClassesListModel.setUseKey (true);
      this.availableClassesList.setModel (availableClassesListModel);
   }


   /**
    * Fills the displayedClasses and the initiallyDisplayedClasses
    * Collections with all classes that are displayed in the
    * class diagram that is edited.
    */
   private void fillDisplayedClasses()
   {
      this.displayedClasses.clear();
      this.initiallyDisplayedClasses.clear();

      Iterator iter = classDiag.iteratorOfElements();

      while (iter.hasNext())
      {
         Object item = iter.next();
         if (item instanceof UMLClass)
         {
            this.displayedClasses.put ( ((UMLClass) item).getFullClassName(), item);
            this.initiallyDisplayedClasses.add (item);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param pack  No description provided
    */
   private void fillAvailableClasses (UMLPackage pack)
   {
      Object obj;
      Enumeration enumeration = pack.elementsOfAllChildren();
      String key;
      while (enumeration.hasMoreElements())
      {
         obj = enumeration.nextElement();
         if (obj instanceof UMLPackage)
         {
            this.fillAvailableClasses ((UMLPackage) obj);
         }
         else if (obj instanceof UMLClass)
         {
            key =  ((UMLClass) obj).getFullClassName();
            if (this.displayedClasses.get (key) == null)
            {
               this.availableClasses.put (key, obj);
            }
         }
      }
   }

   // ######################################################################

   /**
    * dialog to UML-AST
    */
   public void parse()
   {
      boolean diagramNameValid = true;
      String newName = diagramNameTextField.getText();

      if (newName == null || newName.trim().equals (""))
      {
         WarningMessages.warnUnnamedDiagram (FrameMain.get(), "class");
         diagramNameValid = false;
      }

      Iterator iter = UMLProject.get().iteratorOfDiags();
      while (diagramNameValid && iter.hasNext())
      {
         ASGDiagram diagram = (ASGDiagram) iter.next();
         if ( (diagram instanceof UMLClassDiagram) &&
             (diagram != this.classDiag) &&
             (diagram.getName().equals (newName)))
         {
            WarningMessages.warnDoubleIdentifiedDiagram (FrameMain.get(), "class", newName);
            diagramNameValid = false;
         }
      }

      if (diagramNameValid)
      {
         this.classDiag.setName (diagramNameTextField.getText());
      } // end of if ()

      //----- remove classes from diagram
      // remove only those classes, that were selected to be removed!!!!
      Collection classesToRemove = this.initiallyDisplayedClasses;
      classesToRemove.removeAll (this.displayedClasses.values());
      iter = classesToRemove.iterator();
      while (iter.hasNext())
      {
         Object obj = iter.next();
         if (obj instanceof UMLClass)
         {
            UMLClass clazz = (UMLClass) obj;

            //remove the comment associated with clazz
            UMLCommentary comment = clazz.getComment();
            if (comment != null)
            {
               this.classDiag.removeFromElements (comment);
               //the dashed line between comment and class will not be removed
               //but if someone is familiar with fsa stuff ...
            }

            // remove generalizations
            Iterator genIter = clazz.iteratorOfRevSubclass();
            while (genIter.hasNext())
            {
               UMLGeneralization generalization = (UMLGeneralization) genIter.next();
               classDiag.removeFromElements (generalization);
            }
            genIter = clazz.iteratorOfRevSuperclass();
            while (genIter.hasNext())
            {
               UMLGeneralization generalization = (UMLGeneralization) genIter.next();
               classDiag.removeFromElements (generalization);
            }
            // remove assocs
            Iterator rolesIter = clazz.iteratorOfRoles();
            while (rolesIter.hasNext())
            {
               UMLRole role = (UMLRole) rolesIter.next();
               classDiag.removeFromElements (role.getAssoc());
            }

            String key =  ((UMLClass) obj).getFullClassName();
            if (this.displayedClasses.get (key) == null)
            {
               this.classDiag.removeFromElements (clazz);
            }
         }
      }

      //----- add classes to diagram
      // add only those classes, that were selected to be added!!!!
      Collection classesToAdd = this.displayedClasses.values();
      classesToAdd.removeAll (this.initiallyDisplayedClasses);
      iter = classesToAdd.iterator();
      while (iter.hasNext())
      {
         UMLClass clazz = (UMLClass) iter.next();
         this.classDiag.addToElements (clazz);

         //add the classes comment
         UMLCommentary comment = clazz.getComment();
         if (comment != null)
         {
            this.classDiag.addToElements (comment);
         }

         Enumeration enumGen;
         UMLGeneralization gen;
         enumGen = clazz.elementsOfRevSubclass();
         while (enumGen.hasMoreElements())
         {
            gen = (UMLGeneralization) enumGen.nextElement();
            if (this.classDiag.hasInElements (gen.getSuperclass()))
            {
               this.classDiag.addToElements (gen);
            }
         }

         enumGen = clazz.elementsOfRevSuperclass();
         while (enumGen.hasMoreElements())
         {
            gen = (UMLGeneralization) enumGen.nextElement();
            if (this.classDiag.hasInElements (gen.getSubclass()))
            {
               this.classDiag.addToElements (gen);
            }
         }

         Iterator roleIter = clazz.iteratorOfRoles();
         while (roleIter.hasNext())
         {
            UMLRole role = (UMLRole) roleIter.next();
            UMLAssoc assoc = role.getAssoc();

            if (role.getPartnerRole() != null)
            {
               UMLClass partner = role.getPartnerRole().getTarget();
               if (this.classDiag.hasInElements (partner))
               {
                  this.classDiag.addToElements (assoc);
               }
            } // end of if ()

         }
      }
      FrameMain.get().createNewTreeItems();

      UMLProject.get().refreshDisplay();
   }

   // ######################################################################

   /**
    * Access method for an one to n association.
    *
    * @param add      The object added.
    * @param context  The object added.
    */
   void addToDisplayedClasses (Object[] add, boolean context)
   {
      this.moveClasses (this.displayedClasses, this.availableClasses, add, context);
   }


   /**
    * Access method for an one to n association.
    *
    * @param add      The object added.
    * @param context  The object added.
    */
   void addToAvailableClasses (Object[] add, boolean context)
   {
      this.moveClasses (this.availableClasses, this.displayedClasses, add, context);
   }


   /**
    * Wrapper for moveClasses(TreeMap, TreeMap, Object[]) that additionally
    * includes the context classes.
    *
    * @param pasteInto  The object added.
    * @param cutFrom    The object added.
    * @param classKeys  The object added.
    * @param context    The object added.
    * @return           A set of classes that contains all classes that have been
    * successfully moved.
    * @see              EditClassDiagDialog#moveClasses(TreeMap, TreeMap, Object[])
    * @see              EditClassDiagDialog#getContext(UMLClass)
    */
   private Set moveClasses (TreeMap pasteInto, TreeMap cutFrom,
                            Object[] classKeys, boolean context)
   {
      Set movedClasses = new HashSet();

      if (classKeys != null)
      {
         if (context == true)
         {
            Object value;
            for (int i = 0; i < classKeys.length; i++)
            {
               value = cutFrom.get (classKeys[i]);
               if (value instanceof UMLClass)
               {
                  Set movedClassesWithContext = this.moveClasses (pasteInto, cutFrom, this.getContext ((UMLClass) value));
                  movedClasses.addAll (movedClassesWithContext);
               }
            }
         }
         Set movedClassesWithoutContext = this.moveClasses (pasteInto, cutFrom, classKeys);
         movedClasses.addAll (movedClassesWithoutContext);

         updateGuiList();
      }

      return movedClasses;
   }


   /**
    * Moves a set of classes from one TreeMap to another.
    *
    * @param pasteInto  The object added.
    * @param cutFrom    The object added.
    * @param classKeys  The object added.
    * @return           A set of classes that contains all classes that have been
    * successfully moved.
    */
   private Set moveClasses (TreeMap pasteInto, TreeMap cutFrom,
                            Object[] classKeys)
   {
      Set movedClasses = new HashSet();

      if (classKeys != null)
      {
         Object value;
         for (int i = 0; i < classKeys.length; i++)
         {
            value = cutFrom.get (classKeys[i]);
            if (value instanceof UMLClass)
            {
               cutFrom.remove (classKeys[i]);
               pasteInto.put (classKeys[i], value);

               // remember all classes that have been moved
               movedClasses.add (value);
            }
         }
      }

      return movedClasses;
   }


   /**
    * Get the context classes of the specified class.
    * Context classes are sub- and superclasses and
    * all classes that are connected to the class via
    * an association.
    *
    * @param clazz  No description provided
    * @return       The context value
    */
   private Object[] getContext (UMLClass clazz)
   {
      LinkedList context = new LinkedList();
      Enumeration enumeration;
      UMLGeneralization gen;

      //----- add parents
      enumeration = clazz.elementsOfRevSubclass();
      while (enumeration.hasMoreElements())
      {
         gen = (UMLGeneralization) enumeration.nextElement();
         context.add (gen.getSuperclass().getFullClassName());
      }

      //----- add children
      enumeration = clazz.elementsOfRevSuperclass();
      while (enumeration.hasMoreElements())
      {
         gen = (UMLGeneralization) enumeration.nextElement();
         context.add (gen.getSubclass().getFullClassName());
      }

      //----- add all class of the assocs
      Iterator iter = clazz.iteratorOfRoles();
      while (iter.hasNext())
      {
         UMLRole role = (UMLRole) iter.next();
         UMLRole partner = role.getPartnerRole();
         context.add (partner.getTarget().getFullClassName());
      }

      return context.toArray();
   }

   // ######################################################################
   // ----- some action classes
   // ######################################################################

   /*
    *  action-stuff
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.29.2.3 $
    */
   class ButtonAddActionListener implements java.awt.event.ActionListener
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void actionPerformed (ActionEvent e)
      {
         addToDisplayedClasses (availableClassesList.getSelectedValues(), false);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.29.2.3 $
    */
   class ButtonAddContextActionListener implements java.awt.event.ActionListener
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void actionPerformed (ActionEvent e)
      {
         addToDisplayedClasses (availableClassesList.getSelectedValues(), true);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.29.2.3 $
    */
   class ButtonDeleteActionListener implements java.awt.event.ActionListener
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void actionPerformed (ActionEvent e)
      {
         addToAvailableClasses (classesInDiagramList.getSelectedValues(), false);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.29.2.3 $
    */
   class ButtonDeleteContextActionListener implements java.awt.event.ActionListener
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void actionPerformed (ActionEvent e)
      {
         addToAvailableClasses (classesInDiagramList.getSelectedValues(), true);
      }
   }

}

/*
 * $Log: EditClassDiagDialog.java,v $
 * Revision 1.29.2.3  2006/02/09 15:30:37  lowende
 * Some compile warnings removed.
 *
 */
