/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.gui;

import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.*;

import javax.swing.*;

import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.asg.ASGDiagram;
import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.uml.*;
import de.uni_paderborn.fujaba.views.ViewDiagram;
import de.uni_paderborn.tools.util.ClassDeclaration;


/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: fklar $
 * @version   $Revision: 1.132.2.3 $
 */
public class PEClass extends PropertyEditor
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (PEClass.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private PECheck staticModifier;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private PECheck abstractModifier;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private PETextField className;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   PEListIncr generalList;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   PEListIncr stereotypeList;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   PEListIncr basicStereotypeList;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private PEButton resetGeneral;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private PEButton resetBasic;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private PEButton resetStereo;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private PEComboBox packageName;

   /**
    * editing the default icon for objects of this class.
    */
   private PETextField defaultIcon;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private PEButton browseButton;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLTypeList typeList = null;


   /**
    * Constructor for class PEClass
    *
    * @param frame  No description provided
    */
   public PEClass (JFrame frame)
   {
      super (frame);
      this.setTitle ("Class Editor");

      initPE();

      addFocusListener (new CMAFocusListener (className));
      className.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent e)
            {
               buttonOK_actionPerformed (e);
            }
         }
         );
   }


   /**
    * Get the propertyName attribute of the PEClass object
    *
    * @return   The propertyName value
    */
   protected String getPropertyName()
   {
      return "New Class";
   }


   /**
    * Sets the increment attribute of the PEClass object
    *
    * @param asgElement  The new increment value
    */
   public void setIncrement (ASGElement asgElement)
   {
      getFrame().setCursor (Cursor.getPredefinedCursor (Cursor.WAIT_CURSOR));
      setReadOnly (isReadOnly());

      super.setIncrement (asgElement);
      className.setReadOnly (isReadOnly());
      //FIX ME:packageName.setReadOnly (isReadOnly());
      if (asgElement instanceof UMLClass)
      {
         setTitle ("Class Editor: " +  ((UMLClass) asgElement).getName());
      }
      getFrame().setCursor (Cursor.getPredefinedCursor (Cursor.DEFAULT_CURSOR));
   }


   /**
    * Fill the dialog with values from the edited class.
    */
   protected void unparse()
   {
      ASGElement asgElement = getIncrement();

      // unparse all Stereotypes from project
      fillStereotypeList();

      //fill package list
      Set packageNames = new TreeSet();
      for (Iterator it = UMLProject.get().getTypeList().iteratorOfTypes(); it.hasNext(); )
      {
         FType type = (FType) it.next();
         this.packageName.removeAllItems();
         if (type instanceof FClass)
         {
            FClass cls =  ((FClass) type);
            if (cls.getFDeclaredInPackage() != null)
            {
               String packageName = cls.getFDeclaredInPackage().getFullPackageName();
               packageNames.add (packageName);
            }
         }
      }
      for (Iterator it = packageNames.iterator(); it.hasNext(); )
      {
         String packageName = (String) it.next();
         this.packageName.add (packageName);
      }

      if (asgElement instanceof UMLClass)
      {
         UMLClass clazz = (UMLClass) asgElement;

         // if the selected class is an inner class
         // its outer class defines the packagename
         UMLClass outerClass = clazz.getDeclaredInClass();
         UMLPackage thePackage = null;

         if (outerClass != null)
         {
            // the selected class is an inner class!

            // we have to use '$' and not '.' as separator, between outer class
            // and inner class, because otherwise method
            // 'syntaxCheck(ClassDeclaration)' will use the wrong
            // packagename (e.g. 'java.util.Map' instead of 'java.util') and
            // won't determine this is an iner class
            className.setText (outerClass.getName() + "$" + clazz.getText());

            thePackage = outerClass.getDeclaredInPackage();
         }
         else
         {
            // the selected class is a 'normal' class

            className.setText (clazz.getName());

            thePackage = clazz.getDeclaredInPackage();
         }

         if (thePackage != null)
         {
            this.packageName.setSelectedItem (thePackage.getFullPackageName());
         }

         // is class static
         staticModifier.setSelected (clazz.isStatic());

         // is class abstract
         abstractModifier.setSelected (clazz.isAbstract());

         // unparse all types from project
         fillTypeList();
         // search for generalized objects
         Enumeration superclasses = clazz.elementsOfRevSubclass();
         if (superclasses != null)
         {
            int count = 0;
            while (superclasses.hasMoreElements())
            {
               count++;
               superclasses.nextElement();
            }
            superclasses = clazz.elementsOfRevSubclass();
            UMLGeneralization gen;
            ASGElement[] selectedClasses = new ASGElement[count];
            int i = 0;
            while (superclasses.hasMoreElements())
            {
               gen = (UMLGeneralization) superclasses.nextElement();
               selectedClasses[i] = gen.getSuperclass();
               i++;
            }
            generalList.selectIncrements (selectedClasses);
         }
         selectStereotypes (clazz);

         defaultIcon.setText (clazz.getDefaultIcon());
      }
      else if (asgElement instanceof UMLDiagram)
      {
         // new class selected
         fillTypeList();
      }
   }


   /**
    * ************************************************************ parse the package data
    *
    * @param clazz  No description provided
    */
   private void parsePackage (UMLClass clazz)
   {
      String tmpPackageName = (String) packageName.getSelectedItem();
      if (tmpPackageName == null || tmpPackageName.length() == 0)
      {
         clazz.setDeclaredInPackage (UMLProject.get().getDefaultPackage());
      }
      else
      {
         UMLPackage tmpPackage = UMLProject.get().getNewFromPackages (tmpPackageName);

         if (clazz.getDeclaredInPackage() != tmpPackage)
         {
            clazz.setDeclaredInPackage (tmpPackage);

            // subclasses and partner classes of assocs must import the new package.
            // so add the new package to the import list of their files.
            // deleting the old package from the import list is not possible.
            // the current class cannot know if the other class needs the old
            // package to import a second here unknown class.
            /*
             *  let's do this in codegen instead, where it belongs. shall we?
             *  / inform all super classes.
             *  Iterator genIter = clazz.iteratorOfRevSuperclass();
             *  while (genIter.hasNext())
             *  {
             *  UMLGeneralization gen = (UMLGeneralization) genIter.next();
             *  UMLClass subClass = gen.getSubclass();
             *  if (subClass != null &&
             *  subClass.getDeclaredInPackage() != clazz.getDeclaredInPackage())
             *  {
             *  UMLFile file = subClass.getFile();
             *  if (file != null)
             *  {
             *  file.addToImportedPackages (clazz.getDeclaredInPackage());
             *  }
             *  }
             *  }
             *  / inform all assocs
             *  Iterator roleIter = clazz.iteratorOfRoles();
             *  while (roleIter.hasNext())
             *  {
             *  UMLRole role = (UMLRole) roleIter.next();
             *  UMLRole partnerRole = role.getPartnerRole();
             *  if (partnerRole != null &&
             *  partnerRole.getAdornment() != FRole.REFERENCE)
             *  {
             *  UMLClass partnerClass = partnerRole.getTarget();
             *  if (partnerClass != null &&
             *  partnerClass.getDeclaredInPackage() != clazz.getDeclaredInPackage())
             *  {
             *  UMLFile file = partnerClass.getFile();
             *  if (file != null)
             *  {
             *  file.addToImportedPackages (clazz.getDeclaredInPackage());
             *  }
             *  }
             *  }
             *  }
             */
         }
      }
   }


   /**
    * ************************************************************ parse the static and interface
    * data
    *
    * @param clazz  No description provided
    */
   private void parseStatic (UMLClass clazz)
   {
      // set class modifier
      clazz.setStatic (staticModifier.isSelected());
   }


   /**
    * ************************************************************ parse the persistent data
    *
    * @param clazz  No description provided
    */
   private void parseAbstract (UMLClass clazz)
   {
      // set class modifier
      clazz.setAbstract (abstractModifier.isSelected());
   }


   /**
    * ************************************************************ parse the 'Sterteotypes'
    * data
    *
    * @param clazz  No description provided
    */
   private void parseStereotypes (UMLClass clazz)
   {
      if (clazz != null)
      {
         clazz.removeAllFromStereotypes();
         UMLStereotype selectedStereotype;

         int indices[] = basicStereotypeList.getSelectedIndices();
         for (int i = 0; i < indices.length; i++)
         {
            selectedStereotype = (UMLStereotype) basicStereotypeList.getItemByIndex (indices[i]).getIncrement();
            clazz.addToStereotypes (selectedStereotype);
         }

         indices = stereotypeList.getSelectedIndices();
         for (int i = 0; i < indices.length; i++)
         {
            selectedStereotype = (UMLStereotype) stereotypeList.getItemByIndex (indices[i]).getIncrement();
            clazz.addToStereotypes (selectedStereotype);
         }
      }
   }


   /**
    * ************************************************************ parse the 'derived from'
    * data
    *
    * @param clazz  No description provided
    */
   private void parseGeneralisation (UMLClass clazz)
   {
      // first: test generalisation list for killed ancestors
      UMLGeneralization gen = null;
      Enumeration enumeration = clazz.elementsOfRevSubclass();

      while (enumeration.hasMoreElements())
      {
         gen = (UMLGeneralization) enumeration.nextElement();
         if (!generalList.isIncrementSelected (gen.getSuperclass()))
         {
            // delete generalization object
            gen.removeYou();
         }
      }

      // second: test selection for new ancestors
      UMLClass selectedClass = null;
      boolean genExists = false;
      int indices[] = generalList.getSelectedIndices();
      for (int i = 0; i < indices.length; i++)
      {
         selectedClass = (UMLClass) generalList.getItemByIndex (indices[i]).getIncrement();

         Enumeration genList = selectedClass.elementsOfRevSuperclass();
         genExists = false;
         while (genList.hasMoreElements() && genExists == false)
         {
            gen = (UMLGeneralization) genList.nextElement();
            if (gen.getSubclass() == clazz)
            {
               genExists = true;
            }
         }
         if (!genExists)
         {
            gen = new UMLGeneralization ("test", null, clazz, selectedClass);
         }

         UMLDiagram item = UMLProject.get().getCurrentUMLDiagram();
         if (item != null &&
             (item instanceof UMLClassDiagram) ||
             (item instanceof ViewDiagram &&
             ((ViewDiagram) item).getOriginalDiagram() instanceof UMLClassDiagram))
         {
            if (item.hasInElements (clazz) && item.hasInElements (selectedClass))
            {
               if (!item.hasInElements (gen))
               {
                  item.addToElements (gen);
               }
            }
         }

         Iterator diagrams = UMLProject.get().iteratorOfDiags();
         while (diagrams.hasNext())
         {
            ASGDiagram tmpItem = (ASGDiagram) diagrams.next();
            if (tmpItem instanceof UMLDiagram)
            {
               item = (UMLDiagram) tmpItem;
               if (item.hasInElements (clazz) && item.hasInElements (selectedClass))
               {
                  if (!item.hasInElements (gen))
                  {
                     item.addToElements (gen);
                  }
               }
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @throws ParseException  Exception description not provided
    */
   protected void parse() throws ParseException
   {

      if (className.getText().length() == 0)
      {
         JOptionPane.showMessageDialog (getFrame(), "You entered no name for the class!", "Class Editor",
            JOptionPane.ERROR_MESSAGE);
         throw new ParseException ("no class name");
      }

      UMLFile myFile = null;
      UMLClass clazz = null;
      UMLStereotype referenceType = UMLStereotypeManager.get().
         getFromStereotypes (UMLStereotypeManager.REFERENCE);

      //Runtime.getRuntime().traceMethodCalls(true);
      ASGElement asgElement = getIncrement();
      if (asgElement instanceof UMLClass)
      {
         if (log.isDebugEnabled())
         {
            log.debug ("------class:" + asgElement.getName());
         }
         clazz = (UMLClass) asgElement;
      }

      final Object pkg = packageName.getSelectedItem();
      String fullClassName = pkg == null || "".equals (pkg)
         ? className.getText()
         : pkg + "." + className.getText();
      if ( (clazz == null || !fullClassName.equals (clazz.getFullClassName())))
      {
         if (UMLProject.get().hasInClasses (fullClassName))
         {
            JOptionPane.showMessageDialog (getFrame(), "Please enter a unique name for the class!", "Class Editor",
               JOptionPane.ERROR_MESSAGE);
            throw new ParseException ("class name not unique");
         }
      }

      if (asgElement instanceof UMLDiagram)
      {
         if (log.isDebugEnabled())
         {
            log.debug ("------diagram:" + asgElement.getName());
         }

         clazz = new UMLClass (className.getText(), null, typeList, null);
      }

      if (clazz != null)
      {
         // set the properties of the class.
         clazz.setName (className.getText());
         UMLClass parentClass = clazz.getDeclaredInClass();
         this.parsePackage ( (parentClass == null ? clazz : parentClass));
         this.parseStereotypes (clazz);

         // a real class must have a file, a reference not.
         myFile = clazz.getFile();
         if (clazz.hasInStereotypes (referenceType))
         {
            clazz.setDeclaredInReferences (UMLProject.get());
            clazz.setFile (null);

            if (myFile != null)
            {
               myFile.removeYou();
            }
         } // end of if ()
         else
         {
            clazz.setDeclaredInReferences (null);
            if (myFile == null)
            {
               myFile = new UMLFile (clazz);
            } // end of if ()
         }

         UMLDiagram diag = UMLProject.get().getCurrentUMLDiagram();
         if (diag instanceof UMLClassDiagram)
         {
            diag.addToElements (clazz);
         } // end of if ()

         this.parseStatic (clazz);
         this.parseAbstract (clazz);
         this.parseGeneralisation (clazz);

         clazz.setDefaultIcon (defaultIcon.getText());
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void cancel() { }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param clazz  No description provided
    */
   private void selectStereotypes (UMLClass clazz)
   {
      if (clazz != null)
      {
         int size = clazz.sizeOfStereotypes();
         if (size != 0)
         {
            Iterator it = clazz.iteratorOfStereotypes();
            ASGElement[] selectedStereotypes = new ASGElement[size];
            for (int i = 0; i < size; i++)
            {
               selectedStereotypes[i] = (ASGElement) it.next();
            }
            basicStereotypeList.selectIncrements (selectedStereotypes);
            stereotypeList.selectIncrements (selectedStereotypes);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void fillStereotypeList()
   {
      basicStereotypeList.removeAll();
      stereotypeList.removeAll();

      TreeMap basicTypes = new TreeMap();
      TreeMap stereoTypes = new TreeMap();

      Iterator iter = UMLStereotypeManager.get().iteratorOfBasicStereotypes();
      while (iter.hasNext())
      {
         UMLStereotype stereotype = (UMLStereotype) iter.next();
         basicTypes.put (stereotype.getText(), stereotype);
      } // end of while ()

      iter = UMLStereotypeManager.get().iteratorOfStereotypes();
      while (iter.hasNext())
      {
         UMLStereotype type = (UMLStereotype) iter.next();

         if (!basicTypes.containsValue (type))
         {
            stereoTypes.put (type.getText(), type);
         } // end of if ()
      }

      iter = basicTypes.keySet().iterator();
      while (iter.hasNext())
      {
         basicStereotypeList.add ((ASGElement) basicTypes.get (iter.next()));
      } // end of while ()

      iter = stereoTypes.keySet().iterator();
      while (iter.hasNext())
      {
         stereotypeList.add ((ASGElement) stereoTypes.get (iter.next()));
      } // end of while ()
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void fillTypeList()
   {
      UMLType type = null;
      typeList = UMLProject.get().getTypeList();
      Iterator list = typeList.sortedIteratorOfTypes();
      generalList.removeAll();
      while (list.hasNext())
      {
         type = (UMLType) list.next();
         // only classes possible
         if (type instanceof UMLClass)
         {
            // only different types from selected class allowed
            if ( ((UMLClass) type) != getClassIncrement())
            {
               generalList.add ((ASGElement) type);
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param panel  No description provided
    */
   protected void additionalProperties (PEEditPanel panel)
   {
      className = new PETextField (this, "Class name", "NewClass");
      className.setStatus ("Enter the new class name");
      className.selectAll();

      staticModifier = new PECheck (this, "Static modifier", "Class is static");
      abstractModifier = new PECheck (this, "Abstract modifier", "Class is abstract");

      generalList = new PEListIncr (this, "Derived from");
      generalList.setStatus ("Select all generalized classes");
      generalList.setSingleSelection (false);
      resetGeneral = new PEButton (this, "reset selection");
      resetGeneral.addActionListener (createResetGeneralizationListener());

      basicStereotypeList = new PEListIncr (this, "UML stereotypes");
      basicStereotypeList.setStatus ("Select UML stereotypes");
      basicStereotypeList.setSingleSelection (false);
      resetBasic = new PEButton (this, "reset selection");
      resetBasic.addActionListener (createResetBasicStereotypesListener());

      stereotypeList = new PEListIncr (this, "User defined stereotypes");
      stereotypeList.setStatus ("Select user defined stereotypes");
      stereotypeList.setSingleSelection (false);
      resetStereo = new PEButton (this, "reset selection");
      resetStereo.addActionListener (createResetStereotypesListener());

      packageName = new PEComboBox();
      packageName.setEditable (true);
      packageName.setPreferredSize (new Dimension (300, (int) packageName.getPreferredSize().getHeight()));
      //FIX ME: why is PEComboBox no PEHeaderComponent!?
      // packageName.setStatus ("Enter the package including this class");
      SDMObjectDialog.enableCompletion (packageName);

      PEColumn leftFirst = new PEColumn (this);
      PEColumn rightFirst = new PEColumn (this);

      PEColumn generalColumn = new PEColumn (this);
      generalColumn.add (generalList);
      generalColumn.add (resetGeneral);

      PEColumn basicColumn = new PEColumn (this);
      basicColumn.add (basicStereotypeList);
      basicColumn.add (resetBasic);

      PEColumn stereoColumn = new PEColumn (this);
      stereoColumn.add (stereotypeList);
      stereoColumn.add (resetStereo);

      leftFirst.add (className);
      //FIX ME: why is PEComboBox no PEHeaderComponent!?
      leftFirst.add (new JLabel ("Package name"));
      leftFirst.add (packageName);
      leftFirst.add (staticModifier);
      leftFirst.add (abstractModifier);

      rightFirst.add (basicColumn);

      PERow firstRow = new PERow (this);
      firstRow.add (leftFirst);
      firstRow.add (rightFirst);

      panel.add (firstRow);

      PEColumn leftSnd = new PEColumn (this);
      PEColumn rightSnd = new PEColumn (this);

      leftSnd.add (generalColumn);
      rightSnd.add (stereoColumn);

      PERow sndRow = new PERow (this);
      sndRow.add (leftSnd);
      sndRow.add (rightSnd);

      panel.add (sndRow);

      PERow iconRow = new PERow (this);
      defaultIcon = new PETextField (this, "Default icon");
      iconRow.add (defaultIcon);

      browseButton = new PEButton (this, "Browse ...");
      browseButton.addActionListener (createBrowseListener());
      iconRow.add (browseButton);

      panel.add (iconRow);
   }


   /**
    * Overwritten to enable syntax checking.
    *
    * @param e  No description provided
    */
   public void buttonOK_actionPerformed (ActionEvent e)
   { // public needed temporarily, JW.

      ClassDeclaration decl = new ClassDeclaration (className.getText());

      if (!syntaxCheck (decl))
      {
         return;
      }
      else
      {
         super.buttonOK_actionPerformed (e);

         Iterator iter = decl.getConstructors().iterator();

         while (iter.hasNext())
         {
            Vector vector = (Vector) iter.next();

            ASGElement asgElement = getIncrement();
            if (asgElement instanceof UMLDiagram)
            {
               Iterator it =  ((UMLDiagram) asgElement).iteratorOfElements();

               while (it.hasNext())
               {
                  Object o = it.next();
                  {
                     if (o instanceof UMLClass)
                     {
                        String name =  ((UMLClass) o).getName();
                        if (name.equalsIgnoreCase (decl.getName()))
                        {
                           UMLClass umlclass = (UMLClass) o;

                           UMLMethod method = new UMLMethod();
                           method.setName (umlclass.getName());
                           method.setResultType (UMLProject.get().getTypeList().getFromTypes (UMLBaseTypes.CONSTRUCTOR));
                           method.setVisibility (FDeclaration.PUBLIC);

                           for (int i = 0; i < vector.size(); i += 2)
                           {
                              method.addToParam (new UMLParam ((String) vector.get (i + 1), (String) vector.get (i)));
                           }

                           umlclass.addToMethods (method);
                        }
                     }
                  }
               }
            }
            else
            {
               return;
            }
         }
      }
   }


   /**
    * Checks the syntax of the arguments typed in the textfield
    *
    * @param decl  No description provided
    * @return      No description provided
    */
   private boolean syntaxCheck (ClassDeclaration decl)
   {
      if (!decl.isValid())
      {
         JOptionPane.showMessageDialog (null, decl.getClassError());
         return false;
      }

      if (ClassDeclaration.ILLEGAL_CONSTRUCTORS_DECLARATION.equals (decl.getClassError()))
      {
         JOptionPane.showMessageDialog (null, ClassDeclaration.ILLEGAL_CONSTRUCTORS_DECLARATION + ".\n\nNo constructor will be created.");
      }

      className.setText (decl.getName());
      staticModifier.setSelected (decl.isStatic() || staticModifier.isSelected());
      abstractModifier.setSelected (decl.isAbstract() || abstractModifier.isSelected());

      if (decl.getPackage().length() != 0)
      {
         packageName.setSelectedItem (decl.getPackage());
      }

      if (ClassDeclaration.NO_CONSTRUCTORS_ALLOWED.equals (decl.getClassError()))
      {
         JOptionPane.showMessageDialog (null, ClassDeclaration.NO_CONSTRUCTORS_ALLOWED + ".\n\nConstructor declaration will be ignored.");
         return false;
      }

      return true;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private ActionListener createResetGeneralizationListener()
   {
      ActionListener listener =
         new ActionListener()
         {
            public void actionPerformed (ActionEvent event)
            {
               generalList.clearSelection();
            }
         };
      return listener;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private ActionListener createResetBasicStereotypesListener()
   {
      ActionListener listener =
         new ActionListener()
         {
            public void actionPerformed (ActionEvent event)
            {
               basicStereotypeList.clearSelection();
            }
         };
      return listener;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private ActionListener createBrowseListener()
   {
      ActionListener listener =
         new ActionListener()
         {
            public void actionPerformed (ActionEvent event)
            {
               browse();
            }
         };
      return listener;
   }


   /**
    * Opens a JFileChooser so you can browse to get your favourite directory.
    */
   void browse()
   {
      JFileChooser jf = new JFileChooser (".");
      int opt = jf.showDialog (this, "Choose an icon");

      if (opt == JFileChooser.APPROVE_OPTION)
      {
         defaultIcon.setText (jf.getSelectedFile().getPath());
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private ActionListener createResetStereotypesListener()
   {
      ActionListener listener =
         new ActionListener()
         {
            public void actionPerformed (ActionEvent event)
            {
               stereotypeList.clearSelection();
            }
         };
      return listener;
   }
}

/*
 * $Log: PEClass.java,v $
 * Revision 1.132.2.3  2005/10/25 09:12:50  fklar
 * better support for inner classes (check if UML class to be edited is an inner class - setting Java-style inner class name in name field)
 *
 */
