/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.gui;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.*;

import javax.swing.*;

import org.apache.log4j.Logger;

import de.tu_bs.coobra.LocalRepository;
import de.uni_paderborn.fujaba.asg.ASGElement;

import de.uni_paderborn.fujaba.basic.FD;
import de.uni_paderborn.fujaba.coobra.FujabaChangeManager;
import de.uni_paderborn.fujaba.gui.comp.FujabaDialog;
import de.uni_paderborn.fujaba.metamodel.FClass;
import de.uni_paderborn.fujaba.metamodel.FMethod;
import de.uni_paderborn.fujaba.uml.*;


/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.24.2.3 $
 */
public class RepairProjectDialog extends FujabaDialog implements ActionListener
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (RepairProjectDialog.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final JButton jCallGC = new JButton ("Call the garbage collector");
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final JButton jRebuildFragments = new JButton ("Rebuild fragments of all classes");
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final JButton jRemoveMethods = new JButton ("Remove all method with no result type");
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final JButton jRemoveGenMethods = new JButton ("Remove all generated methods");
   //private final JButton jCheckObjects = new JButton ("check all Objects");
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final JButton jRemoveParentlessElements = new JButton ("Remove all parentless elements");
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final JButton jRemoveUnusedClasses = new JButton ("Remove unused classes");

   // ######################################################################

   /**
    * Constructor for class RepairProjectDialog
    *
    * @param frame  No description provided
    */
   public RepairProjectDialog (JFrame frame)
   {
      super (frame, "Repair Project Dialog -- ONLY FOR DEVELOPER", true);

      this.guiInit();
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final void guiInit()
   {
      java.awt.Container container = this.getContentPane();
      container.setLayout (new BorderLayout());
      container.add (this.guiWorkPanel(), BorderLayout.CENTER);
      container.add (this.guiPanelCloseHelp(), BorderLayout.SOUTH);

      //----- setup all the listeners
      jCallGC.addActionListener (this);
      jRebuildFragments.addActionListener (this);
      jRemoveMethods.addActionListener (this);
      jRemoveGenMethods.addActionListener (this);
      jRemoveParentlessElements.addActionListener (this);
      jRemoveUnusedClasses.addActionListener (this);
      //jCheckObjects.addActionListener (this);

      this.pack();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private final JPanel guiWorkPanel()
   {
      JPanel panel = new JPanel();
      panel.setLayout (new BoxLayout (panel, BoxLayout.Y_AXIS));
      panel.add (jCallGC);
      panel.add (jRebuildFragments);
      panel.add (jRemoveMethods);
      panel.add (jRemoveGenMethods);
      panel.add (jRemoveParentlessElements);
      panel.add (jRemoveUnusedClasses);
      //panel.add (jCheckObjects);
      return panel;
   }

   // ######################################################################
   // ----- begin methods from interface ActionListener
   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void actionPerformed (ActionEvent e)
   {
      Object source = e.getSource();

      if (source == jCallGC)
      {
         callGC();
      }
      else if (source == jRebuildFragments)
      {
         rebuildFragments();
      }
      else if (source == jRemoveMethods)
      {
         if (log.isInfoEnabled())
         {
            log.info ("remove all methods without result type and adjust the constructor...");
         }
         removeMethods (UMLProject.get().getRootPackage());
      }
      else if (source == jRemoveGenMethods)
      {
         if (log.isInfoEnabled())
         {
            log.info ("remove all generated methods...");
         }
         removeGenMethods (UMLProject.get().getRootPackage());
      }
      else if (source == jRemoveParentlessElements)
      {
         cleanUp (FujabaChangeManager.getVMRepository());
      }
      else if (source == jRemoveUnusedClasses)
      {
         removeUnusedClasses (UMLProject.get());
      }
      else
      {
         if (log.isInfoEnabled())
         {
            log.info (e);
         }
      }
   }

   // ######################################################################
   // ----- end of methods from interface ActionListener
   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final void callGC()
   {
      System.out.print ("call gc ...");
      System.gc();
      if (log.isInfoEnabled())
      {
         log.info (" - ok");
      }
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final void rebuildFragments()
   {
      //      UMLProject.get ().rebuildFragments ();
      if (log.isInfoEnabled())
      {
         log.info ("Rebuild fragments for project.");
      }
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param pack  No description provided
    */
   private final void removeMethods (UMLPackage pack)
   {
      Iterator iter;

      iter = pack.iteratorOfPackages();
      while (iter.hasNext())
      {
         removeMethods ((UMLPackage) iter.next());
      }

      iter = pack.iteratorOfDeclares();
      while (iter.hasNext())
      {
         removeMethods ((UMLClass) iter.next());
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param project  No description provided
    */
   private final void removeUnusedClasses (UMLProject project)
   {
      LocalRepository repo = project.getRepository();
      Iterator iter = repo.iteratorOfKnownObjects();
      HashSet objects = new HashSet();
      while (iter.hasNext())
      {
         Object obj = iter.next();
         if (obj instanceof UMLClass)
         {
            UMLClass clazz = (UMLClass) obj;
            if ( (clazz.getName() == null) &&  (clazz.getRevTypes() == null))
            {
               objects.add (clazz);
            }
         }
      }
      if (objects.size() > 0)
      {
         if (log.isInfoEnabled())
         {
            log.info ("Found " + objects.size() + " unused classes.");
         }
         iter = objects.iterator();
         while (iter.hasNext())
         {
             ((UMLClass) iter.next()).removeYou();
         }
         repo.compact();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param cls  No description provided
    */
   private final void removeMethods (UMLClass cls)
   {
      Iterator iter;

      iter = cls.iteratorOfDeclares();
      while (iter.hasNext())
      {
         removeMethods ((UMLClass) iter.next());
      }

      if (log.isInfoEnabled())
      {
         log.info ("  check class " + cls.getFullClassName());
      }

      UMLMethod method;
      String clsName = cls.getName();
      UMLType retType = UMLProject.get().getFromBaseTypes (UMLBaseTypes.CONSTRUCTOR);

      iter = cls.iteratorOfMethods();
      while (iter.hasNext())
      {
         method = (UMLMethod) iter.next();

         System.out.print ("    check method " + method.getFullMethodName());
         if (method.getResultType() == null)
         {
            if (clsName.equals (method.getName()))
            {
               if (log.isInfoEnabled())
               {
                  log.info (" - set result type to constructor");
               }
               method.setResultType (retType);
            }
            else
            {
               if (log.isInfoEnabled())
               {
                  log.info (" - remove method");
               }
               method.removeYou();
            }
         }
         else
         {
            if (log.isInfoEnabled())
            {
               log.info (" - OK");
            }
         }
      }
   }
   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param pack  No description provided
    */
   public final void removeGenMethods (UMLPackage pack)
   {
      Iterator iter;

      iter = pack.iteratorOfPackages();
      while (iter.hasNext())
      {
         removeGenMethods ((UMLPackage) iter.next());
      }

      iter = pack.iteratorOfDeclares();
      while (iter.hasNext())
      {
         removeGenMethods ((UMLClass) iter.next());
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param cls  No description provided
    */
   private final void removeGenMethods (UMLClass cls)
   {
      Iterator iter;

      iter = cls.iteratorOfDeclares();
      while (iter.hasNext())
      {
         removeGenMethods ((UMLClass) iter.next());
      }

      if (log.isInfoEnabled())
      {
         log.info ("  check " + FD.toString ((FClass) cls));
      }

      UMLMethod method;
      iter = cls.iteratorOfMethods();
      while (iter.hasNext())
      {
         method = (UMLMethod) iter.next();

         System.out.print ("    check " + FD.toString ((FMethod) method));
         if (method.isGenerated())
         {
            if (log.isInfoEnabled())
            {
               log.info (" - remove method");
            }
            method.removeYou();
         }
         else
         {
            if (log.isInfoEnabled())
            {
               log.info (" - ok");
            }
         }
      }
   }


   /**
    * A unit test for JUnit
    *
    * @param repository  No description provided
    */
   private void cleanUp (LocalRepository repository)
   {
      Map objectCountByType = new TreeMap();
      Map getParentFailures = new TreeMap();
      Set parentless = new HashSet();
      int total = 0;

      if (repository != null)
      {
         Iterator it = repository.iteratorOfKnownObjects();
         while (it.hasNext())
         {
            Object obj = it.next();

            String className = obj.getClass().getName();
            count (objectCountByType, className);
            total++;
            try
            {
               if ( ((ASGElement) obj).getParentElement() == null)
               {
                  if (! (obj instanceof UMLProject))
                  {
                     parentless.add (obj);
                  }
                  count (objectCountByType, "*" + className);
               }
            }
            catch (Throwable e)
            {
               getParentFailures.put (className, e);
            }
         }
      }

      for (Iterator it = getParentFailures.entrySet().iterator(); it.hasNext(); )
      {
         Map.Entry entry = (Map.Entry) it.next();
         log.error (entry.getKey());
          ((Throwable) entry.getValue()).printStackTrace();
      }
      for (Iterator it = objectCountByType.entrySet().iterator(); it.hasNext(); )
      {
         Map.Entry entry = (Map.Entry) it.next();
         String classname = (String) entry.getKey();
         if (!classname.startsWith ("*"))
         {
            Integer nullCount = (Integer) objectCountByType.get ("*" + classname);
            if (log.isInfoEnabled())
            {
               log.info (entry.getValue() + "\twhere\t" +  (nullCount != null ? nullCount.intValue() : 0) + "\tfloating\t" + classname);
            }
         }
      }
      if (log.isInfoEnabled())
      {
         log.info (total + " total");
      }

      //print some more stats
      if (log.isInfoEnabled())
      {
         log.info ("Type\t attrs\t attrs in hierarchy\t methods");
      }
      for (Iterator it = UMLProject.get().getTypeList().iteratorOfTypes(); it.hasNext(); )
      {
         UMLType type = (UMLType) it.next();
         if (type instanceof UMLClass)
         {
            final UMLClass cls =  ((UMLClass) type);
            UMLClass parent = cls;
            int numAttrInHier = 0;
            while (parent != null)
            {
               numAttrInHier += parent.sizeOfAttrs();
               parent = parent.getSuperClass();
            }
            if (log.isInfoEnabled())
            {
               log.info (cls.getFullClassName() + "\t" + cls.sizeOfAttrs() + "\t" + numAttrInHier + "\t" + cls.sizeOfMethods());
            }
         }
      }
      //finished stats

      int choice = JOptionPane.showOptionDialog (this, "Delete " + parentless.size() + " floating objects?", "Repair",
         JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE, null, null, null);
      if (choice == JOptionPane.YES_OPTION)
      {
         for (Iterator it = parentless.iterator(); it.hasNext(); )
         {
            ASGElement element = (ASGElement) it.next();
            element.removeYou();
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param objectCountByType  No description provided
    * @param className          No description provided
    */
   private void count (Map objectCountByType, String className)
   {
      Integer count = (Integer) objectCountByType.get (className);
      if (count == null)
      {
         count = new Integer (0);
      }
      objectCountByType.put (className, new Integer (count.intValue() + 1));
   }
}

/*
 * $Log: RepairProjectDialog.java,v $
 * Revision 1.24.2.3  2005/09/30 18:56:51  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
