/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.gxl;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.*;

import javax.swing.*;
import javax.swing.border.EtchedBorder;
import javax.swing.border.TitledBorder;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.log4j.Logger;
import org.w3c.dom.*;

import de.uni_paderborn.fujaba.app.FrameMain;
import de.uni_paderborn.fujaba.asg.ASGDiagram;
import de.uni_paderborn.fujaba.basic.SchemaFilter;
import de.uni_paderborn.fujaba.fsa.unparse.UnparseManager;
import de.uni_paderborn.fujaba.gui.comp.FujabaDialog;
import de.uni_paderborn.fujaba.uml.*;


/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: mtt $
 * @version   $Revision: 1.28.2.2 $
 */
public class XSLTransformerDialog extends FujabaDialog
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (XSLTransformerDialog.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int EXPORT = 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int IMPORT = 2;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int status = 0;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Hashtable stylesheets;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private File stylesheetDir;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private File stylesheetList;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Iterator iter;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String text;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JFileChooser fileChooser;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JTextField file;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JPanel buttonPanel;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JButton okButton;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JButton cancelButton;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JList availableSheetList;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JList appliedSheetList;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private DefaultListModel importModel;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private DefaultListModel exportModel;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JCheckBox addExport;


   /**
    * Constructor for class XSLTransformerDialog
    *
    * @param parent  No description provided
    * @param iter    No description provided
    */
   public XSLTransformerDialog (JFrame parent, Iterator iter)
   {
      super (parent);
      this.iter = iter;
      stylesheetDir = new File (System.getProperty ("user.dir") + File.separatorChar + "stylesheets");
      stylesheetList = new File (stylesheetDir.getAbsolutePath() + File.separatorChar + "stylesheet.xml");
      fileChooser = new JFileChooser();
      readStylesheets();

      appliedSheetList = new JList (new Vector());
      availableSheetList = new JList (getStylesheets());

      //the FHashMap is filled with the UMLClassDiagramFilter and the UMLActivitiyDiagramFilter
      GXLFilterMap.get().addToFilter (UMLClassDiagram.class, new UMLClassDiagramFilter());
      GXLFilterMap.get().addToFilter (UMLActivityDiagram.class, new UMLActivityDiagramFilter());

      //the FHashMap is filled with the UMLClassDiagramMerger and the UMLActivityDiagramMerger
      GXLMergerMap.get().addToMerger (UMLClassDiagram.class, new UMLClassDiagramMerger());
      GXLMergerMap.get().addToMerger (UMLActivityDiagram.class, new UMLActivityDiagramMerger());
   }


   /**
    * Sets title and status for import dialog.
    */
   public void showImportDialog()
   {
      status = IMPORT;
      setTitle ("importing GXL");
      text = "read file";

      appliedSheetList.setModel (importModel);
      initGUI ("import");

      setSize (300, 600);
      super.showCentered();
   }


   /**
    * Sets title and status for export dialog.
    */
   public void showExportDialog()
   {
      status = EXPORT;
      setTitle ("exporting GXL");
      text = "write file";

      appliedSheetList.setModel (exportModel);
      initGUI ("export");

      setSize (300, 600);
      super.showCentered();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param text  No description provided
    */
   private void initGUI (String text)
   {
      JPanel panel = new JPanel();
      panel.setLayout (new BorderLayout());
      panel.add (this.guiSingleFilePanel(), BorderLayout.NORTH);
      panel.add (this.guiWorkPanel(), BorderLayout.CENTER);
      panel.add (this.guiButtonPanel (text), BorderLayout.SOUTH);
      this.getContentPane().add (panel);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private JPanel guiSingleFilePanel()
   {
      JPanel panel = new JPanel();

      panel.setBorder (new TitledBorder (new EtchedBorder(), text));

      file = new JTextField (30);
      panel.add (file); //, constraints);
      JButton browseButton = new JButton ("Browse...");
      browseButton.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent e)
            {
               fileSelection();
            }
         });
      panel.add (browseButton);
      return panel;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private JPanel guiWorkPanel()
   {
      JPanel panel = new JPanel();
      JPanel tmpPanel;

      //----- set gridbaglayout to the panel
      GridBagLayout gridbag = new GridBagLayout();
      GridBagConstraints constraints = new GridBagConstraints();
      panel.setLayout (gridbag);

      //----- some pre-settings to the constraints
      constraints.gridwidth = 1;
      constraints.gridheight = 1;
      constraints.fill = GridBagConstraints.BOTH;
      constraints.ipadx = 5;
      constraints.ipady = 5;
      constraints.insets = new Insets (5, 0, 5, 5);
      constraints.anchor = GridBagConstraints.NORTHWEST;
      constraints.weightx = 1.0;
      constraints.weighty = 1.0;

      panel.add (guiLeftSide(), constraints);

      constraints.fill = GridBagConstraints.HORIZONTAL;
      constraints.anchor = GridBagConstraints.CENTER;
      constraints.insets = new Insets (5, 5, 5, 5);
      constraints.weightx = 0.0;
      constraints.weighty = 0.0;
      tmpPanel = this.guiMiddle();
      gridbag.setConstraints (tmpPanel, constraints);
      panel.add (tmpPanel);

      constraints.anchor = GridBagConstraints.NORTHWEST;
      constraints.gridwidth = GridBagConstraints.REMAINDER;
      constraints.fill = GridBagConstraints.BOTH;
      constraints.insets = new Insets (5, 5, 5, 0);
      constraints.weightx = 1.0;
      constraints.weighty = 1.0;
      panel.add (guiRightSide(), constraints);

      return panel;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private JPanel guiLeftSide()
   {
      //----- add all gui-stuff to the panel
      JPanel leftPanel = new JPanel (new BorderLayout());
      JScrollPane sPane = new JScrollPane (this.appliedSheetList);
      leftPanel.setBorder (new TitledBorder (new EtchedBorder(), "applied stylesheets"));
      leftPanel.add (sPane, BorderLayout.CENTER);

      leftPanel.setPreferredSize (new Dimension (150, 200));
      return leftPanel;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private JPanel guiRightSide()
   {
      //----- add all gui-stuff to the panel
      JPanel rightPanel = new JPanel (new BorderLayout());
      JScrollPane sPane = new JScrollPane (this.availableSheetList);
      rightPanel.setBorder (new TitledBorder (new EtchedBorder(), "available stylesheets"));
      rightPanel.add (sPane, BorderLayout.CENTER);
      rightPanel.setPreferredSize (new Dimension (150, 200));
      return rightPanel;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private JPanel guiMiddle()
   {
      //----- set gridbaglayout to the panel
      GridBagLayout gridbag = new GridBagLayout();
      GridBagConstraints constraints = new GridBagConstraints();
      //----- some pre-settings to the constraints
      constraints.gridwidth = GridBagConstraints.REMAINDER;
      constraints.gridheight = 1;
      constraints.fill = GridBagConstraints.HORIZONTAL;
      constraints.anchor = GridBagConstraints.CENTER;
      constraints.weightx = 0.0;
      constraints.weighty = 0.0;
      constraints.insets = new Insets (0, 5, 0, 5);

      JPanel panel = new JPanel();

      panel.setLayout (gridbag);
      JButton addButton = new JButton ("<< add");
      addButton.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent ae)
            {
               if (availableSheetList.getSelectedIndex() > -1)
               {
                   ((DefaultListModel) appliedSheetList.getModel()).addElement (availableSheetList.getSelectedValue());
               }
            }
         });
      panel.add (addButton, constraints);
      JButton removeButton = new JButton ("remove >>");
      removeButton.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent ae)
            {
               if (appliedSheetList.getSelectedIndex() > -1)
               {
                   ((DefaultListModel) appliedSheetList.getModel()).removeElement (appliedSheetList.getSelectedValue());
               }
            }
         });
      panel.add (removeButton, constraints);

      if (status == XSLTransformerDialog.EXPORT)
      {
         String info = "If no stylesheet is selected, the xml-stylesheet will be taken automatically!";
         JLabel infoLabel = new JLabel (info);
         panel.add (infoLabel, constraints);

      }
      if (status == XSLTransformerDialog.IMPORT)
      {
         String info = "It is not necessary to specify a stylesheet for the import of RTSCs!";
         JLabel infoLabel = new JLabel (info);
         panel.add (infoLabel, constraints);
      }

      constraints.insets = new Insets (10, 0, 0, 0);
      return panel;
   }


   /**
    * Returns a sorted list of all available stylesheets.
    *
    * @return   The stylesheets value
    */
   private Vector getStylesheets()
   {
      Vector sheets = new Vector (0);
      for (Enumeration enumeration = stylesheets.keys(); enumeration.hasMoreElements(); )
      {
         sheets.add (enumeration.nextElement());
      }
      Collections.sort (sheets);
      return sheets;
   }


   /**
    * Creates a panel with ok and cancel button.
    *
    * @param text  No description provided
    * @return      No description provided
    */
   private JPanel guiButtonPanel (String text)
   {
      buttonPanel = new JPanel();
      buttonPanel.setLayout (new BorderLayout());

      if (text.equals ("export"))
      {
         addExport = new JCheckBox ("include additional diagrams");
         buttonPanel.add (addExport, BorderLayout.NORTH);
      }
      JPanel tmpPanel = new JPanel();
      okButton = new JButton();
      okButton.setText ("OK");
      okButton.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent e)
            {
               okButtonActionPerformed (e);
            }
         }
         );
      okButton.setBounds (600, 15, 90, 25);
      tmpPanel.add (okButton);

      cancelButton = new JButton();
      cancelButton.setText ("Cancel");
      cancelButton.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent e)
            {
               setVisible (false);
            }
         }
         );
      cancelButton.setBounds (600, 15, 90, 25);
      tmpPanel.add (cancelButton);
      buttonPanel.setBounds (5, 260, 600, 50);
      buttonPanel.add (tmpPanel, BorderLayout.CENTER);
      return buttonPanel;
   }


   /**
    * Creates a import/export dialog and updates the JTextField. (depending on internal status)
    */
   void fileSelection()
   {
      fileChooser.setFileSelectionMode (JFileChooser.FILES_AND_DIRECTORIES);
      fileChooser.setCurrentDirectory (new File (System.getProperty ("user.dir")));
      if (log.isInfoEnabled())
      {
         log.info (String.valueOf (getAppliedStylesheets().length));
      }
      SchemaFilter filter;
      String target = "";
      if (getAppliedStylesheets().length > 0)
      {
         if (status == XSLTransformerDialog.EXPORT)
         {
            int length = getAppliedStylesheets().length;
            String stylesheet = getAppliedStylesheets()[length - 1].getName();
            target = stylesheet.split ("2")[1];
            target = target.split (".xsl")[0];
         }
         else
         {
            String stylesheet = getAppliedStylesheets()[0].getName();
            target = stylesheet.split ("2")[1];
            target = stylesheet.split ("2")[0];
         }

         if (target.equals ("fujaba"))
         {
            target = "fpr";
         }

         filter = new SchemaFilter ("." + target, target.toUpperCase() + " Files (*." + target + ", *." + target + ".gz)");
         filter.addExtension (target + ".gz");

      }
      else
      {
         target = "xml";
         filter = new SchemaFilter (".xml", "XML Files (*.xml, *.xml.gz)");
         filter.addExtension ("xml.gz");
      }
      fileChooser.setFileFilter (filter);

      if (status == XSLTransformerDialog.EXPORT)
      {
         if (fileChooser.showSaveDialog (this) == JFileChooser.APPROVE_OPTION)
         {
            String tmpFile = fileChooser.getSelectedFile().getPath();
            if (!tmpFile.endsWith ("." + target + ".gz"))
            {
               if (!tmpFile.endsWith ("." + target))
               {
                  tmpFile += "." + target;
               }
               if (!tmpFile.endsWith (".gz"))
               {
                  tmpFile += ".gz";
               }
            }
            file.setText (tmpFile);
         }
      }
      else if (status == XSLTransformerDialog.IMPORT)
      {
         if (fileChooser.showOpenDialog (this) == JFileChooser.APPROVE_OPTION)
         {
            String tmpFile = fileChooser.getSelectedFile().getPath();
            file.setText (tmpFile);
         }
      }
   }


   /**
    * Returns root element of XML file which contains the list of available stylesheets.
    *
    * @param xmlFile  No description provided
    * @return         No description provided
    */
   private Element readXMLFile (File xmlFile)
   {
      if (xmlFile.exists())
      {
         try
         {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder parser = factory.newDocumentBuilder();
            Document doc = parser.parse (xmlFile);
            Element root = doc.getDocumentElement();
            return root;
         }
         catch (Exception e)
         {
            log.error ("file \"" + xmlFile.getName() + "\" seems to be corrupted.\n" + e);
            return null;
         }
      }
      else
      {
         log.error ("file \"" + xmlFile.getName() + "\" not found.");
         return null;
      }
   }


   /**
    * Generates an array of File out of the JList configured by user.
    *
    * @return   The appliedStylesheets value
    */
   private File[] getAppliedStylesheets()
   {
      DefaultListModel model = (DefaultListModel) appliedSheetList.getModel();
      File[] files = new File[model.getSize()];
      int i = 0;
      for (Enumeration enumeration = model.elements(); enumeration.hasMoreElements(); )
      {
         String realFileName = (String) stylesheets.get (enumeration.nextElement());
         files[i] = new File (stylesheetDir.getAbsolutePath() + "/" + realFileName);
         i++;
      }
      return files;
   }


   /**
    * Calling FileSaver or GXLImport depending on internal status.
    *
    * @param e  No description provided
    */
   void okButtonActionPerformed (ActionEvent e)
   {
      boolean exportFlag = false;
      boolean importFlag = false;
      if (file.getText().equals (""))
      {
         log.error ("No file specified.");
         return;
      }
      if (getAppliedStylesheets().length == 0)
      {
         if (status == XSLTransformerDialog.EXPORT)
         {
            FileSaver fileSaver = new FileSaver (iter, new File (file.getText()), addExport.isSelected());
            fileSaver.save();
            exportFlag = true;
         }
         else if (status == XSLTransformerDialog.IMPORT)
         {
            importFlag = true;
         }
      }
      if ( (status == XSLTransformerDialog.EXPORT) &&  (!exportFlag))
      {
         //no stylesheet specidfied
         FileSaver fileSaver = new FileSaver (iter, new File (file.getText()), getAppliedStylesheets(), addExport.isSelected());
         fileSaver.save();
      }
      else if (status == XSLTransformerDialog.IMPORT)
      {
         Object tmpObject = null;
         if (iter.hasNext())
         {
            tmpObject = iter.next();
         }
         if (tmpObject != null && tmpObject instanceof ASGDiagram)
         {
            ASGDiagram tmpIncr = (ASGDiagram) tmpObject;
            Validator myValidator = null;
            if (!importFlag)
            {
               myValidator = new Validator (new File (file.getText()), getAppliedStylesheets());
            }
            else
            {
               myValidator = new Validator (new File (file.getText()));
            }
            try
            {
               //checking for fujaba validity implied
               FXMLTree tmpTree = myValidator.getDOMTree();
               //stopping property processing
               UnparseManager.get().pausePropertyProcessing();
               
               //import gxl file into diagram
               new GXLImport (tmpTree, tmpIncr);
               
               //display diagram and resume property processing
               FrameMain.get().createNewTreeItems();
               UnparseManager.get().resumePropertyProcessing();
               UMLProject.get().refreshDisplay();
            }
            catch (Exception ex)
            {
               log.error ("file not valid");
               ex.printStackTrace();
            }
         }
         else
         {
            log.error ("Not a diagram");
         }
      }
      setVisible (false);
   }


   /**
    * Reads in list of all available stylesheets and stores them in a Hashtable.
    */
   private void readStylesheets()
   {
      stylesheets = new Hashtable();
      Element root = readXMLFile (stylesheetList);
      NodeList nl = root.getElementsByTagName ("stylesheet");
      importModel = new DefaultListModel();
      exportModel = new DefaultListModel();
      for (int i = 0; i < nl.getLength(); i++)
      {
         NamedNodeMap attrs = nl.item (i).getAttributes();
         String name = attrs.getNamedItem ("name").getNodeValue();
         stylesheets.put (name, attrs.getNamedItem ("file").getNodeValue());
         Node node = attrs.getNamedItem ("export");
         String isExport =  (node == null ? "false" : node.getNodeValue());
         node = attrs.getNamedItem ("import");
         String isImport =  (node == null ? "false" : node.getNodeValue());

         if ("true".equals (isExport))
         {
            exportModel.addElement (name);
         }
         if ("true".equals (isImport))
         {
            importModel.addElement (name);
         }
      }
   }
}

/*
 * $Log: XSLTransformerDialog.java,v $
 * Revision 1.28.2.2  2006/04/04 08:45:09  mtt
 * moved gui-code to Dialog
 *
 * Revision 1.28.2.1  2005/09/30 18:57:22  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
