/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.layout;

import java.util.Iterator;

import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.basic.SwingWorker;
import de.uni_paderborn.fujaba.fsa.FSAContainer;
import de.uni_paderborn.fujaba.fsa.FSAObject;
import de.uni_paderborn.fujaba.fsa.unparse.LogicUnparseInterface;
import de.uni_paderborn.fujaba.layout.classdiag.ClassdiagramLayouter;
import de.uni_paderborn.fujaba.layout.options.LayoutPreferences;
import de.uni_paderborn.fujaba.uml.*;


/**
 * Singleton class provided for invoking the layouting process
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.34 $
 */
public class MrLayout
{
   /**
    * log4j logging
    */
   final static transient Logger log = Logger.getLogger (MrLayout.class);

   // Some defines
   /**
    * UMLAttribute: '+ TREE_LAYOUT : Integer = 0'
    */
   public final static int TREE_LAYOUT = 0;

   /**
    * UMLAttribute: '+ SPRING_EMBEDDER_LAYOUT : Integer = 1'
    */
   public final static int SPRING_EMBEDDER_LAYOUT = 1;

   /**
    * UMLAttribute: '+ CONTROL_FLOW_LAYOUT : Integer = 2'
    */
   public final static int CONTROL_FLOW_LAYOUT = 2;

   /**
    * UMLAttribute: '+ CLASS_DIAGRAM_LAYOUT : Integer = 3'
    */
   public final static int CLASS_DIAGRAM_LAYOUT = 3;

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   private static MrLayout myMrLayout;


   /**
    * Constructor of Class MrLayout
    */
   private MrLayout() { } // MrLayout


   /**
    * Use this method to get a reference to singleton class MrLayout
    *
    * @return   reference to singleton instance of MrLayout
    */
   public static MrLayout get()
   {
      if (myMrLayout == null)
      {
         myMrLayout = new MrLayout();
      }
      // simply return reference to myMrLayout
      return myMrLayout;
   } // get


   /**
    * The function which starts the Layout process on the current canvas
    *
    * @param currentCanvas  the Canvas which to be layouted
    */
   public void reLayout (FSAContainer currentCanvas)
   {
      // First lets check, if there are any elements on the canvas
      // if not stop
      if ( (currentCanvas == null) ||  (currentCanvas.sizeOfChildren() == 0))
      {
         return;
      }

      // Set Options before start
      LayoutPreferences options = LayoutPreferences.get();

      whichLayout = options.getWhichLayout();
      whichActivityDiagramLayout = options.getWhichActivityDiagramLayout();

      // Now get reference of chosen Layouter
      LogicUnparseInterface lui = currentCanvas.getFSAInterface().getLogic();
      if (lui instanceof UMLActivityDiagram)
      {
         // layout on canvas of an activity diagram
         switch (whichActivityDiagramLayout)
         {
            case TREE_LAYOUT:
               myLayouter = TreeLayout.get();
               break;
            case SPRING_EMBEDDER_LAYOUT:
               myLayouter = SpringEmbedderLayout.get();
               break;
            case CONTROL_FLOW_LAYOUT:
               myLayouter = ControlFlowLayout.get();
               break;
            default:
               break;
         }
      }
      else if (lui instanceof UMLClassDiagram)
      {
         // layout on canvas of any other diagram
         switch (whichLayout)
         {
            case TREE_LAYOUT:
               myLayouter = TreeLayout.get();
               break;
            case SPRING_EMBEDDER_LAYOUT:
               myLayouter = SpringEmbedderLayout.get();
               break;
            case CLASS_DIAGRAM_LAYOUT:
               myLayouter = ClassdiagramLayouter.get();
               break;
            default:
               break;
         }
      }
      else
      {
         // layout on canvas of any other diagram
         switch (whichLayout)
         {
            case TREE_LAYOUT:
               myLayouter = TreeLayout.get();
               break;
            case SPRING_EMBEDDER_LAYOUT:
               myLayouter = SpringEmbedderLayout.get();
               break;
            default:
               break;
         }
      }

      // start layouting...as thread or as task!
      if (myLayouter != null)
      {
         if (myLayouter instanceof ControlFlowLayout)
         {
            // ...as task
            try
            {
               myLayouter.reLayout (currentCanvas);
               Iterator childrenIterator = currentCanvas.iteratorOfChildren();
               while (childrenIterator.hasNext())
               {
                  FSAObject fsaObject = (FSAObject) childrenIterator.next();
                  if (fsaObject.getLogic() instanceof ASGElement)
                  {
                      ((ASGElement) fsaObject.getLogic()).saveFSAProperties();
                  }
               }
            }
            catch (InterruptedException exept)
            {
               if (log.isDebugEnabled())
               {
                  log.debug ("Layouter: Error!!!");
               }
            }
         }
         else
         {
            // ...as thread
            LayoutThread layoutThread = new LayoutThread (myLayouter, currentCanvas);
            layoutThread.start();
         }
      }
   } // reLayout


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   private int whichLayout;


   /**
    * Get current selected Layout
    *
    * @return   Currently selected Layouter
    */
   public int getWhichLayout()
   {
      return whichLayout;
   } // getWhichLayout


   /**
    * Select an alternative Layouter
    *
    * @param whichLayout  The new Layouter to be set
    * @return             The new selected Layouter
    */
   public int setWhichLayout (int whichLayout)
   {
      if ( (whichLayout == TREE_LAYOUT) ||  (whichLayout == SPRING_EMBEDDER_LAYOUT))
      {
         this.whichLayout = whichLayout;
      }

      return this.whichLayout;
   } // setWhichLayout


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   private int whichActivityDiagramLayout;


   /**
    * Get current selected Layout for Activity Diagrams
    *
    * @return   Currently selected Layouter
    */
   public int getWhichActivityDiagramLayout()
   {
      return whichActivityDiagramLayout;
   } // getWhichActivityDiagramLayout


   /**
    * Select an alternative Layouter for Activity Diagrams
    *
    * @param whichActivityDiagramLayout  The new Layouter to be set
    * @return                            The new selected Layouter
    */
   public int setWhichActivityDiagramLayout (int whichActivityDiagramLayout)
   {
      if ( (whichActivityDiagramLayout == TREE_LAYOUT)
         ||  (whichActivityDiagramLayout == SPRING_EMBEDDER_LAYOUT)
         ||  (whichActivityDiagramLayout == CONTROL_FLOW_LAYOUT))
      {
         this.whichActivityDiagramLayout = whichActivityDiagramLayout;
      } // if

      return this.whichActivityDiagramLayout;
   } // setWhichActivityDiagramLayout


   /**
    * 1 myLayouter 1 MrLayout -------------------- AbstractLayouter
    */
   private AbstractLayouter myLayouter; // reverse attribute revMyLayouter


   /**
    * Get the myLayouter attribute of the MrLayout object
    *
    * @return   The myLayouter value
    */
   public AbstractLayouter getMyLayouter()
   {
      return myLayouter;
   } // getMyLayouter


   /**
    * Sets the myLayouter attribute of the MrLayout object
    *
    * @param myLayouter  The new myLayouter value
    */
   public void setMyLayouter (AbstractLayouter myLayouter)
   {
      if (this.myLayouter != myLayouter)
      {
         // newPartner
         if (this.myLayouter != null)
         {
            // inform old partner
            AbstractLayouter oldMyLayouter = this.myLayouter;
            this.myLayouter = null;

            oldMyLayouter.setRevMyLayouter (null);
         } // if

         this.myLayouter = myLayouter;
         if (myLayouter != null)
         {
            // inform new partner
            myLayouter.setRevMyLayouter (this);
         } // if

      } // if

   } // setMyLayouter


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   public void removeYou()
   {
      setMyLayouter (null);
   } // removeYou


   /**
    * This inner classes starts if called a new layout threat based on the
    * selected layouter
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.34 $
    */
   private class LayoutThread extends SwingWorker
   {
      /**
       * No comment provided by developer, please add a comment to improve
       * documentation.
       */
      AbstractLayouter myLayout;

      /**
       * No comment provided by developer, please add a comment to improve
       * documentation.
       */
      FSAContainer currentCanvas;


      /**
       * The constructor starts the layout process, see SwingWorker.
       *
       * @param myLayout       The desired layouter
       * @param currentCanvas  The Canvas to be layout
       */
      public LayoutThread (AbstractLayouter myLayout, FSAContainer currentCanvas)
      {
         super();
         this.myLayout = myLayout;
         this.currentCanvas = currentCanvas;
      }


      /**
       * This Method is invoked by a new thread.
       *
       * @return   Nothing to care about.
       */
      public Object construct()
      {
         try
         {
            Thread.sleep (100);
            myLayout.reLayout (currentCanvas);
            Iterator childrenIterator = currentCanvas.iteratorOfChildren();
            while (childrenIterator.hasNext())
            {
               FSAObject fsaObject = (FSAObject) childrenIterator.next();
               if (fsaObject.getLogic() instanceof ASGElement)
               {
                   ((ASGElement) fsaObject.getLogic()).saveFSAProperties();
               }
            }
         }
         catch (Exception except)
         {
            except.printStackTrace();
            if (log.isDebugEnabled())
            {
               log.debug ("Layout: Error!!!");
            }
         }
         return "";
      } // construct


      /**
       * This Method is invoked after the layouter is ready.
       */
      public void finished()
      {
         // The Canvas should be redrawn
         UMLProject.get().refreshDisplay();
      } // finished

   } // LayoutThread

}

/*
 * $Log: MrLayout.java,v $
 * Revision 1.34  2005/01/28 11:36:28  lowende
 * Removed some compile warnings.
 *
 */
