/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.layout;

import java.awt.*;
import java.util.Iterator;
import java.util.Map;

import de.uni_paderborn.fujaba.app.ProgressBarWrapper;
import de.uni_paderborn.fujaba.fsa.*;
import de.uni_paderborn.fujaba.layout.options.LayoutPreferences;
import de.upb.tools.fca.FEmptyIterator;
import de.upb.tools.fca.FHashMap;


/**
 * Class 'SpringEmbedderLayout' specified in class diagram 'TreeLayout.java diagram'.
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.29.2.1 $
 */
public class SpringEmbedderLayout extends AbstractLayouter
{
   /**
    * The children of the abstract Layouter are singletons, so define singleton instance.
    */
   private static SpringEmbedderLayout myLayouter = null;


   /**
    * Constructor of the class SpringEmbedderLayout
    */
   private SpringEmbedderLayout()
   {
      super();
   }


   /**
    * Use this method to get a reference to singleton layouter classes the class var is defined
    * in AbstractLayouter.
    *
    * @return   reference to singleton instance of Layout-Classes.
    */
   public static SpringEmbedderLayout get()
   {
      if (myLayouter == null)
      {
         myLayouter = new SpringEmbedderLayout();
      }
      // simply return reference to myLayouter
      return myLayouter;
   } // get



   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   final static int minDist = 100;


   /**
    * Layout Algorithm doing a Spring Embedder Layout
    *
    * @param currentCanvas  the Canvas which has to be layouted
    */
   private void springEmbedderLayout (FSAContainer currentCanvas)
   {
      Iterator myIter = currentCanvas.iteratorOfChildren();
      int vx;
      int vy;
      int xCorrection = 0;
      int yCorrection = 0;
      Object myElement;
      Object mySister;
      FSAObject fromFrame = null;
      FSAObject toFrame = null;
      FSABendLine myLine = null;
      int len = 0;

      // Initialize the preferred Pos of the DisFrames
      while (myIter.hasNext())
      {
         myElement = myIter.next();
         if (myElement instanceof FSABendLine)
         {
            myLine = (FSABendLine) myElement;
            if (!myLine.isVisible())
            {
               continue;
            }
            // If we have a disLine, lets look for the connected DisFrames
            fromFrame = getFromToFrame (myLine, 1);
            toFrame = getFromToFrame (myLine, 2);

            if (fromFrame == null || toFrame == null)
            {
               continue;
            }

            vx = toFrame.getLocation().x - fromFrame.getLocation().x;
            vy = toFrame.getLocation().y - fromFrame.getLocation().y;

            // now calculate the feather power for the frames to be moved
            len = (int) Math.sqrt (vx * vx + vy * vy);
            double f =  ((double)  (getPreferredLen (myLine) - len)) /
                ((double)  (len * 3));
            int dx = (int)  (f * vx);
            int dy = (int)  (f * vy);
            //log.info( "vx:" + vx + " vy:" + vy + " len:" + len + " f:" + f + " dx:" + dx + " dy:" + dy + " getLen:" + getLen(myLine));

            // line straightening
            Point leftPoint = myLine.getStartPoint();
            Point rightPoint = myLine.getEndPoint();
            int xDiff = rightPoint.x - leftPoint.x;
            double absXDiff = Math.abs (xDiff);
            int yDiff = rightPoint.y - leftPoint.y;
            double absYDiff = Math.abs (yDiff);

            if ( (absXDiff >= 20) &&  ( (absYDiff / absXDiff) < 0.2))
            {
               // should be straightened
               // if (log.isInfoEnabled()) log.info ("straightening dy " + myLine.getFrontGrab ().getRole ()
               //                    + " " + myLine.getBackGrab ().getRole ()
               //                    + " by " + yDiff /2 );
               dy -= yDiff / 2;
               if (Math.abs (yDiff) == 1)
               {
                  yCorrection = -1;
               }
            }

            if ( (absYDiff >= 20) &&  ( (absXDiff / absYDiff) < 0.2))
            {
               // should be straightened
               dx -= xDiff / 2;
               if (Math.abs (xDiff) == 1)
               {
                  xCorrection = -1;
               }
            }

            // Set the new preferred Position with the calculated feather
            // power dx, dy
            Point newPos;

            newPos = new Point (getFromPreferredPos (toFrame).x + dx + xCorrection,
               getFromPreferredPos (toFrame).y + dy + yCorrection);
            addToPreferredPos (toFrame, newPos);
            //log.info( "SpringEmb: (toFrame) " + newPos);

            newPos = new Point (getFromPreferredPos (fromFrame).x - dx,
               getFromPreferredPos (fromFrame).y - dy);
            addToPreferredPos (fromFrame, newPos);
            //log.info( "SpringEmb: (fromFrame) " + newPos);

         }
      }

      // calculate the preferred pos for the DisFrames
      myIter = currentCanvas.iteratorOfChildren();
      while (myIter.hasNext())
      {
         myElement = myIter.next();
         if (myElement instanceof FSAPanel)
         {
            if (! ((FSAObject) myElement).isVisible())
            {
               continue;
            }
            int dx = 0;
            int dy = 0;

            // Go through all connected DisFrames
            Iterator mySistersnBrothersIter = currentCanvas.iteratorOfChildren();
            while (mySistersnBrothersIter.hasNext())
            {
               mySister = mySistersnBrothersIter.next();
               // Here comes the springing
               if ( (mySister != myElement) &&  (mySister instanceof FSAPanel))
               {
                  if (! ((FSAObject) mySister).isVisible())
                  {
                     continue;
                  }
                  vx =  ((FSAObject) myElement).getLocation().x -
                      ((FSAObject) mySister).getLocation().x;

                  vy =  ((FSAObject) myElement).getLocation().y -
                      ((FSAObject) mySister).getLocation().y;

                  int lenSquare = vx * vx + vy * vy;

                  if (lenSquare == 0)
                  {
                     dx += Math.random() * 40 - 20;
                     dy += Math.random() * 40 - 20;
                  }
                  else if (lenSquare < minDist * minDist)
                  {
                     int dl = (int)  (minDist - Math.sqrt (lenSquare));
                     dx +=  (vx == 0 ? Math.random() * 40 - 20 : dl * vx /
                        Math.abs (vx)) / 2;
                     dy +=  (vy == 0 ? Math.random() * 40 - 20 : dl * vy /
                        Math.abs (vy)) / 2;
                  }
               }
            }
            double dlen = dx * dx + dy * dy;
            if (dlen > 0)
            {
               //log.info( "dlen: " + dlen);
               addToPreferredPos ((FSAObject) myElement, new Point (
                  getFromPreferredPos ((FSAObject) myElement).x + dx,
                  getFromPreferredPos ((FSAObject) myElement).y + dy));
            }
         }
      }

      // Now lets layout the canvas to the new calculated values
      myIter = currentCanvas.iteratorOfChildren();
      while (myIter.hasNext())
      {
         myElement = myIter.next();
         if (myElement instanceof FSAPanel)
         {
            { //if (!((FSAObject) myElement).isFixed ())

               // Do not move more than 5 pixels
               int x;
               // Do not move more than 5 pixels
               int
                  y;
               x =  ((FSAObject) myElement).getLocation().x +
                  Math.max (-5, Math.min (5, getFromPreferredPos ((FSAObject) myElement).x));

               y =  ((FSAObject) myElement).getLocation().y +
                  Math.max (-5, Math.min (5, getFromPreferredPos ((FSAObject) myElement).y));

                ((FSAObject) myElement).setLocation (x, y);

               if ( ((FSAObject) myElement).getLocation().x < 0)
               {
                  Point myRec =  ((FSAObject) myElement).getLocation();
                  myRec.x = 0;
                   ((FSAObject) myElement).setLocation (0, myRec.y);
               }

               if ( ((FSAObject) myElement).getLocation().y < 0)
               {
                  Point myRec =  ((FSAObject) myElement).getLocation();
                  myRec.y = 0;
                   ((FSAObject) myElement).setLocation (myRec.x, 0);
               }
            }
            addToPreferredPos ((FSAObject) myElement, new Point (getFromPreferredPos ((FSAObject) myElement).x / 2,
               getFromPreferredPos ((FSAObject) myElement).y / 2));
         }
      }
      removeAllFromPreferredPos();
      //currentCanvas.redraw ();
   } // springEmbedderLayout


   /**
    * UMLMethod: '+ reLayout (currentCanvas : DisResizeable) : Void' Method for invoking a
    * Spring Embedder Layout sequence.
    *
    * @param currentCanvas          The Canvas which should be layouted.
    * @throws InterruptedException  Exception description not provided
    */
   public void reLayout (FSAContainer currentCanvas)
       throws InterruptedException
   {
      // Set Options before start
      LayoutPreferences options = LayoutPreferences.get();

      setHorizDist (options.getHorizDist());
      setVertDist (options.getVertDist());
      setPresetAdjustment (options.getStretchFactor());

      // get preset value for number of Iterations to do
      int numberIterations = options.getNumberIterations();

      ProgressBarWrapper myProgress = new ProgressBarWrapper (getFrame (currentCanvas), "Spring embedder layout", numberIterations);

      myProgress.openWindow();
      try
      {
         while ( (numberIterations > 0) &&  (!myProgress.isAborted()))
         {
            springEmbedderLayout (currentCanvas);
            myProgress.increment ("Layouter working...");
            numberIterations--;
         }
      }
      finally
      {
         myProgress.closeWindow();
      }
   } // reLayout


   /**
    * <pre>
    *                      ------------- 0..1            Assoc             0..1
    * SpringEmbedderLayout | fsaObject |---------------------------------------> Point
    *                      ------------- springEmbedderLayout      preferredPos
    * </pre>
    */
   private FHashMap preferredPos;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfPreferredPos()
   {
      return  ( (this.preferredPos == null)
         ? 0
         : this.preferredPos.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      super.removeYou();
      removeAllFromPreferredPos();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean removeKeyFromPreferredPos (FSAObject key)
   {
      boolean changed = false;
      if ( (this.preferredPos != null) &&  (key != null))
      {
         Point tmpValue = (Point) this.preferredPos.get (key);
         if (tmpValue != null)
         {
            this.preferredPos.remove (key);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key    No description provided
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromPreferredPos (FSAObject key, Point value)
   {
      boolean changed = false;
      if ( (this.preferredPos != null) &&  (value != null) &&  (key != null))
      {
         Point oldValue = (Point) this.preferredPos.get (key);
         if (oldValue == value)
         {
            this.preferredPos.remove (key);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromPreferredPos (Point value)
   {
      boolean changed = false;
      if ( (this.preferredPos != null) &&  (value != null))
      {
         Iterator iter = this.entriesOfPreferredPos();
         Map.Entry entry;
         while (iter.hasNext())
         {
            entry = (Map.Entry) iter.next();
            if (entry.getValue() == value)
            {
               changed = changed || this.removeFromPreferredPos ((FSAObject) entry.getKey(), value);
            }
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromPreferredPos()
   {
      Iterator iter = entriesOfPreferredPos();
      Map.Entry entry;
      while (iter.hasNext())
      {
         entry = (Map.Entry) iter.next();
         removeFromPreferredPos ((FSAObject) entry.getKey(), (Point) entry.getValue());
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator keysOfPreferredPos()
   {
      return  ( (this.preferredPos == null)
         ? FEmptyIterator.get()
         : this.preferredPos.keySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfPreferredPos()
   {
      return  ( (this.preferredPos == null)
         ? FEmptyIterator.get()
         : this.preferredPos.values().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean hasKeyInPreferredPos (FSAObject key)
   {
      return  ( (this.preferredPos != null) &&
          (key != null) &&
         this.preferredPos.containsKey (key));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key    No description provided
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInPreferredPos (FSAObject key, Point value)
   {
      return  ( (this.preferredPos != null) &&
          (value != null) &&  (key != null) &&
          (this.preferredPos.get (key) == value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInPreferredPos (Point value)
   {
      return  ( (this.preferredPos != null) &&
          (value != null) &&
         this.preferredPos.containsValue (value));
   }


   /**
    * Get the fromPreferredPos attribute of the SpringEmbedderLayout object
    *
    * @param key  No description provided
    * @return     The fromPreferredPos value
    */
   public Point getFromPreferredPos (FSAObject key)
   {
      Point p =  ( (this.preferredPos == null) ||  (key == null))
         ? null
         : (Point) this.preferredPos.get (key);
      return  (p == null ? new Point (0, 0) : p);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator entriesOfPreferredPos()
   {
      return  ( (this.preferredPos == null)
         ? FEmptyIterator.get()
         : this.preferredPos.entrySet().iterator());
   }


   /**
    * Access method for an one to n association.
    *
    * @param key    The object added.
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToPreferredPos (FSAObject key, Point value)
   {
      boolean changed = false;
      if ( (value != null) &&  (key != null))
      {
         if (this.preferredPos == null)
         {
            this.preferredPos = new FHashMap(); // or FTreeMap
         }
         Point oldValue = (Point) this.preferredPos.put (key, value);
         if (oldValue != value)
         {
            changed = true;
         }
      }
      return changed;
   }


   /**
    * Access method for an one to n association.
    *
    * @param entry  The object added.
    * @return       No description provided
    */
   public boolean addToPreferredPos (Map.Entry entry)
   {
      return addToPreferredPos ((FSAObject) entry.getKey(), (Point) entry.getValue());
   }

}

/*
 * $Log: SpringEmbedderLayout.java,v $
 * Revision 1.29.2.1  2005/09/30 18:57:24  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
