/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.mpEdit;

import java.awt.*;
import java.io.*;
import java.util.*;
import java.util.zip.GZIPInputStream;

import javax.swing.text.*;

import org.apache.log4j.Logger;


/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.35.2.2 $
 */
class DocMan implements DocInterface
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (DocMan.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private mpEDIT mpEdit; // the boss

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private ResourceBundle strings;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Properties props; // the user preferences

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Hilite hilite;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Ruler ruler;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private LineMan lines; // the document data

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private DocOwnerInterface docOwner;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Vector frames; // the document frames

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Vector textMenus; // attached edit menu handlers

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private File file; // the document file

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JournalItem undo_list; // where we are in undo

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JournalItem redo_list; // where we are in redo

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean dirty;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean neverNamed;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean privateProps;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean highlighting;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean hiding;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Boolean saveReadOnly = null; //if hiding is enabled, readOnly is always set to true, this saves the original state
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean readOnly;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String lineSeparator;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int REPLACE_LINE = 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int SPLIT_LINE = 2;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int JOIN_LINE = 3;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int INSERT = 4;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int DELETE = 5;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int SWAP_LINES = 6;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int DELETE_LINE = 7;


   /**
    * Constructor for class DocMan
    *
    * @param mpe  No description provided
    * @param str  No description provided
    * @param pr   No description provided
    */
   public DocMan (mpEDIT mpe, ResourceBundle str, Properties pr)
   {
      lines = new LineMan();
      frames = new Vector (4, 4);
      textMenus = new Vector (4, 4);

      lines.addElement ("");

      mpEdit = mpe;
      strings = str;
      props = pr;

      hilite = new Hilite (lines, 0, true);
      ruler = new Ruler (this);

      lineSeparator = System.getProperty ("line.separator");

      undo_list = redo_list = null;

      setProperty ("mpEDIT.hide", String.valueOf (false));
   }


   // methods implementing DocInterface

   /**
    * Sets the owner attribute of the DocMan object
    *
    * @param o  The new owner value
    */
   public void setOwner (DocOwnerInterface o)
   {
      docOwner = o;
   }


   /**
    * Get the owner attribute of the DocMan object
    *
    * @return   The owner value
    */
   public DocOwnerInterface getOwner()
   {
      return docOwner;
   }


   /**
    * Closes this document. If the "bail" flag is true changes will be discarded, otherwise
    * the user will be queried.
    *
    * @param bail  Exit immediately, discarding changes.
    */
   public void closeDoc (boolean bail)
   {
      int i;
      TextFrame textFrame;

      i = frames.size();

      while (--i >= 0)
      {
         textFrame = (TextFrame) frames.elementAt (i);
         if (bail ||  (i > 1) || !dirty)
         {
            doCloseFrame (textFrame);
         }
         else
         { // show dialog for last frame (if not 'bail', etc.)

            DirtyDialog ab = new DirtyDialog (this,
               textFrame,
               strings,
               strings.getString ("DialogDirty"),
               file.getName());
            ab.show();
         }
      }
   }


   /**
    * Writes out this document.
    */
   public void saveDoc()
   {
      fileSave (anyFrame());
   }


   /**
    * Writes out this document.
    *
    * @param pathname  New document name (full path).
    */
   public void saveAsDoc (String pathname)
   {
      file = new File (pathname);
      setTitles();
      neverNamed = false;
      write (anyFrame(), file);
      scan();
   }


   /**
    * Returns the full path to this document.
    *
    * @return   Document name (full path).
    */
   public String getPathname()
   {
      return file.getPath();
   }


   /**
    * Returns the filename (no path) for this document.
    *
    * @return   Document name (no path).
    */
   public String getFilename()
   {
      return file.getName();
   }


   /**
    * Sets the filename for this document.
    *
    * @param fileName  Document name .
    */
   public void setFilename (String fileName)
   {
      file = new File (fileName);
      setTitles();
      neverNamed = false;
      //      write (anyFrame (), file);
      scan();
   }


   /**
    * Fetch the dirty flag.
    *
    * @return   True if the file has been changed since being written.
    */
   public boolean isDirty()
   {
      return dirty;
   }


   /**
    * This sets the dirty flag. Use this method with special care, if the flag is set to false,
    * there will be no question about saving.
    *
    * @param d  The new dirty value
    */

   public void setDirty (boolean d)
   {
      dirty = d;
   }


   /**
    * Get the line count for a document.
    *
    * @return   The lineCount value
    */
   public int getLineCount()
   {
      return lines.size();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   int nextTag = 123;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param line  No description provided
    * @return      No description provided
    */
   public TagLine tagLine (int line)
   {
      LineInfo li = lines.getLineInfo (line);
      li.tagValue = nextTag++;
      TagLine tag = new TagLine();
      tag.tagValue = li.tagValue;
      tag.lastLine = line;
      return tag;
   }


   /**
    * Set a TagLine (including a tag color), used to track lines even when lines have been
    * inserted or deleted elsewhere in the document.
    *
    * @param color  The Color to display (pass null to clear).
    * @param line   No description provided
    * @return       The TagLine for a given line.
    */
   public TagLine tagLine (int line, Color color)
   {
      LineInfo li = lines.getLineInfo (line);
      li.tagValue = nextTag++;
      li.tagColor = color;
      TagLine tag = new TagLine();
      tag.tagValue = li.tagValue;
      tag.lastLine = line;
      updateFrames (line, line);
      return tag;
   }


   /**
    * Get the current line number for a TagLine.
    *
    * @param tag  No description provided
    * @return     The TagLine for a given line.
    */
   public int lineFromTag (TagLine tag)
   {
      int i = tag.lastLine;

      LineInfo li = lines.getLineInfo (i);

      // first check the old location

      if (li.tagValue == tag.tagValue)
      {
         return i;
      }

      // get fancy later

      i = lines.size();

      while (--i >= 0)
      {
         li = lines.getLineInfo (i);
         if (li.tagValue == tag.tagValue)
         {
            tag.lastLine = i;
            return i;
         }
      }

      return -1;
   }


   /**
    * Bring forward any view window and scroll to the desired line.
    *
    * @param tag  No description provided
    * @return     No description provided
    */
   public boolean showLine (TagLine tag)
   {
      int line = lineFromTag (tag);

      if (line < 0)
      {
         return false;
      }

      TextFrame f = anyFrame();
      f.showLine (line);
      //      f.toFront ();
      return true;
   }


   /**
    * Get the text for a line.
    *
    * @param tag  No description provided
    * @return     The text for the line (null if TagLine not found).
    */
   public String getLine (TagLine tag)
   {
      int line = lineFromTag (tag);

      if (line < 0)
      {
         return null;
      }

      return lines.getString (line);
   }


   /**
    * Set the text for a line.
    *
    * @param tag  The new line value
    * @param s    The new line value
    * @return     Success.
    */
   public boolean setLine (TagLine tag, String s)
   {
      int line = lineFromTag (tag);

      if (line < 0)
      {
         return false;
      }

      // set the text

      lines.setString (s, line);

      // update

      updateFrames (line, line);
      dirty = true;

      return true;
   }


   /**
    * Insert a line of text before the TagLine.
    *
    * @param tag  The object added.
    * @param s    The object added.
    * @return     Success.
    */
   public boolean addLine (TagLine tag, String s)
   {
      int line = lineFromTag (tag);

      if (line < 0)
      {
         return false;
      }

      // set the text

      lines.insertElementAt (s, line);

      // update

      updateFrames (line, line + 1);
      dirty = true;

      return true;
   }


   /**
    * Delete a line.
    *
    * @param tag  No description provided
    * @return     Success.
    */
   public boolean deleteLine (TagLine tag)
   {
      int line = lineFromTag (tag);

      if (line < 0)
      {
         return false;
      }

      // set the text

      lines.removeElementAt (line);

      // update

      updateFrames (line, line + 1);
      dirty = true;

      return true;
   }


   /**
    * Copies the content of this doc into a swing-Document.
    *
    * @return   The document value
    */
   public Document getDocument()
   {
      Document doc = new PlainDocument();
      int i;

      try
      {
         for (i = getLineCount() - 1; i >= 0; i--)
         {
            TagLine tag = tagLine (i);
            doc.insertString (0, getLine (tag) + "\n", null);
         }
      }
      catch (BadLocationException e)
      {
         e.printStackTrace();
      }

      return doc;
   }

//
   // utility functions, manage frames and canvases attached to this document
   //

   /**
    * Sets the untitled attribute of the DocMan object
    *
    * @param count  The new untitled value
    */
   public void setUntitled (int count)
   {
      String filename = strings.getString ("UntitledDoc") +  (count == 1 ? "" : " " + count);
      file = new File (filename);
      setTitles();
      neverNamed = true;
   }


   /**
    * Sets the titles attribute of the DocMan object
    */
   public void setTitles()
   {
      int i;
      int
         max;
      String name = file.toString();

      max = frames.size();

      for (i = 0; i < max; i++)
      {
         if (max > 1)
         {
            name = file.toString() + ":" +  (i + 1);
         }

          ((TextFrame) frames.elementAt (i)).setTitle (name);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param point  No description provided
    * @param size   No description provided
    * @return       No description provided
    */
   public TextFrame newFrame (Point point, Dimension size)
   {
      TextFrame textFrame = new TextFrame (mpEdit, strings, props, this, ruler);
      textFrame.setLocation (point);
      textFrame.setSize (size);
      frames.addElement (textFrame);
      updateMenus();
      return textFrame;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param textFrame  No description provided
    */
   public void closeFrame (TextFrame textFrame)
   {
      int max;

      max = frames.size();

      if ( (max > 1) || !dirty)
      {
         doCloseFrame (textFrame);
      }
      else
      {
         DirtyDialog ab = new DirtyDialog (this,
            textFrame,
            strings,
            strings.getString ("DialogDirty"),
            file.getName());
         ab.show();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param textFrame  No description provided
    */
   public void doCloseFrame (TextFrame textFrame)
   {
      int i;
      int max;

      max = frames.size();

      for (i = 0; i < max; i++)
      {
         if (textFrame == (TextFrame) frames.elementAt (i))
         {
            frames.removeElementAt (i);
            textFrame.dispose();
            max--;
            break;
         }
      }

      if (max == 0)
      {
         mpEdit.closeDoc (this);
      }
      else
      {
         setTitles();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void newDoc()
   {
      int i;
      int
         max;
      lines.removeAllElements();
      lines.addElement ("");
      max = frames.size();
      for (i = 0; i < max; i++)
      {
         TextFrame f = (TextFrame) frames.elementAt (i);
         f.clearCanvas();
         f.redoCanvas();
      }
      setUntitled (mpEdit.getUntitled());
      clear_undo();
      updateMenus();
   }


   /**
    * Get the busy attribute of the DocMan object
    *
    * @return   The busy value
    */
   public boolean isBusy()
   {
      return dirty || !neverNamed;
   }


   /**
    * Get the readOnly attribute of the DocMan object
    *
    * @return   The readOnly value
    */
   public boolean isReadOnly()
   {
      return readOnly;
   }


   /**
    * Sets the readOnly attribute of the DocMan object
    *
    * @param aReadOnly  The new readOnly value
    */
   public void setReadOnly (boolean aReadOnly)
   {
      if (aReadOnly != readOnly)
      {
         readOnly = aReadOnly;
         int max = frames.size();
         int i;
         for (i = 0; i < max; i++)
         {
             ((TextFrame) frames.elementAt (i)).setReadOnly (readOnly);
         }
      }
   }



//
   // utility functions, high-level routines for reading and writing documents
   //

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public TextFrame anyFrame()
   {
      return (TextFrame) frames.firstElement();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param textFrame  No description provided
    * @param filename   No description provided
    */
   public void openDoc (TextFrame textFrame, String filename)
   {
      int i;
      int
         max;

      file = new File (filename);
      if (file.isFile())
      {
         neverNamed = false;
         max = frames.size();
         for (i = 0; i < max; i++)
         {
             ((TextFrame) frames.elementAt (i)).clearCanvas();
         }
         setTitles();
         ruler.invalidate (0, 1000000);
         read (textFrame, filename);
         for (i = 0; i < max; i++)
         {
             ((TextFrame) frames.elementAt (i)).redoCanvas();
         }
         scan();
      }

      textFrame.getCanvas().requestFocus();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param textFrame  No description provided
    */
   public void filePrint (TextFrame textFrame)
   {
      PrintMan pm = new PrintMan (textFrame, this, file.getName(), getFont(), getTabSize());
      pm.start();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param textFrame  No description provided
    * @return           No description provided
    */
   public boolean fileSave (TextFrame textFrame)
   {
      return doSaveFile (textFrame, false);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param textFrame  No description provided
    */
   public void fileSaveAs (TextFrame textFrame)
   {
      doSaveFile (textFrame, true);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param textFrame  No description provided
    * @param As         No description provided
    * @return           No description provided
    */
   private boolean doSaveFile (TextFrame textFrame, boolean As)
   {
      boolean ok = true;
      String filename = null;

      if (neverNamed || As)
      {
         filename = getFileForSave (textFrame);
         if (filename == null)
         {
            ok = false;
         }
         else
         {
            file = new File (filename);
            setTitles();
            neverNamed = false;
         }
      }

      if (ok)
      {
         write (textFrame, file);
         scan();
         if (docOwner != null)
         {
            if (As)
            {
               docOwner.savedAsDoc (this, filename);
            }
            else
            {
               docOwner.savedDoc (this);
            }
         }
      }

      return ok;
   }


   /**
    * Get the fileForSave attribute of the DocMan object
    *
    * @param textFrame  No description provided
    * @return           The fileForSave value
    */
   private String getFileForSave (TextFrame textFrame)
   {
      String prompt;
      String filename;
      String dirname;
      String pathname;

      prompt = strings.getString ("DialogSaveAs");

      FileDialog d = new FileDialog (textFrame, prompt, FileDialog.SAVE);

      d.setFile (file.getName());

      // for a never-named file, pick up the last directory
      // a new file was saved to

      if (neverNamed)
      {
         dirname = mpEdit.getSaveDir();
      }
      else
      {
         dirname = file.getParent();
      }

      d.setDirectory (dirname);

      d.setVisible (true);

      filename = d.getFile();
      dirname = d.getDirectory();
      pathname = dirname + filename;

      d.dispose();

      if (filename != null)
      {
         // remeber where a new file was saved to
         // or should it be where any file was saved ?

         if (neverNamed)
         {
            mpEdit.setSaveDir (dirname);
         }

         return pathname;
      }
      else
      {
         return null;
      }
   }

//
   // utility functions, low-level routines for reading and writing documents
   //

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final int SLOW_READ = 2000;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final int SLOW_WRITE = 10000;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param textFrame  No description provided
    * @param filename   No description provided
    */
   public void read (TextFrame textFrame, String filename)
   {
      Progress readProgress = null;
      String line;

      long max = file.length();
      long tens = max / 10;
      long part = 0;
      long all = 0;
      boolean show_progress = max > SLOW_READ;

      if (show_progress)
      {
         readProgress = new Progress (textFrame, "Reading file ... ");
         readProgress.show();
      }

      lines.freshVectors ((int) Math.max (max / 20, 100), (int) Math.max (max / 100, 100));

      try
      {
         BufferedReader br = null;
         FileReader fr = new FileReader (filename);

         if (filename.endsWith (".zip"))
         {
            br = new BufferedReader (new InputStreamReader
                (new GZIPInputStream (new FileInputStream (filename))), 32768);
         }
         else
         {
            br = new BufferedReader (fr);
         }

         while ( (line = br.readLine()) != null)
         {
            //log.info( "" + Runtime.getRuntime().freeMemory() + "  " + Runtime.getRuntime().totalMemory() );

            if (show_progress)
            {
               part += line.length();
               if (part > tens)
               {
                  all += part;
                  readProgress.update ((int)  (100 * all / max));
                  part = 0;
               }
            }
            lines.addElement (line);
         }

         br.close();
         fr.close();
      }

      catch (IOException e)
      {
         if (log.isInfoEnabled())
         {
            log.info ("Error - could not read file");
         }
      }
      if (readProgress != null)
      {
         readProgress.dispose();
      }
      if (lines.size() < 1)
      {
         lines.addElement ("");
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param textFrame  No description provided
    * @param file       No description provided
    */
   public void write (TextFrame textFrame, File file)
   {
      Progress writeProgress = null;

      long max = lines.size();
      long tens = max / 10;
      long part = 0;
      long all = 0;
      boolean show_progress = max > SLOW_WRITE;

      if (show_progress)
      {
         writeProgress = new Progress (textFrame, "Writing file ... ");
         writeProgress.show();
      }

      try
      {
         FileWriter fw = new FileWriter (file);
         BufferedWriter bw = new BufferedWriter (fw);

         for (int i = 0; i < max; i++)
         {
            String line = lines.getString (i);
            bw.write (line, 0, line.length());
            bw.newLine();
            if (show_progress)
            {
               part++;
               if (part > tens)
               {
                  all += part;
                  writeProgress.update ((int)  (100 * all / max));
                  part = 0;
               }
            }
         }

         bw.close();
         fw.close();

         dirty = false;
      }
      catch (IOException e)
      {
         if (log.isInfoEnabled())
         {
            log.info ("Error - could not write file");
         }
      }
      if (writeProgress != null)
      {
         writeProgress.dispose();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void clear_undo()
   {
      undo_list = redo_list = null;
   }


   //
   // utility functions, manage syntax highlighting
   //

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void scan()
   {
      int i;
      TextFrame textFrame;
      TextCanvas textCanvas;

      highlighting = Boolean.valueOf (getProperty ("mpEDIT.hilite")).booleanValue();

      if (saveReadOnly == null)
      {
         saveReadOnly = Boolean.valueOf (isReadOnly());
      }

      if (highlighting && mpEdit.getHighlightingEnabled())
      { //check if highlighting is enabled by the caller of mpEDIT (e.g. disable for Commentary)

         if (hiding)
         {
            HiliteLife myHilite = new HiliteLife (lines, getTabSize(), false);
            myHilite.scan (lines.size() - 1); //scan the whole doc instead of the visible part only
            if (!myHilite.isHideLevelEmpty())
            {
               if (log.isInfoEnabled())
               {
                  log.info ("mpEDIT: Not all hide tags are closed!");
               }
            }
            hilite = myHilite;
            setReadOnly (true);
         }
         else
         {
            hilite = new HiliteJava (lines, getTabSize(), false);
            setReadOnly (saveReadOnly.booleanValue());
            hilite.scan (getHighest());
         }
      }
      else
      {
         hilite = new Hilite (lines, 0, true);
         highlighting = false;
         hiding = false;
         setReadOnly (saveReadOnly.booleanValue());
         hilite.scan (getHighest());
      }

      i = frames.size();

      while (i-- > 0)
      {
         textFrame = (TextFrame) frames.elementAt (i);
         textCanvas = textFrame.getCanvas();
         textCanvas.repaint();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param highest  No description provided
    */
   public void extendHilite (int highest)
   {
      int temp;

      temp = lines.size() - 1;

      if (highest > temp)
      {
         highest = temp;
      }

      hilite.extendScan (highest);
   }


   /**
    * Get the hilite attribute of the DocMan object
    *
    * @return   The hilite value
    */
   public Hilite getHilite()
   {
      return hilite;
   }


   /**
    * Get the textColor attribute of the DocMan object
    *
    * @return   The textColor value
    */
   public Color getTextColor()
   {
      return hilite.getTextColor();
   }


   /**
    * Get the textXColor attribute of the DocMan object
    *
    * @return   The textXColor value
    */
   public Color getTextXColor()
   {
      return hilite.getTextXColor();
   }


   /**
    * Get the commentColor attribute of the DocMan object
    *
    * @return   The commentColor value
    */
   public Color getCommentColor()
   {
      return hilite.getCommentColor();
   }


   /**
    * Get the commentXColor attribute of the DocMan object
    *
    * @return   The commentXColor value
    */
   public Color getCommentXColor()
   {
      return hilite.getCommentXColor();
   }


   /**
    * Get the keywordColor attribute of the DocMan object
    *
    * @return   The keywordColor value
    */
   public Color getKeywordColor()
   {
      return hilite.getKeywordColor();
   }


   /**
    * Get the keywordXColor attribute of the DocMan object
    *
    * @return   The keywordXColor value
    */
   public Color getKeywordXColor()
   {
      return hilite.getKeywordXColor();
   }


   /**
    * Get the quoteColor attribute of the DocMan object
    *
    * @return   The quoteColor value
    */
   public Color getQuoteColor()
   {
      return hilite.getQuoteColor();
   }


   /**
    * Get the quoteXColor attribute of the DocMan object
    *
    * @return   The quoteXColor value
    */
   public Color getQuoteXColor()
   {
      return hilite.getQuoteXColor();
   }


   /**
    * Get the hideColor attribute of the DocMan object
    *
    * @return   The hideColor value
    */
   public Color getHideColor()
   {
      return hilite.getHideColor();
   }


   /**
    * Get the hideXColor attribute of the DocMan object
    *
    * @return   The hideXColor value
    */
   public Color getHideXColor()
   {
      return hilite.getHideXColor();
   }

//
   // utility functions, update canvases following changes to document
   //

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param first  No description provided
    * @param last   No description provided
    */
   public void updateFrames (int first, int last)
   {
      int i;
      TextFrame textFrame;
      TextCanvas textCanvas;

      ruler.invalidate (first, last);
      last = hilite.update (first, last, getHighest());

      i = frames.size();

      while (i > 0)
      {
         i--;
         textFrame = (TextFrame) frames.elementAt (i);
         textCanvas = textFrame.getCanvas();
         textCanvas.legalizeCursor();
         textCanvas.linesChanged (first, last);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void legalizeCursors()
   {
      int i;
      int
         max;

      max = frames.size();
      for (i = 0; i < max; i++)
      {
         TextFrame f = (TextFrame) frames.elementAt (i);
         f.legalizeCursor();
      }
   }

//
   // utility function, get highest visible line in any canvas
   //

   /**
    * Get the highest attribute of the DocMan object
    *
    * @return   The highest value
    */
   private int getHighest()
   {
      int i;
      int
         temp;
      int
         highest;
      TextFrame textFrame;
      TextCanvas textCanvas;

      i = frames.size();
      highest = 0;

      while (i-- > 0)
      {
         textFrame = (TextFrame) frames.elementAt (i);
         textCanvas = textFrame.getCanvas();
         temp = textCanvas.getHighest();
         if (temp > highest)
         {
            highest = temp;
         }
      }

      temp = lines.size() - 1;

      if (highest > temp)
      {
         highest = temp;
      }

      return highest;
   }

//
   // utility functions, access properties for this document
   //

   /**
    * Get the property attribute of the DocMan object
    *
    * @param p  No description provided
    * @return   The property value
    */
   public String getProperty (String p)
   {
      return props.getProperty (p);
   }


   /**
    * Sets the property attribute of the DocMan object
    *
    * @param p  The new property value
    * @param v  The new property value
    */
   public void setProperty (String p, String v)
   {
      props.put (p, v);
   }


   /**
    * Sets the properties attribute of the DocMan object
    *
    * @param p  The new properties value
    */
   public void setProperties (Properties p)
   {
      props = p;
      privateProps = false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void splitProperties()
   {
      if (!privateProps)
      {
         props = new Properties (props);
         privateProps = true;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param global  No description provided
    */
   public void updateProperties (boolean global)
   {
      if (global)
      {
         mpEdit.updateProperties (props);
      }
      else
      {
         applyProperties();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void applyProperties()
   {
      int i;
      TextFrame textFrame;
      TextCanvas textCanvas;

      scan();

      i = frames.size();

      while (i-- > 0)
      {
         textFrame = (TextFrame) frames.elementAt (i);
         textCanvas = textFrame.getCanvas();
         textCanvas.applyProperties();
      }
   }


   /**
    * Get the fontStyle attribute of the DocMan object
    *
    * @return   The fontStyle value
    */
   public String getFontStyle()
   {
      String style = props.getProperty ("mpEDIT.font.style");

      switch (Integer.valueOf (style).intValue())
      {

         case Font.PLAIN:
            return strings.getString ("ChoicePlain");
         case Font.BOLD:
            return strings.getString ("ChoiceBold");
         case Font.ITALIC:
            return strings.getString ("ChoiceItalic");
         case Font.BOLD | Font.ITALIC:
            return strings.getString ("ChoiceBoldItalic");
         default:
            break;
      }

      return strings.getString ("ChoicePlain");
   }


   /**
    * Sets the fontStyle attribute of the DocMan object
    *
    * @param style  The new fontStyle value
    */
   public void setFontStyle (String style)
   {
      int s = Font.PLAIN;

      if (style.equals (strings.getString ("ChoiceBold")))
      {
         s = Font.BOLD;
      }
      else if (style.equals (strings.getString ("ChoiceItalic")))
      {
         s = Font.ITALIC;
      }
      else if (style.equals (strings.getString ("ChoiceBoldItalic")))
      {
         s = Font.BOLD | Font.ITALIC;
      }

      props.put ("mpEDIT.font.style", String.valueOf (s));
   }


   /**
    * Get the font attribute of the DocMan object
    *
    * @return   The font value
    */
   public Font getFont()
   {
      String name;
      String style;
      String size;

      name = props.getProperty ("mpEDIT.font.name"); // add defaults

      style = props.getProperty ("mpEDIT.font.style");
      size = props.getProperty ("mpEDIT.font.size");

      return new Font (name,
         Integer.valueOf (style).intValue(),
         Integer.valueOf (size).intValue());
   }


   /**
    * Get the tabSize attribute of the DocMan object
    *
    * @return   The tabSize value
    */
   public int getTabSize()
   {
      int tabs;

      try
      {
         tabs = Integer.valueOf (props.getProperty ("mpEDIT.tab.size")).intValue();
      }

      catch (NumberFormatException e)
      {
         props.put ("mpEDIT.tab.size", "4");
         tabs = 4;
      }

      return tabs;
   }


   /**
    * Get the hideLevel attribute of the DocMan object
    *
    * @return   The hideLevel value
    */
   public int getHideLevel()
   {
      int hideLevel;

      try
      {
         hideLevel = Integer.valueOf (props.getProperty ("mpEDIT.hide.level")).intValue();
      }

      catch (NumberFormatException e)
      {
         props.put ("mpEDIT.hide.level", "0");
         hideLevel = 0;
      }

      return hideLevel;
   }


//
   // utility functions, access data in "lines" structure
   //

   /**
    * Get the line attribute of the DocMan object
    *
    * @param i  No description provided
    * @return   The line value
    */
   public String getLine (int i)
   {
      return lines.getString (i);
   }


   /**
    * Get the lineInfo attribute of the DocMan object
    *
    * @param i  No description provided
    * @return   The lineInfo value
    */
   public LineInfo getLineInfo (int i)
   {
      return lines.getLineInfo (i);
   }

//
   // utility functions, update menus to reflect undo/redo and copy/paste status
   //

   /**
    * Access method for an one to n association.
    *
    * @param e  The object added.
    */
   public void addTextMenu (TextMenu e)
   {
      textMenus.addElement (e);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param un  No description provided
    * @param re  No description provided
    */
   public void updateUndoItems (boolean un, boolean re)
   {
      int i;
      int
         max;

      max = frames.size();
      for (i = 0; i < max; i++)
      {
         TextCanvas tc =  ((TextFrame) frames.elementAt (i)).getCanvas();
         tc.updateUndoActions (un, re);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void updateMenus()
   {
      boolean got_undo;
      boolean got_redo;

      got_undo = undo_list != null;
      got_redo = redo_list != null;

      updateUndoItems (got_undo, got_redo);
   }

//
   // The following routines implement high-level actions for undo, and redo
   //

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param textFrame  No description provided
    */
   public void undo (TextFrame textFrame)
   {
      int first;
      int
         last;
      JournalItem temp;
      TextPosition tp;

      if (undo_list != null)
      {
         temp = undo_list;
         undo_list = temp.next;
         first = last = temp.line;

         switch (temp.action)
         {
            case REPLACE_LINE:
               redo_line (temp);
               break;
            case SPLIT_LINE:
               split_or_join (temp, true);
               last++;
               break;
            case JOIN_LINE:
               split_or_join (temp, false);
               break;
            case INSERT:
               copy_or_cut (temp, true);
               break;
            case DELETE:
               tp = insert (temp);
               last = tp.line;
               break;
            case SWAP_LINES:
               internal_swap_lines (temp.line, temp.eline);
               last++;
               break;
            case DELETE_LINE:
               insert (temp);
               last++;
               break;
            default:
               ;
         }

         updateFrames (first, last);

         tp = new TextPosition (temp.line, temp.column);
         textFrame.setPos (tp);

         legalizeCursors();

         temp.next = redo_list;
         redo_list = temp;
      }

      updateMenus();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param textFrame  No description provided
    */
   public void redo (TextFrame textFrame)
   {
      int first;
      int
         last;
      JournalItem temp;
      TextPosition tp;
      int line;
      int
         column;

      if (redo_list != null)
      {
         temp = redo_list;
         redo_list = temp.next;
         first = last = line = temp.line;
         column = temp.column;

         switch (temp.action)
         {
            case REPLACE_LINE:
               column = temp.text.length();
               redo_line (temp);
               break;
            case SPLIT_LINE:
               split_or_join (temp, false);
               column = 0;
               break;
            case JOIN_LINE:
               split_or_join (temp, true);
               last++;
               break;
            case INSERT:
               tp = insert (temp);
               last = tp.line;
               line = temp.eline;
               column = temp.ecolumn;
               break;
            case DELETE:
               copy_or_cut (temp, true);
               break;
            case SWAP_LINES:
               internal_swap_lines (temp.line, temp.eline);
               last++;
               break;
            case DELETE_LINE:
               lines.removeElementAt (temp.line);
               break;
            default:
               ;
         }

         updateFrames (first, last);

         tp = new TextPosition (line, column);
         textFrame.setPos (tp);

         legalizeCursors();

         temp.next = undo_list;
         undo_list = temp;
      }

      updateMenus();
   }

//
   // The following routines implement low-level actions for do, undo, and redo
   //

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param line    No description provided
    * @param column  No description provided
    * @param c       No description provided
    */
   public void insert_char (int line, int column, char c)
   {
      String s;
      s = lines.getString (line);
      remember_line (line, column, s);
      s = s.substring (0, column) + c + s.substring (column, s.length());
      lines.setString (s, line);
      updateFrames (line, line);
      dirty = true;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param line    No description provided
    * @param column  No description provided
    */
   public void delete_char (int line, int column)
   {
      String s;
      s = lines.getString (line);
      remember_line (line, column, s);
      s = s.substring (0, column) + s.substring (column + 1, s.length());
      lines.setString (s, line);
      updateFrames (line, line);
      dirty = true;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param line    No description provided
    * @param column  No description provided
    * @param s       No description provided
    */
   private void remember_line (int line, int column, String s)
   {
      boolean new_line = true;

      if (undo_list != null)
      {
         new_line =  (undo_list.action != REPLACE_LINE) ||  (undo_list.line != line);
      }

      if (new_line)
      {
         JournalItem new_journal = new JournalItem();
         new_journal.action = REPLACE_LINE;
         new_journal.line = line;
         new_journal.column = column;
         new_journal.text = s;
         new_journal.next = undo_list;
         undo_list = new_journal;
      }

      redo_list = null;
      updateMenus();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param i  No description provided
    */
   private void redo_line (JournalItem i)
   {
      String s;
      int line = i.line;
      s = lines.getString (line);
      lines.setString (i.text, line);
      i.text = s;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param line    No description provided
    * @param column  No description provided
    */
   public void split_line (int line, int column)
   {
      JournalItem new_journal = new JournalItem();
      new_journal.action = SPLIT_LINE;
      new_journal.line = line;
      new_journal.column = column;
      split_or_join (new_journal, false);
      new_journal.next = undo_list;
      undo_list = new_journal;
      redo_list = null;
      updateMenus();
      updateFrames (line, line + 1);
      dirty = true;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param line    No description provided
    * @param column  No description provided
    */
   public void join_line (int line, int column)
   {
      JournalItem new_journal = new JournalItem();
      new_journal.action = JOIN_LINE;
      new_journal.line = line;
      new_journal.column = column;
      split_or_join (new_journal, true);
      new_journal.next = undo_list;
      undo_list = new_journal;
      redo_list = null;
      updateMenus();
      updateFrames (line, line + 1);
      dirty = true;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param i     No description provided
    * @param join  No description provided
    */
   public void split_or_join (JournalItem i, boolean join)
   {
      String s;
      int line = i.line;
      int column = i.column;

      if (join)
      {
         s = lines.getString (line);
         s = s.concat (lines.getString (line + 1));
         lines.setString (s, line);
         lines.removeElementAt (line + 1);
         hilite.lineRemoved (line + 1);
      }
      else
      {
         s = lines.getString (line);
         lines.setString (s.substring (0, column), line);
         lines.insertElementAt (s.substring (column, s.length()), ++line);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param line    No description provided
    * @param column  No description provided
    * @param s       No description provided
    * @param update  No description provided
    * @return        No description provided
    */
   public TextPosition insert_section (int line, int column, String s, boolean update)
   {
      TextPosition tp;

      // start creating journal item

      JournalItem new_journal = new JournalItem();
      new_journal.action = INSERT;
      new_journal.text = s;
      new_journal.line = line; // starting line and column

      new_journal.column = column;

      // insert the text

      tp = insert (new_journal);

      // finish creating journal entry

      new_journal.eline = tp.line; // updated line and column

      new_journal.ecolumn = tp.column;
      new_journal.next = undo_list;
      undo_list = new_journal;
      redo_list = null;
      updateMenus();
      if (update)
      {
         updateFrames (line, tp.line);
      }
      dirty = true;

      return tp;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param i  No description provided
    * @return   No description provided
    */
   public TextPosition insert (JournalItem i)
   {
      int line = i.line;
      int column = i.column;
      String s = i.text;

      int charCt;
      int saveCt;
      int charMax;
      char c;
      char
         c2;
      String s2 = null;

      // insert the text

      charMax = s.length();
      charCt = saveCt = 0;

      while (charCt < charMax)
      {
         c = s.charAt (charCt);
         charCt++;
         if ( (c == '\r') ||  (c == '\n'))
         {
            s2 = lines.getString (line);
            lines.setString (s2.substring (0, column) + s.substring (saveCt, charCt - 1), line);
            lines.insertElementAt (s2.substring (column, s2.length()), ++line);
            column = 0;
            if (charCt < charMax)
            {
               c2 = s.charAt (charCt);
               if ( ( (c == '\r') &&  (c2 == '\n')) ||  ( (c2 == '\r') &&  (c == '\n')))
               {
                  charCt++;
               }
            }
            saveCt = charCt;
         }
      }

      if (saveCt < charCt)
      {
         s2 = lines.getString (line);
         s = s.substring (saveCt, charCt);
         if (column == 0)
         {
            s2 = s + s2;
         }
         else
         {
            s2 = s2.substring (0, column) + s + s2.substring (column, s2.length());
         }
         column += s.length();
         lines.setString (s2, line);
      }
      return new TextPosition (line, column);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param line     No description provided
    * @param column   No description provided
    * @param eline    No description provided
    * @param ecolumn  No description provided
    * @param cut      No description provided
    * @return         No description provided
    */
   public String delete_section (int line, int column, int eline, int ecolumn, boolean cut)
   {
      String text;

      JournalItem new_journal = new JournalItem();
      new_journal.action = DELETE;
      new_journal.line = line;
      new_journal.column = column;
      new_journal.eline = eline;
      new_journal.ecolumn = ecolumn;

      text = copy_or_cut (new_journal, cut);

      if (cut)
      {
         new_journal.text = text;
         new_journal.next = undo_list;
         undo_list = new_journal;
         redo_list = null;
         updateMenus();
         dirty = true;
      }

      return text;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param i    No description provided
    * @param cut  No description provided
    * @return     No description provided
    */
   private String copy_or_cut (JournalItem i, boolean cut)
   {
      String s;
      String
         s2;
      String text = null;

      int line = i.line;
      int column = i.column;
      int eline = i.eline;
      int ecolumn = i.ecolumn;

      if (line == eline)
      {
         s = lines.getString (line);
         text = s.substring (column, ecolumn);
         if (cut)
         {
            s = s.substring (0, column) + s.substring (ecolumn, s.length());
            lines.setString (s, line);
         }
      }
      else
      {
         s = lines.getString (line);
         text = s.substring (column, s.length());
         s = s.substring (0, column);

         int diff = eline - line;
         int k = line + 1;

         for (int j = 1; j < diff; j++)
         {
            s2 = lines.getString (k);
            text = text + lineSeparator + s2;
            if (cut)
            {
               lines.removeElementAt (k);
               hilite.lineRemoved (k);
            }
            else
            {
               k++;
            }
         }

         if (k != lines.size())
         {
            s2 = lines.getString (k);
            text = text + lineSeparator + s2.substring (0, ecolumn);
            s = s + s2.substring (ecolumn, s2.length());
            if (cut)
            {
               lines.removeElementAt (k);
               hilite.lineRemoved (k);
            }
         }
         if (cut)
         {
            lines.setString (s, line);
         }
      }
      return text;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param line  No description provided
    * @return      No description provided
    */
   public String delete_line (int line)
   {
      JournalItem ji = new JournalItem();
      ji.action = DELETE_LINE;
      ji.line = line;
      ji.column = 0;
      ji.text = lines.getString (line);
      ji.ecolumn = ji.text.length();
      ji.text += '\n';
      ji.next = undo_list;
      undo_list = ji;
      redo_list = null;

      lines.removeElementAt (line);
      updateMenus();
      updateFrames (line, line);

      return ji.text;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param line  No description provided
    * @return      No description provided
    */
   public String clear_line (int line)
   {
      return delete_section (line, 0, line, lines.getString (line).length(), true);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param after  No description provided
    * @param txt    No description provided
    */
   public void insert_line (int after, String txt)
   {
      insert_section (after, 0, txt + '\n', true);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param first   No description provided
    * @param second  No description provided
    */
   private void internal_swap_lines (int first, int second)
   {
      LineInfo li = lines.getLineInfo (first);

      lines.setLineInfo (lines.getLineInfo (second), first);
      lines.setLineInfo (li, second);

      updateFrames (first, second);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param first   No description provided
    * @param second  No description provided
    */
   public void swap_lines (int first, int second)
   {
      JournalItem ji = new JournalItem();
      ji.action = SWAP_LINES;
      ji.line = first;
      ji.eline = second;
      ji.next = undo_list;
      undo_list = ji;
      redo_list = null;

      internal_swap_lines (first, second);

      updateMenus();
      return;
   }

}


/**
 * JournalItem is used in the redo/undo queues to track changes to a document.
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.35.2.2 $
 */
class JournalItem
{


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public JournalItem next;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public String text;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public int action;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public int line;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public int column;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public int eline;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public int ecolumn;
}

/*
 * $Log: DocMan.java,v $
 * Revision 1.35.2.2  2005/09/30 18:57:15  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
