/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.preferences.gui;

import java.awt.*;
import java.awt.event.*;
import java.util.*;

import javax.swing.*;
import javax.swing.tree.*;
import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.logging.LoggerInfo;
import de.uni_paderborn.fujaba.preferences.LoggingPreferences;


/**
 * A JTree specialised to display information about loggers.
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.5 $
 */
public class LoggerTree extends JTree
{
   /**
    * Log4J logger
    */
   final static Logger logger = Logger.getLogger (LoggerTree.class);

   /**
    * Menu for performing actions on loggers
    */
   JPopupMenu popup;

   /**
    * Submenu for recursively setting levels (need to turn off for leaves)
    */
   JMenu recursiveLevels;

   /**
    * Items which are not valid at the root
    */
   java.util.List nonRoot = new Vector();

   /**
    * Names of loggers to delete
    */
   java.util.List toDelete = new Vector();

   /**
    * Icons for levels
    */
   Map iconMap;


   /**
    * Creates a new tree
    *
    * @param theIconMap  a map of levels to icons
    */
   public LoggerTree (Map theIconMap)
   {
      this.iconMap = theIconMap;

      if (logger.isDebugEnabled())
      {
         logger.debug ("iconMap == null? " +  (iconMap == null));
      }

      loadTree();

      putClientProperty ("JTree.lineStyle", "Angled");

      setCellRenderer (
         new DefaultTreeCellRenderer()
         {
            public Component getTreeCellRendererComponent (JTree tree, Object value,
                                                           boolean sel, boolean expanded,
                                                           boolean leaf, int row, boolean hasFocus)
            {
               super.getTreeCellRendererComponent (tree, value, sel, expanded, leaf, row, hasFocus);

               DefaultMutableTreeNode node = (DefaultMutableTreeNode) value;
               LoggerInfo loggerInfo = (LoggerInfo)  (node.getUserObject());

               String level = loggerInfo.getLevel();

               if (logger.isDebugEnabled())
               {
                  logger.debug ("iconMap == null? " +  (iconMap == null));
               }
               setIcon ((Icon) iconMap.get (level));

               setToolTipText ("Right click to bring up an options menu");

               return this;
            }
         });

      // Add the menu and the listener
      initialiseMenu();
      addMouseListener (new PopupListener());
      ToolTipManager.sharedInstance().registerComponent (this);
   }


   /**
    * Loads the tree with loggers specified in the OptionsLog4J configurations
    */
   public void loadTree()
   {
      // Assume if tree is being reloaded we want to forget about nodes marked for deletion!
      toDelete = new Vector();

      // Set up the root node
      DefaultMutableTreeNode root = new DefaultMutableTreeNode (LoggingPreferences.get().getRootLogger());

      // Set up the rest of the nodes
      addLoggers (root);

      setModel (new DefaultTreeModel (root));
      getSelectionModel().setSelectionMode (TreeSelectionModel.SINGLE_TREE_SELECTION);

   }


   /**
    * Sets up popup menu
    */
   private void initialiseMenu()
   {
      popup = new JPopupMenu();

      JMenu individualLevels = new JMenu ("Set level for this logger");

      individualLevels.add (new LevelSetterItem (LoggerInfo.OFF, false));
      individualLevels.add (new LevelSetterItem (LoggerInfo.FATAL, false));
      individualLevels.add (new LevelSetterItem (LoggerInfo.ERROR, false));
      individualLevels.add (new LevelSetterItem (LoggerInfo.WARN, false));
      individualLevels.add (new LevelSetterItem (LoggerInfo.INFO, false));
      individualLevels.add (new LevelSetterItem (LoggerInfo.DEBUG, false));

      // keep ref to inherited item as we need to turn it off for root logger
      JMenuItem individualInherited = new LevelSetterItem (LoggerInfo.INHERITED, false);
      individualLevels.add (individualInherited);
      nonRoot.add (individualInherited);

      popup.add (individualLevels);

      // keeping ref so we can turn off for leaves
      recursiveLevels = new JMenu ("Set level for this logger and all descendants");

      recursiveLevels.add (new LevelSetterItem (LoggerInfo.OFF, true));
      recursiveLevels.add (new LevelSetterItem (LoggerInfo.FATAL, true));
      recursiveLevels.add (new LevelSetterItem (LoggerInfo.ERROR, true));
      recursiveLevels.add (new LevelSetterItem (LoggerInfo.WARN, true));
      recursiveLevels.add (new LevelSetterItem (LoggerInfo.INFO, true));
      recursiveLevels.add (new LevelSetterItem (LoggerInfo.DEBUG, true));

      // keep ref to inherited item as we need to turn it off for root logger
      JMenuItem recursiveInherited = new LevelSetterItem (LoggerInfo.INHERITED, true);
      recursiveLevels.add (recursiveInherited);
      nonRoot.add (recursiveInherited);

      popup.add (recursiveLevels);

      popup.addSeparator();

      JMenuItem addLoggerItem = new JMenuItem ("Add a logger...");
      addLoggerItem.addActionListener (getAddLoggerListener());
      popup.add (addLoggerItem);

      JMenuItem deleteLoggerItem = new JMenuItem ("Delete logger");
      deleteLoggerItem.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent e)
            {
               DefaultTreeModel model = (DefaultTreeModel) getModel();
               DefaultMutableTreeNode rootLogger = (DefaultMutableTreeNode) model.getRoot();
               DefaultMutableTreeNode currentLogger = (DefaultMutableTreeNode) getLastSelectedPathComponent();

               String name =  ((LoggerInfo) currentLogger.getUserObject()).getName();

               int areYouSure = JOptionPane.showConfirmDialog (FujabaPreferencesDialog.get(), "Are you sure you want to delete logger '" + name + "'?", "Really delete logger?", JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE);
               if (areYouSure == JOptionPane.YES_OPTION)
               {
                  // remove this logger from the tree
                  currentLogger.removeFromParent();

                  // Alternative way of going through kids - take the first kid and
                  // put it somewhere else until there is no first kid anymore
                  while (currentLogger.getChildCount() > 0)
                  {
                     DefaultMutableTreeNode child = (DefaultMutableTreeNode) currentLogger.getFirstChild();
                     placeNodeInTree (child, rootLogger);
                  }

                  // put this logger on the to-delete list
                  toDelete.add (name);
                  // tell tree we've shaken it up a bit
                  model.reload();
               }
            }
         });
      nonRoot.add (deleteLoggerItem);
      popup.add (deleteLoggerItem);
   }


   /**
    * Updates OptionsLog4J with logging options currently being used
    */
   public void updateOptions()
   {
      // delete loggers we already zapped from here
      Iterator deletables = toDelete.iterator();

      while (deletables.hasNext())
      {
         String deleteMe = (String) deletables.next();
         LoggingPreferences.get().deleteLogger (deleteMe);
      }

      // clean out list
      toDelete = new Vector();

      // get root of JTree
      DefaultMutableTreeNode rootNode = (DefaultMutableTreeNode) getModel().getRoot();

      // update loggers
      // get a depth first search enumeration of all nodes in tree
      Enumeration nodes = rootNode.depthFirstEnumeration();
      while (nodes.hasMoreElements())
      {
         DefaultMutableTreeNode currentNode = (DefaultMutableTreeNode)  (nodes.nextElement());
         LoggerInfo currentInfo = (LoggerInfo) currentNode.getUserObject();
         if (currentNode == rootNode)
         {
            LoggingPreferences.get().updateRootLogger (currentInfo);
         }
         else
         {
            LoggingPreferences.get().updateLogger (currentInfo);
         }
      }
   }


   /**
    * Adds loggers to the tree of which <CODE>root</CODE> is the root using information from
    * OptionsLog4J
    *
    * @param root  root node of JTree (represents root logger)
    */
   private void addLoggers (DefaultMutableTreeNode root)
   {
      java.util.Collection loggers = LoggingPreferences.get().getLoggers();

      for (Iterator iter = loggers.iterator(); iter.hasNext(); )
      {
         LoggerInfo loggerInfo = (LoggerInfo) iter.next();

         // create a node for the tree
         DefaultMutableTreeNode loggerNode = new DefaultMutableTreeNode (loggerInfo);

         // now place it in tree
         placeNodeInTree (loggerNode, root);
      }
   }


   /**
    * Determines where in tree <CODE>insertee</CODE> should be added, starting search from
    * <CODE>parent</CODE>.
    *
    * @param insertee  the node to be inserted
    * @param parent    the node under which insertee should be going
    */
   void placeNodeInTree (DefaultMutableTreeNode insertee, DefaultMutableTreeNode parent)
   {
      LoggerInfo inserteeInfo = (LoggerInfo) insertee.getUserObject();

      boolean success = false; // success signifies having found
      // a node under which to place "insertee"
      // node otherwise insertee should be
      // placed directly under parent

      // go through children of "parent" and try to find node
      // under which "insertee" can be added
      for (Enumeration children = parent.children(); children.hasMoreElements(); )
      {
         DefaultMutableTreeNode currentNode = (DefaultMutableTreeNode)  (children.nextElement());
         LoggerInfo currentInfo = (LoggerInfo) currentNode.getUserObject();
         if (inserteeInfo.getName().startsWith (currentInfo.getName() + "."))
         {
            placeNodeInTree (insertee, currentNode);
            success = true;
         }
      }

      // if no node found under which "insertee" node should be placed
      // then add to list of children of "parent" node in alphabetically
      // correct place and check that none of other children should be
      // under this node
      if (!success)
      {
         // NB: this is practically a for loop and you could use an
         // Enumeration but we need the index for insertion at the end
         // Can't simply do insertion within loop as does not deal with
         // poss that there are no children

         int i = 0;
         while (i < parent.getChildCount())
         {
            DefaultMutableTreeNode currentNode = (DefaultMutableTreeNode) parent.getChildAt (i);
            LoggerInfo currentInfo = (LoggerInfo) currentNode.getUserObject();
            if (inserteeInfo.getName().compareTo (currentInfo.getName()) < 0)
            {
               break;
            }
            i++;
         }

         parent.insert (insertee, i);

         // now check that none of the others should be under this node
         int j = 0;
         while (j < parent.getChildCount())
         {
            DefaultMutableTreeNode currentNode = (DefaultMutableTreeNode) parent.getChildAt (j);
            LoggerInfo currentInfo = (LoggerInfo) currentNode.getUserObject();

            if (currentInfo.getName().startsWith (inserteeInfo.getName() + "."))
            {
               placeNodeInTree (currentNode, insertee);
            }
            else
            {
               j++; // only want to increment count if haven't taken a node out
            }
         }
      }
   }


   /**
    * Set the level of the logger represented by <CODE>logger</CODE>.
    *
    * @param logger     the node representing the logger
    * @param level      the level to set the logger to
    * @param recursive  whether the children should be set to this level too
    */
   void setLoggerLevel (DefaultMutableTreeNode logger, String level, boolean recursive)
   {
      LoggerInfo loggerInfo = (LoggerInfo) logger.getUserObject();
      loggerInfo.setLevel (level);
      DefaultTreeModel model = (DefaultTreeModel) getModel();
      model.nodeChanged (logger);

      if (recursive)
      {
         Enumeration children = logger.children();
         while (children.hasMoreElements())
         {
            DefaultMutableTreeNode child = (DefaultMutableTreeNode) children.nextElement();
            setLoggerLevel (child, level, recursive);
         }
      }
   }


   /**
    * Pops the menu up if a popup event is detected
    *
    * @author    $Author: schneider $
    * @version   $Revision: 1.5 $
    */
   class PopupListener extends MouseAdapter
   {
      /**
       * True if a logger is selected
       */
      boolean gotSomething = false;


      /**
       * The mouse has just been pressed, analyse current state of tree
       *
       * @param e  the mouse event
       */
      public void mousePressed (MouseEvent e)
      {
         TreePath selPath = getPathForLocation (e.getX(), e.getY());
         if (selPath != null)
         {
            setSelectionPath (selPath);
            gotSomething = true;
            maybeShowPopup (e);
         }
      }


      /**
       * The mouse has just been released
       *
       * @param e  the mouse event
       */
      public void mouseReleased (MouseEvent e)
      {
         maybeShowPopup (e);
         gotSomething = false;
      }


      /**
       * Maybe show a popup menu
       *
       * @param e  the mouse event just received
       */
      private void maybeShowPopup (MouseEvent e)
      {
         if (e.isPopupTrigger() && gotSomething)
         {
            // turn inherited on or off here
            //DefaultMutableTreeNode rootLogger = (DefaultMutableTreeNode) getModel().getRoot();
            DefaultMutableTreeNode currentLogger = (DefaultMutableTreeNode) getLastSelectedPathComponent();

            //boolean root = (currentLogger == rootLogger);
            boolean root = currentLogger.isRoot();

            Iterator nonRootItems = nonRoot.iterator();

            while (nonRootItems.hasNext())
            {
                ((JComponent) nonRootItems.next()).setEnabled (!root);
            }

            recursiveLevels.setEnabled (!currentLogger.isLeaf());

            popup.show (e.getComponent(), e.getX(), e.getY());
         }
      }
   }


   /**
    * A menu item to set the level of the currently selected logger, and its children if desired
    *
    * @author    $Author: schneider $
    * @version   $Revision: 1.5 $
    */
   class LevelSetterItem extends JMenuItem
   {
      /**
       * The logging level this menu item will set
       */
      String level;

      /**
       * Whether the level should be set recursively
       */
      boolean recursive;


      /**
       * Constructor for class LevelSetterItem
       *
       * @param theLevel  the level the menu item should set
       * @param recurse   whether the level should be set recursively
       */
      public LevelSetterItem (String theLevel, boolean recurse)
      {
         // display level as name
         super (theLevel);

         level = theLevel;
         recursive = recurse;

         setIcon ((Icon) iconMap.get (level));

         addActionListener (
            new ActionListener()
            {
               public void actionPerformed (ActionEvent e)
               {
                  DefaultMutableTreeNode logger = (DefaultMutableTreeNode) getLastSelectedPathComponent();

                  if (logger != null)
                  {
                     setLoggerLevel (logger, level, recursive);
                  }
               }
            });
      }
   }


   /**
    * Returns a listener which will enable the user to add loggers to the tree
    *
    * @return   The addLoggerListener value
    */
   public ActionListener getAddLoggerListener()
   {
      return new AddLoggerListener();
   }


   /**
    * Allows user to add a logger, the name of which is input through a dialog.
    *
    * @author    $Author: schneider $
    * @version   $Revision: 1.5 $
    */
   class AddLoggerListener implements ActionListener
   {
      /**
       * Allow the user to add a logger
       *
       * @param e  the action event
       */
      public void actionPerformed (ActionEvent e)
      {
         String currentLoggerName = null;

         DefaultMutableTreeNode currentLogger = (DefaultMutableTreeNode) getLastSelectedPathComponent();

         if (currentLogger != null && !currentLogger.isRoot())
         {
            LoggerInfo info = (LoggerInfo) currentLogger.getUserObject();
            currentLoggerName = info.getName();
         }

         String loggerName = (String) JOptionPane.showInputDialog (FujabaPreferencesDialog.get(), "Enter the name of the logger:", "Add a logger", JOptionPane.QUESTION_MESSAGE, null, null, currentLoggerName);
         // log.info("Name given was: " + loggerName);
         if (loggerName != null)
         {
            if (!LoggingPreferences.get().isValidLoggerName (loggerName))
            {
               JOptionPane.showMessageDialog (FujabaPreferencesDialog.get(), "'" + loggerName + "' is not a valid logger name.", "Invalid logger name", JOptionPane.ERROR_MESSAGE);
            }
            else
            {
               boolean alreadyThere = false;

               // get the model for the tree
               DefaultTreeModel model = (DefaultTreeModel) getModel();

               // get the root of the tree
               DefaultMutableTreeNode root = (DefaultMutableTreeNode) model.getRoot();

               Enumeration nodes = root.depthFirstEnumeration();
               while (nodes.hasMoreElements())
               {
                  DefaultMutableTreeNode currentNode = (DefaultMutableTreeNode)  (nodes.nextElement());
                  LoggerInfo currentInfo = (LoggerInfo) currentNode.getUserObject();
                  if (currentInfo.getName().equals (loggerName))
                  {
                     JOptionPane.showMessageDialog (FujabaPreferencesDialog.get(), "There is already a logger with the name '" + loggerName + "'.", "Invalid logger name", JOptionPane.ERROR_MESSAGE);
                     alreadyThere = true;
                     break;
                  }
               }

               if (!alreadyThere)
               {
                  // construct the LoggerInfo for the new node
                  LoggerInfo loggerInfo = new LoggerInfo (loggerName, LoggerInfo.INHERITED);

                  // create a node for the tree
                  DefaultMutableTreeNode loggerNode = new DefaultMutableTreeNode (loggerInfo);

                  // now place it in tree
                  placeNodeInTree (loggerNode, root);
                  /*
                   *  TreeNode parent = loggerNode.getParent();
                   *  int index = parent.getIndex(loggerNode);
                   *  int[] indices = new int[] {index};
                   *  model.nodesWereInserted(parent, indices);
                   */
                  // need to assume whole structure has changed as could have just added a logger which is
                  // a parent of many current loggers
                  model.reload();

                  // Make the node visible so that the user can see what's happened
                  TreePath toNewNode = new TreePath (loggerNode.getPath());

                  setSelectionPath (toNewNode);
                  //LoggerTree.this.makeVisible(toNewNode);
               }
            }
         }
      }
   }
}

/*
 * $Log: LoggerTree.java,v $
 * Revision 1.5  2004/10/20 17:50:08  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
