/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.preferences.gui;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.net.URL;
import java.util.*;

import javax.swing.*;
import javax.swing.border.TitledBorder;
import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.app.FrameMain;
import de.uni_paderborn.fujaba.basic.SchemaFilter;
import de.uni_paderborn.fujaba.logging.LoggerInfo;
import de.uni_paderborn.fujaba.logging.LoggingConfigException;
import de.uni_paderborn.fujaba.preferences.LoggingPreferences;
import de.upb.lib.html.HtmlDialog;


/**
 * Options panel for log4j configuration
 *
 * @author    $Author: koenigs $
 * @version   $Revision: 1.6.2.1 $
 */
public class LoggingPreferencesPanel extends PreferencesPanel
{
   /**
    * Log4J logger
    */
   private final static Logger logger = Logger.getLogger (LoggingPreferencesPanel.class);

   /**
    * The main panel containing interfaces for changing appender and logger settings
    */
   private JPanel mainPanel;

   /**
    * JButton for adding loggers
    */
   private JButton addButton = new JButton ("Add a logger...");

   /**
    * JScrollPane for viewing tree of loggers
    */
   private JScrollPane treeScroller = new JScrollPane();

   /**
    * Specialised JTree for viewing logger settings
    */
   private LoggerTree loggerTree;

   /**
    * Specific appender panels within the main appender panel
    */
   private java.util.List appenderPanels = new Vector();

   /**
    * Map of logging levels to icons
    */
   private Map iconMap = new Hashtable (7);



   /**
    * The constructor. It builds a gui interface for the dialog.
    */
   public LoggingPreferencesPanel()
   {
      initialiseIcons();

      // set layout for this panel
      setLayout (new BorderLayout());

      // We want two subpanels of our box - a "main panel" where loggers and
      // appenders can be configured, and a bottom panel with space for the buttons
      // for loading new files, generating test messages etc.

      mainPanel = createMainPanel();
      add (mainPanel, BorderLayout.CENTER);
      add (createBottomPanel(), BorderLayout.SOUTH);
   }


   /**
    * Returns the preferred name of a tab for this panel
    *
    * @return   The preferredTabName value
    */
   public String getPreferredTabName()
   {
      return  ("Logging");
   } // getPreferredTabName


   /**
    * Set the options in the panel, because the options could have changed elsewhere.
    */
   public void setPreferences()
   {
      if (!checkValidConfig())
      {
         return;
      }

      if (loggerTree == null)
      {
         if (logger.isDebugEnabled())
         {
            logger.debug ("iconMap == null? " +  (iconMap == null));
         }
         loggerTree = new LoggerTree (iconMap);
         addButton.addActionListener (loggerTree.getAddLoggerListener());
         treeScroller.setViewportView (loggerTree);
      }
      else
      {
         loggerTree.loadTree();
      }

      // set up appender panels from options
      for (Iterator iter = appenderPanels.iterator(); iter.hasNext(); )
      {
         AppenderPanel panel = (AppenderPanel) iter.next();
         try
         {
            panel.setOptions();
         }
         catch (NullPointerException e)
         {
            System.err.println ("The appender " + panel.getConfigName() + " could not be initialized - try to load default values for logging to fix this problem");
         }
      }
   } // setOptions


   /**
    * Set all options to default values.
    */
   public void setDefaults()
   {
      LoggingPreferences.get().setDefaults();
      setPreferences();
   } // setDefaults


   /**
    * Get all the options from the panel and set the real options.
    */
   public void okPressed()
   {
      LoggingPreferences options = LoggingPreferences.get();

      if (!checkValidConfig())
      {
         return;
      }

      // update options from tree
      loggerTree.updateOptions();

      // update options from appender panels
      for (Iterator iter = appenderPanels.iterator(); iter.hasNext(); )
      {
         AppenderPanel panel = (AppenderPanel) iter.next();
         panel.updateOptions();
      }

      // configure log4j according to updated options
      options.configureLog4J();
   } // okPressed


   /**
    * Sets up the level icons
    */
   private void initialiseIcons()
   {
      String[] imageNames = new String[]{"inherited-grey.gif", "debug-blue.gif",
         "info-green.gif", "warn-yellow.gif",
         "error-orange.gif", "fatal-red.gif",
         "off-black.gif"};

      Class clazz = this.getClass();
      String iconsDir = "/de/uni_paderborn/fujaba/logging/images/";
      URL url;
      for (int i = 0; i < imageNames.length; i++)
      {
         url = clazz.getResource (iconsDir + imageNames[i]);
         if (url != null)
         {
            iconMap.put (LoggerInfo.LEVELS[i], new ImageIcon (url));
         }
      }

      if (logger.isDebugEnabled())
      {
         logger.debug ("Icons initialised; iconMap = " + iconMap);
      }
   }


   /**
    * Create the main panel
    *
    * @return   the main panel
    */
   private JPanel createMainPanel()
   {
      JPanel mainPanel = new JPanel();
      GridLayout grid = new GridLayout (1, 2);
      mainPanel.setLayout (grid);

      // Now want two inner panels, one for loggers and one for appenders

      JPanel loggerPanel = createLoggerPanel();
      mainPanel.add (loggerPanel);

      JPanel appendersPanel = createAppendersPanel();
      mainPanel.add (appendersPanel);

      return mainPanel;
   }


   /**
    * Create the bottom panel
    *
    * @return   the bottom panel
    */
   private JPanel createBottomPanel()
   {
      // this contains a button to load a file and to generate some test messages
      JPanel bottomPanel = new JPanel();
      bottomPanel.setLayout (new GridLayout (1, 4));

      // create the load a file panel
      JPanel loadFilePanel = new JPanel();

      JButton loadFileButton = new JButton ("Load config file...");
      loadFileButton.addActionListener (new LoadFileListener());
      loadFilePanel.add (loadFileButton);

      bottomPanel.add (loadFilePanel);

      // create the apply configuration panel
      JPanel applyConfigPanel = new JPanel();
      JButton applyConfigButton = new JButton ("Apply configuration");
      applyConfigButton.addActionListener (
         new ActionListener()
         {
            /**
             * This just simulates someone pressing ok but leaves the window open so the user
             * can carry on work
             *
             * @param e  No description provided
             */
            public void actionPerformed (ActionEvent e)
            {
               okPressed();
            }
         });

      applyConfigPanel.add (applyConfigButton);

      bottomPanel.add (applyConfigPanel);

      // create the test messages panel
      JPanel testMsgsPanel = new JPanel();
      //JLabel testMsgsLabel = new JLabel ("Generate test messages");
      //testMsgsPanel.add (testMsgsLabel);

      //JButton testMsgsButton = new JButton("Go");
      JButton testMsgsButton = new JButton ("Generate test messages");
      testMsgsButton.addActionListener (
         new ActionListener()
         {
            /**
             * Generate a message of every level to each logger to test configuration
             *
             * @param e  No description provided
             */
            public void actionPerformed (ActionEvent e)
            {
               checkValidConfig();
               LoggingPreferences.get().generateTestMsgs();
            }
         });

      testMsgsPanel.add (testMsgsButton);

      bottomPanel.add (testMsgsPanel);

      // create the help box panel
      JPanel helpBoxPanel = new JPanel();
      JButton helpButton = new JButton ("Help!");
      helpButton.addActionListener (
         new ActionListener()
         {
            /**
             * Pop up a window with some hints for using the logging component
             *
             * @param e  No description provided
             */
            public void actionPerformed (ActionEvent e)
            {
               String helpFile = "/de/uni_paderborn/fujaba/logging/logginghelp.html";
               URL url = this.getClass().getResource (helpFile);
               if (url != null)
               {
                  HtmlDialog html = new HtmlDialog (FrameMain.get(), url, "Help - Fujaba Logging", true);
                  // html.getHtmlPanel().setNavigationBarVisible (false);
                  html.getHtmlPanel().setNavigationBarButtonEnablity (false, false, false);
                  html.show();
               }
               else
               {
                  JOptionPane.showMessageDialog (LoggingPreferencesPanel.this, "Unable to locate help file " + helpFile, "Error loading help file", JOptionPane.ERROR_MESSAGE);
               }
            }
         });

      helpBoxPanel.add (helpButton);

      bottomPanel.add (helpBoxPanel);

      return bottomPanel;
   }


   /**
    * Create the logger panel
    *
    * @return   the logger panel
    */
   private JPanel createLoggerPanel()
   {
      JPanel loggerPanel = new JPanel();

      loggerPanel.setLayout (new BorderLayout());
      loggerPanel.setBorder (new TitledBorder ("Loggers"));

      // add the scroll panel and the combo box
      loggerPanel.add (treeScroller, BorderLayout.CENTER);

      JPanel addPanel = new JPanel();
      addPanel.add (addButton);
      loggerPanel.add (addPanel, BorderLayout.SOUTH);

      return loggerPanel;
   }


   /**
    * Create the appenders panel
    *
    * @return   the appenders panel
    */
   private JPanel createAppendersPanel()
   {
      JPanel containerPanel = new JPanel();
      JScrollPane scrollPane = new JScrollPane();
      JPanel panel = new JPanel();

      scrollPane.setViewportView (panel);
      containerPanel.add (scrollPane);

      containerPanel.setBorder (new TitledBorder ("Appenders"));

      GridBagLayout gridBag = new GridBagLayout();
      GridBagConstraints constraints = new GridBagConstraints();
      panel.setLayout (gridBag);

      AppenderPanel consolePanel = new AppenderPanel ("Console output", "Console");
      AppenderPanel errorConsolePanel = new AppenderPanel ("Error console output", "ErrorConsole");
      consolePanel.addPatternField();
      errorConsolePanel.addPatternField();

      constraints.gridwidth = GridBagConstraints.REMAINDER;
      constraints.fill = GridBagConstraints.HORIZONTAL;
      constraints.weightx = 1.0;
      constraints.weighty = 1.0;

      gridBag.setConstraints (consolePanel, constraints);
      gridBag.setConstraints (errorConsolePanel, constraints);

      panel.add (consolePanel);
      panel.add (errorConsolePanel);
      appenderPanels.add (consolePanel);
      appenderPanels.add (errorConsolePanel);

      AppenderPanel filePanel = new AppenderPanel ("File output", "File");

      filePanel.addFileSelector ("File to log to", "File");
      filePanel.addPatternField();
      filePanel.addCheckBox ("Append to file?", "Append");

      gridBag.setConstraints (filePanel, constraints);

      panel.add (filePanel);
      appenderPanels.add (filePanel);

      // Added SocketHubAppender for use with Loglipse for instance. [ak]
      AppenderPanel socketPanel = new AppenderPanel ("Socket output", "Socket");

      socketPanel.addPortField();

      gridBag.setConstraints (socketPanel, constraints);

      panel.add (socketPanel);
      appenderPanels.add (socketPanel);

      return containerPanel;
   }


   /**
    * Check whether the log4j configuration has been loaded correctly, and notify the user
    * if not.
    *
    * @return   <CODE>true</CODE> if the configuration has been loaded, <CODE>false</CODE>
    *      otherwise
    */
   boolean checkValidConfig()
   {
      LoggingPreferences options = LoggingPreferences.get();

      Iterator appPanelIter = appenderPanels.iterator();

      if (!options.configLoaded())
      {
         addButton.setEnabled (false);
         treeScroller.setEnabled (false);

         while (appPanelIter.hasNext())
         {
            AppenderPanel panel = (AppenderPanel) appPanelIter.next();
            panel.setContentsEnabled (false);
         }

         JOptionPane.showMessageDialog (this, "Fujaba has not been able to successfully load a configuration file for log4j. Please load a valid configuration file.", "No configuration loaded for log4j", JOptionPane.WARNING_MESSAGE);

         return false;
      }
      else
      {
         addButton.setEnabled (true);
         treeScroller.setEnabled (true);

         while (appPanelIter.hasNext())
         {
            AppenderPanel panel = (AppenderPanel) appPanelIter.next();
            panel.setContentsEnabled (true);
         }

         return true;
      }
   }


   /**
    * A listener to allow the user to load a particular config file
    *
    * @author    $Author: koenigs $
    * @version   $Revision: 1.6.2.1 $
    */
   private class LoadFileListener implements ActionListener
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void actionPerformed (ActionEvent e)
      {
         JPanel thisPanel = LoggingPreferencesPanel.this;

         JFileChooser fileChooser = new JFileChooser();

         fileChooser.setFileFilter (new SchemaFilter (".xml", "XML files (*.xml)"));
         fileChooser.setFileSelectionMode (JFileChooser.FILES_ONLY);

         if (fileChooser.showOpenDialog
             (thisPanel) == JFileChooser.APPROVE_OPTION)
         {
            File file = fileChooser.getSelectedFile();

            if (file.exists() && file.isFile())
            {
               try
               {
                  thisPanel.setCursor (new Cursor (Cursor.WAIT_CURSOR));
                  LoggingPreferences.get().loadFile (file);
                  setPreferences();
               }
               catch (LoggingConfigException except)
               {
                  //params: parent, message, title, message type
                  JOptionPane.showMessageDialog (thisPanel, except.getMessage(), "Error loading " + file.getName(), JOptionPane.ERROR_MESSAGE);
                  checkValidConfig();
               }
               finally
               {
                  thisPanel.setCursor (new Cursor (Cursor.DEFAULT_CURSOR));
               }
            }
         }
      }
   }
}

/*
 * $Log: LoggingPreferencesPanel.java,v $
 * Revision 1.6.2.1  2006/06/22 09:04:05  koenigs
 * Added SocketHubAppender support. [ak]
 *
 */
