/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.preferences.gui;

import java.awt.*;
import java.awt.event.*;
import java.util.Iterator;

import javax.swing.*;

import de.uni_paderborn.fujaba.preferences.LoggingPreferences;
import de.uni_paderborn.fujaba.preferences.PreferencesProperties;
import de.upb.tools.fca.FLinkedList;


/**
 * Dialog for the environment options.
 *
 * <h2>Associations</h2>
 *
 * <pre>
 *                    1              PreferencesPanel               n
 * PreferencesDialog ------------------------------------------------- PreferencesPanel
 *                    preferencesDialog   {sorted}  preferencesPanels
 * </pre>
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.7 $
 */
public abstract class PreferencesDialog extends JDialog
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected JTabbedPane tabbedPane;


   /**
    * Constructor for class PreferencesDialog
    *
    * @param frame  No description provided
    * @param title  No description provided
    * @param modal  No description provided
    */
   protected PreferencesDialog (JFrame frame, String title, boolean modal)
   {
      super (frame, title, modal);

      addWindowListener (new DialogClose());

      initPreferencesPanels();

      JPanel container = new JPanel();
      container.setLayout (new BorderLayout());

      tabbedPane = new JTabbedPane();
      container.add (tabbedPane, BorderLayout.CENTER);
      container.add (createButtonPanel(), BorderLayout.SOUTH);

      getContentPane().add (container);
   } // PreferencesDialog


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void show()
   {
      initTabbedPane();
      pack();
      centerDialog();

      setPreferences();

      super.show();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected abstract void initPreferencesPanels();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void initTabbedPane()
   {
      Iterator iter = iteratorOfPreferencesPanels();
      while (iter.hasNext())
      {
         PreferencesPanel tmpOptionsPanel = (PreferencesPanel) iter.next();
         if (tabbedPane.indexOfComponent (tmpOptionsPanel) == -1)
         {
            tabbedPane.addTab (tmpOptionsPanel.getPreferredTabName(), tmpOptionsPanel);
         }
      }
   } // initTabbedPane


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   protected JPanel createButtonPanel()
   {
      JPanel leftButtonPanel = new JPanel();
      leftButtonPanel.setLayout (new FlowLayout (FlowLayout.LEFT));

      JButton setDefaults = new JButton ("Set Defaults");
      setDefaults.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent e)
            {
               defaultPressed (false);
            }
         }
         );
      setDefaults.setToolTipText ("Set defaults for this tab.");
      leftButtonPanel.add (setDefaults);

      JButton setAllDefaults = new JButton ("Set Defaults For All");
      setAllDefaults.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent e)
            {
               defaultPressed (true);
            }
         }
         );
      setAllDefaults.setToolTipText ("Set defaults for all tabs.");
      leftButtonPanel.add (setAllDefaults);

      JPanel rightButtonPanel = new JPanel();
      rightButtonPanel.setLayout (new FlowLayout (FlowLayout.RIGHT));

      JButton ok = new JButton ("OK");
      ok.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent e)
            {
               savePressed();
               okPressed();
            }
         }
         );
      rightButtonPanel.add (ok);

      JButton cancel = new JButton ("Cancel");
      cancel.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent e)
            {
               cancelPressed();
            }
         }
         );
      rightButtonPanel.add (cancel);

      getRootPane().setDefaultButton (ok);

      JPanel buttonPanel = new JPanel();
      buttonPanel.setLayout (new GridLayout (0, 2));
      buttonPanel.add (leftButtonPanel);
      buttonPanel.add (rightButtonPanel);

      return buttonPanel;
   } // createButtonPanel


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void centerDialog()
   {
      Dimension screenSize = this.getToolkit().getScreenSize();
      Dimension size = this.getSize();

      int y =  (screenSize.height - size.height) / 2;
      int x =  (screenSize.width - size.width) / 2;

      setLocation (x, y);
   } // centerDialog


   /**
    * "Set Defaults" or "Set Defaults For All" is pressed.
    *
    * @param all  True, if default values of all tabs should be set.
    */
   public void defaultPressed (boolean all)
   {
      if (all)
      {
         Iterator iter = iteratorOfPreferencesPanels();
         while (iter.hasNext())
         {
             ((PreferencesPanel) iter.next()).setAllDefaults();
         }
      }
      else
      {
         // find the selected PreferencesPanel.
         PreferencesPanel panel = (PreferencesPanel) tabbedPane.getSelectedComponent();
         if (panel != null)
         {
            panel.setDefaults();
         }
      }
   } // defaultPressed


   /**
    * Set all the options of inserted panels, because there options may have changed.
    */
   public void setPreferences()
   {
      Iterator iter = iteratorOfPreferencesPanels();
      while (iter.hasNext())
      {
          ((PreferencesPanel) iter.next()).setPreferences();
      }
   } // setOptions


   /**
    * OK is pressed and set all the options of the inserted panels.
    */
   public void okPressed()
   {
      Iterator iter = iteratorOfPreferencesPanels();
      while (iter.hasNext())
      {
          ((PreferencesPanel) iter.next()).okPressed();
      }

      setVisible (false);
   } // okPressed


   /**
    * Save is pressed and set all the options of the inserted panels. After that save them.
    */
   public void savePressed()
   {
      Iterator iter = iteratorOfPreferencesPanels();
      while (iter.hasNext())
      {
         PreferencesPanel optionsPanel = (PreferencesPanel) iter.next();
         optionsPanel.okPressed();
      }

      PreferencesProperties.saveAll();

      // log4j settings need to be saved seperately as they are not stored in
      // PropertyOptions as other options are
      LoggingPreferences.get().saveSettings();
   } // savePressed


   /**
    * Cancel is pressed and don't set any options.
    */
   public void cancelPressed()
   {
      setVisible (false);
   } // cancelPressed


   /**
    * Sets the selectedTab attribute of the PreferencesDialog object
    *
    * @param panel  The new selectedTab value
    */
   public void setSelectedTab (PreferencesPanel panel)
   {
      if (hasInPreferencesPanels (panel))
      {
         tabbedPane.setSelectedComponent (panel);
      }
   } // setSelectedTab


   /**
    * <pre>
    *                    1              PreferencesPanel               n
    * PreferencesDialog ------------------------------------------------- PreferencesPanel
    *                    preferencesDialog   {sorted}  preferencesPanels
    * </pre>
    */
   private FLinkedList preferencesPanels = new FLinkedList();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    * @return      No description provided
    */
   public boolean hasInPreferencesPanels (PreferencesPanel elem)
   {
      return this.preferencesPanels.contains (elem);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfPreferencesPanels()
   {
      return this.preferencesPanels.iterator();
   }


   /**
    * Access method for an one to n association.
    *
    * @param elem  The object added.
    */
   public void addToPreferencesPanels (PreferencesPanel elem)
   {
      if (elem != null && !this.hasInPreferencesPanels (elem))
      {
         this.preferencesPanels.add (elem);
         elem.setPreferencesDialog (this);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    */
   public void removeFromPreferencesPanels (PreferencesPanel elem)
   {
      if (this.hasInPreferencesPanels (elem))
      {
         this.preferencesPanels.remove (elem);
         elem.setPreferencesDialog (null);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfPreferencesPanels()
   {
      return preferencesPanels.size();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromPreferencesPanels()
   {
      PreferencesPanel item;
      Iterator iter = iteratorOfPreferencesPanels();

      while (iter.hasNext())
      {
         item = (PreferencesPanel) iter.next();
         item.setPreferencesDialog (null);
      }
   }


   //========================= Some inner classes ====================================

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.7 $
    */
   private class DialogClose extends WindowAdapter
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void windowClosing (WindowEvent e)
      {
         cancelPressed();
      } // windowClosing

   } // DialogClose

}

/*
 * $Log: PreferencesDialog.java,v $
 * Revision 1.7  2005/03/16 08:55:34  lowende
 * Tabs in preferences reordered.
 *
 */
