/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml;

import java.beans.PropertyChangeEvent;
import java.util.Iterator;

import de.uni_paderborn.fujaba.sequencer.FlowActivity;
import de.upb.tools.fca.FEmptyIterator;
import de.upb.tools.fca.FPropHashSet;


/**
 * <h2>Associations</h2> <pre>
 *              0..1             N
 * UMLActivity -------------------- UMLTransition
 *              revEntry     entry
 *
 *              0..1             N
 * UMLActivity -------------------- UMLTransition
 *              revExit       exit
 *
 *              0..1                         0..1
 * UMLActivity ----------------------------------- FlowActivity
 *              + UMLActivity      + FlowActivity
 * </pre>
 *
 * @author    $Author: fklar $
 * @version   $Revision: 1.67.2.2 $
 */
public class UMLActivity extends UMLDiagramItem
{
   /**
    * @param coobraPersistent
    */
   protected UMLActivity (boolean coobraPersistent)
   {
      super (coobraPersistent);
   }


   /**
    * Constructor for class UMLActivity
    */
   public UMLActivity()
   {
      super();
   } // UMLActivity


   /**
    * Returns either the activity diagram in which this activity is contained or zero. But
    * then an exception is thrown outside this class.
    *
    * @return   the UMLActivityDiagram
    */
   public UMLActivityDiagram getActivityDiagram()
   {
      Iterator diagsIter = iteratorOfDiagrams();
      return  (diagsIter.hasNext()) ?  ((UMLActivityDiagram) diagsIter.next()) : null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String name;


   /**
    * Get the name attribute of the UMLDiagram object
    *
    * @return   The name value
    */
   public String getName()
   {
      return name;
   }


   /**
    * Sets the name attribute of the UMLDiagram object
    *
    * @param name  The new name value
    */
   public void setName (String name)
   {
      if ( (this.name == null && name != null) ||
          (this.name != null && !this.name.equals (name)))
      {
         String oldValue = this.name;
         this.name = name;
         firePropertyChange (NAME_PROPERTY, oldValue, name);
      }
   }


   /**
    * This number is only used for internal handling, so don't save it.
    */
   private transient int internalCompilationNumber = 0;


   /**
    * Get the internalCompilationNumber attribute of the UMLActivity object
    *
    * @return   The internalCompilationNumber value
    */
   public int getInternalCompilationNumber()
   {
      return internalCompilationNumber;
   } // getInternalCompilationNumber


   /**
    * Sets the internalCompilationNumber attribute of the UMLActivity object
    *
    * @param internalCompilationNumber  The new internalCompilationNumber value
    */
   public void setInternalCompilationNumber (int internalCompilationNumber)
   {
      int oldValue = this.internalCompilationNumber;
      this.internalCompilationNumber = internalCompilationNumber;
      firePropertyChange ("internalCompilationNumber", oldValue, internalCompilationNumber);
   } // setInternalCompilationNumber


   /**
    * <pre>
    *              0..1             N
    * UMLActivity -------------------- UMLTransition
    *              revEntry     entry
    * </pre>
    */
   private FPropHashSet entry;


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToEntry (UMLTransition value)
   {
      boolean changed = false;

      if (value != null && !hasInEntry (value))
      {
         if (this.entry == null)
         {
            this.entry = new FPropHashSet (this, "entry");
         }
         this.entry.add (value);
         value.setRevEntry (this);
         changed = true;
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInEntry (UMLTransition value)
   {
      return  ( (this.entry != null) &&
          (value != null) &&
         this.entry.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfEntry()
   {
      return  ( (this.entry == null)
         ? FEmptyIterator.get()
         : this.entry.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfEntry()
   {
      return  ( (this.entry == null)
         ? 0
         : this.entry.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromEntry (UMLTransition value)
   {
      boolean changed = false;

      if (value != null && hasInEntry (value))
      {
         this.entry.remove (value);
         value.setRevEntry (null);
         changed = true;
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromEntry()
   {
      UMLTransition tmpValue;
      Iterator iter = this.iteratorOfEntry();

      while (iter.hasNext())
      {
         tmpValue = (UMLTransition) iter.next();
         this.removeFromEntry (tmpValue);
      }
   }


   /**
    * <pre>
    *              0..1             N
    * UMLActivity -------------------- UMLTransition
    *              revExit       exit
    * </pre>
    */
   private FPropHashSet exit = null;


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToExit (UMLTransition value)
   {
      boolean changed = false;

      if (value != null && !hasInExit (value))
      {
         if (this.exit == null)
         {
            this.exit = new FPropHashSet (this, "exit");
         }
         this.exit.add (value);
         value.setRevExit (this);
         changed = true;
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInExit (UMLTransition value)
   {
      return  ( (this.exit != null) &&
          (value != null) &&
         this.exit.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfExit()
   {
      return  ( (this.exit == null)
         ? FEmptyIterator.get()
         : this.exit.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfExit()
   {
      return  ( (this.exit == null)
         ? 0
         : this.exit.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromExit (UMLTransition value)
   {
      boolean changed = false;

      if (value != null && hasInExit (value))
      {
         this.exit.remove (value);
         value.setRevExit (null);
         changed = true;
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromExit()
   {
      UMLTransition tmpValue;
      Iterator iter = this.iteratorOfExit();

      while (iter.hasNext())
      {
         tmpValue = (UMLTransition) iter.next();
         this.removeFromExit (tmpValue);
      }
   }


   /**
    * <pre>
    *              0..1                         0..1
    * UMLActivity ----------------------------------- FlowActivity
    *              + UMLActivity      + FlowActivity
    * </pre>
    */
   private transient FlowActivity flowActivity;


   /**
    * Get the flowActivity attribute of the UMLActivity object
    *
    * @return   The flowActivity value
    */
   public FlowActivity getFlowActivity()
   {
      return flowActivity;
   } // getFlowActivity


   /**
    * Sets the flowActivity attribute of the UMLActivity object
    *
    * @param flowActivity  The new flowActivity value
    */
   public void setFlowActivity (FlowActivity flowActivity)
   {
      if ( (this.flowActivity == null && flowActivity != null) ||
          (this.flowActivity != null && !this.flowActivity.equals (flowActivity)))
      {
         FlowActivity oldFlowActivity = this.flowActivity;
         // newPartner
         if (this.flowActivity != null)
         {
            // inform old partner
            this.flowActivity = null;
            oldFlowActivity.setUMLActivity (null);
         }

         this.flowActivity = flowActivity;
         if (this.flowActivity != null)
         {
            // inform new partner
            this.flowActivity.setUMLActivity (this);
         }

         firePropertyChange ("flowActivity", oldFlowActivity, flowActivity);
      }
   } // setFlowActivity


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void removeAllTransitions()
   {
      UMLTransition transition;

      Iterator iter = iteratorOfEntry();
      while (iter.hasNext())
      {
         transition = (UMLTransition) iter.next();
         removeFromEntry (transition);
         transition.removeYou();
      }

      iter = iteratorOfExit();
      while (iter.hasNext())
      {
         transition = (UMLTransition) iter.next();
         removeFromExit (transition);
         transition.removeYou();
      }
   }


   /**
    * Returns the name of the activity
    *
    * @return   The text value
    */
   public String getText()
   {
      return "";
   }


   /**
    * Isolates the object so the garbage collector can remove it.
    */
   public void removeYou()
   {
      removeAllTransitions();
      setFlowActivity (null);

      super.removeYou();
   }


   /**
    * Get the persistencyChange attribute of the UMLActivity object
    *
    * @param e  No description provided
    * @return   The persistencyChange value
    */
   protected boolean isPersistencyChange (PropertyChangeEvent e)
   {
      return super.isPersistencyChange (e) &&
         ! ("flowActivity".equals (e.getPropertyName()));
   }
}

/*
 * $Log: UMLActivity.java,v $
 * Revision 1.67.2.2  2005/11/18 09:38:53  fklar
 * added missing access-methods for property 'name' (overriding methods from ASGElement)
 *
 */
