/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml;

import java.util.*;

import de.uni_paderborn.fujaba.metamodel.*;
import de.upb.tools.fca.*;
import de.upb.tools.pcs.CollectionChangeEvent;


/**
 * <h2>Associations</h2> <pre>
 *          ------------- 0..n    stereotype     0..1
 * UMLAssoc | getText() |----------------------------- UMLStereotype
 *          ------------- uMLAssoc      uMLStereotype
 * </pre>
 *
 * @author    $Author: mtt $
 * @version   $Revision: 1.148.2.2 $
 */
public class UMLAssoc extends UMLConnection implements FAssoc
{
   /**
    * direction = {LeftRight, RightLeft}
    */
   public final static transient int LEFTRIGHT = 10;

   /**
    * direction = {LeftRight, RightLeft}
    */
   public final static transient int RIGHTLEFT = 11;


   /**
    * Constructor for class UMLAssoc
    */
   public UMLAssoc()
   {
      super();
   }


   /**
    * Constructor for class UMLAssoc
    *
    * @param name           No description provided
    * @param leftClass      No description provided
    * @param leftRoleName   No description provided
    * @param leftCard       No description provided
    * @param rightClass     No description provided
    * @param rightRoleName  No description provided
    * @param rightCard      No description provided
    */
   public UMLAssoc (String name,
                    UMLClass leftClass,
                    String leftRoleName,
                    String leftCard,
                    UMLClass rightClass,
                    String rightRoleName,
                    String rightCard)
   {
      this (name, LEFTRIGHT, null,
         new UMLRole (leftRoleName, leftClass, leftCard),
         new UMLRole (rightRoleName, rightClass, rightCard));
   }


   /**
    * Constructor for class UMLAssoc
    *
    * @param name        No description provided
    * @param direction   No description provided
    * @param constraint  No description provided
    * @param leftRole    No description provided
    * @param rightRole   No description provided
    */
   public UMLAssoc (String name, int direction,
                    UMLConstraint constraint, UMLRole leftRole,
                    UMLRole rightRole)
   {
      super();
      setName (name);
      setDirection (direction);
      addToConstraints (constraint);
      setLeftRole (leftRole);
      setRightRole (rightRole);
   }


   /**
    * Boolean static method to check wether two classes are connected via the passed association.
    *
    * @param firstClass      No description provided
    * @param firstRoleName   No description provided
    * @param secondClass     No description provided
    * @param secondRoleName  No description provided
    * @return                The assocForItems value
    */
   public static UMLAssoc[] getAssocForItems (UMLClass firstClass,
                                              String firstRoleName,
                                              UMLClass secondClass,
                                              String secondRoleName)
   {
      // has the current class such a Role attached
      Enumeration firstRolesEnum = firstClass.elementsOfRoles();
      UMLRole firstRole = null;
      UMLRole secondRole = null;
      Vector foundAssoc = new Vector();
      UMLAssoc assocs[] = null;

      while (firstRolesEnum.hasMoreElements())
      {
         firstRole = (UMLRole) firstRolesEnum.nextElement();

         if (firstRoleName == null || firstRoleName.equals (firstRole.getName()))
         {
            // Ups, firstClass has the expected Role
            // does firstRole acutally belong to the expected assoc
            // look for other role and class
            secondRole = firstRole.getPartnerRole();

            if (secondRoleName == null || secondRoleName.equals (secondRole.getName()))
            {
               if (secondClass == null || secondClass.isChildOf (secondRole.getTarget()))
               {
                  foundAssoc.add (firstRole.getAssoc());
               }
            }
         }
      }

      assocs = new UMLAssoc[foundAssoc.size()];

      System.arraycopy (foundAssoc.toArray(), 0,
         assocs, 0, foundAssoc.size());

      return assocs;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theClass     No description provided
    * @param theRoleName  No description provided
    * @return             No description provided
    */
   public static boolean existsRef (UMLClass theClass, String theRoleName)
   {
      // has the current class such a Role attached
      Enumeration theRolesEnum = theClass.elementsOfRoles();
      UMLRole theRole = null;

      while (theRolesEnum.hasMoreElements())
      {
         theRole = (UMLRole) theRolesEnum.nextElement();
         //if (log.isInfoEnabled()) log.info ("Found role " + theRole.getPartnerRole().getAttrName ());
         if (theRole.getPartnerRole() != null &&
            theRole.getPartnerRole().getAttrName() != null)
         {
            //log.info("hm......");
            if (theRole.getPartnerRole().getAttrName().equals (theRoleName))
            {
               return true;
            }
         }
      }
      return false;
   }


   /**
    * Get the ref attribute of the UMLAssoc class
    *
    * @param theClass     No description provided
    * @param theRoleName  No description provided
    * @return             The ref value
    */
   public static UMLAssoc getRef (UMLClass theClass, String theRoleName)
   {
      // has the current class such a Role attached
      Enumeration theRolesEnum = theClass.elementsOfRoles();
      UMLRole theRole = null;

      while (theRolesEnum.hasMoreElements())
      {
         theRole = (UMLRole) theRolesEnum.nextElement();
         if (theRole.getPartnerRole() != null &&
            theRole.getPartnerRole().getAttrName() != null)
         {
            if (theRole.getPartnerRole().getAttrName().equals (theRoleName))
            {
               return theRole.getAssoc();
            }
         }
      }
      return null;
   } // getRef


   /**
    * Checks whether this is a unidirectional assoc.
    *
    * @return   The realReference value
    */
   public boolean isRealReference()
   {
      return  ( (getLeftRole().getAdornment() == FRole.REFERENCE) ||
          (getRightRole().getAdornment() == FRole.REFERENCE));
   } // isRealReference


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String name = null;


   /**
    * Get the name attribute of the UMLAssoc object
    *
    * @return   The name value
    */
   public String getName()
   {
      return name;
   }


   /**
    * Sets the name attribute of the UMLAssoc object
    *
    * @param name  The new name value
    */
   public void setName (String name)
   {
      if ( (this.name == null && name != null) ||
          (this.name != null && !this.name.equals (name)))
      {
         String oldValue = this.name;
         this.name = name;
         firePropertyChange (NAME_PROPERTY, oldValue, name);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int direction = LEFTRIGHT;


   /**
    * Get the direction attribute of the UMLAssoc object
    *
    * @return   The direction value
    */
   public int getDirection()
   {
      return direction;
   }


   /**
    * Sets the direction attribute of the UMLAssoc object
    *
    * @param direction  The new direction value
    */
   public void setDirection (int direction)
   {
      if (this.direction != direction)
      {
         int oldValue = this.direction;
         this.direction = direction;
         firePropertyChange (DIRECTION_PROPERTY, oldValue, direction);
      }
   }


   /**
    * Get the sortedComparator attribute of the UMLAssoc object. Looks at the roles in the
    * order given by {@link #getDirection()} to retrieve the comparator from them.
    *
    * @return       The sortedComparator value
    * @deprecated   Use {@link de.uni_paderborn.fujaba.uml.UMLRole#getSortedComparator} instead
    */
   public String getSortedComparator()
   {
      UMLRole firstRole;
      UMLRole secondRole;

      if (this.direction == RIGHTLEFT)
      {
         firstRole = this.rightRole;
         secondRole = this.leftRole;
      }
      else
      {
         firstRole = this.leftRole;
         secondRole = this.rightRole;
      }

      if (firstRole.getSortedComparator() != null &&
         secondRole.getAdornment() != FRole.REFERENCE)
      {
         return firstRole.getSortedComparator();
      }
      if (secondRole.getSortedComparator() != null &&
         firstRole.getAdornment() != FRole.REFERENCE)
      {
         return secondRole.getSortedComparator();
      }
      return null;
   }


   /**
    * Sets the sortedComparator attribute of the UMLAssoc object
    *
    * @param comparator  The new sortedComparator value
    * @deprecated        Use {@link de.uni_paderborn.fujaba.uml.UMLRole#setSortedComparator}
    *      instead
    */
   public void setSortedComparator (String comparator)
   {
      UMLRole firstRole;
      UMLRole secondRole;

      if (this.direction == RIGHTLEFT)
      {
         firstRole = this.rightRole;
         secondRole = this.leftRole;
      }
      else
      {
         firstRole = this.leftRole;
         secondRole = this.rightRole;
      }

      if (secondRole.getAdornment() != FRole.REFERENCE)
      {
         firstRole.setSortedComparator (comparator);
         if (secondRole.getSortedComparator() != null)
         {
            secondRole.setSortedComparator (null);
         }
      }
      if (firstRole.getAdornment() != FRole.REFERENCE)
      {
         secondRole.setSortedComparator (comparator);
         if (firstRole.getSortedComparator() != null)
         {
            firstRole.setSortedComparator (null);
         }
      }
   }


   // ######################################################################

   /**
    * Reverse UMLAssoc revLeftRole
    */
   private UMLRole leftRole;

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the leftRole attribute of the UMLAssoc object
    *
    * @return   The leftRole value
    */
   public UMLRole getLeftRole()
   {
      return leftRole;
   }


   /**
    * Get the fLeftRole attribute of the UMLAssoc object
    *
    * @return   The fLeftRole value
    */
   public FRole getFLeftRole()
   {
      return getLeftRole();
   }
   // TODO-END

   /**
    * Sets the leftRole attribute of the UMLAssoc object
    *
    * @param leftRole  The new leftRole value
    * @return          No description provided
    */
   public boolean setLeftRole (FRole leftRole)
   {
      if ( (this.leftRole == null && leftRole != null) ||
          (this.leftRole != null && !this.leftRole.equals (leftRole)))
      { // new partner

         UMLRole oldLeftRole = this.leftRole;
         if (this.leftRole != null)
         { // inform old partner

            this.leftRole = null;
            oldLeftRole.setRevLeftRole (null);
         }
         this.leftRole = (UMLRole) leftRole;
         if (leftRole != null)
         { // inform new partner

            this.leftRole.setRevLeftRole (this);
         }
         firePropertyChange (LEFT_ROLE_PROPERTY, oldLeftRole, leftRole);
         return true;
      }
      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final void removeLeftRole()
   {
      UMLRole incr = this.leftRole;
      if (incr != null)
      {
         this.setLeftRole (null);
         incr.removeYou();
      }
   }


   /**
    * Sets the sourceConnector attribute of the UMLAssoc object
    *
    * @param incr  The new sourceConnector value
    * @return      No description provided
    */
   public boolean setSourceConnector (FDiagramItem incr)
   {
      throw new UnsupportedOperationException ("Cannot set the connector for an UMLAssoc, " +
         "because it needs an UMLRole in between!");
   }


   /**
    * Get the sourceConnector attribute of the UMLAssoc object
    *
    * @return   The sourceConnector value
    */
   public UMLDiagramItem getSourceConnector()
   {
      UMLRole role = null;
      if (getDirection() == RIGHTLEFT)
      {
         role = getRightRole();
      }
      else
      {
         role = getLeftRole();
      }
      return  (role == null ? null : role.getTarget());
   }


   /**
    * Reverse UMLAssoc revRightRole
    */
   private UMLRole rightRole;

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the rightRole attribute of the UMLAssoc object
    *
    * @return   The rightRole value
    */
   public UMLRole getRightRole()
   {
      return rightRole;
   }


   /**
    * Get the fRightRole attribute of the UMLAssoc object
    *
    * @return   The fRightRole value
    */
   public FRole getFRightRole()
   {
      return getRightRole();
   }
   // TODO-END

   /**
    * Sets the rightRole attribute of the UMLAssoc object
    *
    * @param rightRole  The new rightRole value
    * @return           No description provided
    */
   public boolean setRightRole (FRole rightRole)
   {
      if ( (this.rightRole == null && rightRole != null) ||
          (this.rightRole != null && !this.rightRole.equals (rightRole)))
      { // new partner

         UMLRole oldRightRole = this.rightRole;
         if (this.rightRole != null)
         { // inform old partner

            this.rightRole = null;
            oldRightRole.setRevRightRole (null);
         }
         this.rightRole = (UMLRole) rightRole;
         if (rightRole != null)
         { // inform new partner

            this.rightRole.setRevRightRole (this);
         }
         firePropertyChange (RIGHT_ROLE_PROPERTY, oldRightRole, rightRole);
         return true;
      }
      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final void removeRightRole()
   {
      UMLRole incr = this.rightRole;
      if (incr != null)
      {
         this.setRightRole (null);
         incr.removeYou();
      }
   }


   /**
    * Sets the targetConnector attribute of the UMLAssoc object
    *
    * @param incr  The new targetConnector value
    * @return      No description provided
    */
   public boolean setTargetConnector (FDiagramItem incr)
   {
      throw new UnsupportedOperationException ("Cannot set the connector for an UMLAssoc, " +
         "because it needs an UMLRole in between!");
   }


   /**
    * Get the targetConnector attribute of the UMLAssoc object
    *
    * @return   The targetConnector value
    */
   public UMLDiagramItem getTargetConnector()
   {
      UMLRole role = null;
      if (getDirection() == RIGHTLEFT)
      {
         role = getLeftRole();
      }
      else
      {
         role = getRightRole();
      }
      return  (role == null ? null : role.getTarget());
   }


   /**
    * Get the partnerRole attribute of the UMLAssoc object
    *
    * @param role  No description provided
    * @return      The partnerRole value
    */
   public UMLRole getPartnerRole (UMLRole role)
   {
      UMLRole partnerRole = null;

      if (getLeftRole() == role)
      {
         partnerRole = getRightRole();
      }
      else if (getRightRole() == role)
      {
         partnerRole = getLeftRole();
      }
      else
      {
         throw new IllegalArgumentException();
      }

      return partnerRole;
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private FTreeSet instances = null;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    * @return      No description provided
    */
   public boolean hasInInstances (UMLLink elem)
   {
      if (this.instances == null)
      {
         return false;
      }
      return this.instances.contains (elem);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Enumeration elementsOfInstances()
   {
      return new EnumerationForAnIterator (iteratorOfInstances());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfInstances()
   {
      if (this.instances == null)
      {
         return FEmptyIterator.get();
      }
      return instances.iterator();
   }


   /**
    * Access method for an one to n association.
    *
    * @param elem  The object added.
    */
   public void addToInstances (UMLLink elem)
   {
      if (elem != null && !this.hasInInstances (elem))
      {
         if (this.instances == null)
         {
            this.instances = new FTreeSet();
         }
         this.instances.add (elem);
         elem.setInstanceOf (this);
         firePropertyChange (CollectionChangeEvent.get (this, "instances", this.instances, null,
            elem, CollectionChangeEvent.ADDED));
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    */
   public void removeFromInstances (UMLLink elem)
   {
      if (elem != null && this.hasInInstances (elem))
      {
         this.instances.remove (elem);
         elem.setInstanceOf (null);
         firePropertyChange (CollectionChangeEvent.get (this, "instances", this.instances, elem,
            null, CollectionChangeEvent.REMOVED));
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final void removeAllFromInstances()
   {
      UMLLink item = null;
      Iterator iter = iteratorOfInstances();

      while (iter.hasNext())
      {
         item = (UMLLink) iter.next();
         removeFromInstances (item);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean alreadyRemoved = false;


   /**
    * Get the alreadyRemoved attribute of the UMLAssoc object
    *
    * @return   The alreadyRemoved value
    */
   public boolean isAlreadyRemoved()
   {
      return alreadyRemoved;
   }


   /**
    * @return   whether this assoc should not generate any code
    */
   public boolean isVirtualPath()
   {
      return UMLStereotypeManager.get().getFromStereotypes (UMLStereotypeManager.VIRTUAL_PATH).hasInIncrements (this);
   }


   /**
    * Isolates the object so the garbage collector can remove it.
    */
   public void removeYou()
   {
      this.removeLeftRole();
      this.removeRightRole();
      this.removeAllFromInstances();
      alreadyRemoved = true;

      super.removeYou();
   }


   /**
    * Query the logical parent of this element (e.g. package of a class, diagram of an object).
    * An assoc is the child of the left target class!
    *
    * @return   the logical parent of this element;
    */
   public FElement getParentElement()
   {
      return UMLProject.get();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAttrs()
   {
      UMLRole role = getLeftRole();
      if (role != null)
      {
         UMLAttr attr = role.getAssociatedAttribute();
         if (attr != null)
         {
            attr.removeYou();
         }
      }
      role = getRightRole();
      if (role != null)
      {
         UMLAttr attr = role.getAssociatedAttribute();
         if (attr != null)
         {
            attr.removeYou();
         }
      }
   }


   /**
    * Get the text attribute of the UMLAssoc object
    *
    * @return   The text value
    */
   public String getText()
   {
      return getName();
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      StringBuffer result = new StringBuffer();

      result.append ("UMLAssoc[");
      result.append ("name=");
      result.append (getName());
      result.append (",leftRole=");
      result.append (getLeftRole());
      result.append (",rightRole=");
      result.append (getRightRole());
      result.append ("]");

      return result.toString();
   }


   /**
    * Sets the stereotype attribute of the UMLAssoc object
    *
    * @param obj    The new stereotype value
    * @deprecated   use stereotypes/UMLStereotype instead
    */
   public void setStereotype (FStereotype obj)
   {
      if (obj == null)
      {
         removeAllFromStereotypes();
      }
      else
      {
         addToStereotypes (obj);
      }
   }


   /**
    * Get the stereotype attribute of the UMLAssoc object
    *
    * @return       The stereotype value
    * @deprecated   use stereotypes/UMLStereotype instead
    */
   public UMLStereotype getStereotype()
   {
      return null;
   }


   /**
    * Get the stereotypeName attribute of the UMLAssoc object
    *
    * @return       The stereotypeName value
    * @deprecated   use stereotypes/UMLStereotype instead
    */
   public String getStereotypeName()
   {
      return "";
   }


   /**
    * @return   true if attributes for roles should be flagged transient
    */
   public boolean isRolesTransient()
   {
      return rolesTransient;
   }


   /**
    * @param rolesTransient  true if attributes for roles should be flagged transient
    */
   public void setRolesTransient (boolean rolesTransient)
   {
      if (this.rolesTransient != rolesTransient)
      {
         boolean oldValue = this.rolesTransient;
         this.rolesTransient = rolesTransient;
         firePropertyChange ("rolesTransient", oldValue, rolesTransient);
      }
   }


   /**
    * true if attributes for roles should be flagged transient
    */
   private boolean rolesTransient;

}

/*
 * $Log: UMLAssoc.java,v $
 * Revision 1.148.2.2  2006/01/09 14:32:34  mtt
 * fixed storing of comparator class for sorted assocs, code generation for sorted assocs does work now
 *
 */
