/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml;


import java.util.*;

import de.uni_paderborn.fujaba.app.FrameMain;
import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.basic.FujabaComparator;
import de.uni_paderborn.fujaba.basic.KeyValuePair;
import de.uni_paderborn.fujaba.coobra.FujabaChangeManager;
import de.uni_paderborn.fujaba.messages.Warning;
import de.uni_paderborn.fujaba.metamodel.*;
import de.upb.tools.fca.*;
import de.upb.tools.pcs.CollectionChangeEvent;
import de.upb.tools.sdm.Path;


/**
 * UMLClass is the core element of UMLClass-Diagrams.
 *
 * <h2>Associations</h2>
 *
 * <pre>
 *          +--------------------+ 1                1
 * UMLClass | getFullClassName() +-------------------- UMLMethod
 *          +--------------------+ parent     methods
 *
 *          +-----------+ 1              1
 * UMLClass | getName() +------------------ UMLAttr
 *          +-----------+ parent     attrs
 *
 *         +-----------+ 1                   1
 * UMLFile | getName() +----------------------- UMLClass
 *         +-----------+ file         contains
 *
 *          +-----------+ 1                         1
 * UMLClass | getName() +----------------------------- UMLMethod
 *          +-----------+ declares   declaredInMethod
 *
 *          +-----------+ 1                        1
 * UMLClass | getName() +---------------------------- UMLClass
 *          +-----------+ declares   declaredInClass
 *
 *          +-----------+ 1                          1
 * UMLClass | getName() +------------------------------ UMLPackage
 *          +-----------+ declares   declaredInPackage
 *
 *            +---------+ 1                                   1
 * UMLProject + name    +--------------------------------------- UMLClass
 *            +---------+ declaredInReferences       references
 *
 *                 parsedMembers          0..n
 * UMLClass -----------------------------------> UMLDeclaration
 *                               parsedMembers
 * </pre>
 *
 * @author    $Author: creckord $
 * @version   $Revision: 1.461.2.10 $
 */
public class UMLClass extends UMLDeclaration implements UMLType, FClass
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String PROPERTY_NAME = FElement.NAME_PROPERTY;


   /**
    * Constructor for class UMLClass
    */
   public UMLClass()
   {
      super();
      setVisibility (getVisibility()); // ensure that attribute and getter are in sync
   } // UMLClass


   /**
    * Constructor for class UMLClass
    *
    * @param coobraPersistent  No description provided
    */
   public UMLClass (boolean coobraPersistent)
   {
      super (coobraPersistent);
   }


   /**
    * Create a UMLClass with specified name and {@link UMLStereotypeManager#REFERENCE} as Stereotype
    * if reference parameter is true
    *
    * @param name       name of the new class
    * @param reference  if true the reference stereotype is added as stereotype of the class
    */
   public UMLClass (String name, boolean reference)
   {
      this (name);
      addToStereotypes (UMLStereotypeManager.get().getFromStereotypes (UMLStereotypeManager.REFERENCE));
   }


   /**
    * Constructor for class UMLClass
    *
    * @param name        No description provided
    * @param stereotype  No description provided
    * @param typeList    No description provided
    * @param declaredIn  No description provided
    */
   public UMLClass (String name, UMLStereotype stereotype,
                    UMLTypeList typeList, UMLPackage declaredIn)
   {
      this();

      // before name is set, package of the class must be set
      if (declaredIn != null)
      {
         setDeclaredInPackage (declaredIn);
      }
      else if (this.declaredInPackage == null && this.declaredInClass == null &&
         this.declaredInMethod == null && !FujabaChangeManager.isInUndoRedo())
      {
         setDeclaredInPackage (UMLProject.get().getDefaultPackage());
      }

      setName (name);
      addToStereotypes (stereotype);
      if (typeList == null)
      {
         typeList = UMLProject.get().getTypeList();
      }
      setRevTypes (typeList);

      UMLStereotype referenceType = UMLStereotypeManager.get().
         getFromStereotypes (UMLStereotypeManager.REFERENCE);

      if (hasInStereotypes (referenceType) ||  (this.declaredInClass == null &&
         this.declaredInMethod == null && this.declaredInPackage == null))
      {
         setDeclaredInReferences (UMLProject.get());
      }
   } // UMLClass


   /**
    * Constructor for class UMLClass
    *
    * @param name  No description provided
    */
   public UMLClass (String name)
   {
      this (name, null,
         UMLProject.get().getTypeList(),
         UMLProject.get().getDefaultPackage());
   } // UMLClass


   /**
    * Get the project attribute of the UMLClass object
    *
    * @return   The project value
    */
   public UMLProject getProject()
   {
      if (this.declaredInReferences != null)
      {
         return this.declaredInReferences;
      }
      return UMLProject.get();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String name;


   /**
    * Get the name attribute of the UMLClass object
    *
    * @return   The name value
    */
   public String getName()
   {
      return this.name;
   } // getName


   /**
    * Sets the name attribute of the UMLClass object.
    * <pre>
    *
    * The new name of the UMLClass can either be a short name,
    * a full qualified name or an inner class definition in Java-style.
    * Note that the intention of this method is only
    * to set the SHORT name of the class!
    * To specify the full qualified name, use method 'setDeclaredInPackage'.
    * To define an inner class, use method 'setDeclaredInClass'.
    *
    * Example:
    *   Map (short)
    *   java.util.Map (full qualified)
    *   java.util.Map$Entry (inner class)
    *   java.util.Map.Entry (inner class, alternative)
    * </pre>
    *
    * @param name  The new name value
    */
   public void setName (String name)
   {
      if (name == null)
      {
         return;
      }

      String oldName = getName();
      String shortName = name;

      // extract package prefix from name
      String packageName = UMLProject.getPackageOfFullQualifiedType (name);

      // extract short name if the name describes an inner class
      String parentClassFullName = UMLProject.getOuterClassOfFullQualifiedType (name);
      if (!"".equals (parentClassFullName))
      {
         shortName = name.substring (parentClassFullName.length() + 1);
      }
      else
      {
         if (packageName.length() > 0)
         {
            shortName = name.substring (packageName.length() + 1);
         }
      }

      // we have to make sure that some other objects
      // change the key used to identify this UMLClass,
      // because the name (which is part of the key) is
      // about to change
      UMLTypeList oldTypeList = this.getRevTypes();
      this.setRevTypes (null);

      UMLProject oldPrj = this.getDeclaredInReferences();
      this.setDeclaredInReferences (null);

      UMLPackage oldPack = this.getDeclaredInPackage();
      this.setDeclaredInPackage (null);

      UMLClass oldClass = this.getDeclaredInClass();
      this.setDeclaredInClass (null);

      UMLMethod oldMethod = this.getDeclaredInMethod();
      this.setDeclaredInMethod (null);

      UMLFile oldFile = this.getFile();
      this.setFile (null);

      // TODO: what if this UMLClass declares other UMLClasses
      // (i.e., has inner classes)?
      // do we have to inform them about the name-change?


      // BEGIN change the (full qualified) name
      if (this.name == null || !this.name.equals (shortName))
      {
         this.name = shortName;
         firePropertyChange (FElement.NAME_PROPERTY, oldName, shortName);
      }

      this.setDeclaredInReferences (oldPrj);
      this.setDeclaredInMethod (oldMethod);

      // this class should be an inner class
      if (!parentClassFullName.equals (""))
      {
         UMLClass parentClass = getProject().getOrNewClass (parentClassFullName);
         setDeclaredInClass (parentClass);
         // the package of an inner class is defined by its outer class
         setDeclaredInPackage (null);
      }
      // this class should be contained in a package
      else if (!packageName.equals (""))
      {
         UMLPackage tmpPkg = getProject().getNewFromPackages (packageName);
         setDeclaredInPackage (tmpPkg);
      }
      else
      {
         this.setDeclaredInPackage (oldPack);
         this.setDeclaredInClass (oldClass);
      }

      // END change the (full qualified) name
      //////////

      // if the (full qualified) name of this UMLClass has changed,
      // we have to inform all partners, that use this name as a key
      this.setRevTypes (oldTypeList);

      this.setFile (oldFile);

      //----- reestablish assocs
      // if the name changes inform all UMLParams
      Iterator paramIter = new Path (this, "revParamType.revParam");
      UMLMethod tmpMethod;
      while (paramIter.hasNext())
      {
         tmpMethod = (UMLMethod) paramIter.next();
         tmpMethod.updateAssocOverFullMethodName();
      }

      // TODO: why do we fire another property-change-event?
      if (this.name == null || !this.name.equals (shortName))
      {
         firePropertyChange (FElement.NAME_PROPERTY, oldName, shortName);
      }
   } // setName


   /**
    * Get the text attribute of the UMLClass object
    *
    * @return   The text value
    */
   public String getText()
   {
      return this.getName();
   } // getText


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public String toString()
   {
      return this.getName();
   } // toString


   /**
    * This function computs the full qualified classname of this class e. g. if the class <code>FooBar</code>
    * is defined in the package <code>upb</code> and <code>upb</code> is in <code>de</code>
    * then this function will return <code>de.upb.FooBar</code>
    *
    * @return   the full qualified class name
    */
   public String getFullClassName()
   {
      String strg = this.getName();
      UMLClass outerClass = getDeclaredInClass();
      UMLPackage pack = getDeclaredInPackage();

      if (outerClass != null)
      {
         strg = outerClass.getFullClassName() + "$" + strg;
      }
      else if (pack != null &&
          (pack != getProject().getDefaultPackage() &&
         pack != getProject().getRootPackage()))
      {
         strg = pack.getFullPackageName() + "." + strg;
      }
      return strg;
   } // getFullClassName


   /**
    * Get the typeName attribute of the UMLClass object
    *
    * @return   The typeName value
    */
   public String getTypeName()
   {
      return getFullClassName();
   }


   /**
    * Sets the typeName attribute of the UMLClass object
    *
    * @param name  The new typeName value
    */
   public void setTypeName (String name)
   {
      setName (name);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String defaultIcon;


   /**
    * Get the defaultIcon attribute of the UMLClass object
    *
    * @return   The defaultIcon value
    */
   public String getDefaultIcon()
   {
      return this.defaultIcon;
   }


   /**
    * Sets the defaultIcon attribute of the UMLClass object
    *
    * @param newVal  The new defaultIcon value
    */
   public void setDefaultIcon (String newVal)
   {
      if (newVal == null || !newVal.equals (this.defaultIcon))
      {
         String oldVal = this.defaultIcon;
         this.defaultIcon = newVal;
         firePropertyChange ("defaultIcon", oldVal, newVal);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean umlStatic = false;


   /**
    * Sets the static attribute of the UMLClass object
    *
    * @param umlStatic  The new static value
    */
   public void setStatic (boolean umlStatic)
   {
      boolean oldValue = this.umlStatic;
      this.umlStatic = umlStatic;

      firePropertyChange ("umlStatic", oldValue, umlStatic);
   } // setStatic


   /**
    * Get the static attribute of the UMLClass object
    *
    * @return   The static value
    */
   public boolean isStatic()
   {
      return umlStatic;
   } // isStatic


   /**
    * if (codeGenDenied) UMLProject is not allowed to create the corresponding UMLFile
    */
   private boolean codeGenDenied;


   /**
    * Get the value of codeGenDenied.
    *
    * @return   Value of codeGenDenied.
    */
   public boolean isCodeGenDenied()
   {
      return this.codeGenDenied;
   }


   /**
    * Set the value of codeGenDenied.
    *
    * @param codeGenDenied  Value to assign to codeGenDenied.
    */
   public void setCodeGenDenied (boolean codeGenDenied)
   {
      if (this.codeGenDenied != codeGenDenied)
      {
         this.codeGenDenied = codeGenDenied;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean umlAbstract = false;


   /**
    * Sets the umlAbstract attribute of the UMLClass object
    *
    * @param umlAbstract  The new umlAbstract value
    * @return             No description provided
    */
   public boolean setUmlAbstract (boolean umlAbstract)
   {
      boolean changed = false;
      if (this.umlAbstract != umlAbstract)
      {
         this.umlAbstract = umlAbstract;
         changed = true;
         firePropertyChange ("umlAbstract", !this.umlAbstract, this.umlAbstract);
      }
      return changed;
   } // setUmlAbstract


   /**
    * Sets the abstract attribute of the UMLClass object
    *
    * @param umlAbstract  The new abstract value
    * @return             No description provided
    */
   public boolean setAbstract (boolean umlAbstract)
   {
      return setUmlAbstract (umlAbstract);
   }


   /**
    * Get the umlAbstract attribute of the UMLClass object
    *
    * @return   The umlAbstract value
    */
   public boolean isUmlAbstract()
   {
      return this.umlAbstract;
   } // isUmlAbstract


   /**
    * Get the abstract attribute of the UMLClass object
    *
    * @return   The abstract value
    */
   public boolean isAbstract()
   {
      return isUmlAbstract();
   }


   /**
    * Get the final attribute of the UMLClass object
    *
    * @return   The final value
    */
   public boolean isFinal()
   {
      return false;
   } // isFinal


   /**
    * Get the visibility attribute of the UMLClass object
    *
    * @return   The visibility value
    */
   public int getVisibility()
   {
      return  (this.getDeclaredInPackage() != null)
         ? FDeclaration.PUBLIC
         : FDeclaration.PACKAGE;
   } // isPublic


   /**
    * Repair the visibility attribute. Needed for import in Fujaba 5.
    *
    * @param visibility
    */
   public void setVisibility (int visibility)
   {
      super.setVisibility (visibility);
      super.setVisibility (getVisibility());
   }


   /**
    * Get the protected attribute of the UMLClass object
    *
    * @return   The protected value
    */
   public boolean isProtected()
   {
      return false;
   } // isProtected


   /**
    * UMLMethod: isReference : boolean
    *
    * @return   The reference value
    */
   public boolean isReference()
   {
      UMLStereotype referenceType = UMLStereotypeManager.get().
         getFromStereotypes (UMLStereotypeManager.REFERENCE);

      return hasInStereotypes (referenceType);
   } // isReference


   /**
    * <pre>
    *         +-----------+ 1                   1
    * UMLFile | getName() +----------------------- UMLClass
    *         +-----------+ file         contains
    * </pre>
    */
   private UMLFile file;


   /**
    * Sets the file attribute of the UMLClass object
    *
    * @param obj  The new file value
    * @return     No description provided
    */
   public boolean setFile (FFile obj)
   {
      boolean changed = false;

      if (this.file != obj)
      {
         UMLFile oldValue = this.file;
         if (oldValue != null)
         {
            this.file = null;
            oldValue.removeFromContains (this);
         }
         this.file = (UMLFile) obj;
         if (obj != null)
         {
            this.file.addToContains (this);
         }
         changed = true;

         firePropertyChange (FILE_PROPERTY, oldValue, this.file);
      }

      return changed;
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the file attribute of the UMLClass object
    *
    * @return   The file value
    */
   public UMLFile getFile()
   {
      if ( (file == null) &&  (this.getDeclaredInPackage() != null))
      {
         UMLFile tmpFile = new UMLFile (this.getName());
         this.setFile (tmpFile);
      }

      return this.file;
   }


   /**
    * Get the fFile attribute of the UMLClass object
    *
    * @return   The fFile value
    * @see      de.uni_paderborn.fujaba.metamodel.FClass#getFFile()
    */
   public FFile getFFile()
   {
      return getFile();
   }
   // TODO-END

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  No description provided
    */
   public void removeFromDiagrams (FDiagram obj)
   {
      if (obj != null)
      {
         super.removeFromDiagrams (obj);
         Iterator iter;

         //----- remove all assocs
         UMLAssoc assoc;
         UMLRole role;
         iter = iteratorOfRoles();
         while (iter.hasNext())
         {
            role = (UMLRole) iter.next();
            assoc = role.getAssoc();
            obj.removeFromElements (assoc);
         }

         //----- remove all inheritance
         UMLGeneralization gen;
         iter = iteratorOfRevSubclass();
         while (iter.hasNext())
         {
            gen = (UMLGeneralization) iter.next();
            obj.removeFromElements (gen);
         }
         iter = iteratorOfRevSuperclass();
         while (iter.hasNext())
         {
            gen = (UMLGeneralization) iter.next();
            obj.removeFromElements (gen);
         }
      }
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private FTreeSet revImportedClasses = new FTreeSet();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    * @return      No description provided
    */
   public boolean hasInRevImportedClasses (FFile elem)
   {
      return this.revImportedClasses.contains (elem);
   } // hasInRevImportedClasses


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Enumeration elementsOfRevImportedClasses()
   {
      return new EnumerationForAnIterator (iteratorOfRevImportedClasses());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfRevImportedClasses()
   {
      return revImportedClasses.iterator();
   }


   /**
    * Access method for an one to n association.
    *
    * @param elem  The object added.
    */
   public void addToRevImportedClasses (FFile elem)
   {
      if (elem != null && !this.hasInRevImportedClasses (elem))
      {
         this.revImportedClasses.add (elem);
         elem.addToImportedClasses (this);
         firePropertyChange (CollectionChangeEvent.get (this, "revImportedClasses", this.revImportedClasses, null,
            elem, CollectionChangeEvent.ADDED));
      }
   } // addToRevImportedClasses


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    */
   public void removeFromRevImportedClasses (FFile elem)
   {
      if (this.hasInRevImportedClasses (elem))
      {
         this.revImportedClasses.remove (elem);
         elem.removeFromImportedClasses (this);
         firePropertyChange (CollectionChangeEvent.get (this, "revImportedClasses", this.revImportedClasses, elem,
            null, CollectionChangeEvent.REMOVED));
      }
   } // removeFromRevImportedClasses


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final void removeAllFromRevImportedClasses()
   {
      UMLFile item;
      Iterator iter = iteratorOfRevImportedClasses();

      while (iter.hasNext())
      {
         item = (UMLFile) iter.next();
         item.removeFromImportedClasses (this);
         firePropertyChange (CollectionChangeEvent.get (this, "revImportedClasses", this.revImportedClasses, item,
            null, CollectionChangeEvent.REMOVED));
      }
   } // removeAllRevImportedClasses


   // ######################################################################

   /**
    * Implements the association of UMLType. Used for example for parsed attributes.
    */
   private transient FPropTreeSet revAttrType = null;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInRevAttrType (FAttr value)
   {
      return  ( (this.revAttrType != null) &&
          (value != null) &&
         this.revAttrType.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfRevAttrType()
   {
      return  ( (this.revAttrType == null)
         ? FEmptyIterator.get()
         : this.revAttrType.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfRevAttrType()
   {
      return  ( (this.revAttrType == null)
         ? 0
         : this.revAttrType.size());
   }


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToRevAttrType (FAttr value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.revAttrType == null)
         {
            this.revAttrType = new FPropTreeSet (this, "revAttrType");
         }
         changed = this.revAttrType.add (value);
         if (changed)
         {
            value.setAttrType (this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromRevAttrType (FAttr value)
   {
      boolean changed = false;
      if ( (this.revAttrType != null) &&  (value != null))
      {
         changed = this.revAttrType.remove (value);
         if (changed)
         {
            value.setAttrType ((UMLType) null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromRevAttrType()
   {
      UMLAttr tmpValue;
      Iterator iter = this.iteratorOfRevAttrType();
      while (iter.hasNext())
      {
         tmpValue = (UMLAttr) iter.next();
         this.removeFromRevAttrType (tmpValue);
      }
   }


   // ######################################################################

   /**
    * Used for keeping the order of the parsed attributes, methods and inner classes.
    *
    * <pre>
    *                 parsedMembers          0..n
    * UMLClass -----------------------------------> UMLDeclaration
    *                               parsedMembers
    * </pre>
    */
   private FLinkedList parsedMembers;


   /**
    * Access method for a To N-association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToParsedMembers (FDeclaration value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.parsedMembers == null)
         {
            this.parsedMembers = new FLinkedList();
         }

         changed = this.parsedMembers.add (value);
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInParsedMembers (FDeclaration value)
   {
      return  ( (this.parsedMembers != null) &&
          (value != null) &&
         this.parsedMembers.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfParsedMembers()
   {
      return  ( (this.parsedMembers == null)
         ? FEmptyIterator.get()
         : this.parsedMembers.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromParsedMembers()
   {
      FDeclaration tmpValue;
      Iterator iter = this.iteratorOfParsedMembers();
      while (iter.hasNext())
      {
         tmpValue = (FDeclaration) iter.next();
         this.removeFromParsedMembers (tmpValue);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromParsedMembers (FDeclaration value)
   {
      boolean changed = false;
      if ( (this.parsedMembers != null) &&  (value != null))
      {
         changed = this.parsedMembers.remove (value);
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfParsedMembers()
   {
      return  ( (this.parsedMembers == null)
         ? 0
         : this.parsedMembers.size());
   }


   // ######################################################################

   /**
    * <pre>
    *          +-----------+ 1              1
    * UMLClass | getName() +------------------ UMLAttr
    *          +-----------+ parent     attrs
    * </pre>
    */
   private FPropTreeMap attrs;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public boolean hasInAttrs (FAttr obj)
   {
      return  ( (this.attrs != null) &&
          (obj != null) &&  (obj.getName() != null) &&
          (this.attrs.get (obj.getName()) == obj));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean hasKeyInAttrs (String key)
   {
      return  ( (this.attrs != null) &&
          (key != null) &&
         this.attrs.containsKey (key));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfAttrs()
   {
      return  ( (this.attrs == null)
         ? FEmptyIterator.get()
         : this.attrs.values().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator keysOfAttrs()
   {
      return  ( (this.attrs == null)
         ? FEmptyIterator.get()
         : this.attrs.keySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator entriesOfAttrs()
   {
      return  ( (this.attrs == null)
         ? FEmptyIterator.get()
         : this.attrs.entrySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfAttrs()
   {
      return  ( (this.attrs == null)
         ? 0
         : this.attrs.size());
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the fromAttrs attribute of the UMLClass object
    *
    * @param key  No description provided
    * @return     The fromAttrs value
    */
   public UMLAttr getFromAttrs (String key)
   {
      return  ( ( (this.attrs == null) ||  (key == null))
         ? null
         : (UMLAttr) this.attrs.get (key));
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FClass#getFromFAttrs(java.lang.String)
    */
   /**
    * Get the fromFAttrs attribute of the UMLClass object
    *
    * @param key  No description provided
    * @return     The fromFAttrs value
    */
   public FAttr getFromFAttrs (String key)
   {
      return getFromAttrs (key);
   }
   // TODO-END


   /**
    * Access method for an one to n association.
    *
    * @param obj  The object added.
    * @return     No description provided
    */
   public boolean addToAttrs (FAttr obj)
   {
      boolean changed = false;

      if ( (obj != null) &&  (obj.getName() != null))
      {
         if (this.attrs == null)
         {
            this.attrs = new FPropTreeMap (FujabaComparator.getLessString(),
               this, ATTRS_PROPERTY);
         }
         UMLAttr oldValue = (UMLAttr) this.attrs.put (obj.getName(), obj);
         if (oldValue != obj)
         {
            if (oldValue != null)
            {
               oldValue.setParent (null);
               oldValue.removeYou();
            }
            obj.setParent (this);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * this method should be protected in order to enable the fpr file mechanism for subclasses,
    * too. AZ
    *
    * @param entry  The object added.
    */
   protected void addToAttrs (Map.Entry entry)
   {
      String key = (String) entry.getKey();
      UMLAttr attr = (UMLAttr) entry.getValue();

      attr.setName (key);
      addToAttrs (attr);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public boolean removeFromAttrs (FAttr obj)
   {
      boolean changed = false;

      if ( (this.attrs != null) &&  (obj != null) &&  (obj.getName() != null))
      {
         UMLAttr oldValue = (UMLAttr) this.attrs.get (obj.getName());
         if (oldValue == obj)
         {
            this.attrs.remove (obj.getName());
            obj.setParent (null);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean removeKeyFromAttrs (String key)
   {
      boolean changed = false;

      if ( (this.attrs != null) &&  (key != null))
      {
         UMLAttr tmpObj = (UMLAttr) this.attrs.get (key);
         if (tmpObj != null)
         {
            this.attrs.remove (key);
            tmpObj.setParent (null);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromAttrs()
   {
      Iterator iter = this.iteratorOfAttrs();
      while (iter.hasNext())
      {
          ((UMLAttr) iter.next()).removeYou();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Map collectionOfAttrs()
   {
      return this.attrs;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Enumeration elementsOfAllChildren()
   {
      return new EnumerationForAnIterator (Path.iterUnion (Path.iterUnion (iteratorOfMethods(), iteratorOfAttrs()), iteratorOfDeclares()));
   } // elementsOfAllChildren


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private FTreeSet roles = new FTreeSet();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfRoles()
   {
      return roles.size();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    * @return      No description provided
    */
   public boolean hasInRoles (FRole elem)
   {
      return this.roles.contains (elem);
   } // hasInRoles


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Enumeration elementsOfRoles()
   {
      return new EnumerationForAnIterator (iteratorOfRoles());
   } // elementsOfRoles


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfRoles()
   {
      return roles.iterator();
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the fromRoles attribute of the UMLClass object
    *
    * @param roleName  No description provided
    * @return          The fromRoles value
    */
   public UMLRole getFromRoles (String roleName)
   {
      Enumeration rolesEnum = this.elementsOfRoles();
      UMLRole tmpRole = null;
      while (rolesEnum.hasMoreElements())
      {
         tmpRole =  ((UMLRole) rolesEnum.nextElement());
         if (roleName.equals (tmpRole.getName()))
         {
            return tmpRole;
         }
      }
      return null;
   } // getFromRoles


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FClass#getFromFRoles(java.lang.String)
    */
   /**
    * Get the fromFRoles attribute of the UMLClass object
    *
    * @param roleName  No description provided
    * @return          The fromFRoles value
    */
   public FRole getFromFRoles (String roleName)
   {
      return getFromRoles (roleName);
   }
   // TODO-END

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the fromRevRoles attribute of the UMLClass object
    *
    * @param roleName  No description provided
    * @return          The fromRevRoles value
    */
   public UMLRole getFromRevRoles (String roleName)
   {
      Enumeration rolesEnum = this.elementsOfRoles();
      UMLRole tmpRole = null;
      while (rolesEnum.hasMoreElements())
      {
         tmpRole =  ((UMLRole) rolesEnum.nextElement());
         tmpRole = tmpRole.getPartnerRole();
         if (tmpRole.getName().equals (roleName))
         {
            return tmpRole;
         }
      }
      return null;
   } // getFromRoles


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FClass#getFromFRevRoles(java.lang.String)
    */
   /**
    * Get the fromFRevRoles attribute of the UMLClass object
    *
    * @param roleName  No description provided
    * @return          The fromFRevRoles value
    */
   public FRole getFromFRevRoles (String roleName)
   {
      return getFromRevRoles (roleName);
   }
   // TODO-END

   /**
    * Access method for an one to n association.
    *
    * @param elem  The object added.
    */
   public void addToRoles (FRole elem)
   {
      if (elem != null && !this.hasInRoles (elem))
      {
         this.roles.add (elem);
         elem.setTarget (this);

         firePropertyChange (CollectionChangeEvent.get (this, ROLES_PROPERTY, this.roles, null,
            elem, CollectionChangeEvent.ADDED));
      }
   } // addToRoles


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    */
   public void removeFromRoles (FRole elem)
   {
      if (this.hasInRoles (elem))
      {
         this.roles.remove (elem);
         elem.setTarget (null);

         firePropertyChange (CollectionChangeEvent.get (this, ROLES_PROPERTY, this.roles, elem,
            null, CollectionChangeEvent.REMOVED));
      }
   } // removeFromRoles


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final void removeAllFromRoles()
   {
      UMLRole item;
      Iterator iter = iteratorOfRoles();

      while (iter.hasNext())
      {
         item = (UMLRole) iter.next();
         item.setTarget (null);
         item.removeYou();
         firePropertyChange (CollectionChangeEvent.get (this, ROLES_PROPERTY, this.roles, item,
            null, CollectionChangeEvent.REMOVED));
      }
   } // removeAllRoles


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private FTreeSet instances = new FTreeSet();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    * @return      No description provided
    */
   public boolean hasInInstances (UMLObject elem)
   {
      return this.instances.contains (elem);
   } // hasInInstances


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Enumeration elementsOfInstances()
   {
      return new EnumerationForAnIterator (iteratorOfInstances());
   } // elementsOfInstances


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfInstances()
   {
      return instances.iterator();
   }


   /**
    * Access method for an one to n association.
    *
    * @param elem  The object added.
    */
   public void addToInstances (UMLObject elem)
   {
      if (elem != null && !this.hasInInstances (elem))
      {
         this.instances.add (elem);
         elem.setInstanceOf (this);

         firePropertyChange (CollectionChangeEvent.get (this, "instances", this.instances, null,
            elem, CollectionChangeEvent.ADDED));
      }
   } // addToInstances


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    */
   public void removeFromInstances (UMLObject elem)
   {
      if (this.hasInInstances (elem))
      {
         this.instances.remove (elem);
         elem.setInstanceOf (null);
         firePropertyChange (CollectionChangeEvent.get (this, "instances", this.instances, elem,
            null, CollectionChangeEvent.REMOVED));
      }
   } // removeFromInstances


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final void removeAllFromInstances()
   {

      UMLObject item;
      Iterator iter = iteratorOfInstances();

      while (iter.hasNext())
      {
         item = (UMLObject) iter.next();
         item.setInstanceOf (null);
         firePropertyChange (CollectionChangeEvent.get (this, "instances", this.instances, item,
            null, CollectionChangeEvent.REMOVED));
      }
   } // removeAllInstances


   /**
    * Get the uMLType attribute of the UMLClass object
    *
    * @return   The uMLType value
    */
   public UMLType getUMLType()
   {
      return this;
   }


   /**
    * <pre>
    *          +---------------------+ 0..1              0..1
    * UMLClass | getFullMethodName() +------------------------ UMLMethod
    *          +---------------------+ parent         methods
    * </pre>
    */
   private Set methods = null;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public boolean hasInMethods (FMethod obj)
   {
      return methods != null ? methods.contains (obj) : false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean hasKeyInMethods (String key)
   {
      return getFromMethods (key) != null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfMethods()
   {
      return  ( (this.methods == null)
         ? FEmptyIterator.get()
         : this.methods.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfMethods()
   {
      return  ( (this.methods == null)
         ? 0
         : this.methods.size());
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the fromMethods attribute of the UMLClass object
    *
    * @param key  No description provided
    * @return     The fromMethods value
    */
   public UMLMethod getFromMethods (String key)
   {
      if (key != null)
      {
         for (Iterator it = iteratorOfMethods(); it.hasNext(); )
         {
            UMLMethod method = (UMLMethod) it.next();
            if (key.equals (method.getFullMethodName()))
            {
               return method;
            }
         }
      }
      return null;
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FClass#getFromFMethods(java.lang.String)
    */
   /**
    * Get the fromFMethods attribute of the UMLClass object
    *
    * @param key  No description provided
    * @return     The fromFMethods value
    */
   public FMethod getFromFMethods (String key)
   {
      return getFromMethods (key);
   }
   // TODO-END

   /**
    * Access method for an one to n association.
    *
    * @param obj  The object added.
    * @return     No description provided
    */
   public boolean addToMethods (FMethod obj)
   {
      boolean changed = false;

      if (obj != null)
      {
         if (this.methods == null)
         {
            this.methods = new FPropHashSet (this, METHODS_PROPERTY); //can't be treeset as method name changes!
         }

         changed = this.methods.add (obj);
         if (changed)
         {
            obj.setParent (this);
            if (obj.isSignal())
            {
               firePropertyChange (CollectionChangeEvent.get (this, "signals", methods, null,
                  obj, obj.getFullMethodName(),
                  CollectionChangeEvent.ADDED));
            }
            changed = true;
         }
      }
      return changed;
   }


   /**
    * Access method for an one to n association.
    *
    * @param entry  The object added.
    */
   private void addToMethods (Map.Entry entry)
   {
      String key = (String) entry.getKey();
      UMLMethod method = (UMLMethod) entry.getValue();

      if (method != null)
      {
         method.setFullMethodName (key);
      } // end of if ()

      this.addToMethods (method);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public boolean removeFromMethods (FMethod obj)
   {
      boolean changed = false;

      if ( (this.methods != null) &&
          (obj != null))
      {
         changed = methods.remove (obj);
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean removeKeyFromMethods (String key)
   {
      boolean changed = false;
      UMLMethod method;
      while ( (method = getFromMethods (key)) != null)
      {
         removeFromMethods (method);
         changed = true;
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromMethods()
   {
      Iterator iter = this.iteratorOfMethods();
      while (iter.hasNext())
      {
          ((UMLMethod) iter.next()).removeYou();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public boolean hasInSignals (UMLMethod obj)
   {
      return  (obj != null && obj.isSignal() && hasInMethods (obj));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean hasKeyInSignals (String key)
   {
      UMLMethod signal = getFromMethods (key);
      return  (signal != null && signal.isSignal());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfSignals()
   {
      Iterator methods = iteratorOfMethods();
      return new FFilterIterator (SignalFilter.get(), methods);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfSignals()
   {
      int size = 0;
      Iterator iter = iteratorOfSignals();
      while (iter.hasNext())
      {
         iter.next();
         size++;
      }
      return size;
   }


   /**
    * Get the fromSignals attribute of the UMLClass object
    *
    * @param key  No description provided
    * @return     The fromSignals value
    */
   public UMLMethod getFromSignals (String key)
   {
      UMLMethod method = getFromMethods (key);
      return  (method.isSignal() ? method : null);
   }


   /**
    * Access method for an one to n association.
    *
    * @param obj  The object added.
    * @return     No description provided
    */
   public boolean addToSignals (UMLMethod obj)
   {
      boolean changed = false;

      if ( (obj != null) &&  (obj.getFullMethodName() != null))
      {
         obj.setSignal (true);
         changed = addToMethods (obj);
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public boolean removeFromSignals (UMLMethod obj)
   {
      boolean changed = false;

      if ( (obj != null) &&  (obj.getFullMethodName() != null) &&  (obj.isSignal()))
      {
         changed = removeFromMethods (obj);
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean removeKeyFromSignals (String key)
   {
      boolean changed = false;

      if (key != null)
      {
         UMLMethod tmpObj = getFromMethods (key);
         if (tmpObj != null && tmpObj.isSignal())
         {
            changed = removeKeyFromMethods (key);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromSignals()
   {
      Iterator iter = this.iteratorOfSignals();
      while (iter.hasNext())
      {
          ((UMLMethod) iter.next()).removeYou();
      }
   }

   // ######################################################################

   // ---- old stuff for the methods attribut

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param method  No description provided
    */
   public void replaceMethod (UMLMethod method)
   {
      UMLMethod oldMethod = this.getFromMethods (method.getFullMethodName());

      if (oldMethod != null)
      {
         oldMethod.removeYou();
      }

      this.addToMethods (method);
   }


   /**
    * Searches the ASG tree for a given id
    *
    * @param id  The id to search for.
    * @return    The Element with the given id, null if not found.
    */
   public ASGElement searchID (String id)
   {
      ASGElement elem = super.searchID (id);
      Iterator iter = iteratorOfMethods();
      while ( (elem == null) &&  (iter.hasNext()))
      {
         elem =  ((UMLMethod) iter.next()).searchID (id);
      }
      iter = iteratorOfAllAttrs();
      while ( (elem == null) &&  (iter.hasNext()))
      {
         elem =  ((UMLAttr) iter.next()).searchID (id);
      }
      iter = iteratorOfRoles();
      while ( (elem == null) &&  (iter.hasNext()))
      {
         elem =  ((UMLRole) iter.next()).searchID (id);
      }
      if (elem == null)
      {
         UMLFile f = getFile();
         if (f != null)
         {
            elem = getFile().searchID (id);
         }
      }
      return elem;
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the fromMethodsByShortName attribute of the UMLClass object
    *
    * @param key  No description provided
    * @return     The fromMethodsByShortName value
    */
   public UMLMethod getFromMethodsByShortName (String key)
   {
      Iterator iter = this.iteratorOfMethods();
      while (iter.hasNext())
      {
         UMLMethod tmpMethod = (UMLMethod) iter.next();
         if (tmpMethod.getName().equals (key))
         {
            return tmpMethod;
         }
      }
      return null;
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FClass#getFromFMethodsByShortName(java.lang.String)
    */
   /**
    * Get the fromFMethodsByShortName attribute of the UMLClass object
    *
    * @param name  No description provided
    * @return      The fromFMethodsByShortName value
    */
   public FMethod getFromFMethodsByShortName (String name)
   {
      return getFromMethodsByShortName (name);
   }
   // TODO-END


   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Searches recursivly the inheritance hierarchy and looks for methods.
    *
    * @param key  No description provided
    * @return     The fromMethodsByShortNameIncludingInherited value
    */
   public UMLMethod getFromMethodsByShortNameIncludingInherited (String key)
   {
      // get method from this class
      UMLMethod firstMethodFound = getFromMethodsByShortName (key);

      if (firstMethodFound == null)
      {
         // get method from super-classes.
         UMLClass tmpClass;
         Enumeration enumGen = elementsOfRevSubclass();
         while (enumGen.hasMoreElements() && firstMethodFound == null)
         {
            tmpClass =  ((UMLGeneralization) enumGen.nextElement()).getSuperclass();
            firstMethodFound = tmpClass.getFromMethodsByShortNameIncludingInherited (key);
         }
      }
      return firstMethodFound;
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FClass#getFromFMethodsByShortNameIncludingInherited(java.lang.String)
    */
   /**
    * Get the fromFMethodsByShortNameIncludingInherited attribute of the UMLClass object
    *
    * @param name  No description provided
    * @return      The fromFMethodsByShortNameIncludingInherited value
    */
   public FMethod getFromFMethodsByShortNameIncludingInherited (String name)
   {
      return getFromMethodsByShortNameIncludingInherited (name);
   }
   // TODO-END

   /**
    * <pre>
    *            +------+ 1                                   1
    * UMLProject + name +--------------------------------------- UMLClass
    *            +------+ declaredInReferences       references
    * </pre>
    */
   private UMLProject declaredInReferences;


   /**
    * Sets the declaredInReferences attribute of the UMLClass object
    *
    * @param obj  The new declaredInReferences value
    * @return     No description provided
    */
   public boolean setDeclaredInReferences (UMLProject obj)
   {
      boolean changed = false;

      if (this.declaredInReferences != obj)
      {
         //side effect as FQN is key in type list
         final UMLTypeList oldRevTypes = getRevTypes();
         setRevTypes (null);

         UMLProject oldValue = null;
         if (this.declaredInReferences != null)
         {
            oldValue = this.declaredInReferences;
            this.declaredInReferences = null;
            oldValue.removeFromReferences (this);
         }
         this.declaredInReferences = obj;
         firePropertyChange ("declaredInReferences", oldValue, obj);
         if (obj != null)
         {
            obj.addToReferences (this);
         }
         changed = true;

         // side effects
         setRevTypes (oldRevTypes);
      }
      return changed;
   }


   /**
    * Get the declaredInReferences attribute of the UMLClass object
    *
    * @return   The declaredInReferences value
    */
   public UMLProject getDeclaredInReferences()
   {
      if ( (this.declaredInReferences == null) &&
          (this.getDeclaredInClass() == null) &&
          (this.getDeclaredInMethod() == null))
      {
         //this.setDeclaredInReferences (UMLProject.get());
      }

      return this.declaredInReferences;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfUndefinedMethods()
   {
      Vector methodVector = new Vector();

      if (hasKeyInStereotypes (UMLStereotypeManager.INTERFACE))
      {
         return FEmptyIterator.get();
      } // end of if ()

      Iterator methodIter = iteratorOfMethods();
      while (methodIter.hasNext())
      {
         UMLMethod method = (UMLMethod) methodIter.next();
         if (!method.isAbstract() &&
            method.getRevSpec() == null)
         {
            methodVector.add (method);
         } // end of if ()
      } // end of while ()

      return methodVector.iterator();
   }


   /**
    * <pre>
    *          +-----------+ 1                          1
    * UMLClass | getName() +------------------------------ UMLPackage
    *          +-----------+ declares   declaredInPackage
    * </pre>
    */
   private UMLPackage declaredInPackage;


   /**
    * Sets the declaredInPackage attribute of the UMLClass object
    *
    * @param obj  The new declaredInPackage value
    * @return     No description provided
    */
   public boolean setDeclaredInPackage (FPackage obj)
   {
      boolean changed = false;

      if (this.declaredInPackage != obj)
      {
         UMLPackage oldValue = this.declaredInPackage;

         //side effect as FQN is key in type list
         final UMLTypeList oldRevTypes = getRevTypes();
         setRevTypes (null);
         //end of side effect

         if (this.declaredInPackage != null)
         {
            this.declaredInPackage = null;
            oldValue.removeFromDeclares (this);
         }
         else if (obj != null)
         {
            if (this.declaredInMethod != null)
            {
               setDeclaredInMethod (null);
            }
            else if (this.declaredInClass != null)
            {
               setDeclaredInClass (null);
            }
         }

         this.declaredInPackage = (UMLPackage) obj;
         firePropertyChange (DECLARED_IN_PACKAGE_PROPERTY, oldValue, obj);
         if (obj != null)
         {
            obj.addToDeclares (this);
         }
         changed = true;

         // side effects
         //side effect as FQN is key in type list
         setRevTypes (oldRevTypes);
         setVisibility (getVisibility()); // ensure that attribute and getter are in sync
         //end of side effect

      }

      return changed;
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the declaredInPackage attribute of the UMLClass object
    *
    * @return   The declaredInPackage value
    */
   public UMLPackage getDeclaredInPackage()
   {
      return this.declaredInPackage;
   } // getDeclaredInPackage


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FClass#getFDeclaredInPackage()
    */
   /**
    * Get the fDeclaredInPackage attribute of the UMLClass object
    *
    * @return   The fDeclaredInPackage value
    */
   public FPackage getFDeclaredInPackage()
   {
      return getDeclaredInPackage();
   }
   // TODO-END

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * @return   package this class is enclosed in - even for inner and anonymous classes
    */
   public UMLPackage findPackage()
   {
      if (this.declaredInPackage != null)
      {
         return this.declaredInPackage;
      }
      if (this.declaredInClass != null)
      {
         return this.declaredInClass.findPackage();
      }
      if (this.declaredInMethod != null)
      {
         return this.declaredInMethod.getParent().findPackage();
      }
      UMLProject project = this.declaredInReferences;
      if (project == null)
      {
         project = UMLProject.get();
      }
      return project.getDefaultPackage();
   }


   /**
    * @return       No description provided
    * @deprecated   will be renamed when using JDK 1.5 (without F)
    */
   public FPackage findFPackage()
   {
      return findPackage();
   }
   // TODO-END

   /**
    * <pre>
    *          +-----------+ 1                          1
    * UMLClass | getName() +------------------------------ UMLMethod
    *          +-----------+ declares    declaredInMethod
    * </pre>
    */
   private UMLMethod declaredInMethod;


   /**
    * Sets the declaredInMethod attribute of the UMLClass object
    *
    * @param obj  The new declaredInMethod value
    * @return     No description provided
    */
   public boolean setDeclaredInMethod (FMethod obj)
   {
      boolean changed = false;

      if (this.declaredInMethod != obj)
      {
         //side effect as FQN is key in type list
         final UMLTypeList oldRevTypes = getRevTypes();
         setRevTypes (null);

         UMLMethod oldValue = this.declaredInMethod;
         if (this.declaredInMethod != null)
         {
            this.declaredInMethod = null;
            oldValue.removeFromDeclares (this);
         }
         else if (obj != null)
         {
            if (this.declaredInClass != null)
            {
               setDeclaredInClass (null);
            }
            else if (this.declaredInPackage != null)
            {
               setDeclaredInPackage (null);
            }
         }
         this.declaredInMethod = (UMLMethod) obj;
         firePropertyChange (DECLARED_IN_METHOD_PROPERTY, oldValue, obj);
         if (obj != null)
         {
            this.declaredInMethod.addToDeclares (this);
         }
         changed = true;

         // side effects
         setRevTypes (oldRevTypes);
      }

      return changed;
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the declaredInMethod attribute of the UMLClass object
    *
    * @return   The declaredInMethod value
    */
   public UMLMethod getDeclaredInMethod()
   {
      return this.declaredInMethod;
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FClass#getFDeclaredInMethod()
    */
   /**
    * Get the fDeclaredInMethod attribute of the UMLClass object
    *
    * @return   The fDeclaredInMethod value
    */
   public FMethod getFDeclaredInMethod()
   {
      return getDeclaredInMethod();
   }
   // TODO-END

   /**
    * <pre>
    *          +-----------+ 1                        1
    * UMLClass | getName() +---------------------------- UMLClass
    *          +-----------+ declares   declaredInClass
    * </pre>
    */
   private UMLClass declaredInClass;


   /**
    * Sets the declaredInClass attribute of the UMLClass object
    *
    * @param obj  The new declaredInClass value
    * @return     No description provided
    */
   public boolean setDeclaredInClass (FClass obj)
   {
      boolean changed = false;

      if (this.declaredInClass != obj)
      {
         //side effect as FQN is key in type list
         final UMLTypeList oldRevTypes = getRevTypes();
         setRevTypes (null);

         UMLClass oldValue = this.declaredInClass;
         if (this.declaredInClass != null)
         {
            this.declaredInClass = null;
            oldValue.removeFromDeclares (this);
         }
         else if (obj != null)
         {
            if (this.declaredInMethod != null)
            {
               setDeclaredInMethod (null);
            }
            else if (this.declaredInPackage != null)
            {
               setDeclaredInPackage (null);
            }
         }
         this.declaredInClass = (UMLClass) obj;
         if (obj != null)
         {
            obj.addToDeclares (this);
         }
         changed = true;

         // side effects
         setRevTypes (oldRevTypes);

         firePropertyChange (DECLARED_IN_CLASS_PROPERTY, oldValue, obj);
      }

      return changed;
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the declaredInClass attribute of the UMLClass object
    *
    * @return   The declaredInClass value
    */
   public UMLClass getDeclaredInClass()
   {
      return this.declaredInClass;
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FClass#getFDeclaredInClass()
    */
   /**
    * Get the fDeclaredInClass attribute of the UMLClass object
    *
    * @return   The fDeclaredInClass value
    */
   public FClass getFDeclaredInClass()
   {
      return getDeclaredInClass();
   }
   // TODO-END

   /**
    * <pre>
    *          +-----------+ 1                        1
    * UMLClass | getName() +---------------------------- UMLClass
    *          +-----------+ declares   declaredInClass
    * </pre>
    */
   private transient FPropTreeMap declares;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public boolean hasInDeclares (FClass obj)
   {
      return  ( (this.declares != null) &&
          (obj != null) &&  (obj.getName() != null) &&
          (this.declares.get (obj.getName()) == obj));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean hasKeyInDeclares (String key)
   {
      return  ( (this.declares != null) &&
          (key != null) &&
         this.declares.containsKey (key));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfDeclares()
   {
      return  ( (this.declares == null)
         ? FEmptyIterator.get()
         : this.declares.values().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator keysOfDeclares()
   {
      return  ( (this.declares == null)
         ? FEmptyIterator.get()
         : this.declares.keySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator entriesOfDeclares()
   {
      return  ( (this.declares == null)
         ? FEmptyIterator.get()
         : this.declares.entrySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfDeclares()
   {
      return  ( (this.declares == null)
         ? 0
         : this.declares.size());
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the fromDeclares attribute of the UMLClass object
    *
    * @param key  No description provided
    * @return     The fromDeclares value
    */
   public UMLClass getFromDeclares (String key)
   {
      return  ( ( (this.declares == null) ||  (key == null))
         ? null
         : (UMLClass) this.declares.get (key));
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FClass#getFromFDeclares(java.lang.String)
    */
   /**
    * Get the fromFDeclares attribute of the UMLClass object
    *
    * @param key  No description provided
    * @return     The fromFDeclares value
    */
   public FClass getFromFDeclares (String key)
   {
      return getFromDeclares (key);
   }
   // TODO-END

   /**
    * Access method for an one to n association.
    *
    * @param obj  The object added.
    * @return     No description provided
    */
   public boolean addToDeclares (FClass obj)
   {
      boolean changed = false;

      if ( (obj != null) &&  (obj.getName() != null))
      {
         if (this.declares == null)
         {
            this.declares = new FPropTreeMap (this, DECLARES_PROPERTY);
         }

         UMLClass oldValue = (UMLClass) this.declares.put (obj.getName(), obj);
         if (oldValue != obj)
         {
            if (oldValue != null)
            {
               oldValue.setDeclaredInClass (null);
            }
            obj.setDeclaredInClass (this);
            changed = true;
         }
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public boolean removeFromDeclares (FClass obj)
   {
      boolean changed = false;

      if ( (this.declares != null) &&  (obj != null) &&  (obj.getName() != null))
      {
         UMLClass oldValue = (UMLClass) this.declares.get (obj.getName());
         if (oldValue == obj)
         {
            this.declares.remove (obj.getName());
            obj.setDeclaredInClass (null);
            changed = true;
         }
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean removeKeyFromDeclares (String key)
   {
      boolean changed = false;

      if ( (this.declares != null) &&  (key != null))
      {
         UMLClass tmpObj = (UMLClass) this.declares.get (key);
         if (tmpObj != null)
         {
            this.declares.remove (key);
            tmpObj.setDeclaredInClass (null);
            changed = true;
         }
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromDeclares()
   {
      UMLClass tmpObj;
      Iterator iter = this.iteratorOfDeclares();

      while (iter.hasNext())
      {
         tmpObj = (UMLClass) iter.next();
         this.removeFromDeclares (tmpObj);
      }
   }


   /**
    * This method is needed only for loading FTreeMaps, don't use it in other cases.
    *
    * @param pair  The object added.
    */
   protected void addToDeclares (KeyValuePair pair)
   {
      if (pair != null)
      {
         UMLClass elem = (UMLClass) pair.getValue();
         String key = (String) pair.getKey();
         if ( (elem != null) &&  (key != null) && ! (key.equals (elem.getName())))
         {
            elem.setName (key);
         }

         addToDeclares (elem);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private FTreeSet revSubclass = new FTreeSet();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    * @return      No description provided
    */
   public boolean hasInRevSubclass (FGeneralization elem)
   {
      return this.revSubclass.contains (elem);
   } // hasInRevSubclass


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfRevSubclass()
   {
      return this.revSubclass.size();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfRevSubclass()
   {
      return this.revSubclass.iterator();
   }


   /**
    * Get a Path object containing all subclasses of this UMLClass.
    *
    * @return   A Path object containing all subclasses of this UMLClass.
    */
   public Path iteratorOfSubClasses()
   {
      return new From (this, "revSuperclass.subclass");
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Enumeration elementsOfRevSubclass()
   {
      return new EnumerationForAnIterator (iteratorOfRevSubclass());
   } // elementsOfRevSubclass


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Enumeration elementsOfAllAttrs()
   {
      return new EnumerationForAnIterator (iteratorOfAllAttrs());
   }


   /**
    * @return       No description provided
    * @deprecated   please use {@link #iteratorOfSuperClasses()}
    */
   public Path elementsOfSuperClasses()
   {
      return iteratorOfSuperClasses();
   }


   /**
    * Get a Path object containing all superclasses of this UMLClass.
    *
    * @return   A Path object containing all superclasses of this UMLClass.
    */
   public Path iteratorOfSuperClasses()
   {
      return new From (this, "revSubclass.superclass");
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * This method returns the super class of this class. This method treats no implements-generalizations.
    *
    * @return   the extended class
    */
   public UMLClass getSuperClass()
   {
      UMLStereotype interfacetype = UMLStereotypeManager.get().
         getFromStereotypes (UMLStereotypeManager.INTERFACE);

      boolean isInterface = hasInStereotypes (interfacetype);

      From path = (From) iteratorOfSuperClasses();
      while (path.hasNext())
      {
         UMLClass tmpClass = (UMLClass) path.next();

         if (tmpClass.hasInStereotypes (interfacetype) == isInterface)
         {
            return tmpClass;
         }
      }

      return null;
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FClass#getFSuperClass()
    */
   /**
    * Get the fSuperClass attribute of the UMLClass object
    *
    * @return   The fSuperClass value
    */
   public FClass getFSuperClass()
   {
      return getSuperClass();
   }
   // TODO-END

   /**
    * Access method for an one to n association.
    *
    * @param elem  The object added.
    */
   public void addToRevSubclass (FGeneralization elem)
   {
      if (elem != null && !this.hasInRevSubclass (elem))
      {
         this.revSubclass.add (elem);
         elem.setSubclass (this);

         firePropertyChange (CollectionChangeEvent.get (this, REV_SUBCLASS_PROPERTY, this.revSubclass, null,
            elem, CollectionChangeEvent.ADDED));
      }
   } // addToRevSubclass


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    */
   public void removeFromRevSubclass (FGeneralization elem)
   {
      if (this.hasInRevSubclass (elem))
      {
         this.revSubclass.remove (elem);
         elem.setSubclass (null);

         firePropertyChange (CollectionChangeEvent.get (this, REV_SUBCLASS_PROPERTY, this.revSubclass, elem,
            null, CollectionChangeEvent.REMOVED));
      }
   } // removeFromRevSubclass


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final void removeAllFromRevSubclass()
   {
      Enumeration enumeration = elementsOfRevSubclass();
      UMLGeneralization incr;
      while (enumeration.hasMoreElements())
      {
         incr = (UMLGeneralization) enumeration.nextElement();
         removeFromRevSubclass (incr);
         incr.removeYou();
      }
   } // removeAllRevSubclass


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private FTreeSet revSuperclass = new FTreeSet();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    * @return      No description provided
    */
   public boolean hasInRevSuperclass (FGeneralization elem)
   {
      return this.revSuperclass.contains (elem);
   } // hasInRevSuperclass


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Enumeration elementsOfRevSuperclass()
   {
      return new EnumerationForAnIterator (iteratorOfRevSuperclass());
   } // elementsOfRevSuperclass


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfRevSuperclass()
   {
      return revSuperclass.iterator();
   }


   /**
    * Access method for an one to n association.
    *
    * @param elem  The object added.
    */
   public void addToRevSuperclass (FGeneralization elem)
   {
      if (elem != null && !this.hasInRevSuperclass (elem))
      {
         this.revSuperclass.add (elem);
         elem.setSuperclass (this);

         firePropertyChange (CollectionChangeEvent.get (this, REV_SUPERCLASS_PROPERTY, this.revSuperclass, null,
            elem, CollectionChangeEvent.ADDED));
      }
   } // addToRevSuperclass


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    */
   public void removeFromRevSuperclass (FGeneralization elem)
   {
      if (this.hasInRevSuperclass (elem))
      {
         this.revSuperclass.remove (elem);
         elem.setSuperclass (null);

         firePropertyChange (CollectionChangeEvent.get (this, REV_SUPERCLASS_PROPERTY, this.revSuperclass, elem,
            null, CollectionChangeEvent.REMOVED));
      }
   } // removeFromRevSuperclass


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final void removeAllFromRevSuperclass()
   {
      UMLGeneralization item;
      Iterator iter = iteratorOfRevSuperclass();

      while (iter.hasNext())
      {
         item = (UMLGeneralization) iter.next();
         item.removeYou();
         firePropertyChange (CollectionChangeEvent.get (this, REV_SUPERCLASS_PROPERTY, this.revSuperclass, item,
            null, CollectionChangeEvent.REMOVED));
      }
   } // removeAllRevSuperclass


   /**
    * Returns true, if this class is subclass of the param.<p>
    * <p/>
    * Otherwise the param clazz isn't a superclass of this class. FIX ME: there can be a deadlock,
    * when a loop in the generalization exists.
    *
    * @param clazz  the UMLClass to be checked.
    * @return       returns true if this is a subclass of the param.
    */
   public boolean isChildOf (UMLClass clazz)
   {
      if (this.equals (clazz))
      {
         return true;
      }
      else
      {
         boolean result = false;
         Enumeration enumeration = this.elementsOfRevSubclass();

         while (!result && enumeration.hasMoreElements())
         {
            UMLGeneralization general = (UMLGeneralization) enumeration.nextElement();
            UMLClass superClass = general.getSuperclass();
            if (superClass != null)
            {
               result = superClass.isChildOf (clazz);
            }
         }

         return result;
      }
   } // isChildOf


   /**
    * To get the generalization between this child class and a parent class.
    *
    * @param parent  No description provided
    * @return        the generalization between this class and the parent class
    */
   public UMLGeneralization getParentGeneralization (UMLClass parent)
   {
      UMLGeneralization umlGen = null;
      Enumeration enumeration = this.elementsOfRevSubclass();
      while (umlGen == null &&
         enumeration.hasMoreElements())
      {
         umlGen = (UMLGeneralization) enumeration.nextElement();
         if (umlGen.getSuperclass() != parent)
         {
            umlGen = null;
         }
      }
      return umlGen;
   } // getParentGeneralization


   /**
    * interface implementation remember to initialize the variables
    *
    * @return   The progLangType value
    */
   public String getProgLangType()
   {
      String tmpString = getName();

      if (tmpString != null)
      {
         if (tmpString.trim().equals (UMLBaseTypes.BOOLEAN))
         {
            tmpString = "boolean";
         }
         else if (tmpString.trim().equals ("Short"))
         {
            tmpString = "short";
         }
         else if (tmpString.trim().equals (UMLBaseTypes.INTEGER))
         {
            tmpString = "int";
         }
         else if (tmpString.trim().equals (UMLBaseTypes.FLOAT))
         {
            tmpString = "float";
         }
         else if (tmpString.trim().equals (UMLBaseTypes.DOUBLE))
         {
            tmpString = "double";
         }
      }
      return tmpString;
   } // getProgLangType


   /**
    * Sets the progLangType attribute of the UMLClass object
    *
    * @param progLangType  The new progLangType value
    */
   public void setProgLangType (String progLangType)
   {
      setName (progLangType);
   } // setProgLangType


   /**
    * <pre>
    *           0..1                     0..n
    * UMLCLass ------------------------------- UMLMethod
    *           resultType       revResultType
    * </pre>
    */
   private transient FPropTreeSet revResultType;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInRevResultType (FMethod value)
   {
      return  ( (this.revResultType != null) &&
          (value != null) &&
         this.revResultType.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfRevResultType()
   {
      return  ( (this.revResultType == null)
         ? FEmptyIterator.get()
         : this.revResultType.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfRevResultType()
   {
      return  ( (this.revResultType == null)
         ? 0
         : this.revResultType.size());
   }


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToRevResultType (FMethod value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.revResultType == null)
         {
            this.revResultType = new FPropTreeSet (this, "revResultType");
         }
         changed = this.revResultType.add (value);
         if (changed)
         {
            value.setResultType (this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromRevResultType (FMethod value)
   {
      boolean changed = false;
      if ( (this.revResultType != null) &&  (value != null))
      {
         changed = this.revResultType.remove (value);
         if (changed)
         {
            value.setResultType (null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromRevResultType()
   {
      UMLMethod tmpValue;
      Iterator iter = this.iteratorOfRevResultType();
      while (iter.hasNext())
      {
         tmpValue = (UMLMethod) iter.next();
         this.removeFromRevResultType (tmpValue);
      }
   }


   /**
    * <pre>
    *           0..1                   0..n
    * UMLClass ----------------------------- UMLParam
    *           paramTyp        revParamTyp
    * </pre>
    */
   private transient FPropTreeSet revParamType;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInRevParamType (FParam value)
   {
      return  ( (this.revParamType != null) &&
          (value != null) &&
         this.revParamType.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfRevParamType()
   {
      return  ( (this.revParamType == null)
         ? FEmptyIterator.get()
         : this.revParamType.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfRevParamType()
   {
      return  ( (this.revParamType == null)
         ? 0
         : this.revParamType.size());
   }


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToRevParamType (FParam value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.revParamType == null)
         {
            this.revParamType = new FPropTreeSet (this, "revParamType");
         }
         changed = this.revParamType.add (value);
         if (changed)
         {
            value.setParamType (this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromRevParamType (FParam value)
   {
      boolean changed = false;
      if ( (this.revParamType != null) &&  (value != null))
      {
         changed = this.revParamType.remove (value);
         if (changed)
         {
            value.setParamType (null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromRevParamType()
   {
      UMLParam tmpValue;
      Iterator iter = this.iteratorOfRevParamType();
      while (iter.hasNext())
      {
         tmpValue = (UMLParam) iter.next();
         this.removeFromRevParamType (tmpValue);
      }
   }


   /**
    * <pre>
    *             +------+ 1                 1
    * UMLTypeList | name +--------------------- UMLType
    *             +------+ revTypes      types
    * </pre>
    */
   private transient UMLTypeList revTypes;


   /**
    * Sets the revTypes attribute of the UMLClass object
    *
    * @param obj  The new revTypes value
    * @return     No description provided
    */
   public boolean setRevTypes (FTypeList obj)
   {
      boolean changed = false;

      if (this.revTypes != obj)
      {
         UMLTypeList oldValue = null;
         if (this.revTypes != null)
         {
            oldValue = this.revTypes;
            this.revTypes = null;
            oldValue.removeFromTypes (this);
         }
         this.revTypes = (UMLTypeList) obj;
         if (obj != null)
         {
            obj.addToTypes (this);
         }
         changed = true;

         // side effects

         firePropertyChange ("revTypes", oldValue, obj);
      }

      return changed;
   }


   /**
    * Get the revTypes attribute of the UMLClass object
    *
    * @return   The revTypes value
    */
   public UMLTypeList getRevTypes()
   {
      //Arg! The following breaks the typelist if setName() of a Class is called!
//      if (this.revTypes == null)
//      {
//         setRevTypes (UMLProject.get().getTypeList());
//      }
      return this.revTypes;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLArray revArrayType = null;

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the revArrayType attribute of the UMLClass object
    *
    * @return   The revArrayType value
    */
   public UMLArray getRevArrayType()
   {
      return this.revArrayType;
   } // getRevArrayType


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FType#getFRevArrayType()
    */
   /**
    * Get the fRevArrayType attribute of the UMLClass object
    *
    * @return   The fRevArrayType value
    */
   public FArray getFRevArrayType()
   {
      return getRevArrayType();
   }
   // TODO-END

   /**
    * Sets the revArrayType attribute of the UMLClass object
    *
    * @param revArrayType  The new revArrayType value
    */
   public void setRevArrayType (FArray revArrayType)
   {
      if (this.revArrayType != revArrayType)
      {
         UMLArray oldRevArrayType = this.revArrayType;
         // new partner
         if (this.revArrayType != null)
         {
            // inform old partner
            this.revArrayType = null;
            oldRevArrayType.setArrayType (this);
         }
         this.revArrayType = (UMLArray) revArrayType;
         if (this.revArrayType != null)
         {
            // inform new partner
            this.revArrayType.setArrayType (this);
         }
         firePropertyChange ("revArrayType", oldRevArrayType, revArrayType);
      }
   } // setRevArrayType

   // TODO-Begin: Merge with JDK 1.5
   /**
    * Get the realType attribute of the UMLClass object
    *
    * @return   The realType value
    */
   public UMLType getRealType()
   {
      return this;
   } // getRealType


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FType#getFRealType()
    */
   /**
    * Get the fRealType attribute of the UMLClass object
    *
    * @return   The fRealType value
    */
   public FType getFRealType()
   {
      return getRealType();
   }
   // TODO-End


   /**
    * Get the dimension attribute of the UMLClass object
    *
    * @return   The dimension value
    */
   public int getDimension()
   {
      return 0;
   } // getDimension


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public boolean necessaryToGenerateSourceCode()
   {
      return ! (hasInStereotypes (UMLStereotypeManager.get().getFromStereotypes (UMLStereotypeManager.REFERENCE)) ||
         hasInStereotypes (UMLStereotypeManager.get().getFromStereotypes (UMLStereotypeManager.IMPORT)));
   } // necessaryToGenerateSourceCode


   /**
    * Returns all associations which can be accessed from this class. Inherited assocs will
    * be also covered.
    *
    * @return   The allAssocs value
    */
   public FTreeSet getAllAssocs()
   {
      FTreeSet allAssocs = new FTreeSet();

      // Get all assoEnumeration enumRoles = elementsOfRoles ()
      // which are implemented in this class.
      Iterator iter = iteratorOfRoles();
      while (iter.hasNext())
      {
         UMLAssoc tmpAssoc =  ((UMLRole) iter.next()).getAssoc();
         allAssocs.add (tmpAssoc);
      }

      iter = iteratorOfRevSubclass();
      while (iter.hasNext())
      {
         UMLGeneralization generalization =  ((UMLGeneralization) iter.next());
         UMLClass tmpClass = generalization.getSuperclass();

         if (tmpClass != null)
         {
            // Get all assocs which are implemented in this class.
            allAssocs.addAll (tmpClass.getAllAssocs());
         }
         else
         {
            Warning warning = new Warning ("Generalization without superclass");
            warning.addToContext (this);
            warning.addToContext (generalization);
            FrameMain.get().getMessageView().addToMessages (warning);
         }
      }

      return allAssocs;
   } // getAllAssocs


   /**
    * returns all assocs of this class an his parents
    *
    * @return   The allAssocsNew value
    */
   public TreeSet getAllAssocsNew()
   {
      TreeSet assocSet = new TreeSet();
      LinkedList classes = new LinkedList();

      classes.add (this);
      UMLClass cls;
      UMLRole tmpRole;
      UMLAssoc tmpAssoc;
      UMLGeneralization tmpGen;
      UMLClass tmpClass;
      Iterator iter;
      while (!classes.isEmpty())
      {
         cls = (UMLClass) classes.removeFirst();

         //----- add all assocs
         iter = cls.iteratorOfRoles();
         while (iter.hasNext())
         {
            tmpRole = (UMLRole) iter.next();
            tmpAssoc = tmpRole.getAssoc();
            assocSet.add (tmpAssoc);
         }

         //----- add all superclasses to classes
         iter = cls.iteratorOfRevSubclass();
         while (iter.hasNext())
         {
            tmpGen = (UMLGeneralization) iter.next();
            tmpClass = tmpGen.getSuperclass();
            classes.addLast (tmpClass);
         }
      }
      return assocSet;
   } // getAllAssocs


   /**
    * Check if an class has an public accessible Attribut including derived Attributs with
    * given name
    *
    * @param key  The name of the attribut
    * @return     True if the class or its superclass has an public attribut with the given
    *         name
    */
   public boolean hasPubAttrWithKey (String key)
   {
      LinkedList classes = new LinkedList();

      classes.add (this);
      UMLClass cls;
      UMLAttr tmpAttr = null;
      UMLGeneralization tmpGen;
      UMLClass tmpClass;
      Iterator iter;
      boolean found = false;
      while ( (!classes.isEmpty()) &&  (found == false))
      {
         cls = (UMLClass) classes.removeFirst();

         //----- look for the attr
         tmpAttr = cls.getFromAttrs (key);
         if (tmpAttr != null)
         {
            found = true;
         } // if

         else
         {
            //----- add all superclasses to classes
            iter = cls.iteratorOfRevSubclass();
            while (iter.hasNext())
            {
               tmpGen = (UMLGeneralization) iter.next();
               tmpClass = tmpGen.getSuperclass();
               classes.addLast (tmpClass);
            }
         } // else

      } // while

      if (found == false)
      {
         return false;
      }

      return  (tmpAttr.getVisibility() == FDeclaration.PUBLIC);
   } // hasPubAttrWithKey


   /**
    * Check if an class has an public accessible Attribut including derived Attributs with
    * given name and deliver it when found
    *
    * @param key  The name of the attribut
    * @return     The attribut if found or null
    */
   public UMLAttr getPubAttrWithKey (String key)
   {
      LinkedList classes = new LinkedList();

      classes.add (this);
      UMLClass cls;
      UMLAttr tmpAttr = null;
      UMLGeneralization tmpGen;
      UMLClass tmpClass;
      Iterator iter;
      boolean found = false;
      while ( (!classes.isEmpty()) &&  (found == false))
      {
         cls = (UMLClass) classes.removeFirst();

         //----- look for the attr
         tmpAttr = cls.getFromAttrs (key);
         if (tmpAttr != null)
         {
            found = true;
         }
         else
         {
            //----- add all superclasses to classes
            iter = cls.iteratorOfRevSubclass();
            while (iter.hasNext())
            {
               tmpGen = (UMLGeneralization) iter.next();
               tmpClass = tmpGen.getSuperclass();
               classes.addLast (tmpClass);
            }
         }
      }

      if (found == false)
      {
         return null;
      }

      if (tmpAttr.getVisibility() == FDeclaration.PUBLIC)
      {
         return tmpAttr;
      }

      return null;
   } // getPubAttrWithKey

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the fromAllAttrs attribute of the UMLClass object
    *
    * @param key  No description provided
    * @return     The fromAllAttrs value
    */
   public UMLAttr getFromAllAttrs (String key)
   {
      UMLAttr theAttr = getFromAttrs (key);

      if (theAttr == null)
      {
         UMLGeneralization tmpGen = null;
         UMLClass tmpClass = null;

         //----- add all superclasses to classes
         Iterator iter = iteratorOfRevSubclass();
         while (theAttr == null &&
            iter.hasNext())
         {
            tmpGen = (UMLGeneralization) iter.next();
            tmpClass = tmpGen.getSuperclass();
            theAttr = tmpClass.getFromAllAttrs (key);
         }
      }

      return theAttr;
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FClass#getFromAllFAttrs(java.lang.String)
    */
   /**
    * Get the fromAllFAttrs attribute of the UMLClass object
    *
    * @param key  No description provided
    * @return     The fromAllFAttrs value
    */
   public FAttr getFromAllFAttrs (String key)
   {
      return getFromAllAttrs (key);
   }
   // TODO-END

   /**
    * Get the fromAllRoles attribute of the UMLClass object
    *
    * @param key  No description provided
    * @return     The fromAllRoles value
    */
   public UMLRole getFromAllRoles (String key)
   {
      UMLRole theRole = (UMLRole) getFromFRoles (key);

      if (theRole == null)
      {
         UMLGeneralization tmpGen = null;
         UMLClass tmpClass = null;

         //----- add all superclasses to classes
         Iterator iter = iteratorOfRevSubclass();
         while (theRole == null &&
            iter.hasNext())
         {
            tmpGen = (UMLGeneralization) iter.next();
            tmpClass = tmpGen.getSuperclass();
            theRole = tmpClass.getFromAllRoles (key);
         }
      }

      return theRole;
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the fromAllMethods attribute of the UMLClass object
    *
    * @param key  No description provided
    * @return     The fromAllMethods value
    */
   public UMLMethod getFromAllMethods (String key)
   {
      UMLMethod method = getFromMethods (key);

      if (method == null)
      {
         Map methods = getAllAccessibleMethodObjects();
         method = (UMLMethod) methods.get (key);
      }

      return method;
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FClass#getFromAllFMethods(java.lang.String)
    */
   /**
    * Get the fromAllFMethods attribute of the UMLClass object
    *
    * @param key  No description provided
    * @return     The fromAllFMethods value
    */
   public FMethod getFromAllFMethods (String key)
   {
      return getFromAllMethods (key);
   }
   // TODO-END

   /**
    * Get the fromAllPartnerRoles attribute of the UMLClass object
    *
    * @param key  No description provided
    * @return     The fromAllPartnerRoles value
    */
   public UMLRole getFromAllPartnerRoles (String key)
   {
      UMLRole theRole = getFromPartnerRoles (key);

      if (theRole == null)
      {
         UMLGeneralization tmpGen = null;
         UMLClass tmpClass = null;

         //----- add all superclasses to classes
         Iterator iter = iteratorOfRevSubclass();
         while (theRole == null &&
            iter.hasNext())
         {
            tmpGen = (UMLGeneralization) iter.next();
            tmpClass = tmpGen.getSuperclass();
            theRole = tmpClass.getFromAllPartnerRoles (key);
         }
      }

      return theRole;
   }


   /**
    * Get the fromPartnerRoles attribute of the UMLClass object
    *
    * @param key  No description provided
    * @return     The fromPartnerRoles value
    */
   public UMLRole getFromPartnerRoles (String key)
   {
      Iterator iter = iteratorOfRoles();
      while (iter.hasNext())
      {
         UMLRole role = (UMLRole) iter.next();
         UMLRole partnerRole = role.getPartnerRole();
         String name = partnerRole.getName();
         if ( (name == null && key == null) ||
             (name != null && name.equals (key)))
         {
            return partnerRole;
         }
      }
      return null;
   }


   /**
    * Get the fromAllRevRoles attribute of the UMLClass object
    *
    * @param key  No description provided
    * @return     The fromAllRevRoles value
    */
   public UMLRole getFromAllRevRoles (String key)
   {
      UMLRole theRole = getFromRevRoles (key);

      if (theRole == null)
      {
         UMLGeneralization tmpGen = null;
         UMLClass tmpClass = null;

         //----- add all superclasses to classes
         Iterator iter = iteratorOfRevSubclass();
         while (theRole == null &&
            iter.hasNext())
         {
            tmpGen = (UMLGeneralization) iter.next();
            tmpClass = tmpGen.getSuperclass();
            theRole = tmpClass.getFromAllRevRoles (key);
         }
      }

      return theRole;
   }


   /**
    * Get the treeSetOfAllAttrs attribute of the UMLClass object
    *
    * @return   The treeSetOfAllAttrs value
    */
   private FTreeSet getTreeSetOfAllAttrs()
   {
      FTreeSet theSet = new FTreeSet();

      Iterator iter = iteratorOfAttrs();

      while (iter.hasNext())
      {
         UMLAttr theAttr = (UMLAttr) iter.next();
         theSet.add (theAttr);
      } // end of while ()

      UMLGeneralization tmpGen = null;
      UMLClass tmpClass = null;

      //----- add all superclasses to classes
      iter = iteratorOfRevSubclass();
      while (iter.hasNext())
      {
         tmpGen = (UMLGeneralization) iter.next();
         tmpClass = tmpGen.getSuperclass();
         if (tmpClass != null)
         {
            theSet.addAll (tmpClass.getTreeSetOfAllAttrs());
         }
      }

      return theSet;
   }


   /**
    * Get the treeSetOfAllRoles attribute of the UMLClass object
    *
    * @return   The treeSetOfAllRoles value
    */
   private FTreeSet getTreeSetOfAllRoles()
   {
      FTreeSet theSet = new FTreeSet();

      Iterator iter = iteratorOfRoles();

      while (iter.hasNext())
      {
         UMLRole theRole = (UMLRole) iter.next();
         theSet.add (theRole);
      } // end of while ()

      UMLGeneralization tmpGen = null;
      UMLClass tmpClass = null;

      //----- add all superclasses to classes
      iter = iteratorOfRevSubclass();
      while (iter.hasNext())
      {
         tmpGen = (UMLGeneralization) iter.next();
         tmpClass = tmpGen.getSuperclass();
         theSet.addAll (tmpClass.getTreeSetOfAllRoles());
      }

      return theSet;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfAllAttrs()
   {
      return getTreeSetOfAllAttrs().iterator();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfAllRoles()
   {
      return getTreeSetOfAllRoles().iterator();
   }


   /**
    * Get all Atrributs which could be accessed from inside the class including derived attributs
    *
    * @return   A set whith all Attrs accessible from inside the class
    */
   public FTreeSet getAllAccessibleAttrs()
   {
      FTreeSet attribs = new FTreeSet();
      FTreeSet attribNames = new FTreeSet();

      LinkedList classes = new LinkedList();

      UMLClass cls;
      UMLAttr tmpAttr;
      UMLGeneralization tmpGen;
      UMLClass tmpClass;
      Iterator iter;

      //  first add all the attrs of this class
      if (attrs != null)
      {
         attribs.addAll (attrs.values());
         attribNames.addAll (attrs.keySet());
      }
      //----- add superclasses to classes
      iter = iteratorOfRevSubclass();
      while (iter.hasNext())
      {
         tmpGen = (UMLGeneralization) iter.next();
         tmpClass = tmpGen.getSuperclass();
         classes.addLast (tmpClass);
      }

      while (! (classes.isEmpty()))
      {
         cls = (UMLClass) classes.removeFirst();

         //----- get the attrs from class

         iter = cls.iteratorOfAttrs();
         while (iter.hasNext())
         {
            tmpAttr = (UMLAttr) iter.next();

            if (tmpAttr.getVisibility() != FDeclaration.PRIVATE)
            {

               if (tmpAttr.getVisibility() != FDeclaration.PACKAGE)
               {
                  if (! (attribNames.contains (tmpAttr.getName())))
                  {
                     attribs.add (tmpAttr);
                  }
               }
               else
               {
                  if (cls.getDeclaredInPackage() == this.getDeclaredInPackage())
                  {
                     if (! (attribNames.contains (tmpAttr.getName())))
                     {
                        attribs.add (tmpAttr);
                     }
                  }
               }
            }
            attribNames.add (tmpAttr.getName());
         }

         //----- add all superclasses to classes
         iter = cls.iteratorOfRevSubclass();
         while (iter.hasNext())
         {
            tmpGen = (UMLGeneralization) iter.next();
            tmpClass = tmpGen.getSuperclass();
            classes.addLast (tmpClass);
         }
      }

      return attribs;
   } // getAllAccessibleAttrs


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfAllAccessibleMethods()
   {
      return this.getAllAccessibleMethods().iterator();
   }


   /**
    * Get all Methods which could be accessed from inside the class including derived methods
    *
    * @return   A set whith all Methods accessible from inside the class
    */
   public SortedMap getAllAccessibleMethodObjects()
   {
      FTreeMap accMethods = new FTreeMap();

      LinkedList classes = new LinkedList();

      UMLClass cls;
      UMLMethod tmpMeth;
      UMLGeneralization tmpGen;
      UMLClass tmpClass;
      Iterator iter;

      //  first add all the methods of this class
      for (Iterator it = iteratorOfMethods(); it.hasNext(); )
      {
         UMLMethod method = (UMLMethod) it.next();
         accMethods.put (method.getFullMethodName(), method);
      }

      //----- add superclasses to classes
      iter = iteratorOfRevSubclass();
      while (iter.hasNext())
      {
         tmpGen = (UMLGeneralization) iter.next();
         tmpClass = tmpGen.getSuperclass();
         classes.addLast (tmpClass);
      } // while

      while (! (classes.isEmpty()))
      {
         cls = (UMLClass) classes.removeFirst();

         //----- get the methods from class

         iter = cls.iteratorOfMethods();
         while (iter.hasNext())
         {
            tmpMeth = (UMLMethod) iter.next();

            if (tmpMeth.getVisibility() != FDeclaration.PRIVATE)
            {

               if (tmpMeth.getVisibility() != FDeclaration.PACKAGE)
               {
                  if (! (accMethods.containsKey (tmpMeth.getFullMethodName())))
                  {
                     accMethods.put (tmpMeth.getFullMethodName(), tmpMeth);
                  } // if

               } // if

               else
               {
                  if (cls.getDeclaredInPackage() == this.getDeclaredInPackage())
                  {
                     if (! (accMethods.containsKey (tmpMeth.getFullMethodName())))
                     {
                        accMethods.put (tmpMeth.getFullMethodName(), tmpMeth);
                     } // if

                  } // if

               } // else

            } // if
         } // while

         //----- add all superclasses to classes
         iter = cls.iteratorOfRevSubclass();
         while (iter.hasNext())
         {
            tmpGen = (UMLGeneralization) iter.next();
            tmpClass = tmpGen.getSuperclass();
            classes.addLast (tmpClass);
         } // while

      } // while

      return accMethods;
   } // getAllAccessibleMethods


   /**
    * Get the allAccessibleMethods attribute of the UMLClass object
    *
    * @return   The allAccessibleMethods value
    */
   public Collection getAllAccessibleMethods()
   {
      return getAllAccessibleMethodObjects().values();
   } // getAllAccessibleMethods


   /**
    * Get all roles which are used in this class or in a superclass
    *
    * @return   A set whith all roles use by the class or its superclass
    */
   public FTreeSet getAllUsedRoles()
   {
      FTreeSet theRoles = new FTreeSet();

      LinkedList classes = new LinkedList();

      UMLClass cls;
      UMLGeneralization tmpGen;
      UMLClass tmpClass;
      Iterator iter;

      //  first add all the roles of this class
      if (roles != null)
      {
         theRoles.addAll (roles);
      } // if

      //----- add superclasses to classes
      iter = iteratorOfRevSubclass();
      while (iter.hasNext())
      {
         tmpGen = (UMLGeneralization) iter.next();
         tmpClass = tmpGen.getSuperclass();
         classes.addLast (tmpClass);
      } // while

      while (! (classes.isEmpty()))
      {
         cls = (UMLClass) classes.removeFirst();

         //----- get the roles from class

         theRoles.addAll (cls.getAllUsedRoles());

      } // while

      return theRoles;
   } // getAllUsedRoles


   /**
    * Get all classes wich are derived direct or indirect
    *
    * @return   A set whith all direct and indirect derived Subclasses
    */
   public FTreeSet getAllDerivedClasses()
   {
      LinkedList classes = new LinkedList();

      FTreeSet derived = new FTreeSet();

      classes.add (this);
      UMLClass cls;
      UMLGeneralization tmpGen;
      UMLClass tmpClass;
      Iterator iter;
      while (!classes.isEmpty())
      {
         cls = (UMLClass) classes.removeFirst();

         //----- add all subclasses to derived
         iter = cls.iteratorOfRevSuperclass();
         while (iter.hasNext())
         {
            tmpGen = (UMLGeneralization) iter.next();
            tmpClass = tmpGen.getSubclass();
            classes.addLast (tmpClass);
            derived.add (tmpClass);
         } // while

      } // while

      return derived;
   } // getAllDerivedClasses


   /**
    * Check if an class has an public accessible method including derived methods with given
    * (full) name
    *
    * @param key  The (full) name of the method
    * @return     True if the class or its superclass has an public method with the given (full)
    *         name
    */
   public boolean hasPubMethWithKey (String key)
   {
      LinkedList classes = new LinkedList();

      classes.add (this);
      UMLClass cls;
      UMLMethod tmpMeth = null;
      UMLGeneralization tmpGen;
      UMLClass tmpClass;
      Iterator iter;
      boolean found = false;
      while ( (!classes.isEmpty()) &&  (found == false))
      {
         cls = (UMLClass) classes.removeFirst();

         //----- look for the attr
         tmpMeth = cls.getFromMethods (key);
         if (tmpMeth != null)
         {
            found = true;
         } // if

         else
         {
            //----- add all superclasses to classes
            iter = cls.iteratorOfRevSubclass();
            while (iter.hasNext())
            {
               tmpGen = (UMLGeneralization) iter.next();
               tmpClass = tmpGen.getSuperclass();
               classes.addLast (tmpClass);
            }
         } // else

      } // while

      if (found == false)
      {
         return false;
      }

      return  (tmpMeth.getVisibility() == FDeclaration.PUBLIC);
   } // hasPubMethWithKey


   /**
    * Check if an class has an public accessible method including derived methods with given
    * (full) name and return it
    *
    * @param key  The (full) name of the method
    * @return     The found method, or null if not found
    */
   public UMLMethod getPubMethWithKey (String key)
   {
      LinkedList classes = new LinkedList();

      classes.add (this);
      UMLClass cls;
      UMLMethod tmpMeth = null;
      UMLGeneralization tmpGen;
      UMLClass tmpClass;
      Iterator iter;
      boolean found = false;
      while ( (!classes.isEmpty()) &&  (found == false))
      {
         cls = (UMLClass) classes.removeFirst();

         //----- look for the attr
         tmpMeth = cls.getFromMethods (key);
         if (tmpMeth != null)
         {
            found = true;
         } // if

         else
         {
            //----- add all superclasses to classes
            iter = cls.iteratorOfRevSubclass();
            while (iter.hasNext())
            {
               tmpGen = (UMLGeneralization) iter.next();
               tmpClass = tmpGen.getSuperclass();
               classes.addLast (tmpClass);
            }
         } // else

      } // while

      if ( (found == false) ||  (tmpMeth.getVisibility() != FDeclaration.PUBLIC))
      {
         return null;
      }

      return tmpMeth;
   } // getPubMethWithKey


   /**
    * <pre>
    *                     0..1       statechart       0..1
    * UMLReactiveDiagram ---------------------------------- UMLClass
    *                     uMLReactiveDiagram      uMLClass
    * </pre>
    */
   private UMLReactiveDiagram uMLReactiveDiagram;


   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * UMLMethod: '+ getUMLReactiveDiagram (): UMLReactiveDiagram'.
    *
    * @return   The uMLReactiveDiagram value
    */
   public UMLReactiveDiagram getUMLReactiveDiagram()
   {
      return this.uMLReactiveDiagram;
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FClass#getFReactiveDiagram()
    */
   /**
    * Get the fReactiveDiagram attribute of the UMLClass object
    *
    * @return   The fReactiveDiagram value
    */
   public FDiagram getFReactiveDiagram()
   {
      return getUMLReactiveDiagram();
   }
   // TODO-END

   /**
    * UMLMethod: '+ setUMLReactiveDiagram (value: UMLReactiveDiagram): Boolean'.
    *
    * @param value  The new uMLReactiveDiagram value
    * @return       No description provided
    */
   public boolean setUMLReactiveDiagram (UMLReactiveDiagram value)
   {
      boolean changed = false;
      if (this.uMLReactiveDiagram != value)
      {
         if (this.uMLReactiveDiagram != null)
         {
            UMLReactiveDiagram oldValue = this.uMLReactiveDiagram;
            this.uMLReactiveDiagram = null;
            oldValue.setUMLClass (null);
         }
         this.uMLReactiveDiagram = value;
         if (value != null)
         {
            value.setUMLClass (this);
         }
         changed = true;
      }
      return changed;
   }


   /**
    * This method checks if all grabbed roles are references. Only then this class can be marked
    * as TYPE, which means that the reference is hidden and the corresponding attribute is
    * shown.
    *
    * @return   The typeMarkable value
    */
   public boolean isTypeMarkable()
   {
      boolean result = true;
      UMLRole tmpRole;
      Enumeration enumeration = elementsOfRoles();
      while (result && enumeration.hasMoreElements())
      {
         tmpRole = (UMLRole) enumeration.nextElement();
         result =  (tmpRole.getAdornment() == FRole.REFERENCE);
      }
      return result;
   } // isTypeMarkable

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * @return   the statechart for this class, null if not applicable
    */
   public UMLStatechart getStatechart()
   {
      if (getStartOfStateChart() != null)
      {
         for (Iterator it = getStartOfStateChart().iteratorOfDiagrams(); it.hasNext(); )
         {
            Object diagram = it.next();
            if (diagram instanceof UMLStatechart)
            {
               return (UMLStatechart) diagram;
            }
         }
      }
      return null;
   }


   /**
    * Get the fStatechart attribute of the UMLClass object
    *
    * @return   The fStatechart value
    */
   public FDiagram getFStatechart()
   {
      return getStatechart();
   }
   // TODO-END

   /**
    * @return       getStatechart()
    * @deprecated   use {@link #getStatechart()}
    */
   public UMLActivityDiagram getStoryDiagram()
   {
      return getStatechart();
   }


   /**
    * <pre>
    *           0..1   startOfStateChart   0..1
    * UMLClass --------------------------------- UMLStartActivity
    *           +                            +
    * </pre>
    */
   private UMLStartActivity startOfStateChart;


   /**
    * Get the startOfStateChart attribute of the UMLClass object
    *
    * @return   The startOfStateChart value
    */
   public UMLStartActivity getStartOfStateChart()
   {
      return startOfStateChart;
   } // getStartOfStateChart


   /**
    * Sets the startOfStateChart attribute of the UMLClass object
    *
    * @param startOfStateChart  The new startOfStateChart value
    */
   public void setStartOfStateChart (UMLStartActivity startOfStateChart)
   {
      if (this.startOfStateChart != startOfStateChart)
      {
         // newPartner
         UMLStartActivity oldStartOfStateChart = this.startOfStateChart;
         if (this.startOfStateChart != null)
         {
            // inform old partner
            this.startOfStateChart = null;

            oldStartOfStateChart.setRevStartOfStateChart (null);
         }

         this.startOfStateChart = startOfStateChart;
         if (startOfStateChart != null)
         {
            // inform new partner
            startOfStateChart.setRevStartOfStateChart (this);
         }

         firePropertyChange ("startOfStateChart", oldStartOfStateChart, startOfStateChart);
      }
   } // setStartOfStateChart


   /**
    * Get the allOutGoingRoles attribute of the UMLClass object
    *
    * @return   The allOutGoingRoles value
    */
   public final TreeMap getAllOutGoingRoles()
   {
      TreeMap outGoing = new TreeMap();

      //----- search the assoc, which 'leaves' this class
      Iterator iter = iteratorOfRoles();
      while (iter.hasNext())
      {
         UMLRole role = (UMLRole) iter.next();
         UMLAssoc assoc = role.getAssoc();
         int direction = assoc.getDirection();
         UMLRole leftRole = assoc.getLeftRole();
         if ( ( (direction == UMLAssoc.LEFTRIGHT) &&  (leftRole == role)) ||
             ( (direction == UMLAssoc.RIGHTLEFT) &&  (leftRole != role)))
         {
            outGoing.put (assoc.getName(), role);
         }
      }
      return outGoing;
   }


   /**
    * Use this method to determine wether this instance of UMLClass is a subclass of the given
    * UMLClass clazz
    *
    * @param clazz  a possible superclass
    * @return       true if this is a subclass of clazz, false otherwise. notice: isSubClassOf(this)
    *         returns true
    */
   public boolean isSubClassOf (UMLClass clazz)
   {
      if (clazz != null)
      {
         if (this == clazz)
         {
            return true;
         }
         UMLClass superClazz = this.getSuperClass();
         while (superClazz != null)
         {
            if (clazz == superClazz)
            {
               return true;
            }
            else
            {
               superClazz = superClazz.getSuperClass();
            }
         }
      }
      return false;
   }


   /**
    * Get the assignableFrom attribute of the UMLClass object
    *
    * @param clazz  No description provided
    * @return       The assignableFrom value
    */
   public boolean isAssignableFrom (UMLClass clazz)
   {
      if (clazz != null)
      {
         if (this == clazz)
         {
            return true;
         }
         boolean result = false;
         Iterator generalizations = clazz.iteratorOfRevSubclass();
         while (generalizations.hasNext() && !result)
         {
            UMLGeneralization gen = (UMLGeneralization) generalizations.next();
            clazz = gen.getSuperclass();
            if (clazz == this)
            {
               return true;
            }
            else
            {
               result = isAssignableFrom (clazz);
            }
         }
      }
      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient FPropTreeSet revType;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInRevType (FQualifier value)
   {
      return  ( (this.revType != null) &&
          (value != null) &&
         this.revType.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfRevType()
   {
      return  ( (this.revType == null)
         ? FEmptyIterator.get()
         : this.revType.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfRevType()
   {
      return  ( (this.revType == null)
         ? 0
         : this.revType.size());
   }


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToRevType (FQualifier value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.revType == null)
         {
            this.revType = new FPropTreeSet (this, "revType");
         }
         changed = this.revType.add (value);
         if (changed)
         {
            value.setType (this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromRevType (FQualifier value)
   {
      boolean changed = false;
      if ( (this.revType != null) &&  (value != null))
      {
         changed = this.revType.remove (value);
         if (changed)
         {
            value.setType (null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromRevType()
   {
      UMLQualifier tmpValue;
      Iterator iter = this.iteratorOfRevType();
      while (iter.hasNext())
      {
         tmpValue = (UMLQualifier) iter.next();
         this.removeFromRevType (tmpValue);
      }
   }


   /**
    * Isolates the object so the garbage collector can remove it.
    */
   public void removeYou()
   {
      this.removeAllFromStereotypes();
      this.setFile (null);
      this.removeAllFromRevImportedClasses();
      this.removeAllFromRevAttrType();
      this.removeAllFromParsedMembers();
      this.removeAllFromAttrs();
      this.removeAllFromRoles();
      this.removeAllFromInstances();
      this.removeAllFromMethods();
      this.setDeclaredInReferences (null);
      this.setDeclaredInPackage (null);
      this.setDeclaredInMethod (null);
      this.setDeclaredInClass (null);
      this.removeAllFromDeclares();
      this.removeAllFromRevSubclass();
      this.removeAllFromRevSuperclass();
      this.removeAllFromRevResultType();
      this.removeAllFromRevParamType();
      this.setRevTypes (null);
      this.setRevArrayType (null);
      this.setStartOfStateChart (null);
      this.removeAllFromUnparseInformations();

      UMLReactiveDiagram tmpUMLReactiveDiagram = getUMLReactiveDiagram();
      if (tmpUMLReactiveDiagram != null)
      {
         setUMLReactiveDiagram (null);
      } // if

      super.removeYou();
   } // removeYou


   /**
    * Query the logical parent of this element (e.g. package of a class, diagram of an object).
    *
    * @return   the logical parent of this element;
    */
   public FElement getParentElement()
   {
      if (getDeclaredInPackage() != null)
      {
         return getDeclaredInPackage();
      }
      else if (getDeclaredInMethod() != null)
      {
         return getDeclaredInMethod();
      }
      else if (getDeclaredInClass() != null)
      {
         return getDeclaredInClass();
      }
      else if (getDeclaredInReferences() != null)
      {
         return getDeclaredInReferences();
      }
      else
      {
         //declared nowhere
         return null;
      }
   }


   /**
    * Add all methods with given signature to the methods set, found in the subclasses and -interfaces.
    *
    * @param signature  signature to match
    * @return           set of found method
    */
   public Set findMethodsWithSignatureInSubclasses (String signature)
   {
      HashSet result = new HashSet();
      findMethodsWithSignatureInSubclasses (signature, result);
      return result;
   }


   /**
    * Add all methods with given signature to the methods set, found in the subclasses and -interfaces.
    *
    * @param signature  signature to match
    * @param methods    result set
    */
   private void findMethodsWithSignatureInSubclasses (String signature, Set methods)
   {
      for (Iterator it = iteratorOfRevSuperclass(); it.hasNext(); )
      {
         UMLGeneralization gen = (UMLGeneralization) it.next();
         UMLClass superclass = gen.getSubclass();
         UMLMethod method = superclass.getFromMethods (signature);
         if (method != null)
         {
            if (methods.contains (method))
            {
               throw new IllegalStateException ("Cyclic class hierarchy!");
            }
            methods.add (method);
         }
         superclass.findMethodsWithSignatureInSubclasses (signature, methods);
      }
   }


   /**
    * Add all methods with given signature to the methods set, found in the superclasses and -interfaces.
    *
    * @param signature  signature to match
    * @return           set of found method
    */
   public Set findMethodsWithSignatureInSuperclasses (String signature)
   {
      HashSet result = new HashSet();
      findMethodsWithSignatureInSuperclasses (signature, result);
      return result;
   }


   /**
    * Add all methods with given signature to the methods set, found in the superclasses and -interfaces.
    *
    * @param signature  signature to match
    * @param methods    result set
    */
   private void findMethodsWithSignatureInSuperclasses (String signature, Set methods)
   {
      for (Iterator it = iteratorOfRevSubclass(); it.hasNext(); )
      {
         UMLGeneralization gen = (UMLGeneralization) it.next();
         UMLClass superclass = gen.getSuperclass();
         UMLMethod method = superclass.getFromMethods (signature);
         if (method != null)
         {
            if (methods.contains (method))
            {
               throw new IllegalStateException ("Cyclic class hierarchy!");
            }
            methods.add (method);
         }
         superclass.findMethodsWithSignatureInSuperclasses (signature, methods);
      }
   }


   // ============================ inner classes =====================================

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: creckord $
    * @version   $Revision: 1.461.2.10 $ $Date: 2006/06/07 09:13:32 $
    */
   private final static class SignalFilter implements FFilterIterator.Filter, FFilterIterator.Translator
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private static SignalFilter singleton = null;


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @return   No description provided
       */
      public static SignalFilter get()
      {
         if (singleton == null)
         {
            singleton = new SignalFilter();
         }
         return singleton;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param arg  No description provided
       * @return     No description provided
       */
      public boolean include (Object arg)
      {
         if (arg instanceof Map.Entry)
         {
            arg =  ((Map.Entry) arg).getValue();
         }
         return  ( (arg instanceof UMLMethod) &&  ((UMLMethod) arg).isSignal());
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param arg  No description provided
       * @return     No description provided
       */
      public Object translate (Object arg)
      {
         return  ((Map.Entry) arg).getKey();
      }
   }

}

/*
 * $Log: UMLClass.java,v $
 * Revision 1.461.2.10  2006/06/07 09:13:32  creckord
 * - UMLTransitionGuard can be null instead of UMLTransitionGuard.NONE
 * - old "repair assoc" code removed (access methods are not kept, so repairing them is no longer needed)
 * - loop bends for assocs are removed when loop is removed
 *
 */
