/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml;

import java.beans.PropertyChangeEvent;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import de.uni_paderborn.fujaba.metamodel.FElement;
import de.upb.tools.fca.EnumerationForAnIterator;
import de.upb.tools.fca.FLinkedList;
import de.upb.tools.pcs.CollectionChangeEvent;


/**
 * <h2>Associations</h2>
 *
 * <pre>
 *                0..n    callTarget    0..1
 * UMLCollabStat ---------------------------- UMLObject
 *                collabStats     callTarget
 *
 *            0..1    callSource    0..n
 * UMLObject ---------------------------- UMLCollabStat
 *            callSource      collabStat
 *
 *                0..1    masterCollabStat    0..1
 * UMLCollabStat ---------------------------------- UMLStoryPattern
 *                revMasterCollabStat    myPattern
 *
 *                0..n    subStats    0..1
 * UMLCollabStat -------------------------- UMLCollabStat
 *                subStats      fatherStat
 *
 *             0..1   masterCollabStat   0..1
 * UMLDiagram -------------------------------- UMLCollabStat
 *             diag                collabStat
 * </pre>
 *
 * @author    $Author: l3_g5 $
 * @version   $Revision: 1.81.2.8 $
 */
public class UMLCollabStat extends UMLDiagramItem
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static Pattern PATH_PATTERN = Pattern.compile ("(\\d+)([_a-zA-Z]\\w*)?(?:\\.(.+))?");

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int TYPE_CALL = 0;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int TYPE_COND = 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int TYPE_LOOP = 2;


   /**
    * Constructor for class UMLCollabStat
    */
   public UMLCollabStat() { }


   /**
    * Constructor for class UMLCollabStat
    *
    * @param father  No description provided
    * @param src     No description provided
    * @param tgt     No description provided
    * @param diag    No description provided
    */
   public UMLCollabStat (UMLCollabStat father, UMLObject src, UMLObject tgt, UMLDiagram diag)
   {
      this.setFatherStat (father);
      this.setCallSource (src);
      this.setCallTarget (tgt);
      this.addToDiagrams (diag);
   }


   /**
    * If callOnElementsOfSet is true, the Collaboration Statement is executed on every element
    * of the set instead of executing it on the set itself. callOnElementsOfSet should do nothing
    * if the set is delete type. The Collaboration Editor should also allow to set this flag
    * only if the UMLObject is a set and of create/none type. The flag is used in JavaFactory
    * to generate code to iterate through the set and call the method.
    */
   private boolean callOnElementsOfSet = false;


   /**
    * Get the callOnElementsOfSet attribute of the UMLCollabStat object
    *
    * @return   The callOnElementsOfSet value
    */
   public boolean isCallOnElementsOfSet()
   {
      return this.callOnElementsOfSet;
   }


   /**
    * Sets the callOnElementsOfSet attribute of the UMLCollabStat object
    *
    * @param flag  The new callOnElementsOfSet value
    */
   public void setCallOnElementsOfSet (boolean flag)
   {
      if (this.callOnElementsOfSet != flag)
      {
         boolean oldFlag = this.callOnElementsOfSet;
         this.callOnElementsOfSet = flag;
         firePropertyChange ("callOnElementsOfSet", Boolean.valueOf (oldFlag), Boolean.valueOf (flag));
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int number = 0;


   /**
    * Get the number attribute of the UMLCollabStat object
    *
    * @return   The number value
    */
   public int getNumber()
   {
      return this.number;
   }


   /**
    * Sets the number attribute of the UMLCollabStat object
    *
    * @param number  The new number value
    */
   public void setNumber (int number)
   {
      if (this.number != number)
      {
         int oldNumber = this.number;
         this.number = number;
         firePropertyChange ("number", oldNumber, number);
         createNoText();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient String noText = null;


   /**
    * Get the noText attribute of the UMLCollabStat object
    *
    * @return   The noText value
    */
   public String getNoText()
   {
      if (noText == null)
      {
         createNoText();
      }
      return noText;
   } // getNoText


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void createNoText()
   {
      String oldNoText = this.noText;
      String newNoText = null;
      if (this.fatherStat == null)
      {
         //rootStat has no text
      }
      else
      {
         String fatherText = this.fatherStat.getNoText();
         if (fatherText != null && fatherText.length() > 0)
         {
            StringBuffer buffer = new StringBuffer (fatherText);
            buffer.append ('.');
            buffer.append (this.number);
            newNoText = buffer.toString();
         }
         else
         {
            newNoText = Integer.toString (this.number);
         }
      }

      if ( (oldNoText == null && newNoText != null) ||
          (oldNoText != null && !oldNoText.equals (newNoText)))
      {
         if (this.noText == null && this.text == null)
         {
            //avoid endless loop if (full) text is not computed yet
            this.noText = "null";
         }
         String oldFullText = getText();
         this.noText = newNoText;
         fireTextChange ("noText", oldNoText, newNoText, oldFullText);
         for (Iterator subStats = this.iteratorOfSubStats(); subStats.hasNext(); )
         {
            UMLCollabStat subStat = (UMLCollabStat) subStats.next();
            subStat.createNoText();
         }
      }
   }


   /**
    * Sets the noText attribute of the UMLCollabStat object
    *
    * @param noText                  The new noText value
    * @return                        No description provided
    * @deprecated                    Use setNumber and setFatherStat instead.
    * @throws NumberFormatException  Exception description not provided
    */
   public String setNoText (String noText) throws NumberFormatException
   {
      if (noText != null && !noText.equals (""))
      {
         int lastDot = noText.lastIndexOf ('.');
         String parentPath = null;
         if (lastDot > -1)
         {
            parentPath = noText.substring (0, lastDot);
            noText = noText.substring (lastDot + 1);
         }
         int number = Integer.parseInt (noText);
         setNumber (number);

         //don't create the path up to this collab stat
         //while loading, because it's done by setting the
         //parent
         if (!UMLProject.isLoading() && lastDot > -1)
         {
            UMLCollabStat master = getRootFatherStat();
            if (master != this)
            {
               UMLCollabStat parent = master.findSubStat (parentPath);
               if (parent == null)
               {
                  parent = master.createSubStat (parentPath);
               }
               parent.addToSubStats (this);
            }
         }
      }
      else
      {
         setNumber (0);
      }

      return this.getNoText();
   } // setNoText


   /**
    * Get the noDepth attribute of the UMLCollabStat object
    *
    * @return   The noDepth value
    */
   public int getNoDepth()
   {
      return getNoDepth (0);
   }


   /**
    * Get the noDepth attribute of the UMLCollabStat object
    *
    * @param i  No description provided
    * @return   The noDepth value
    */
   private int getNoDepth (int i)
   {
      UMLCollabStat father = getFatherStat();
      if (father == null)
      {
         return i;
      }
      else
      {
         return father.getNoDepth (++i);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String callText = "";


   /**
    * Get the callText attribute of the UMLCollabStat object
    *
    * @return   The callText value
    */
   public String getCallText()
   {
      return  (callText == null) ? "" : callText;
   } // getCallText


   /**
    * Sets the callText attribute of the UMLCollabStat object
    *
    * @param callText  The new callText value
    * @return          No description provided
    */
   public String setCallText (String callText)
   {
      if ( (this.callText == null) ||  (this.callText != null && !this.callText.equals (callText)))
      {
         String oldFullText = getText();
         String oldValue = this.getText();
         this.callText = callText;
         fireTextChange ("callText", oldValue, callText, oldFullText);
      }

      return this.callText;
   } // setCallText


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String assignTgtText = "";


   /**
    * Get the assignTgtText attribute of the UMLCollabStat object
    *
    * @return   The assignTgtText value
    */
   public String getAssignTgtText()
   {
      return  (assignTgtText == null) ? "" : assignTgtText;
   } // getAssignTgtText


   /**
    * Sets the assignTgtText attribute of the UMLCollabStat object
    *
    * @param assignTgtText  The new assignTgtText value
    * @return               No description provided
    */
   public String setAssignTgtText (String assignTgtText)
   {
      if ( (this.assignTgtText == null) ||  (this.assignTgtText != null && !this.assignTgtText.equals (assignTgtText)))
      {
         String oldFullText = getText();
         String oldValue = this.assignTgtText;
         this.assignTgtText = assignTgtText;
         fireTextChange ("assignTgtText", oldValue, assignTgtText, oldFullText);
      }

      return this.assignTgtText;
   } // setAssignTgtText


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLType assignTgtType;


   /**
    * Get the assignTgtType attribute of the UMLCollabStat object
    *
    * @return   The assignTgtText value
    */
   public UMLType getAssignTgtType()
   {
      return assignTgtType;
   } // getAssignTgtType


   /**
    * Sets the assignTgtType attribute of the UMLCollabStat object
    *
    * @param assignTgtType  The new assignTgtText value
    * @return               No description provided
    */
   public UMLType setAssignTgtType (UMLType assignTgtType)
   {
      if ( (this.assignTgtType == null) ||  (this.assignTgtType != null && !this.assignTgtType.equals (assignTgtType)))
      {
         String oldFullText = getText();
         UMLType oldValue = this.assignTgtType;
         this.assignTgtType = assignTgtType;
         fireTextChange ("assignTgtType", oldValue, assignTgtType, oldFullText);
      }

      return this.assignTgtType;
   } // setAssignTgtType


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String ifCondText = "";


   /**
    * Get the ifCondText attribute of the UMLCollabStat object
    *
    * @return   The ifCondText value
    */
   public String getIfCondText()
   {
      return  (ifCondText == null) ? "" : ifCondText;
   } // getIfCondText


   /**
    * Sets the ifCondText attribute of the UMLCollabStat object
    *
    * @param ifCondText  The new ifCondText value
    * @return            No description provided
    */
   public String setIfCondText (String ifCondText)
   {
      if ( (this.ifCondText == null) ||  (this.ifCondText != null && !this.ifCondText.equals (ifCondText)))
      {
         String oldFullText = getText();
         String oldValue = this.ifCondText;
         this.ifCondText = ifCondText;
         fireTextChange ("ifCondText", oldValue, ifCondText, oldFullText);
      }
      this.setStatType (TYPE_COND);
      return this.ifCondText;
   } // setIfCondText


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String loopVarName = "";


   /**
    * Get the loopVarName attribute of the UMLCollabStat object
    *
    * @return   The loopVarName value
    */
   public String getLoopVarName()
   {
      return  (loopVarName == null) ? "" : loopVarName;
   } // getLoopVarName


   /**
    * Sets the loopVarName attribute of the UMLCollabStat object
    *
    * @param loopVarName  The new loopVarName value
    * @return             No description provided
    */
   public String setLoopVarName (String loopVarName)
   {
      if ( (this.loopVarName == null) ||  (this.loopVarName != null && !this.loopVarName.equals (loopVarName)))
      {
         String oldFullText = getText();
         String oldValue = this.loopVarName;
         this.loopVarName = loopVarName;
         fireTextChange ("loopVarName", oldValue, loopVarName, oldFullText);
      }
      this.setStatType (TYPE_LOOP);
      return this.loopVarName;
   } // setLoopVarName


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String loopStartVal = "";


   /**
    * Get the loopStartVal attribute of the UMLCollabStat object
    *
    * @return   The loopStartVal value
    */
   public String getLoopStartVal()
   {
      return  (loopStartVal == null) ? "" : loopStartVal;
   } // getLoopStartVal


   /**
    * Sets the loopStartVal attribute of the UMLCollabStat object
    *
    * @param loopStartVal  The new loopStartVal value
    * @return              No description provided
    */
   public String setLoopStartVal (String loopStartVal)
   {
      if ( (this.loopStartVal == null) ||  (this.loopStartVal != null && !this.loopStartVal.equals (loopStartVal)))
      {
         String oldFullText = getText();
         String oldValue = this.loopStartVal;
         this.loopStartVal = loopStartVal;
         fireTextChange ("loopStartVal", oldValue, loopStartVal, oldFullText);
      }
      this.setStatType (TYPE_LOOP);
      return this.loopStartVal;
   } // setLoopStartVal


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String loopStopVal = "";


   /**
    * Get the loopStopVal attribute of the UMLCollabStat object
    *
    * @return   The loopStopVal value
    */
   public String getLoopStopVal()
   {
      return  (loopStopVal == null) ? "" : loopStopVal;
   } // getLoopStopVal


   /**
    * Sets the loopStopVal attribute of the UMLCollabStat object
    *
    * @param loopStopVal  The new loopStopVal value
    * @return             No description provided
    */
   public String setLoopStopVal (String loopStopVal)
   {
      if ( (this.loopStopVal == null) ||  (this.loopStopVal != null && !this.loopStopVal.equals (loopStopVal)))
      {
         String oldFullText = getText();
         String oldValue = this.loopStopVal;
         this.loopStopVal = loopStopVal;
         fireTextChange ("loopStopVal", oldValue, loopStopVal, oldFullText);
      }

      return this.loopStopVal;
   } // setLoopStopVal


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String whileLoopText = "";


   /**
    * Get the whileLoopText attribute of the UMLCollabStat object
    *
    * @return   The whileLoopText value
    */
   public String getWhileLoopText()
   {
      return  (whileLoopText == null) ? "" : whileLoopText;
   } // getWhileLoopText


   /**
    * Sets the whileLoopText attribute of the UMLCollabStat object
    *
    * @param whileLoopText  The new whileLoopText value
    * @return               No description provided
    */
   public String setWhileLoopText (String whileLoopText)
   {
      if ( (this.whileLoopText == null) ||  (this.whileLoopText != null && !this.whileLoopText.equals (whileLoopText)))
      {
         String oldFullText = getText();
         String oldValue = this.whileLoopText;
         this.whileLoopText = whileLoopText;
         fireTextChange ("whileLoopText", oldValue, whileLoopText, oldFullText);
      }
      this.setStatType (TYPE_LOOP);
      return this.whileLoopText;
   } // setWhileLoopText


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String threadId = "";


   /**
    * Get the threadId attribute of the UMLCollabStat object
    *
    * @return   The threadId value
    */
   public String getThreadId()
   {
      return  (threadId == null) ? "" : threadId;
   } // getThreadId


   /**
    * Sets the threadId attribute of the UMLCollabStat object
    *
    * @param threadId  The new threadId value
    * @return          No description provided
    */
   public String setThreadId (String threadId)
   {
      if ( (this.threadId == null) ||  (this.threadId != null && !this.threadId.equals (threadId)))
      {
         String oldFullText = getText();
         String oldValue = this.threadId;
         this.threadId = threadId;
         fireTextChange ("threadId", oldValue, threadId, oldFullText);
      }

      return this.threadId;
   } // setThreadId


   /**
    * <pre>
    *                0..n    callTarget    0..1
    * UMLCollabStat ---------------------------- UMLObject
    *                collabStats     callTarget
    * </pre>
    */
   private UMLObject callTarget;


   /**
    * Get the callTarget attribute of the UMLCollabStat object
    *
    * @return   The callTarget value
    */
   public UMLObject getCallTarget()
   {
      return callTarget;
   } // getCallTarget


   /**
    * Sets the callTarget attribute of the UMLCollabStat object
    *
    * @param callTarget  The new callTarget value
    */
   public void setCallTarget (UMLObject callTarget)
   {
      if (this.callTarget != callTarget)
      {
         // newPartner
         UMLObject oldCallTarget = this.callTarget;
         if (this.callTarget != null)
         {
            // inform old partner
            this.callTarget = null;
            oldCallTarget.removeFromCollabStats (this);
         }

         this.callTarget = callTarget;
         if (callTarget != null)
         {
            // inform new partner
            callTarget.addToCollabStats (this);
         }

         firePropertyChange ("callTarget", oldCallTarget, callTarget);
      }
   } // setCallTarget


   /**
    * <pre>
    *            0..1    callSource    0..n
    * UMLObject ---------------------------- UMLCollabStat
    *            callSource      collabStat
    * </pre>
    */
   private UMLObject callSource;


   /**
    * Sets the callSource attribute of the UMLCollabStat object
    *
    * @param value  The new callSource value
    * @return       No description provided
    */
   public boolean setCallSource (UMLObject value)
   {
      boolean changed = false;
      if (this.callSource != value)
      {
         if (this.callSource != null)
         {
            UMLObject oldValue = this.callSource;
            this.callSource = null;
            oldValue.removeFromCallSourceCollabStats (this);
         }
         this.callSource = value;
         if (value != null)
         {
            value.addToCallSourceCollabStats (this);
         }
         changed = true;
      }
      return changed;
   }


   /**
    * Get the callSource attribute of the UMLCollabStat object
    *
    * @return   The callSource value
    */
   public UMLObject getCallSource()
   {
      return this.callSource;
   }


   /**
    * <pre>
    *                0..1    masterCollabStat    0..1
    * UMLCollabStat ---------------------------------- UMLStoryPattern
    *                revMasterCollabStat    myPattern
    * </pre>
    */
   private UMLStoryPattern myPattern;


   /**
    * Get the myPattern attribute of the UMLCollabStat object
    *
    * @return   The myPattern value
    */
   public UMLStoryPattern getMyPattern()
   {
      return myPattern;
   } // getMyPattern


   /**
    * Sets the myPattern attribute of the UMLCollabStat object
    *
    * @param myPattern  The new myPattern value
    */
   public void setMyPattern (UMLStoryPattern myPattern)
   {
      if (this.myPattern != myPattern)
      {
         // newPartner
         UMLStoryPattern oldMyPattern = this.myPattern;
         if (this.myPattern != null)
         {
            // inform old partner
            this.myPattern = null;
            oldMyPattern.setRevMasterCollabStat (null);
         }

         this.myPattern = myPattern;
         firePropertyChange ("myPattern", oldMyPattern, myPattern);

         if (myPattern != null)
         {
            // inform new partner
            myPattern.setRevMasterCollabStat (this);
         }

      }
   } // setMyPattern


   /**
    * <pre>
    *                0..n    subStats    0..1
    * UMLCollabStat -------------------------- UMLCollabStat
    *                subStats      fatherStat
    * </pre>
    */
   private UMLCollabStat fatherStat;


   /**
    * Get the fatherStat attribute of the UMLCollabStat object
    *
    * @return   The fatherStat value
    */
   public UMLCollabStat getFatherStat()
   {
      return fatherStat;
   } // getFatherStat


   /**
    * Sets the fatherStat attribute of the UMLCollabStat object
    *
    * @param fatherStat  The new fatherStat value
    */
   public void setFatherStat (UMLCollabStat fatherStat)
   {
      if (this.fatherStat != fatherStat)
      {
         // newPartner
         UMLCollabStat oldFatherStat = this.fatherStat;
         if (this.fatherStat != null)
         {
            // inform old partner
            this.fatherStat = null;
            oldFatherStat.removeFromSubStats (this);
         }
         this.fatherStat = fatherStat;
         firePropertyChange ("fatherStat", oldFatherStat, fatherStat);
         if (fatherStat != null)
         {
            // inform new partner
            fatherStat.addToSubStats (this);
         }
         createNoText();
      }
   } // setFatherStat


   /**
    * <pre>
    *                0..n    subStats    0..1
    * UMLCollabStat -------------------------- UMLCollabStat
    *                subStats      fatherStat
    * </pre>
    */
   private transient FLinkedList subStats = new FLinkedList();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    * @return      No description provided
    */
   public boolean hasInSubStats (UMLCollabStat elem)
   {
      return this.subStats.contains (elem);
   } // hasInSubStats


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Enumeration elementsOfSubStats()
   {
      return new EnumerationForAnIterator (iteratorOfSubStats());
   } // elementsOfSubStats


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfSubStats()
   {
      return this.subStats.iterator();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfSubStats()
   {
      return  (this.subStats.size());
   } // sizeOfSubStats


   /**
    * Get the lastOfSubStats attribute of the UMLCollabStat object
    *
    * @return   The lastOfSubStats value
    */
   public UMLCollabStat getLastOfSubStats()
   {
      return  ( (this.subStats.isEmpty() == true)
         ? null
         : (UMLCollabStat) this.subStats.getLast());
   }


   /**
    * Access method for an one to n association.
    *
    * @param elem  The object added.
    */
   public void addToSubStats (UMLCollabStat elem)
   {
      if (elem != null && !hasInSubStats (elem))
      {
         this.subStats.add (elem);
         elem.setFatherStat (this);
         firePropertyChange (CollectionChangeEvent.get (this, "subStats", this.subStats, null,
            elem, subStats.size() - 1, CollectionChangeEvent.ADDED));
      }
   } // addToSubStats


   /**
    * Access method for a To N-association.
    *
    * @param elem  The object added.
    */
   public void addFirstToSubStats (UMLCollabStat elem)
   {
      if (elem != null && !hasInSubStats (elem))
      {
         this.subStats.addFirst (elem);
         elem.setFatherStat (this);
         firePropertyChange (CollectionChangeEvent.get (this, "subStats", this.subStats, null,
            elem, 0, CollectionChangeEvent.ADDED));
      }
   } // addToSubStats


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    * @return      No description provided
    */
   public int indexOfSubStats (UMLCollabStat elem)
   {
      return this.subStats.indexOf (elem);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    */
   public void removeFromSubStats (UMLCollabStat elem)
   {
      int index = this.subStats.indexOf (elem);
      if (index > -1)
      {
         this.subStats.remove (index);
         elem.setFatherStat (null);
         firePropertyChange (CollectionChangeEvent.get (this, "subStats", this.subStats, elem,
            null, index, CollectionChangeEvent.REMOVED));
      }
   } // removeFromSubStats


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromSubStats()
   {
      UMLCollabStat item;
      Iterator iter = iteratorOfSubStats();

      while (iter.hasNext())
      {
         item = (UMLCollabStat) iter.next();
         item.setFatherStat (null);
         firePropertyChange (CollectionChangeEvent.get (this, "subStats", this.subStats, item,
            null, 0, CollectionChangeEvent.REMOVED));
      }
   }


   /**
    * Get the prevFromPreorderEnum attribute of the UMLCollabStat object
    *
    * @return   The prevFromPreorderEnum value
    */
   public UMLCollabStat getPrevFromPreorderEnum()
   {
      UMLCollabStat prevPreorderStat = null;
      UMLCollabStat prevStat = getPrevCollabStat();
      if (prevStat != null)
      {
         prevPreorderStat = prevStat;
         while ( (prevStat = prevStat.getLastOfSubStats()) != null)
         {
            prevPreorderStat = prevStat;
         }
      }
      else
      {
         prevPreorderStat = getFatherStat();
      }
      return prevPreorderStat;
   }


   /**
    * Get the nextFromPreorderEnum attribute of the UMLCollabStat object
    *
    * @return   The nextFromPreorderEnum value
    */
   public UMLCollabStat getNextFromPreorderEnum()
   {
      if (sizeOfSubStats() > 0)
      {
         return (UMLCollabStat) subStats.getFirst();
      }
      else
      {
         return getNextFromPreorderEnum (this.getNextCollabStat());
      }
   }


   /**
    * Get the nextFromPreorderEnum attribute of the UMLCollabStat object
    *
    * @param collabStat  No description provided
    * @return            The nextFromPreorderEnum value
    */
   private UMLCollabStat getNextFromPreorderEnum (UMLCollabStat collabStat)
   {
      if (collabStat == null)
      {
         collabStat = getFatherStat();
         if (collabStat != null)
         {
            return collabStat.getNextFromPreorderEnum (collabStat.getNextCollabStat());
         }
      }
      return collabStat;
   }


   /**
    * Get the prevCollabStat attribute of the UMLCollabStat object
    *
    * @return   The prevCollabStat value
    */
   public UMLCollabStat getPrevCollabStat()
   {
      UMLCollabStat prevStat = null;
      if (fatherStat != null && this != fatherStat.subStats.getFirst())
      {
         int position = fatherStat.subStats.indexOf (this);
         prevStat = (UMLCollabStat) fatherStat.subStats.get (position - 1);
      }
      return prevStat;
   }


   /**
    * Get the nextCollabStat attribute of the UMLCollabStat object
    *
    * @return   The nextCollabStat value
    */
   public UMLCollabStat getNextCollabStat()
   {
      UMLCollabStat nextStat = null;
      if (fatherStat != null && this != fatherStat.subStats.getLast())
      {
         int position = fatherStat.subStats.indexOf (this);
         nextStat = (UMLCollabStat) fatherStat.subStats.get (position + 1);
      }
      return nextStat;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param path  No description provided
    * @return      No description provided
    */
   public UMLCollabStat createSubStat (String path)
   {
      path = path.trim();
      Matcher matcher = PATH_PATTERN.matcher (path);
      if (matcher.matches())
      {
         int number = Integer.parseInt (matcher.group (1));
         String thread = matcher.group (2);
         if (thread != null && thread.length() == 0)
         {
            thread = null;
         }
         String subPath = matcher.group (3);
         if (subPath != null && subPath.length() > 0)
         {
            UMLCollabStat subStat = findSubStat (number, thread);
            if (subStat == null)
            {
               subStat = createSubStat (number, thread);
            }
            return subStat.createSubStat (subPath);
         }
         else
         {
            return createSubStat (number, thread);
         }
      }
      else
      {
         throw new IllegalArgumentException ("Malformed sequence path expression: " + path);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param number  No description provided
    * @param thread  No description provided
    * @return        No description provided
    */
   protected UMLCollabStat createSubStat (int number, String thread)
   {
      UMLCollabStat newStat = new UMLCollabStat();
      newStat.setNumber (number);
      if (thread != null)
      {
         newStat.setThreadId (thread);
      }
      this.addToSubStats (newStat);
      return newStat;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param path  No description provided
    * @return      No description provided
    */
   public UMLCollabStat findSubStat (String path)
   {
      if (sizeOfSubStats() > 0)
      {
         path = path.trim();
         Matcher matcher = PATH_PATTERN.matcher (path);
         if (matcher.matches())
         {
            int number = Integer.parseInt (matcher.group (1));
            String thread = matcher.group (2);
            if (thread != null && thread.length() == 0)
            {
               thread = null;
            }
            String subPath = matcher.group (3);
            UMLCollabStat subStat = findSubStat (number, thread);
            if (subPath != null && subPath.length() > 0 && subStat != null)
            {
               subStat = subStat.findSubStat (subPath);
            }
            return subStat;
         }
         else
         {
            throw new IllegalArgumentException ("Malformed sequence path expression: " + path);
         }
      }
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param number  No description provided
    * @param thread  No description provided
    * @return        No description provided
    */
   public UMLCollabStat findSubStat (int number, String thread)
   {
      Iterator iter = iteratorOfSubStats();
      while (iter.hasNext())
      {
         UMLCollabStat collabStat = (UMLCollabStat) iter.next();
         if (collabStat.getNumber() == number &&
             ( (thread == null && collabStat.getThreadId().length() == 0) ||
             (thread != null && thread.equals (collabStat.getThreadId()))))
         {
            return collabStat;
         }
      }
      return null;
   }


   /**
    * <pre>
    *             0..1   masterCollabStat   0..1
    * UMLDiagram -------------------------------- UMLCollabStat
    *             diag                collabStat
    * </pre>
    */
   private UMLDiagram diag;


   /**
    * Sets the diag attribute of the UMLCollabStat object
    *
    * @param value  The new diag value
    * @return       No description provided
    */
   public boolean setDiag (UMLDiagram value)
   {
      boolean changed = false;
      if (this.diag != value)
      {
         UMLDiagram oldValue = this.diag;
         if (this.diag != null)
         {
            this.diag = null;
            oldValue.setCollabStat (null);
         }
         this.diag = value;
         firePropertyChange ("diag", oldValue, value);
         if (value != null)
         {
            value.setCollabStat (this);
         }
         changed = true;
      }
      return changed;
   }


   /**
    * Get the diag attribute of the UMLCollabStat object
    *
    * @return   The diag value
    */
   public UMLDiagram getDiag()
   {
      return this.diag;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      setDiag (null);
      setCallSource (null);
      setCallTarget (null);
      setMyPattern (null);
      setFatherStat (null);
      setMethod (null);
      removeAllFromSubStats();

      super.removeYou();
   } // removeYou


   /**
    * Query the logical parent of this element (e.g. package of a class, diagram of an object).
    *
    * @return   the logical parent of this element;
    */
   public FElement getParentElement()
   {
      if (getMyPattern() != null)
      {
         return getMyPattern();
      }
      else
      {
         return getFatherStat();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient String text;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient boolean parsing = false;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name         No description provided
    * @param oldValue     No description provided
    * @param newValue     No description provided
    * @param oldFullText  No description provided
    */
   private void fireTextChange (String name, Object oldValue, Object newValue, String oldFullText)
   {
      if (!parsing)
      {
         this.text = null;
         firePropertyChange (name, oldValue, newValue);
         firePropertyChange ("text", oldFullText, getText());
      }
   }


   /**
    * Sets the text attribute of the UMLCollabStat object
    *
    * @param text  The new text value
    */
   public void setText (String text)
   {
      if (UMLProject.isLoading())
      {
         //Parsing is not necessary for loading since all
         //involved attributes are saved explicitly.
         //Since parsing is also extremely buggy, skip it here
         return;
      }
      //FIXME: Parsing cannot handle texts properly that contain [, ] or : in condition/loop/call parts
      String oldtext = this.getText();
      if ( (oldtext == null && text != null) ||  (oldtext != null && !oldtext.equals (text)))
      {
         parsing = true;
         try
         {

            String oldNoText = getNoText();
            String oldIfCondText = getIfCondText();
            String oldWhileLoopText = getWhileLoopText();
            String oldLoopVarName = getLoopVarName();
            UMLType oldAssignTgtType = getAssignTgtType();
            String oldAssignTgtText = getAssignTgtText();
            String oldCallText = getCallText();

            if (text.startsWith (":"))
            {
               text = "1" + text;
            }
            // this method tries to parse a collab stat and to derive its parts.
            // just a test. Parsing is not yet complete. Albert
            StringTokenizer tokens = new StringTokenizer (text, "[");
            String word = null;

            this.setIfCondText ("");
            this.setWhileLoopText ("");
            this.setLoopVarName ("");
            this.setAssignTgtType (null);

            int i = tokens.countTokens();
            if (i > 1)
            { // yes a square brace exists

               if (tokens.hasMoreElements())
               {
                  word = tokens.nextToken();
                  this.setNoText (word.trim());
               }
               if (tokens.hasMoreElements())
               {
                  word = tokens.nextToken ("]");

                  // skip "["
                  word = word.substring (1);

                  // is it a while loop?
                  if (word.startsWith ("while "))
                  {
                     this.setWhileLoopText (word.substring (6).trim());
                     this.setStatType (TYPE_LOOP);
                  }
                  else
                  {
                     this.setIfCondText (word.trim());
                     this.setStatType (TYPE_COND);
                  }
               }

               // skip ":"
               if (tokens.hasMoreElements())
               {
                  tokens.nextToken (":");
               }
            }
            else
            {
               // no square brackets, up to ":" belongs to number
               if (tokens.hasMoreElements())
               {
                  word = tokens.nextToken (":");
                  this.setNoText (word.trim());
               }
            }
            if (tokens.hasMoreElements())
            {
               word = tokens.nextToken ("=").substring (1);
               if (word.endsWith (":"))
               {
                  word = word.substring (0, word.length() - 1);
               }
            }
            i = tokens.countTokens();
            if (i > 0)
            {
               // an assignment "x = m1 ()" exists
               // get the assign target variable
               String assignTxt = word.trim(); //.substring (1);
               int index = assignTxt.indexOf (" ");
               if (index > 0)
               {
                  this.setAssignTgtType (UMLProject.get().getTypeList().getFromTypes (assignTxt.substring (0, index)));
                  if (this.getAssignTgtType() != null)
                  {
                     assignTxt = assignTxt.substring (index + 1);
                  }
               }
               this.setAssignTgtText (assignTxt);

               // now look up the call text
               if (tokens.hasMoreElements())
               {
                  word = tokens.nextToken();
                  this.setCallText (word.trim());
               }
            }
            else
            {
               this.setAssignTgtText ("");
               this.setCallText (word.substring (1).trim());
            }
            this.text = null;

            firePropertyChange ("noText", oldNoText, getNoText());
            firePropertyChange ("ifCondText", oldIfCondText, getIfCondText());
            firePropertyChange ("whileLoopText", oldWhileLoopText, getWhileLoopText());
            firePropertyChange ("loopVarName", oldLoopVarName, getLoopVarName());
            firePropertyChange ("assignTgtType", oldAssignTgtType, getAssignTgtType());
            firePropertyChange ("assignTgtText", oldAssignTgtText, getAssignTgtText());
            firePropertyChange ("callText", oldCallText, getCallText());

            String tmpNewText = this.getText();
            firePropertyChange ("text", oldtext, tmpNewText);
         }
         /*
          *  catch (Exception e2)
          *  {
          *  e2.printStackTrace();
          *  /         }
          */
         finally
         {
            parsing = false;
         }
      }
   } // setText


   /**
    * Get the text attribute of the UMLCollabStat object
    *
    * @return   The text value
    */
   public String getText()
   {
      if (this.text == null)
      {
         StringBuffer buffer = new StringBuffer();
         String tmpString;

         buffer.append (getNoText());
         buffer.append (getThreadId());

         tmpString = getIfCondText();
         if ( (tmpString != null) &&  (tmpString.length() != 0))
         {
            buffer.append (" [");
            buffer.append (tmpString);
            buffer.append ("]");
         }

         tmpString = getWhileLoopText();
         if ( (tmpString != null) &&  (tmpString.length() != 0))
         {
            buffer.append (" [while ");
            buffer.append (tmpString);
            buffer.append ("]");
         }

         tmpString = getLoopVarName();
         // take loopVarName as an indicator for the existance of a loop statement
         if ( (tmpString != null) &&  (tmpString.length() != 0))
         {
            buffer.append (" [");
            buffer.append (tmpString);
            buffer.append (":=");
            buffer.append (getLoopStartVal());
            buffer.append ("..");
            buffer.append (getLoopStopVal());
            buffer.append ("]");
         }

         buffer.append (": ");

         tmpString = getAssignTgtText();
         if ( (tmpString != null) &&  (tmpString.length() != 0))
         {
            UMLType tmpType = getAssignTgtType();
            if (tmpType != null)
            {
               buffer.append (tmpType.getName());
               buffer.append (" ");
            }
            buffer.append (tmpString);
            buffer.append (" := ");
         }
         buffer.append (getCallText());

         this.text = buffer.toString();
      }

      return this.text;
   } // getText


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public String toString()
   {
      return getText();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public UMLStoryPattern findStoryPattern()
   {
      UMLCollabStat fatherStat = this;
      UMLCollabStat lastFatherStat = this;

      do
      {
         lastFatherStat = fatherStat;
         fatherStat = fatherStat.getFatherStat();
      } while (fatherStat != null);

      return lastFatherStat.getMyPattern();
   } // findStoryPattern


   /**
    * Get the rootFatherStat attribute of the UMLCollabStat object
    *
    * @return   The rootFatherStat value
    */
   public UMLCollabStat getRootFatherStat()
   {
      if (this.fatherStat == null)
      {
         return this;
      }
      else
      {
         return fatherStat.getRootFatherStat();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLMethod method;


   /**
    * Sets the value attribute of the UMLCollabStat object
    *
    * @param value  The new value value
    */
   public void setMethod (UMLMethod value)
   {
      if (this.method != value)
      {
         UMLMethod oldValue = this.method;
         this.method = value;
         firePropertyChange ("method", oldValue, value);
      }
   }


   /**
    * Get the method attribute of the UMLCollabStat object
    *
    * @return   The method value
    */
   public UMLMethod getMethod()
   {
      return method;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean expanded = false;


   /**
    * Sets the expanded attribute of the UMLCollabStat object
    *
    * @param value  The new expanded value
    */
   public void setExpanded (boolean value)
   {
      expanded = value;
   }


   /**
    * Get the expanded attribute of the UMLCollabStat object
    *
    * @return   The expanded value
    */
   public boolean isExpanded()
   {
      return expanded;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int statType = TYPE_CALL;


   /**
    * Sets the statType attribute of the UMLCollabStat object
    *
    * @param value  The new statType value
    */
   public void setStatType (int value)
   {
      if (this.statType != value)
      {
         int oldValue = this.statType;
         this.statType = value;
         firePropertyChange ("statType", oldValue, value);
      }
   }


   /**
    * Get the statType attribute of the UMLCollabStat object
    *
    * @return   The statType value
    */
   public int getStatType()
   {
      return statType;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void autoNumber()
   {
      autoNumber (1);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param number  No description provided
    */
   private void autoNumber (int number)
   {
      int kidNo = 0;
      setNumber (number);
      Iterator iter = this.iteratorOfSubStats();
      while (iter.hasNext())
      {
         UMLCollabStat kidStat = (UMLCollabStat) iter.next();
         kidNo++;
         kidStat.autoNumber (kidNo);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void renumberCollabStats()
   {
      Iterator statIter;
      UMLCollabStat aCollabStat;

      LinkedList stats = new LinkedList (this.subStats);

      Collections.sort (stats, new CollabStatLessThan());
      statIter = stats.iterator();

      removeAllFromSubStats();
      int lastNumber = 0;
      int oldNumberOfPrevious = -1;
      String threadOfPrevious = null;
      while (statIter.hasNext())
      {
         aCollabStat = (UMLCollabStat) statIter.next();
         String thread = aCollabStat.getThreadId();
         if (thread != null && thread.length() > 0)
         {
            int number = aCollabStat.getNumber();
            if (threadOfPrevious == null || oldNumberOfPrevious != number)
            {
               threadOfPrevious = thread;
               oldNumberOfPrevious = number;
               aCollabStat.setNumber (++lastNumber);
            }
            else
            {
               threadOfPrevious = thread;
               aCollabStat.setNumber (lastNumber);
            }
         }
         else
         {
            threadOfPrevious = null;
            aCollabStat.setNumber (++lastNumber);
         }
         addToSubStats (aCollabStat);
         aCollabStat.renumberCollabStats();
      }
   } // renumberCollabStats


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: l3_g5 $
    * @version   $Revision: 1.81.2.8 $
    */
   private class CollabStatLessThan implements java.util.Comparator
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param first   No description provided
       * @param second  No description provided
       * @return        No description provided
       */
      public int compare (Object first, Object second)
      {
         if (first == second)
         {
            return 0;
         }
         else if (first == null)
         {
            return Integer.MIN_VALUE;
         }
         else if (second == null)
         {
            return Integer.MAX_VALUE;
         }

         UMLCollabStat stat1 = (UMLCollabStat) first;
         UMLCollabStat stat2 = (UMLCollabStat) second;

         int depth1 = stat1.getNoDepth();
         int depth2 = stat2.getNoDepth();
         UMLCollabStat ancestor1 = stat1;
         UMLCollabStat ancestor2 = stat2;
         for (int i = 0; i < depth1 - depth2; i++)
         {
            ancestor1 = ancestor1.getFatherStat();
         }
         for (int i = 0; i < depth2 - depth1; i++)
         {
            ancestor2 = ancestor2.getFatherStat();
         }

         if (ancestor1 == ancestor2)
         {
            return depth1 - depth2; //one is substat of other
         }

         //seek common father stat
         UMLCollabStat last1 = ancestor1;
         UMLCollabStat last2 = ancestor2;
         while (ancestor1 != ancestor2)
         {
            last1 = ancestor1;
            last2 = ancestor2;
            ancestor1 = ancestor1.getFatherStat();
            ancestor2 = ancestor2.getFatherStat();
         }

         int result = last1.getNumber() - last2.getNumber(); //compare by number
         if (result == 0)
         { //now compare by thread

            result = compareStr (last1.getThreadId(), last2.getThreadId());
            if (result == 0)
            { //fallback

               if (ancestor1 != null)
               {
                  int pos1 = ancestor1.indexOfSubStats (last1);
                  int pos2 = ancestor1.indexOfSubStats (last2);
                  result = pos1 - pos2;
               }
               else
               { //ultimate fallback ;P

                  result = compareStr (stat1.getText(), stat2.getText());
               }
            }
         }
         return result;
      } // compare


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param str1  No description provided
       * @param str2  No description provided
       * @return      No description provided
       */
      private final int compareStr (final String str1, final String str2)
      {
         if (str1 == str2)
         {
            return 0;
         }
         else if (str1 == null)
         {
            return Integer.MIN_VALUE;
         }
         else if (str2 == null)
         {
            return Integer.MAX_VALUE;
         }
         return str1.compareTo (str2);
      }
   } // CollabStatLessThan


   /**
    * Hangs the current ASGElement into the ASG-tree. Needed for cut'n'paste.
    *
    * @param parent
    */
   public void setCutCopyPasteParent (FElement parent)
   {
      if (parent instanceof UMLStoryActivity)
      {
         UMLStoryActivity activity = (UMLStoryActivity) parent;
         this.removeAllFromDiagrams();
         activity.getStoryPattern().addToElements (this);
      }
      else
      {
         super.setCutCopyPasteParent (parent);
      }
   }


   /**
    * Get the persistencyChange attribute of the UMLCollabStat object
    *
    * @param e  No description provided
    * @return   The persistencyChange value
    */
   protected boolean isPersistencyChange (PropertyChangeEvent e)
   {
      return super.isPersistencyChange (e) &&
         !"noText".equals (e.getPropertyName()) && !"text".equals (e.getPropertyName());
   }
}

/*
 * $Log: UMLCollabStat.java,v $
 * Revision 1.81.2.8  2006/03/14 13:26:59  l3_g5
 * bugfix
 *
 */
