/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml;

import java.beans.PropertyChangeEvent;
import java.util.HashMap;
import java.util.TreeMap;

import org.apache.log4j.Logger;
import de.uni_paderborn.fujaba.basic.RuntimeExceptionWithContext;

import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.metamodel.FDiagramItem;
import de.uni_paderborn.fujaba.metamodel.FElement;


/**
 * <h2>Associations</h2>
 *
 * <pre>
 *          0..n    source    0..1
 * UMLLink ------------------------ UMLObject
 *          revSource       source
 *
 *          0..n    target    0..1
 * UMLLink ------------------------ UMLObject
 *          revTarget       target
 *
 *          0..n    instanceOf    0..1
 * UMLLink ---------------------------- UMLAssoc
 *          instances       instanceOf
 *
 *          0..1     sourceLink     0..1
 * UMLLink ------------------------------ UMLMultiLink
 *          revSourceLink     sourceLink
 *
 *          0..1     targetLink     0..1
 * UMLLink ------------------------------ UMLMultiLink
 *          targetLink     revTargetLink
 * </pre>
 *
 * @author    $Author: creckord $
 * @version   $Revision: 1.176.2.2 $
 */
public class UMLLink extends UMLConnection implements Traversable
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (UMLLink.class);

   // ------------------------------- Constants ---------------------------------------

   // type = {}
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int NULL = 0;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int NEGATIVE = 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int OPTIONAL = 2;

   // modifier = {}
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int NONE = UMLStoryPattern.NONE;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int CREATE = UMLStoryPattern.CREATE;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int DELETE = UMLStoryPattern.DELETE;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int P_CHECK = 0;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int P_TO_ONE = P_CHECK + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int P_CHECK_TO_MANY = P_TO_ONE + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int P_MULTILINK_CHECK = P_CHECK_TO_MANY + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int P_MULTILINK_FIRST = P_MULTILINK_CHECK + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int P_MULTILINK_LAST = P_MULTILINK_FIRST + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int P_MULTILINK_BOUND_TO_UNBOUND = P_MULTILINK_LAST + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int P_MULTILINK_UNBOUND_TO_BOUND = P_MULTILINK_BOUND_TO_UNBOUND + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int P_MULTILINK_ENTRY = P_MULTILINK_UNBOUND_TO_BOUND + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int P_MULTILINK_PATH = P_MULTILINK_ENTRY + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int P_TO_MANY = P_MULTILINK_PATH + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int P_MULTILINK_ENTRY_OPTIONAL = P_TO_MANY + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int P_MULTILINK_PATH_OPTIONAL = P_MULTILINK_ENTRY_OPTIONAL + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int P_OPTIONAL = 20;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int P_OPTIONAL_CHECK = P_OPTIONAL;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int P_OPTIONAL_TO_ONE = P_OPTIONAL_CHECK + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int P_OPTIONAL_TO_MANY = P_OPTIONAL_TO_ONE + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int P_SET = 40;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int P_NEGATIVE = 60;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int P_NONE = -1;

   // generateJavaOptions
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int SEARCH = 0;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int MODIFY = SEARCH + 1;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String VIA_ASGELEMENTREF_SUFFIX = " (via ASGElementRef)";

   // ------------------------------- Constructors ------------------------------------

   /**
    * Constructor for class UMLLink
    */
   public UMLLink()
   {
      super();
   } // constructor


   /**
    * Constructor for class UMLLink
    *
    * @param name        No description provided
    * @param type        No description provided
    * @param modifier    No description provided
    * @param range       No description provided
    * @param source      No description provided
    * @param target      No description provided
    * @param instanceOf  No description provided
    */
   public UMLLink (String name, int type, int modifier,
                   String range, UMLObject source, UMLObject target, UMLAssoc instanceOf)
   {
      super();
      setName (name);
      setType (type);
      setModifier (modifier);
      setRange (range);
      setSource (source);
      setTarget (target);
      setInstanceOf (instanceOf);
   } // constructor


   // --------------------------------- Attributes ------------------------------------

   /**
    * This attribute is needed to hide via ASGElementRef links is object diagrams. status:
    * current implementation ... DO NOT REMOVE ... (JPW)
    */
   private boolean isViaASGElementRef = false;


   /**
    * Get the isViaASGElementRef attribute of the UMLLink object
    *
    * @return   The isViaASGElementRef value
    */
   public boolean getIsViaASGElementRef()
   {
      return isViaASGElementRef;
   }


   /**
    * Sets the isViaASGElementRef attribute of the UMLLink object
    *
    * @param value  The new isViaASGElementRef value
    */
   public void setIsViaASGElementRef (boolean value)
   {
      isViaASGElementRef = value;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String name;


   /**
    * Get the name attribute of the UMLLink object
    *
    * @return   The name value
    */
   public String getName()
   {
      if (name == null && this.instanceOf != null)
      {
         return this.instanceOf.getName();
      }
      return name;
   }


   /**
    * Sets the name attribute of the UMLLink object
    *
    * @param name  The new name value
    */
   public void setName (String name)
   {
      if (name == null || !name.equals (getName()))
      {
         String oldValue = this.name;
         this.name = name;
         firePropertyChange ("name", oldValue, name);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int type = NULL;


   /**
    * Get the type attribute of the UMLLink object
    *
    * @return   The type value
    */
   public int getType()
   {
      return type;
   }


   /**
    * Sets the type attribute of the UMLLink object
    *
    * @param type  The new type value
    */
   public void setType (int type)
   {
      if (NULL <= type && type <= OPTIONAL)
      {
         int oldValue = this.type;
         this.type = type;
         firePropertyChange ("type", oldValue, type);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int modifier = NONE;


   /**
    * Get the modifier attribute of the UMLLink object
    *
    * @return   The modifier value
    */
   public int getModifier()
   {
      return modifier;
   }


   /**
    * Sets the modifier attribute of the UMLLink object
    *
    * @param modifier  The new modifier value
    */
   public void setModifier (int modifier)
   {
      int oldValue = this.modifier;
      this.modifier = modifier;
      firePropertyChange ("modifier", oldValue, modifier);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean totality = false;


   /**
    * Get the totality attribute of the UMLLink object
    *
    * @return   The totality value
    */
   public boolean getTotality()
   {
      return totality;
   }


   /**
    * Sets the totality attribute of the UMLLink object
    *
    * @param totality  The new totality value
    */
   public void setTotality (boolean totality)
   {
      boolean oldValue = this.totality;
      this.totality = totality;
      firePropertyChange ("totality", oldValue, totality);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String range;


   /**
    * Get the range attribute of the UMLLink object
    *
    * @return   The range value
    */
   public String getRange()
   {
      return range;
   }


   /**
    * Sets the range attribute of the UMLLink object
    *
    * @param range  The new range value
    */
   public void setRange (String range)
   {
      String oldValue = this.range;
      this.range = range;
      firePropertyChange ("range", oldValue, range);
   }

   // ------------------------------- Associations ----------------------------------

   /**
    * <pre>
    *          0..n    source    0..1
    * UMLLink ------------------------ UMLObject
    *          revSource       source
    * </pre>
    */
   private UMLObject source;


   /**
    * Get the source attribute of the UMLLink object
    *
    * @return   The source value
    */
   public UMLObject getSource()
   {
      return source;
   }


   /**
    * Sets the source attribute of the UMLLink object
    *
    * @param source  The new source value
    * @return        No description provided
    */
   public boolean setSource (UMLObject source)
   {
      boolean changed = false;

      if ( (this.source == null && source != null) ||
          (this.source != null && !this.source.equals (source)))
      {
         // new partner
         UMLObject oldSource = this.source;
         UMLRole oldRole = getSourceRole();
         if (this.source != null)
         {
            // inform old partner
            this.source = null;
            oldSource.removeFromRevSource (this);
         }
         this.source = source;
         if (source != null)
         {
            // inform new partner
            source.addToRevSource (this);
         }
         firePropertyChange ("source", oldSource, source);
         UMLRole newRole = getSourceRole();
         if (newRole != oldRole)
         {
            firePropertyChange ("sourceRole", oldRole, newRole);
         }
         changed = true;
      }

      return changed;
   } // setSource


   /**
    * <pre>
    *          0..n    target    0..1
    * UMLLink ------------------------ UMLObject
    *          revTarget       target
    * </pre>
    */
   private UMLObject target;


   /**
    * Get the target attribute of the UMLLink object
    *
    * @return   The target value
    */
   public UMLObject getTarget()
   {
      return target;
   }


   /**
    * Sets the target attribute of the UMLLink object
    *
    * @param target  The new target value
    * @return        No description provided
    */
   public boolean setTarget (UMLObject target)
   {
      boolean changed = false;

      if ( (this.target == null && target != null) ||
          (this.target != null && !this.target.equals (target)))
      {
         // new partner
         UMLObject oldTarget = this.target;
         UMLRole oldRole = getTargetRole();
         if (this.target != null)
         {
            // inform old partner
            this.target = null;
            oldTarget.removeFromRevTarget (this);
         }
         this.target = target;
         if (target != null)
         {
            // inform new partner
            target.addToRevTarget (this);
         }
         firePropertyChange ("target", oldTarget, target);
         UMLRole newRole = getTargetRole();
         if (newRole != oldRole)
         {
            firePropertyChange ("targetRole", oldRole, newRole);
         }
         changed = true;
      }

      return changed;
   }


   /**
    * <pre>
    *          0..n    instanceOf    0..1
    * UMLLink ---------------------------- UMLAssoc
    *          instances       instanceOf
    * </pre>
    */
   private transient UMLAssoc instanceOf;


   /**
    * Sets the instanceOf attribute of the UMLLink object
    *
    * @param instanceOf  The new instanceOf value
    */
   public void setInstanceOf (UMLAssoc instanceOf)
   {
      if (this.instanceOf != instanceOf)
      {
         // new partner
         UMLAssoc oldInstanceOf = this.instanceOf;
         if (this.instanceOf != null)
         {
            // inform old partner
            this.instanceOf = null;
            oldInstanceOf.removeFromInstances (this);
         }
         this.instanceOf = instanceOf;
         if (instanceOf != null)
         {
            // inform new partner
            instanceOf.addToInstances (this);
         }
         firePropertyChange ("instanceOf", oldInstanceOf, instanceOf);
         if (instanceOf != null)
         {
            setName (null);
         }
      }
   } // setInstanceOf


   /**
    * Get the instanceOf attribute of the UMLLink object
    *
    * @return   The instanceOf value
    */
   public UMLAssoc getInstanceOf()
   {
      return instanceOf;
   } // getInstanceOf


   /**
    * <pre>
    *          0..1     sourceLink     0..1
    * UMLLink ------------------------------ UMLMultiLink
    *          revSourceLink     sourceLink
    * </pre>
    */
   private UMLMultiLink revSourceLink;


   /**
    * Sets the revSourceLink attribute of the UMLLink object
    *
    * @param revSourceLink  The new revSourceLink value
    */
   public void setRevSourceLink (UMLMultiLink revSourceLink)
   {
      if (this.revSourceLink != revSourceLink)
      {
         // new partner
         UMLMultiLink oldRevSourceLink = this.revSourceLink;
         if (this.revSourceLink != null)
         {
            // inform old partner
            this.revSourceLink = null;
            oldRevSourceLink.setSourceLink (null);
         }
         this.revSourceLink = revSourceLink;
         if (revSourceLink != null)
         {
            // inform new partner
            revSourceLink.setSourceLink (this);
         }
         firePropertyChange ("revSourceLink", oldRevSourceLink, revSourceLink);
      }
   } // setRevSourceLink


   /**
    * Get the revSourceLink attribute of the UMLLink object
    *
    * @return   The revSourceLink value
    */
   public UMLMultiLink getRevSourceLink()
   {
      return this.revSourceLink;
   } // getRevSourceLink


   /**
    * <pre>
    *          0..1     targetLink     0..1
    * UMLLink ------------------------------ UMLMultiLink
    *          targetLink     revTargetLink
    * </pre>
    */
   private UMLMultiLink revTargetLink;


   /**
    * Sets the revTargetLink attribute of the UMLLink object
    *
    * @param revTargetLink  The new revTargetLink value
    */
   public void setRevTargetLink (UMLMultiLink revTargetLink)
   {
      if (this.revTargetLink != revTargetLink)
      {
         // new partner
         UMLMultiLink oldRevTargetLink = this.revTargetLink;
         if (this.revTargetLink != null)
         {
            // inform old partner
            this.revTargetLink = null;
            oldRevTargetLink.setTargetLink (null);
         }
         this.revTargetLink = revTargetLink;
         if (revTargetLink != null)
         {
            // inform new partner
            revTargetLink.setTargetLink (this);
         }
         firePropertyChange ("revTargetLink", oldRevTargetLink, revTargetLink);
      }
   } // setTargetLink


   /**
    * Get the revTargetLink attribute of the UMLLink object
    *
    * @return   The revTargetLink value
    */
   public UMLMultiLink getRevTargetLink()
   {
      return this.revTargetLink;
   } // getRevTargetLink


   // ------------------------------- Other Methods ---------------------------------------

   /**
    * Get the successorLink attribute of the UMLLink object
    *
    * @param objectsMap  No description provided
    * @return            The successorLink value
    */
   public UMLLink getSuccessorLink (TreeMap objectsMap)
   {
      UMLLink succLink = null;
      UMLObject source = (UMLObject) objectsMap.get (getSource());
      UMLObject target = (UMLObject) objectsMap.get (getTarget());

      if (source != null && target != null && getModifier() != DELETE)
      {
         succLink = new UMLLink (getName(), getType(), NONE, getRange(), source, target, getInstanceOf());
      }

      return succLink;
   } // getSuccessorLink


   /**
    * Get the reference attribute of the UMLLink object
    *
    * @return   The reference value
    */
   public boolean isReference()
   {
      UMLAssoc instOf = getInstanceOf();

      if (instOf != null)
      {
         return  ( (instOf.getLeftRole() != null && instOf.getLeftRole().getAdornment() == FRole.REFERENCE) ||
             (instOf.getRightRole() != null && instOf.getRightRole().getAdornment() == FRole.REFERENCE));
      }
      else
      {
         return false;
      }
   } // isReference


   /**
    * Get the navigable attribute of the UMLLink object
    *
    * @param source  No description provided
    * @return        The navigable value
    */
   public boolean isNavigable (UMLObject source)
   {
      return getCorrespondingRole (source).getAdornment() != FRole.REFERENCE;
   }


   /**
    * Get the targetsUpperBound attribute of the UMLLink object
    *
    * @param target  No description provided
    * @return        The targetsUpperBound value
    */
   private int getTargetsUpperBound (UMLObject target)
   {
      UMLClass objectInstance = null;
      if (target != null)
      {
         objectInstance = target.getInstanceOf();
      }

      if (objectInstance == null)
      {
         throw new RuntimeExceptionWithContext ("UMLObject has no UMLClass: " + target, target);
      }

      UMLAssoc linkInstance = getInstanceOf();

      if (linkInstance == null)
      {
         throw new RuntimeExceptionWithContext ("UMLLink has no UMLAssoc: " + getName(), this);
      }

      UMLRole role = getCorrespondingRole (target);
      int card = 1;

      if ( (role != null) &&
          (role.getCard() != null))
      {
         UMLCardinality cardinality = role.getCard();
         card = cardinality.getUpperBound();
      }
      return card;
   }


   /**
    * Get the priority attribute of the UMLLink object
    *
    * @param boundObjects        No description provided
    * @param isomorphicBindings  No description provided
    * @return                    The priority value
    */
   public int getPriority (HashMap boundObjects, HashMap isomorphicBindings)
   {
      UMLObject sourceObject =  (boundObjects.get (getSource().getID()) == getSource()) ? getSource() : getTarget();
      UMLObject targetObject =  (getTarget() == sourceObject) ? getSource() : getTarget();
      int priority = P_NONE;
      int type = getType();

      if (boundObjects.get (sourceObject.getID()) != sourceObject)
      {
         throw new RuntimeExceptionWithContext ("The source object is not bound", sourceObject);
      }

      // check if the reference can be traversed
      if (isReference() && getSource() == targetObject &&
         boundObjects.get (targetObject.getID()) != targetObject)
      {
         return P_NONE;
      }

      boolean typeCastNeeded = !targetObject.getObjectType().equals (getCorrespondingRole (targetObject).getTarget().getName());
      boolean targetNegative = targetObject.isNegative();
      boolean targetBound = boundObjects.get (targetObject.getID()) != null;
      boolean boundSourceSet = sourceObject.isSet();
      int targetLinks =  (targetObject.sizeOfRevSource() + targetObject.sizeOfRevTarget());
      boolean targetObjectHasOutGoingLinks =  (targetLinks > 1);

      UMLClass targetClass = targetObject.getInstanceOf();
      String targetClassName = targetClass.getFullClassName();
      boolean targetIsoCheck =  (!targetBound &&  (isomorphicBindings.get (targetClassName) != null));

      if (type == NEGATIVE ||  (targetNegative && !targetBound && !targetObject.isSet() &&
         !targetObjectHasOutGoingLinks && !targetIsoCheck))
      {
         if (getRevSourceLink() != null || getRevTargetLink() != null)
         {
            return priority = P_NONE;
         }

         if (!targetBound && targetObjectHasOutGoingLinks && !targetNegative)
         {
            if (!isReference())
            {
               return priority = P_NONE;
            }
         }
         if ( (!targetBound && !toOneAccess (getCorrespondingRole (targetObject)) &&  (targetObject.iteratorOfAttrs().hasNext() || typeCastNeeded))
            || boundSourceSet ||  (boundSourceSet && targetBound))
         {
            if (targetNegative)
            {
               return priority = P_CHECK_TO_MANY + P_OPTIONAL;
            }
            else
            {
               return priority = P_CHECK_TO_MANY;
            }
         }
         return priority = P_CHECK;
      }

      if ( ( (getModifier() == CREATE && getType() != OPTIONAL) ||
          (targetObject.getModifier() == UMLObject.CREATE) && targetObject.getType() != UMLObject.OPTIONAL) &&  (priority != P_CHECK))
      {
         return P_NONE;
      }

      priority = P_CHECK;

      if (!boundObjects.containsValue (targetObject))
      {
         if (getTargetsUpperBound (targetObject) == 1)
         {
            if (getCorrespondingRole (sourceObject).getQualifier() != null &&
                (getRange() == null || getRange().trim().equals ("")))
            {
               priority += P_TO_MANY;
            }
            else
            {
               if (!targetObject.isSet())
               {
                  priority += P_TO_ONE;
               }
               else
               {
                  priority += P_TO_MANY;
               }
            }
         }
         else
         {
            priority += P_TO_MANY;
         }
      }

      if (targetObject.isOptional() || getType() == OPTIONAL)
      {
         priority += P_OPTIONAL;
      }

      if (targetObject.isSet() || sourceObject.isSet())
      {
         priority += P_SET;
      }

      return priority;
   } // getPriority


   /**
    * Get the unboundObject attribute of the UMLLink object
    *
    * @param boundObjects  No description provided
    * @return              The unboundObject value
    */
   public UMLObject getUnboundObject (HashMap boundObjects)
   {
      UMLObject source = (UMLObject) boundObjects.get (getSource().getID());
      UMLObject target = (UMLObject) boundObjects.get (getTarget().getID());

      if (source == null && target == null)
      {
         throw new RuntimeExceptionWithContext ("Neither source nor target object is bound", this);
      }

      if (source != null)
      {
         return getTarget();
      }
      else
      {
         if (target != null)
         {
            return getSource();
         }
         else
         {
            throw new RuntimeExceptionWithContext ("This link is a checklink", this);
         }
      }
   } // getUnboundObject


   /**
    * Wrapper method for getName.
    *
    * @return   The text value
    * @see      #getName
    */
   public String getText()
   {
      return getName();
   } // getText


   /**
    * Get the absoluteModifier attribute of the UMLLink object
    *
    * @return   The absoluteModifier value
    */
   public int getAbsoluteModifier()
   {
      int result = getModifier() | getSource().getModifier() | getTarget().getModifier();

      if ( (result != CREATE) &&  (result != DELETE) &&  (result != NONE))
      {
         throw new RuntimeExceptionWithContext ("Error: Link " + this + " connects to a create and to a delete object", this);
      }

      return result;
   } // getAbsoluteModifier


   /**
    * Wrapper method for setSource.
    *
    * @param incr  The new sourceConnector value
    * @return      No description provided
    * @see         #setSource
    */
   public boolean setSourceConnector (FDiagramItem incr)
   {
      if (! (incr instanceof UMLObject))
      {
         throw new IllegalArgumentException ("Argument is no UMLObject");
      }
      return setSource ((UMLObject) incr);
   } // setSourceConnector


   /**
    * Wrapper method for getSource.
    *
    * @return   The sourceConnector value
    * @see      #getSource
    */
   public UMLDiagramItem getSourceConnector()
   {
      return getSource();
   } // getSourceConnector


   /**
    * Wrapper method for setTarget.
    *
    * @param incr  The new targetConnector value
    * @return      No description provided
    * @see         #setTarget
    */
   public boolean setTargetConnector (FDiagramItem incr)
   {
      if (! (incr instanceof UMLObject))
      {
         throw new IllegalArgumentException ("Argument is no UMLObject");
      }
      return setTarget ((UMLObject) incr);
   } // setTargetConnector


   /**
    * Wrapper method for getTarget.
    *
    * @return   The targetConnector value
    * @see      #getTarget
    */
   public UMLDiagramItem getTargetConnector()
   {
      return getTarget();
   } // getTargetConnector


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param role  No description provided
    * @return      No description provided
    */
   public boolean toOneAccess (UMLRole role)
   {
      return  ( (role.getCard() != null) ?  (role.getCard().getUpperBound() == 1 && role.getPartnerRole().getQualifier() == null) : true);
   } // toOneAccess


   /**
    * Boolean method to check whether this link connects the two objects given by their names.
    *
    * @param firstObject   No description provided
    * @param secondObject  No description provided
    * @return              The linkBetween value
    */
   public boolean isLinkBetween (String firstObject, String secondObject)
   {
      return  ( (getSource().getObjectName().equals (firstObject) && getTarget().getObjectName().equals (secondObject))
         ||  (getSource().getObjectName().equals (secondObject) && getTarget().getObjectName().equals (firstObject)));
   } // isLinkBetween


   /**
    * Get the correspondingRole attribute of the UMLLink object
    *
    * @param object  No description provided
    * @return        The correspondingRole value
    */
   public UMLRole getCorrespondingRole (UMLObject object)
   {
      if (object == null || getSource() == null || getTarget() == null || getInstanceOf() == null)
      {
         return null;
      }
      UMLRole correspondingRole = null;
      UMLAssoc myAssoc = getInstanceOf();

      if (object == getSource())
      {
         if (myAssoc.getDirection() == UMLAssoc.LEFTRIGHT)
         {
            correspondingRole = myAssoc.getLeftRole();
         }
         else
         {
            correspondingRole = myAssoc.getRightRole();
         }
      }
      else if (object == getTarget())
      {
         if (myAssoc != null)
         {
            if (myAssoc.getDirection() == UMLAssoc.LEFTRIGHT)
            {
               correspondingRole = myAssoc.getRightRole();
            }
            else
            {
               correspondingRole = myAssoc.getLeftRole();
            }
         }
      }
      else
      {
         throw new IllegalArgumentException ("UMLLink: UMLObject '" + object.getObjectName()
            + "' does not belong to UMLLink '"
            + getName() + "'");
      }

      return correspondingRole;
   } // getCorrespondingRole


   /**
    * Get the sourceRole attribute of the UMLLink object
    *
    * @return   The sourceRole value
    */
   public UMLRole getSourceRole()
   {
      if (getSource() == null)
      {
         return null;
      }

      UMLRole role = null;
      try
      {
         role = getCorrespondingRole (getSource());
      }
      catch (UnsupportedOperationException e)
      {
      }

      return role;
   }


   /**
    * Get the targetRole attribute of the UMLLink object
    *
    * @return   The targetRole value
    */
   public UMLRole getTargetRole()
   {
      if (getTarget() == null)
      {
         return null;
      }

      UMLRole role = null;
      try
      {
         role = getCorrespondingRole (getTarget());
      }
      catch (UnsupportedOperationException e)
      {
      }

      return role;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param object  No description provided
    * @return        No description provided
    */
   public boolean accessable (UMLObject object)
   {
      return ! ( (isReference()) &&  (getTarget() != object));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param link  No description provided
    * @return      No description provided
    */
   public static boolean checkSwap (UMLLink link)
   {
      boolean result = false;

      try
      {
         UMLAssoc assoc = link.getInstanceOf();
         if (assoc != null)
         {
            UMLClass assocTgtClass = assoc.getLeftRole().getTarget();
            UMLClass assocSrcClass = assoc.getRightRole().getTarget();

            UMLClass linkTgtClass = link.getTarget().getInstanceOf();
            UMLClass linkSrcClass = link.getSource().getInstanceOf();

            // standard
            result = linkSrcClass.isChildOf (assocSrcClass);
            result &= linkTgtClass.isChildOf (assocTgtClass);

            // 69
            result &= linkSrcClass.isChildOf (assocTgtClass);
            result &= linkTgtClass.isChildOf (assocSrcClass);
         }
      }
      catch (Exception e)
      {
         if (link != null)
         {
            log.debug ("exception occured during checking swap. Link deleted.");
            e.printStackTrace();
            link.removeYou();
         } // end of if ()

         result = false;
      }

      return result;
   } // checkSwap


   /**
    * Isolates the object so the garbage collector can remove it.
    */
   public void removeYou()
   {
      setInstanceOf (null);
      setSource (null);
      setTarget (null);

      if (getRevSourceLink() != null)
      {
         getRevSourceLink().removeYou();
      }
      if (getRevTargetLink() != null)
      {
         getRevTargetLink().removeYou();
      }

      setRevSourceLink (null);
      setRevTargetLink (null);

      super.removeYou();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param sourceObject  No description provided
    * @param targetObject  No description provided
    * @param priority      No description provided
    * @return              No description provided
    */
   public boolean checkNegativeError (UMLObject sourceObject, UMLObject targetObject, int priority)
   {
      boolean sourceBound = sourceObject.isBound();
      boolean targetBound = targetObject.isBound();
      boolean targetOptional = targetObject.isOptional();
      boolean sourceNegative = sourceObject.isNegative();
      boolean targetNegative = targetObject.isNegative();
      boolean linkNegative = getType() == NEGATIVE;
      boolean sourceHasUMLAttrExprPair = sourceObject.iteratorOfAttrs().hasNext();
      boolean targetHasUMLAttrExprPair = targetObject.iteratorOfAttrs().hasNext();

      boolean result = false;

      switch (priority)
      {
         case P_CHECK:
         case P_CHECK_TO_MANY:
         {
            if ( (sourceBound && sourceNegative && !sourceHasUMLAttrExprPair) ||
                (targetBound && targetNegative && !targetHasUMLAttrExprPair) ||
                (linkNegative && targetNegative && !targetBound) ||
                (targetOptional && targetNegative && !targetHasUMLAttrExprPair))
            {
               result = true;
            }
         }
            break;
         case P_TO_ONE:
         case P_TO_MANY:
         {
            if ( (sourceBound && sourceNegative && !sourceHasUMLAttrExprPair) ||
                (targetBound && targetNegative && !targetHasUMLAttrExprPair) ||
                (linkNegative))
            {
               result = true;
            }

         }
            break;
      }

      return result;
   } // checkNegativeError


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      StringBuffer result = new StringBuffer();

      result.append ("UMLLink[name=");
      result.append (getName());
      result.append (",type=");
      result.append (getType());
      result.append (",modifier=");
      result.append (getModifier());
      result.append (",sourceObject=");
      result.append (getSource());
      result.append (",targetObject=");
      result.append (getTarget());
      result.append (",instanceof=");
      result.append (getInstanceOf());
      result.append ("]");

      return result.toString();
   }


   /**
    * Hangs the current ASGElement into the ASG-tree. Needed for cut'n'paste.
    *
    * @param parent
    */
   public void setCutCopyPasteParent (FElement parent)
   {
      if (parent instanceof UMLStoryActivity)
      {
         UMLStoryActivity activity = (UMLStoryActivity) parent;
         this.removeAllFromDiagrams();
         activity.getStoryPattern().addToElements (this);
      }
      else
      {
         super.setCutCopyPasteParent (parent);
      }
   }


   /**
    * Get the persistencyChange attribute of the UMLLink object
    *
    * @param e  No description provided
    * @return   The persistencyChange value
    */
   protected boolean isPersistencyChange (PropertyChangeEvent e)
   {
      return super.isPersistencyChange (e) &&
         ! ("sourceRole".equals (e.getPropertyName()) || "targetRole".equals (e.getPropertyName()));
   }

}

/*
 * $Log: UMLLink.java,v $
 * Revision 1.176.2.2  2006/04/27 10:29:20  creckord
 * Fixed some codegen exception issues
 *
 */
