/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml;

import java.util.Enumeration;
import java.util.Iterator;

import de.uni_paderborn.fujaba.metamodel.FElement;
import de.upb.tools.fca.*;
import de.upb.tools.sdm.Path;


/**
 * <h2>Associations</h2>
 *
 * <pre>
 *            0..1     returnParam      0..1
 * UMLObject -------------------------------- UMLStopActivity
 *            returnParam     revReturnParam
 *
 *            0..n     instanceOf     0..1
 * UMLObject ------------------------------ UMLClass
 *            instances         instanceOf
 *
 *            0..1     source     0..n
 * UMLObject -------------------------- UMLLink
 *            source         revSource
 *
 *            0..1     target     0..n
 * UMLObject -------------------------- UMLLink
 *            target         revTarget
 *
 *            0..1     toSetObject     0..n
 * UMLObject ------------------------------- UMLLinkSet
 *            toSetObject    revToSetObject
 *
 *            0..1     attrs     0..n
 * UMLObject ------------------------- UMLAttrExprPair
 *            revAttrs          attrs
 *
 *            0..1    callTarget     0..n
 * UMLObject ----------------------------- UMLCollabStat
 *            callTarget      collabStats
 *
 *            0..n    boundTo    0..1
 * UMLObject ------------------------- UMLObject
 *            bindings        boundTo
 *
 *            0..1       containerObject       0..n
 * UMLObject --------------------------------------- UMLMultiLink
 *            containerObject    revContainerObject
 *
 * </pre> Valid combinations of modifier and type (X=valid/O=invalid): <pre>
 * modifier\type  |  normal  |  optional  |  negative  |  set
 * -------------------------------------------------------------
 * none           |    X     |      X     |      X     |   X
 * -------------------------------------------------------------
 * create         |    X     |      O     |      O     |   O
 * -------------------------------------------------------------
 * delete         |    X     |      X     |      O     |   X
 * </pre>
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.150.2.1 $
 */
public class UMLObject extends UMLDiagramItem
{
   /**
    * Type
    *
    * @see   #setType
    * @see   #getType
    */
   public final static transient int NORM = 0;

   /**
    * Type
    *
    * @see   #setType
    * @see   #getType
    */
   public final static transient int OPTIONAL = 1;

   /**
    * Type
    *
    * @see   #setType
    * @see   #getType
    */
   public final static transient int NEGATIVE = 2;

   /**
    * Type
    *
    * @see   #setType
    * @see   #getType
    */
   public final static transient int SET = 3;

   /**
    * Modifier
    *
    * @see   #setModifier
    * @see   #getModifier
    */
   public final static transient int NONE = UMLStoryPattern.NONE;

   /**
    * Modifier
    *
    * @see   #setModifier
    * @see   #getModifier
    */
   public final static transient int CREATE = UMLStoryPattern.CREATE;

   /**
    * Modifier
    *
    * @see   #setModifier
    * @see   #getModifier
    */
   public final static transient int DELETE = UMLStoryPattern.DELETE;


   /**
    * Constructor for class UMLObject
    */
   public UMLObject()
   {
      super();
   } // UMLObject


   /**
    * Constructor for class UMLObject
    *
    * @param objectName  No description provided
    * @param objectType  No description provided
    * @param type        No description provided
    * @param bound       No description provided
    * @param modifier    No description provided
    * @param instanceOf  No description provided
    */
   public UMLObject (String objectName, String objectType, int type,
                     boolean bound, int modifier, UMLClass instanceOf)
   {
      super();
      setObjectName (objectName);
      if (objectType != null)
      {
         if (instanceOf == null)
         {
            instanceOf = UMLProject.get().findClass (null, objectType, false);
            if (instanceOf == null)
            {
               setObjectType (objectType);
            }
         }
         else if (!objectType.equals (instanceOf.getName()))
         {
            setObjectType (objectType);
         }
      }
      setType (type);
      setBound (bound);
      setModifier (modifier);
      setInstanceOf (instanceOf);
   } // UMLObject


   /**
    * Constructor for class UMLObject
    *
    * @param objectName  No description provided
    * @param objectType  No description provided
    */
   public UMLObject (String objectName, String objectType)
   {
      super();
      setObjectName (objectName);
      UMLClass instanceOf = UMLProject.get().findClass (null, objectType, false);
      if (instanceOf != null)
      {
         setInstanceOf (instanceOf);
      }
      else
      {
         setObjectType (objectType);
      }
   }


   /**
    * Constructor for class UMLObject
    *
    * @param objectName  No description provided
    * @param objectType  No description provided
    * @param parent      No description provided
    */
   public UMLObject (String objectName, String objectType, UMLDiagram parent)
   {
      this (objectName, objectType);
      this.addToDiagrams (parent);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected String objectName = "";


   /**
    * Get the objectName attribute of the UMLObject object
    *
    * @return   The objectName value
    */
   public String getObjectName()
   {
      return objectName;
   } // getObjectName


   /**
    * Set the value of objectName.
    *
    * @param objectName  Value to assign to objectName.
    */
   public void setObjectName (String objectName)
   {
      if ( (this.objectName == null && objectName != null) ||
          (this.objectName != null && !this.objectName.equals (objectName)))
      {
         String oldValue = this.objectName;

         this.objectName = objectName;

         firePropertyChange ("objectName", oldValue, this.objectName);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String objectType;


   /**
    * Sets the objectType attribute of the UMLObject object
    *
    * @param newVal  The new objectType value
    * @deprecated    use only {@link #setInstanceOf (de.uni_paderborn.fujaba.uml.UMLClass)}.
    *    Using setObjectType for Set Objects and "this" is not needed anymore with
    *    the new StoryPattern codegen.
    */
   public void setObjectType (String newVal)
   {
      if (this.objectType != newVal &&  (newVal == null || this.instanceOf == null ||
         !newVal.equals (this.instanceOf.getName())))
      {
         String oldVal = this.objectType;
         this.objectType = newVal;
         firePropertyChange ("objectType", oldVal, newVal);
      }
   }


   /**
    * Get the objectType attribute of the UMLObject object
    *
    * @return   The objectType value
    */
   public String getObjectType()
   {
      if (this.objectType == null && this.getInstanceOf() != null)
      {
         return this.getInstanceOf().getName();
      }
      return this.objectType;
   } // getObjectType


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String currentIcon;


   /**
    * Get the currentIcon attribute of the UMLClass object
    *
    * @return   The currentIcon value
    */
   public String getCurrentIcon()
   {
      if ( (this.currentIcon == null) &&  (this.getInstanceOf() != null))
      {
         return this.getInstanceOf().getDefaultIcon();
      }
      else
      {
         return this.currentIcon;
      }
   }


   /**
    * Sets the currentIcon attribute of the UMLClass object
    *
    * @param newVal  The new currentIcon value
    */
   public void setCurrentIcon (String newVal)
   {
      if (this.currentIcon != newVal)
      {
         String oldVal = this.currentIcon;
         this.currentIcon = newVal;
         firePropertyChange ("currentIcon", oldVal, newVal);
      }
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      StringBuffer result = new StringBuffer();

      result.append ("UMLObject[");
      result.append (getObjectName());
      result.append (": ");
      result.append (getObjectType());
      result.append (",instanceOf=");
      result.append (getInstanceOf());
      result.append ("]");

      return result.toString();
   }


   /**
    * This method returns the full qualified ObjectName. If the UMLObject is bound this method
    * returns only the objectName, otherwise it will return the objectName + : + objectType.
    *
    * @return   The fullQualifiedName value
    */
   public String getFullQualifiedName()
   {
      String newText = getObjectName();

      if (!isBound())
      {
         newText += ": " + getObjectType();
      }

      return newText;
   } // getFullQualifiedName


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int type = NORM;


   /**
    * Get the type attribute of the UMLObject object
    *
    * @return   The type value
    */
   public int getType()
   {
      return type;
   } // getType


   /**
    * Sets the type attribute of the UMLObject object
    *
    * @param type  The new type value
    */
   public void setType (int type)
   {
      if (NORM <= type && type <= SET && this.type != type)
      {
         int oldValue = this.type;
         this.type = type;
         firePropertyChange ("type", oldValue, type);
      }
   } // setType


   /**
    * Get the optional attribute of the UMLObject object
    *
    * @return   The optional value
    */
   public boolean isOptional()
   {
      return  (getType() == OPTIONAL || getType() == SET);
   } // isOptional


   /**
    * Get the optionalAtPost attribute of the UMLObject object
    *
    * @return   The optionalAtPost value
    */
   public boolean isOptionalAtPost()
   {
      return  (isOptional() && ! (getModifier() == CREATE));
   } // isOptionalAtPost


   /**
    * Get the set attribute of the UMLObject object
    *
    * @return   The set value
    */
   public boolean isSet()
   {
      return  (getType() == SET);
   } // isSet


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean bound = false;


   /**
    * Get the bound attribute of the UMLObject object
    *
    * @return   The bound value
    */
   public boolean isBound()
   {
      return bound;
   } // isBound


   /**
    * Sets the bound attribute of the UMLObject object
    *
    * @param bound  The new bound value
    */
   public void setBound (boolean bound)
   {
      if (this.bound != bound)
      {
         boolean oldValue = this.bound;
         this.bound = bound;
         firePropertyChange ("bound", oldValue, bound);
      }
   } // setBound


   /**
    * Get the negative attribute of the UMLObject object
    *
    * @return   The negative value
    */
   public boolean isNegative()
   {
      return  (getType() == UMLObject.NEGATIVE);
   } // isNegative


   /**
    * Sets the negative attribute of the UMLObject object
    *
    * @param negative  The new negative value
    * @deprecated      Will be removed soon, use
    * @see             #setType instead
    */
   public void setNegative (boolean negative)
   {
      boolean oldValue = isNegative();
      if (oldValue != negative)
      {
         if (negative)
         {
            setType (UMLObject.NEGATIVE);
         }
         else
         {
            setType (UMLObject.NORM);
         }
         firePropertyChange ("negative", oldValue, negative);
      }
   } // setNegative


   /**
    * The boolean value is used only for bound objects for type casting before executing the
    * story pattern.
    */
   private boolean checkTypeCast = false;


   /**
    * Get the checkTypeCast attribute of the UMLObject object
    *
    * @return   The checkTypeCast value
    */
   public boolean isCheckTypeCast()
   {
      return checkTypeCast;
   } // isCheckTypeCast


   /**
    * Sets the checkTypeCast attribute of the UMLObject object
    *
    * @param newValue  The new checkTypeCast value
    */
   public void setCheckTypeCast (boolean newValue)
   {
      if (this.checkTypeCast != newValue)
      {
         boolean oldValue = this.checkTypeCast;
         this.checkTypeCast = newValue;
         firePropertyChange ("checkTypeCast", oldValue, newValue);
      }
   } // setCheckTypeCast


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String typeCastSource;


   /**
    * Sets the typeCastSource attribute of the UMLObject object
    *
    * @param newValue  The new typeCastSource value
    */
   public void setTypeCastSource (String newValue)
   {
      if (this.typeCastSource != newValue)
      {
         String oldValue = this.typeCastSource;
         this.typeCastSource = newValue;
         firePropertyChange ("typeCastSource", oldValue, newValue);
      }
   }


   /**
    * Get the typeCastSource attribute of the UMLObject object
    *
    * @return   The typeCastSource value
    */
   public String getTypeCastSource()
   {
      return this.typeCastSource;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int modifier = NONE;


   /**
    * @return   one of NONE, CREATE, DELETE
    */
   public int getModifier()
   {
      return modifier;
   } // getModifier


   /**
    * @param modifier  one of NONE, CREATE, DELETE
    */
   public void setModifier (int modifier)
   {
      if (this.modifier != modifier)
      {
         int oldValue = this.modifier;
         this.modifier = modifier;
         firePropertyChange ("modifier", oldValue, modifier);
      }
   } // setModifier


   /**
    * <pre>
    *            0..1     returnParam      0..1
    * UMLObject -------------------------------- UMLStopActivity
    *            returnParam     revReturnParam
    * </pre>
    */
   private UMLStopActivity revReturnParam;


   /**
    * Get the revReturnParam attribute of the UMLObject object
    *
    * @return   The revReturnParam value
    */
   public UMLStopActivity getRevReturnParam()
   {
      return revReturnParam;
   } // getRevReturnParam


   /**
    * Sets the revReturnParam attribute of the UMLObject object
    *
    * @param revReturnParam  The new revReturnParam value
    */
   public void setRevReturnParam (UMLStopActivity revReturnParam)
   {
      if (this.revReturnParam != revReturnParam)
      {
         // new partner
         UMLStopActivity oldRevReturnParam = this.revReturnParam;
         if (this.revReturnParam != null)
         {
            // inform old partner
            this.revReturnParam = null;
            oldRevReturnParam.setReturnParam (null);
         }
         this.revReturnParam = revReturnParam;
         if (revReturnParam != null)
         {
            // inform new partner
            revReturnParam.setReturnParam (this);
         }
         firePropertyChange ("revReturnParam", oldRevReturnParam, revReturnParam);
      }
   } // setRevReturnParam


   /**
    * <pre>
    *            0..n     instanceOf     0..1
    * UMLObject ------------------------------ UMLClass
    *            instances         instanceOf
    * </pre>
    */
   private UMLClass instanceOf;


   /**
    * Get the instanceOf attribute of the UMLObject object
    *
    * @return   The instanceOf value
    */
   public UMLClass getInstanceOf()
   {
      return instanceOf;
   } // getInstanceOf


   /**
    * Sets the instanceOf attribute of the UMLObject object
    *
    * @param instanceOf  The new instanceOf value
    */
   public void setInstanceOf (UMLClass instanceOf)
   {
      if (this.instanceOf != instanceOf)
      {
         // new partner
         UMLClass oldInstanceOf = this.instanceOf;
         if (this.instanceOf != null)
         {
            // inform old partner
            this.instanceOf = null;
            oldInstanceOf.removeFromInstances (this);
         }
         this.instanceOf = instanceOf;
         if (instanceOf != null)
         {
            // inform new partner
            instanceOf.addToInstances (this);
         }
         firePropertyChange ("instanceOf", oldInstanceOf, instanceOf);
         if (instanceOf != null)
         {
            setObjectType (null);
         }
      }
   } // setInstanceOf


   /**
    * <pre>
    *            0..1     source     0..n
    * UMLObject -------------------------- UMLLink
    *            source         revSource
    * </pre>
    */
   private FPropTreeSet revSource;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInRevSource (UMLLink value)
   {
      return  ( (this.revSource != null) &&
          (value != null) &&
         this.revSource.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfRevSource()
   {
      return  ( (this.revSource == null)
         ? FEmptyIterator.get()
         : this.revSource.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfRevSource()
   {
      return  ( (this.revSource == null)
         ? 0
         : this.revSource.size());
   }


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToRevSource (UMLLink value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.revSource == null)
         {
            this.revSource = new FPropTreeSet (this, "revSource");
         }
         changed = this.revSource.add (value);
         if (changed)
         {
            value.setSource (this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromRevSource (UMLLink value)
   {
      //log.info("REM " + this + " : " + value);
      boolean changed = false;
      if ( (this.revSource != null) &&  (value != null))
      {
         changed = this.revSource.remove (value);
         if (changed)
         {
            value.setSource (null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromRevSource()
   {
      UMLLink tmpValue;
      Iterator iter = this.iteratorOfRevSource();
      while (iter.hasNext())
      {
         tmpValue = (UMLLink) iter.next();
         tmpValue.removeYou();
      }
   }


   /**
    * <pre>
    *            0..1     target     0..n
    * UMLObject -------------------------- UMLLink
    *            target         revTarget
    * </pre>
    */
   private FPropTreeSet revTarget;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInRevTarget (UMLLink value)
   {
      return  ( (this.revTarget != null) &&
          (value != null) &&
         this.revTarget.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfRevTarget()
   {
      return  ( (this.revTarget == null)
         ? FEmptyIterator.get()
         : this.revTarget.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfRevTarget()
   {
      return  ( (this.revTarget == null)
         ? 0
         : this.revTarget.size());
   }


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToRevTarget (UMLLink value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.revTarget == null)
         {
            this.revTarget = new FPropTreeSet (this, "revTarget");
         }
         changed = this.revTarget.add (value);
         if (changed)
         {
            value.setTarget (this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromRevTarget (UMLLink value)
   {
      boolean changed = false;
      if ( (this.revTarget != null) &&  (value != null))
      {
         changed = this.revTarget.remove (value);
         if (changed)
         {
            value.setTarget (null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromRevTarget()
   {
      UMLLink tmpValue;
      Iterator iter = this.iteratorOfRevTarget();
      while (iter.hasNext())
      {
         tmpValue = (UMLLink) iter.next();
         tmpValue.removeYou();
      }
   }


   /**
    * <pre>
    *            0..1     toSetObject     0..n
    * UMLObject ------------------------------- UMLLinkSet
    *            toSetObject    revToSetObject
    * </pre>
    */
   private FPropTreeSet revToSetObject;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInRevToSetObject (UMLLinkSet value)
   {
      return  ( (this.revToSetObject != null) &&
          (value != null) &&
         this.revToSetObject.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfRevToSetObject()
   {
      return  ( (this.revToSetObject == null)
         ? FEmptyIterator.get()
         : this.revToSetObject.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfRevToSetObject()
   {
      return  ( (this.revToSetObject == null)
         ? 0
         : this.revToSetObject.size());
   }


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToRevToSetObject (UMLLinkSet value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.revToSetObject == null)
         {
            this.revToSetObject = new FPropTreeSet (this, "revToSetObject");
         }
         changed = this.revToSetObject.add (value);
         if (changed)
         {
            value.setToSetObject (this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromRevToSetObject (UMLLinkSet value)
   {
      boolean changed = false;
      if ( (this.revToSetObject != null) &&  (value != null))
      {
         changed = this.revToSetObject.remove (value);
         if (changed)
         {
            value.setToSetObject (null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromRevToSetObject()
   {
      UMLLinkSet tmpValue;
      Iterator iter = this.iteratorOfRevToSetObject();
      while (iter.hasNext())
      {
         tmpValue = (UMLLinkSet) iter.next();
         tmpValue.removeYou();
      }
   }


   /**
    * <pre>
    *            0..1     attrs     0..n
    * UMLObject ------------------------- UMLAttrExprPair
    *            revAttrs          attrs
    * </pre>
    */
   private FPropTreeSet attrs;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInAttrs (UMLAttrExprPair value)
   {
      return  ( (this.attrs != null) &&
          (value != null) &&
         this.attrs.contains (value));
   } // hasInAttrs


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfAttrs()
   {
      return  ( (this.attrs == null)
         ? FEmptyIterator.get()
         : this.attrs.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfAttrs()
   {
      return  ( (this.attrs == null)
         ? 0
         : this.attrs.size());
   }


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToAttrs (UMLAttrExprPair value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.attrs == null)
         {
            this.attrs = new FPropTreeSet (this, "attrs");
         }
         changed = this.attrs.add (value);
         if (changed)
         {
            value.setRevAttrs (this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromAttrs (UMLAttrExprPair value)
   {
      boolean changed = false;
      if ( (this.attrs != null) &&  (value != null))
      {
         changed = this.attrs.remove (value);
         if (changed)
         {
            value.setRevAttrs (null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromAttrs()
   {
      UMLAttrExprPair tmpValue;
      Iterator iter = this.iteratorOfAttrs();
      while (iter.hasNext())
      {
         tmpValue = (UMLAttrExprPair) iter.next();
         tmpValue.removeYou();
      }
   }


   /**
    * <pre>
    *            0..1    callTarget     0..n
    * UMLObject ----------------------------- UMLCollabStat
    *            callTarget      collabStats
    * </pre>
    */
   private transient FPropTreeSet collabStats;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInCollabStats (UMLCollabStat value)
   {
      return  ( (this.collabStats != null) &&
          (value != null) &&
         this.collabStats.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfCollabStats()
   {
      return  ( (this.collabStats == null)
         ? FEmptyIterator.get()
         : this.collabStats.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfRightRoles()
   {
      return  ( (this.collabStats == null)
         ? 0
         : this.collabStats.size());
   }


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToCollabStats (UMLCollabStat value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.collabStats == null)
         {
            this.collabStats = new FPropTreeSet (this, "collabStats");
         }
         changed = this.collabStats.add (value);
         if (changed)
         {
            value.setCallTarget (this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromCollabStats (UMLCollabStat value)
   {
      boolean changed = false;
      if ( (this.collabStats != null) &&  (value != null))
      {
         changed = this.collabStats.remove (value);
         if (changed)
         {
            value.setCallTarget (null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromCollabStats()
   {
      UMLCollabStat tmpValue;
      Iterator iter = this.iteratorOfCollabStats();
      while (iter.hasNext())
      {
         tmpValue = (UMLCollabStat) iter.next();
         this.removeFromCollabStats (tmpValue);
      }
   }


   /**
    * Get the lastFromCollabStats attribute of the UMLObject object
    *
    * @return   The lastFromCollabStats value
    */
   public UMLCollabStat getLastFromCollabStats()
   {
      if (collabStats != null)
      {
         return (UMLCollabStat) collabStats.last();
      }
      return null;
   }


   /**
    * Get the firstFromCollabStats attribute of the UMLObject object
    *
    * @return   The firstFromCollabStats value
    */
   public UMLCollabStat getFirstFromCollabStats()
   {
      if (collabStats != null)
      {
         return (UMLCollabStat) collabStats.first();
      }
      return null;
   }


   /**
    * <pre>
    *                n     callSource      0..1
    * UMLCollabStat ---------------------------- UMLObject
    *                collabStat      callSource
    * </pre>
    */
   private transient FPropTreeSet callSourceCollabStats;


   /**
    * UMLMethod: '+ hasInCallSourceCollabStats (value: UMLCollabStat): Boolean'
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInCallSourceCollabStats (UMLCollabStat value)
   {
      return  ( (this.callSourceCollabStats != null) &&
          (value != null) &&
         this.callSourceCollabStats.contains (value));
   }


   /**
    * UMLMethod: '+ iteratorOfCallSourceCollabStats (): Iterator'
    *
    * @return   No description provided
    */
   public Iterator iteratorOfCallSourceCollabStats()
   {
      return  ( (this.callSourceCollabStats == null)
         ? FEmptyIterator.get()
         : this.callSourceCollabStats.iterator());
   }


   /**
    * UMLMethod: '+ sizeOfCallSourceCollabStats (): Integer'
    *
    * @return   No description provided
    */
   public int sizeOfCallSourceCollabStats()
   {
      return  ( (this.callSourceCollabStats == null)
         ? 0
         : this.callSourceCollabStats.size());
   }


   /**
    * UMLMethod: '+ addToCallSourceCollabStats (value: UMLCollabStat): Boolean'
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToCallSourceCollabStats (UMLCollabStat value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.callSourceCollabStats == null)
         {
            this.callSourceCollabStats = new FPropTreeSet (this, "callSourceCollabStats");

         }
         changed = this.callSourceCollabStats.add (value);
         if (changed)
         {
            value.setCallSource (this);
         }
      }
      return changed;
   }


   /**
    * UMLMethod: '+ removeFromCollabStats (value: UMLCollabStat): Boolean'
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromCallSourceCollabStats (UMLCollabStat value)
   {
      boolean changed = false;
      if ( (this.callSourceCollabStats != null) &&  (value != null))
      {
         changed = this.callSourceCollabStats.remove (value);
         if (changed)
         {
            value.setCallSource (null);
         }
      }
      return changed;
   }


   /**
    * UMLMethod: '+ removeAllFromCollabStat (): Void'
    */
   public void removeAllFromCallSourceCollabStats()
   {
      UMLCollabStat tmpValue;
      Iterator iter = this.iteratorOfCallSourceCollabStats();
      while (iter.hasNext())
      {
         tmpValue = (UMLCollabStat) iter.next();
         this.removeFromCallSourceCollabStats (tmpValue);
      }

   }


   /**
    * Get the lastCallSourceFromCollabStats attribute of the UMLObject object
    *
    * @return   The lastCallSourceFromCollabStats value
    */
   public UMLCollabStat getLastCallSourceFromCollabStats()
   {
      if (callSourceCollabStats != null)
      {
         return (UMLCollabStat) callSourceCollabStats.last();
      }
      return null;
   }


   /**
    * Get the firstFromCallSourceCollabStats attribute of the UMLObject object
    *
    * @return   The firstFromCallSourceCollabStats value
    */
   public UMLCollabStat getFirstFromCallSourceCollabStats()
   {
      if (callSourceCollabStats != null)
      {
         return (UMLCollabStat) callSourceCollabStats.first();
      }
      return null;
   }


   /**
    * <pre>
    *            0..n    boundTo    0..1
    * UMLObject ------------------------- UMLObject
    *            bindings        boundTo
    * </pre>
    */
   private UMLObject boundTo;


   /**
    * Sets the boundTo attribute of the UMLObject object
    *
    * @param value  The new boundTo value
    * @return       No description provided
    */
   public boolean setBoundTo (UMLObject value)
   {
      if (this.boundTo != value)
      {
         if (this.boundTo != null)
         {
            UMLObject oldValue = this.boundTo;
            this.boundTo = null;
            oldValue.removeFromBindings (this);
         }
         this.boundTo = value;
         if (value != null)
         {
            this.boundTo.addToBindings (this);
         }

         return true;
      }

      return false;
   }


   /**
    * Get the boundTo attribute of the UMLObject object
    *
    * @return   The boundTo value
    */
   public UMLObject getBoundTo()
   {
      return this.boundTo;
   }


   /**
    * <pre>
    *            0..n    boundTo    0..1
    * UMLObject ------------------------- UMLObject
    *            bindings        boundTo
    * </pre>
    */
   private FHashSet bindings;


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToBindings (UMLObject value)
   {
      boolean changed = false;

      if (value != null)
      {
         if (this.bindings == null)
         {
            this.bindings = new FHashSet();
         }
         changed = this.bindings.add (value);
         if (changed)
         {
            value.setBoundTo (this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInBindings (UMLObject value)
   {
      return  ( (this.bindings != null) &&
          (value != null) &&
         this.bindings.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfBindings()
   {
      return  ( (this.bindings == null)
         ? FEmptyIterator.get()
         : this.bindings.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfBindings()
   {
      return  ( (this.bindings == null)
         ? 0
         : this.bindings.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromBindings (UMLObject value)
   {
      boolean changed = false;

      if ( (this.bindings != null) &&  (value != null))
      {
         changed = this.bindings.remove (value);
         if (changed)
         {
            value.setBoundTo (null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromBindings()
   {
      UMLObject tmpValue;
      Iterator iter = this.iteratorOfBindings();

      while (iter.hasNext())
      {
         tmpValue = (UMLObject) iter.next();
         this.removeFromBindings (tmpValue);
      }
   }


   /**
    * <pre>
    *            0..1       containerObject       0..n
    * UMLObject --------------------------------------- UMLMultiLink
    *            containerObject    revContainerObject
    * </pre>
    */
   private FPropTreeSet revContainerObject = null;


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToRevContainerObject (UMLMultiLink value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.revContainerObject == null)
         {
            this.revContainerObject = new FPropTreeSet (this, "revContainerObject");
         }
         changed = this.revContainerObject.add (value);
         if (changed)
         {
            value.setContainerObject (this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromRevContainerObject (UMLMultiLink value)
   {
      boolean changed = false;
      if ( (this.revContainerObject != null) &&  (value != null))
      {
         changed = this.revContainerObject.remove (value);
         if (changed)
         {
            value.setContainerObject (null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromRevContainerObject()
   {
      UMLMultiLink tmpValue;
      Iterator iter = this.iteratorOfRevContainerObject();
      while (iter.hasNext())
      {
         tmpValue = (UMLMultiLink) iter.next();
         tmpValue.removeYou();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfRevContainerObject()
   {
      return  ( (this.revContainerObject == null)
         ? FEmptyIterator.get()
         : this.revContainerObject.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInRevContainerObject (UMLMultiLink value)
   {
      return  ( (this.revContainerObject != null) &&
          (value != null) &&
         this.revContainerObject.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfRevContainerObject()
   {
      return  ( (this.revContainerObject == null)
         ? 0
         : this.revContainerObject.size());
   }



   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Enumeration allLinksEnumeration()
   {
      return new EnumerationForAnIterator (Path.iterUnion (iteratorOfRevSource(), iteratorOfRevTarget()));
   } // allLinksEnumeration


   /**
    * Return an successor object of this object
    *
    * @return   The successorObject value
    */
   public UMLObject getSuccessorObject()
   {
      UMLObject succObj = null;
      if (getModifier() != DELETE)
      {
         succObj = new UMLObject (getObjectName(), getObjectType(), getType(), true, NONE, getInstanceOf());

         Iterator iter = iteratorOfConstraints();
         while (iter.hasNext())
         {
            UMLConstraint tmpConstraint = (UMLConstraint) iter.next();
            succObj.addToConstraints (new UMLConstraint (tmpConstraint.getText()));
         }
         iter = iteratorOfAttrs();
         while (iter.hasNext())
         {
            UMLAttrExprPair pair = (UMLAttrExprPair) iter.next();
            int qualifier = pair.getQualifier();
            if (qualifier == UMLAttrExprPair.POST)
            {
               qualifier = UMLAttrExprPair.PRE;
            }
            new UMLAttrExprPair (pair.getName(), qualifier, pair.getOperation(), pair.getExpression(),
               pair.getInstanceOf(), succObj);
         }
      }

      return succObj;
   } // getSuccessorObject


   /**
    * Get the text attribute of the UMLObject object
    *
    * @return   The text value
    */
   public String getText()
   {
      return getObjectName();
   } // getText


   /**
    * Isolates the object so the garbage collector can remove it.
    */
   public void removeYou()
   {
      removeAllFromAttrs();
      removeAllFromRevSource();
      removeAllFromRevTarget();
      removeAllFromRevToSetObject();
      removeAllFromCollabStats();
      removeAllFromBindings();
      removeAllFromRevContainerObject();

      setInstanceOf (null);
      setRevReturnParam (null);
      setBoundTo (null);

      super.removeYou();
   } // removeYou


   /**
    * Hangs the current ASGElement into the ASG-tree. Needed for cut'n'paste.
    *
    * @param parent
    */
   public void setCutCopyPasteParent (FElement parent)
   {
      if (parent instanceof UMLStoryActivity)
      {
         UMLStoryActivity activity = (UMLStoryActivity) parent;
         this.removeAllFromDiagrams();
         activity.getStoryPattern().addToElements (this);
      }
      else
      {
         super.setCutCopyPasteParent (parent);
      }
   }
}

/*
 * $Log: UMLObject.java,v $
 * Revision 1.150.2.1  2005/08/23 08:24:43  lowende
 * Removed compile warnings.
 *
 */
